/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.logging.internal;

import java.time.Clock;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.ArrayUtils;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.internal.CappedLogger;
import org.neo4j.time.Clocks;
import org.neo4j.time.FakeClock;

@RunWith(value=Parameterized.class)
public class CappedLoggerTest {
    private final String logName;
    private final LogMethod logMethod;
    private AssertableLogProvider logProvider;
    private CappedLogger logger;

    @Parameterized.Parameters(name="{0}")
    public static Iterable<Object[]> parameters() {
        LogMethod debug = new LogMethod(){

            @Override
            public void log(CappedLogger logger, String msg) {
                logger.debug(msg);
            }

            @Override
            public void log(CappedLogger logger, String msg, Throwable cause) {
                logger.debug(msg, cause);
            }
        };
        LogMethod info = new LogMethod(){

            @Override
            public void log(CappedLogger logger, String msg) {
                logger.debug(msg);
            }

            @Override
            public void log(CappedLogger logger, String msg, Throwable cause) {
                logger.info(msg, cause);
            }
        };
        LogMethod warn = new LogMethod(){

            @Override
            public void log(CappedLogger logger, String msg) {
                logger.debug(msg);
            }

            @Override
            public void log(CappedLogger logger, String msg, Throwable cause) {
                logger.warn(msg, cause);
            }
        };
        LogMethod error = new LogMethod(){

            @Override
            public void log(CappedLogger logger, String msg) {
                logger.debug(msg);
            }

            @Override
            public void log(CappedLogger logger, String msg, Throwable cause) {
                logger.error(msg, cause);
            }
        };
        return Arrays.asList({"debug", debug}, {"info", info}, {"warn", warn}, {"error", error});
    }

    public CappedLoggerTest(String logName, LogMethod logMethod) {
        this.logName = logName;
        this.logMethod = logMethod;
    }

    public String[] logLines(int lineCount) {
        return this.logLines(lineCount, 0);
    }

    public String[] logLines(int lineCount, int startAt) {
        String[] lines = new String[lineCount];
        for (int i = 0; i < lineCount; ++i) {
            String msg;
            lines[i] = msg = String.format("### %04d ###", startAt + i);
            this.logMethod.log(this.logger, msg);
        }
        return lines;
    }

    public void assertLoggedLines(String[] lines, int count) {
        this.assertLoggedLines(lines, count, 0);
    }

    public void assertLoggedLines(String[] lines, int count, int skip) {
        int i;
        Matcher[] matchers = new Matcher[count];
        for (i = 0; i < skip; ++i) {
            matchers[i] = Matchers.any(String.class);
        }
        while (i < count) {
            String line = lines[i];
            matchers[i] = Matchers.containsString((String)line);
            ++i;
        }
        this.logProvider.rawMessageMatcher().assertContains(skip, matchers);
    }

    @Before
    public void setUp() {
        this.logProvider = new AssertableLogProvider();
        this.logger = new CappedLogger(this.logProvider.getLog(CappedLogger.class));
    }

    @Test
    public void mustLogWithoutLimitConfiguration() {
        int lineCount = 1000;
        String[] lines = this.logLines(lineCount);
        this.assertLoggedLines(lines, lineCount);
    }

    @Test
    public void mustLogExceptions() {
        this.logMethod.log(this.logger, "MESSAGE", new ArithmeticException("EXCEPTION"));
        AssertableLogProvider.MessageMatcher matcher = this.logProvider.internalToStringMessageMatcher();
        matcher.assertContains("MESSAGE");
        matcher.assertContains("ArithmeticException");
        matcher.assertContains("EXCEPTION");
    }

    @Test(expected=IllegalArgumentException.class)
    public void mustThrowOnSettingZeroCountLimit() {
        this.logger.setCountLimit(0);
    }

    @Test(expected=IllegalArgumentException.class)
    public void mustThrowOnSettingNegativeCountLimit() {
        this.logger.setCountLimit(-1);
    }

    @Test(expected=IllegalArgumentException.class)
    public void mustThrowOnZeroTimeLimit() {
        this.logger.setTimeLimit(0L, TimeUnit.MILLISECONDS, Clocks.systemClock());
    }

    @Test(expected=IllegalArgumentException.class)
    public void mustThrowOnNegativeTimeLimit() {
        this.logger.setTimeLimit(-1L, TimeUnit.MILLISECONDS, Clocks.systemClock());
    }

    @Test
    public void mustAllowConfigurationChaining() {
        this.logger.setCountLimit(1).setTimeLimit(10L, TimeUnit.MILLISECONDS, Clocks.systemClock()).unsetCountLimit().unsetTimeLimit().setCountLimit(1);
    }

    @Test
    public void mustLimitByConfiguredCount() {
        int limit = 10;
        this.logger.setCountLimit(limit);
        String[] lines = this.logLines(limit + 1);
        this.assertLoggedLines(lines, limit);
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)lines[limit])));
    }

    @Test
    public void mustLogAfterResetWithCountLimit() {
        int limit = 10;
        this.logger.setCountLimit(limit);
        Object[] lines = this.logLines(limit + 1);
        this.logger.reset();
        Object[] moreLines = this.logLines(1, limit + 1);
        this.assertLoggedLines((String[])ArrayUtils.addAll((Object[])((String[])ArrayUtils.subarray((Object[])lines, (int)0, (int)limit)), (Object[])moreLines), 1 + limit);
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)lines[limit])));
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)moreLines[0]));
    }

    @Test
    public void unsettingCountLimitMustLetMessagesThrough() {
        int limit = 10;
        this.logger.setCountLimit(limit);
        Object[] lines = this.logLines(limit + 1);
        this.logger.unsetCountLimit();
        int moreLineCount = 1000;
        Object[] moreLines = this.logLines(moreLineCount, limit + 1);
        this.assertLoggedLines((String[])ArrayUtils.addAll((Object[])((String[])ArrayUtils.subarray((Object[])lines, (int)0, (int)limit)), (Object[])moreLines), moreLineCount + limit);
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)lines[limit])));
        this.assertLoggedLines((String[])moreLines, moreLineCount, limit);
    }

    @Test
    public void mustNotLogMessagesWithinConfiguredTimeLimit() {
        FakeClock clock = this.getDefaultFakeClock();
        this.logger.setTimeLimit(1L, TimeUnit.MILLISECONDS, (Clock)clock);
        this.logMethod.log(this.logger, "### AAA ###");
        this.logMethod.log(this.logger, "### BBB ###");
        clock.forward(1L, TimeUnit.MILLISECONDS);
        this.logMethod.log(this.logger, "### CCC ###");
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### AAA ###"));
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)"### BBB ###")));
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### CCC ###"));
    }

    @Test
    public void unsettingTimeLimitMustLetMessagesThrough() {
        FakeClock clock = this.getDefaultFakeClock();
        this.logger.setTimeLimit(1L, TimeUnit.MILLISECONDS, (Clock)clock);
        this.logMethod.log(this.logger, "### AAA ###");
        this.logMethod.log(this.logger, "### BBB ###");
        clock.forward(1L, TimeUnit.MILLISECONDS);
        this.logMethod.log(this.logger, "### CCC ###");
        this.logMethod.log(this.logger, "### DDD ###");
        this.logger.unsetTimeLimit();
        this.logMethod.log(this.logger, "### EEE ###");
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### AAA ###"));
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)"### BBB ###")));
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### CCC ###"));
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)"### DDD ###")));
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### EEE ###"));
    }

    @Test
    public void mustLogAfterResetWithTimeLimit() {
        FakeClock clock = this.getDefaultFakeClock();
        this.logger.setTimeLimit(1L, TimeUnit.MILLISECONDS, (Clock)clock);
        this.logMethod.log(this.logger, "### AAA ###");
        this.logMethod.log(this.logger, "### BBB ###");
        this.logger.reset();
        this.logMethod.log(this.logger, "### CCC ###");
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### AAA ###"));
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)"### BBB ###")));
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### CCC ###"));
    }

    @Test
    public void mustOnlyLogMessagesThatPassBothLimits() {
        FakeClock clock = this.getDefaultFakeClock();
        this.logger.setCountLimit(2);
        this.logger.setTimeLimit(1L, TimeUnit.MILLISECONDS, (Clock)clock);
        this.logMethod.log(this.logger, "### AAA ###");
        this.logMethod.log(this.logger, "### BBB ###");
        clock.forward(1L, TimeUnit.MILLISECONDS);
        this.logMethod.log(this.logger, "### CCC ###");
        this.logger.reset();
        this.logMethod.log(this.logger, "### DDD ###");
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### AAA ###"));
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)"### BBB ###")));
        this.logProvider.assertNone(this.currentLog(AssertableLogProvider.inLog(CappedLogger.class), (Matcher<String>)Matchers.containsString((String)"### CCC ###")));
        this.logProvider.rawMessageMatcher().assertContains(Matchers.containsString((String)"### DDD ###"));
    }

    private AssertableLogProvider.LogMatcher currentLog(AssertableLogProvider.LogMatcherBuilder logMatcherBuilder, Matcher<String> stringMatcher) {
        switch (this.logName) {
            case "debug": {
                return logMatcherBuilder.debug(stringMatcher);
            }
            case "info": {
                return logMatcherBuilder.info(stringMatcher);
            }
            case "warn": {
                return logMatcherBuilder.warn(stringMatcher);
            }
            case "error": {
                return logMatcherBuilder.error(stringMatcher);
            }
        }
        throw new RuntimeException("Unknown log name");
    }

    private FakeClock getDefaultFakeClock() {
        return Clocks.fakeClock((long)1000L, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    public static interface LogMethod {
        public void log(CappedLogger var1, String var2);

        public void log(CappedLogger var1, String var2, Throwable var3);
    }
}

