/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.pagecache.impl;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.OpenOption;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.DisabledOnOs;
import org.junit.jupiter.api.condition.OS;
import org.neo4j.adversaries.Adversary;
import org.neo4j.adversaries.RandomAdversary;
import org.neo4j.adversaries.fs.AdversarialFileSystemAbstraction;
import org.neo4j.internal.helpers.NamedThreadFactory;
import org.neo4j.internal.unsafe.UnsafeUtil;
import org.neo4j.io.IOUtils;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.DelegatingFileSystemAbstraction;
import org.neo4j.io.fs.DelegatingStoreChannel;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.fs.StoreFileChannel;
import org.neo4j.io.memory.ByteBuffers;
import org.neo4j.io.pagecache.PageSwapper;
import org.neo4j.io.pagecache.PageSwapperFactory;
import org.neo4j.io.pagecache.PageSwapperTest;
import org.neo4j.io.pagecache.impl.FileLockException;
import org.neo4j.io.pagecache.impl.LockThisFileProgram;
import org.neo4j.io.pagecache.impl.SingleFilePageSwapperFactory;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.test.proc.ProcessUtil;

public class SingleFilePageSwapperTest
extends PageSwapperTest {
    private EphemeralFileSystemAbstraction ephemeralFileSystem;
    private DefaultFileSystemAbstraction fileSystem;
    private File file;
    private ExecutorService operationExecutor;
    private ThreadRegistryFactory threadRegistryFactory;
    private static final int INTERRUPT_ATTEMPTS = 100;

    @BeforeEach
    void setUp() throws IOException {
        this.file = new File("file").getCanonicalFile();
        this.ephemeralFileSystem = new EphemeralFileSystemAbstraction();
        this.fileSystem = new DefaultFileSystemAbstraction();
        this.threadRegistryFactory = new ThreadRegistryFactory();
        this.operationExecutor = Executors.newSingleThreadExecutor((ThreadFactory)((Object)this.threadRegistryFactory));
    }

    @AfterEach
    void tearDown() throws Exception {
        this.operationExecutor.shutdown();
        IOUtils.closeAll((AutoCloseable[])new FileSystemAbstraction[]{this.ephemeralFileSystem, this.fileSystem});
    }

    @Override
    protected PageSwapperFactory swapperFactory(FileSystemAbstraction fileSystem) {
        return new SingleFilePageSwapperFactory(fileSystem);
    }

    @Override
    protected void mkdirs(File dir) throws IOException {
        this.getFs().mkdirs(dir);
    }

    protected File getFile() {
        return this.file;
    }

    @Override
    protected FileSystemAbstraction getFs() {
        return this.getEphemeralFileSystem();
    }

    private FileSystemAbstraction getEphemeralFileSystem() {
        return this.ephemeralFileSystem;
    }

    FileSystemAbstraction getRealFileSystem() {
        return this.fileSystem;
    }

    private void putBytes(long page, byte[] data, int srcOffset, int tgtOffset, int length) {
        for (int i = 0; i < length; ++i) {
            UnsafeUtil.putByte((long)(page + (long)srcOffset + (long)i), (byte)data[tgtOffset + i]);
        }
    }

    @Test
    void swappingInMustFillPageWithData() throws Exception {
        byte[] bytes = new byte[]{1, 2, 3, 4};
        StoreChannel channel = this.getFs().write(this.getFile());
        channel.writeAll(this.wrap(bytes));
        channel.close();
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, this.getFile(), 4, null, false);
        long target = this.createPage(4);
        swapper.read(0L, target);
        Assertions.assertThat((byte[])this.array(target)).containsExactly(bytes);
    }

    @Test
    void mustZeroFillPageBeyondEndOfFile() throws Exception {
        byte[] bytes = new byte[]{1, 2, 3, 4, 5, 6};
        StoreChannel channel = this.getFs().write(this.getFile());
        channel.writeAll(this.wrap(bytes));
        channel.close();
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, this.getFile(), 4, null, false);
        long target = this.createPage(4);
        swapper.read(1L, target);
        Assertions.assertThat((byte[])this.array(target)).containsExactly(new int[]{5, 6, 0, 0});
    }

    @Test
    void uninterruptibleRead() throws Exception {
        byte[] pageContent = new byte[]{1, 2, 3, 4};
        StoreChannel channel = this.getFs().write(this.getFile());
        channel.writeAll(this.wrap(pageContent));
        channel.close();
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, this.getFile(), 4, null, false);
        long target = this.createPage(4);
        CountDownLatch startInterruptsLatch = new CountDownLatch(1);
        AtomicBoolean readFlag = new AtomicBoolean(true);
        Future<?> uninterruptibleFuture = this.operationExecutor.submit(() -> {
            startInterruptsLatch.countDown();
            while (readFlag.get()) {
                try {
                    swapper.read(0L, target);
                }
                catch (Throwable t) {
                    throw new RuntimeException(t);
                }
            }
        });
        startInterruptsLatch.await();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.threadRegistryFactory.getThreads().isEmpty());
        for (int i = 0; i < 100; ++i) {
            this.threadRegistryFactory.getThreads().forEach(Thread::interrupt);
            LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(10L));
        }
        readFlag.set(false);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(uninterruptibleFuture::get);
    }

    @Test
    void uninterruptibleWrite() throws Exception {
        byte[] pageContent = new byte[]{1, 2, 3, 4};
        StoreChannel channel = this.getFs().write(this.getFile());
        channel.writeAll(this.wrap(pageContent));
        channel.close();
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, this.getFile(), 4, null, false);
        long target = this.createPage(4);
        CountDownLatch startInterruptsLatch = new CountDownLatch(1);
        AtomicBoolean writeFlag = new AtomicBoolean(true);
        Future<?> uninterruptibleFuture = this.operationExecutor.submit(() -> {
            startInterruptsLatch.countDown();
            while (writeFlag.get()) {
                try {
                    swapper.write(0L, target);
                }
                catch (Throwable t) {
                    throw new RuntimeException(t);
                }
            }
        });
        startInterruptsLatch.await();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.threadRegistryFactory.getThreads().isEmpty());
        for (int i = 0; i < 100; ++i) {
            this.threadRegistryFactory.getThreads().forEach(Thread::interrupt);
            LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(10L));
        }
        writeFlag.set(false);
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(uninterruptibleFuture::get);
    }

    @Test
    void swappingOutMustWritePageToFile() throws Exception {
        this.getFs().write(this.getFile()).close();
        byte[] expected = new byte[]{1, 2, 3, 4};
        long page = this.createPage(expected);
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, this.getFile(), 4, null, false);
        swapper.write(0L, page);
        try (InputStream stream = this.getFs().openAsInputStream(this.getFile());){
            byte[] actual = new byte[expected.length];
            Assertions.assertThat((int)stream.read(actual)).isEqualTo(actual.length);
            Assertions.assertThat((byte[])actual).containsExactly(expected);
        }
    }

    private long createPage(byte[] expected) {
        long page = this.createPage(expected.length);
        this.putBytes(page, expected, 0, 0, expected.length);
        return page;
    }

    @Test
    void swappingOutMustNotOverwriteDataBeyondPage() throws Exception {
        byte[] initialData = new byte[]{1, 2, 3, 4, 5, 6, 7, 8, 9, 10};
        byte[] finalData = new byte[]{1, 2, 3, 4, 8, 7, 6, 5, 9, 10};
        StoreChannel channel = this.getFs().write(this.getFile());
        channel.writeAll(this.wrap(initialData));
        channel.close();
        byte[] change = new byte[]{8, 7, 6, 5};
        long page = this.createPage(change);
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, this.getFile(), 4, null, false);
        swapper.write(1L, page);
        try (InputStream stream = this.getFs().openAsInputStream(this.getFile());){
            byte[] actual = new byte[(int)this.getFs().getFileSize(this.getFile())];
            Assertions.assertThat((int)stream.read(actual)).isEqualTo(actual.length);
            Assertions.assertThat((byte[])actual).containsExactly(finalData);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void creatingSwapperForFileMustTakeLockOnFile() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory((FileSystemAbstraction)this.fileSystem);
        File file = this.testDir.file("file", new String[0]);
        this.fileSystem.write(file).close();
        try (PageSwapper pageSwapper = this.createSwapper(factory, file, 4, NO_CALLBACK, false);){
            StoreFileChannel channel = this.fileSystem.write(file);
            org.junit.jupiter.api.Assertions.assertThrows(OverlappingFileLockException.class, () -> ((StoreChannel)channel).tryLock());
        }
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void creatingSwapperForInternallyLockedFileMustThrow() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory((FileSystemAbstraction)this.fileSystem);
        File file = this.testDir.file("file", new String[0]);
        StoreFileChannel channel = this.fileSystem.write(file);
        try (FileLock fileLock = channel.tryLock();){
            Assertions.assertThat((Object)fileLock).isNotNull();
            org.junit.jupiter.api.Assertions.assertThrows(FileLockException.class, () -> this.createSwapper(factory, file, 4, NO_CALLBACK, true));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void creatingSwapperForExternallyLockedFileMustThrow() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory((FileSystemAbstraction)this.fileSystem);
        File file = this.testDir.file("file", new String[0]);
        this.fileSystem.write(file).close();
        ProcessBuilder pb = new ProcessBuilder(ProcessUtil.getJavaExecutable().toString(), "-cp", ProcessUtil.getClassPath(), LockThisFileProgram.class.getCanonicalName(), file.getAbsolutePath());
        File wd = new File("target/test-classes").getAbsoluteFile();
        pb.directory(wd);
        Process process = pb.start();
        BufferedReader stdout = new BufferedReader(new InputStreamReader(process.getInputStream()));
        InputStream stderr = process.getErrorStream();
        try {
            Assumptions.assumeTrue((boolean)"locked".equals(stdout.readLine()));
        }
        catch (Throwable e) {
            int b = stderr.read();
            while (b != -1) {
                System.err.write(b);
                b = stderr.read();
            }
            System.err.flush();
            int exitCode = process.waitFor();
            System.out.println("exitCode = " + exitCode);
            throw e;
        }
        try {
            org.junit.jupiter.api.Assertions.assertThrows(FileLockException.class, () -> this.createSwapper(factory, file, 4, NO_CALLBACK, true));
        }
        finally {
            process.getOutputStream().write(0);
            process.getOutputStream().flush();
            process.waitFor();
        }
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void mustUnlockFileWhenThePageSwapperIsClosed() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory((FileSystemAbstraction)this.fileSystem);
        File file = this.testDir.file("file", new String[0]);
        this.fileSystem.write(file).close();
        this.createSwapper(factory, file, 4, NO_CALLBACK, false).close();
        try (StoreFileChannel channel = this.fileSystem.write(file);
             FileLock fileLock = channel.tryLock();){
            Assertions.assertThat((Object)fileLock).isNotNull();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void fileMustRemainLockedEvenIfChannelIsClosedByStrayInterrupt() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory((FileSystemAbstraction)this.fileSystem);
        File file = this.testDir.file("file", new String[0]);
        this.fileSystem.write(file).close();
        try (PageSwapper pageSwapper = this.createSwapper(factory, file, 4, NO_CALLBACK, false);){
            StoreFileChannel channel = this.fileSystem.write(file);
            Thread.currentThread().interrupt();
            pageSwapper.force();
            org.junit.jupiter.api.Assertions.assertThrows(OverlappingFileLockException.class, () -> ((StoreChannel)channel).tryLock());
        }
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void mustCloseFilesIfTakingFileLockThrows() throws Exception {
        final AtomicInteger openFilesCounter = new AtomicInteger();
        PageSwapperFactory factory = this.createSwapperFactory((FileSystemAbstraction)new DelegatingFileSystemAbstraction((FileSystemAbstraction)this.fileSystem){

            public StoreChannel open(File fileName, Set<OpenOption> options) throws IOException {
                openFilesCounter.getAndIncrement();
                return new DelegatingStoreChannel(super.open(fileName, options)){

                    public void close() throws IOException {
                        openFilesCounter.getAndDecrement();
                        super.close();
                    }
                };
            }
        });
        File file = this.testDir.file("file", new String[0]);
        try (StoreFileChannel ch = this.fileSystem.write(file);
             FileLock ignore = ch.tryLock();){
            this.createSwapper(factory, file, 4, NO_CALLBACK, false).close();
            org.junit.jupiter.api.Assertions.fail((String)"Creating a page swapper for a locked channel should have thrown");
        }
        catch (FileLockException fileLockException) {
            // empty catch block
        }
        Assertions.assertThat((int)openFilesCounter.get()).isEqualTo(0);
    }

    private byte[] array(long page) {
        int size = this.sizeOfAsInt(page);
        byte[] array = new byte[size];
        for (int i = 0; i < size; ++i) {
            array[i] = UnsafeUtil.getByte((long)(page + (long)i));
        }
        return array;
    }

    private ByteBuffer wrap(byte[] bytes) {
        ByteBuffer buffer = ByteBuffers.allocate((int)bytes.length, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
        for (byte b : bytes) {
            buffer.put(b);
        }
        buffer.clear();
        return buffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void mustHandleMischiefInPositionedRead() throws Exception {
        int bytesTotal = 512;
        byte[] data = new byte[bytesTotal];
        ThreadLocalRandom.current().nextBytes(data);
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        File file = this.getFile();
        PageSwapper swapper = this.createSwapper(factory, file, bytesTotal, NO_CALLBACK, true);
        try {
            long page = this.createPage(data);
            swapper.write(0L, page);
        }
        finally {
            swapper.close();
            factory.close();
        }
        RandomAdversary adversary = new RandomAdversary(0.5, 0.0, 0.0);
        factory = this.createSwapperFactory((FileSystemAbstraction)new AdversarialFileSystemAbstraction((Adversary)adversary, this.getFs()));
        swapper = this.createSwapper(factory, file, bytesTotal, NO_CALLBACK, false);
        long page = this.createPage(bytesTotal);
        try {
            for (int i = 0; i < 10000; ++i) {
                this.clear(page);
                Assertions.assertThat((long)swapper.read(0L, page)).isEqualTo((long)bytesTotal);
                Assertions.assertThat((byte[])this.array(page)).isEqualTo((Object)data);
            }
        }
        finally {
            swapper.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void mustHandleMischiefInPositionedWrite() throws Exception {
        int bytesTotal = 512;
        byte[] data = new byte[bytesTotal];
        ThreadLocalRandom.current().nextBytes(data);
        long zeroPage = this.createPage(bytesTotal);
        this.clear(zeroPage);
        File file = this.getFile();
        RandomAdversary adversary = new RandomAdversary(0.5, 0.0, 0.0);
        PageSwapperFactory factory = this.createSwapperFactory((FileSystemAbstraction)new AdversarialFileSystemAbstraction((Adversary)adversary, this.getFs()));
        PageSwapper swapper = this.createSwapper(factory, file, bytesTotal, NO_CALLBACK, true);
        long page = this.createPage(bytesTotal);
        try {
            for (int i = 0; i < 10000; ++i) {
                adversary.setProbabilityFactor(0.0);
                swapper.write(0L, zeroPage);
                this.putBytes(page, data, 0, 0, data.length);
                adversary.setProbabilityFactor(1.0);
                Assertions.assertThat((long)swapper.write(0L, page)).isEqualTo((long)bytesTotal);
                this.clear(page);
                adversary.setProbabilityFactor(0.0);
                swapper.read(0L, page);
                Assertions.assertThat((byte[])this.array(page)).isEqualTo((Object)data);
            }
        }
        finally {
            swapper.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void mustHandleMischiefInPositionedVectoredRead() throws Exception {
        int bytesTotal = 512;
        int bytesPerPage = 32;
        int pageCount = bytesTotal / bytesPerPage;
        byte[] data = new byte[bytesTotal];
        ThreadLocalRandom.current().nextBytes(data);
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        File file = this.getFile();
        try (PageSwapper swapper = this.createSwapper(factory, file, bytesTotal, NO_CALLBACK, true);){
            long page = this.createPage(data);
            swapper.write(0L, page);
        }
        RandomAdversary adversary = new RandomAdversary(0.5, 0.0, 0.0);
        factory = this.createSwapperFactory((FileSystemAbstraction)new AdversarialFileSystemAbstraction((Adversary)adversary, this.getFs()));
        swapper = this.createSwapper(factory, file, bytesPerPage, NO_CALLBACK, false);
        long[] pages = new long[pageCount];
        for (int i = 0; i < pageCount; ++i) {
            pages[i] = this.createPage(bytesPerPage);
        }
        byte[] temp = new byte[bytesPerPage];
        try {
            for (int i = 0; i < 10000; ++i) {
                for (long page : pages) {
                    this.clear(page);
                }
                Assertions.assertThat((long)swapper.read(0L, pages, 0, pages.length)).isEqualTo((long)bytesTotal);
                for (int j = 0; j < pageCount; ++j) {
                    System.arraycopy(data, j * bytesPerPage, temp, 0, bytesPerPage);
                    Assertions.assertThat((byte[])this.array(pages[j])).isEqualTo((Object)temp);
                }
            }
        }
        finally {
            swapper.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void mustHandleMischiefInPositionedVectoredWrite() throws Exception {
        int i;
        int bytesTotal = 512;
        int bytesPerPage = 32;
        int pageCount = bytesTotal / bytesPerPage;
        byte[] data = new byte[bytesTotal];
        ThreadLocalRandom.current().nextBytes(data);
        long zeroPage = this.createPage(bytesPerPage);
        this.clear(zeroPage);
        File file = this.getFile();
        RandomAdversary adversary = new RandomAdversary(0.5, 0.0, 0.0);
        PageSwapperFactory factory = this.createSwapperFactory((FileSystemAbstraction)new AdversarialFileSystemAbstraction((Adversary)adversary, this.getFs()));
        PageSwapper swapper = this.createSwapper(factory, file, bytesPerPage, NO_CALLBACK, true);
        long[] writePages = new long[pageCount];
        long[] readPages = new long[pageCount];
        long[] zeroPages = new long[pageCount];
        for (i = 0; i < pageCount; ++i) {
            writePages[i] = this.createPage(bytesPerPage);
            this.putBytes(writePages[i], data, 0, i * bytesPerPage, bytesPerPage);
            readPages[i] = this.createPage(bytesPerPage);
            zeroPages[i] = zeroPage;
        }
        try {
            for (i = 0; i < 10000; ++i) {
                adversary.setProbabilityFactor(0.0);
                swapper.write(0L, zeroPages, 0, pageCount);
                adversary.setProbabilityFactor(1.0);
                swapper.write(0L, writePages, 0, pageCount);
                for (long readPage : readPages) {
                    this.clear(readPage);
                }
                adversary.setProbabilityFactor(0.0);
                Assertions.assertThat((long)swapper.read(0L, readPages, 0, pageCount)).isEqualTo((long)bytesTotal);
                for (int j = 0; j < pageCount; ++j) {
                    Assertions.assertThat((byte[])this.array(readPages[j])).containsExactly(this.array(writePages[j]));
                }
            }
        }
        finally {
            swapper.close();
        }
    }

    private static class ThreadRegistryFactory
    extends NamedThreadFactory {
        private final Set<Thread> threads = ConcurrentHashMap.newKeySet();

        ThreadRegistryFactory() {
            super("SwapperInterruptTestThreads");
        }

        public Thread newThread(Runnable runnable) {
            Thread thread = super.newThread(runnable);
            this.threads.add(thread);
            return thread;
        }

        public Set<Thread> getThreads() {
            return this.threads;
        }
    }
}

