/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.fs.watcher;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.Watchable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.Condition;
import org.assertj.core.api.ListAssert;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.DisabledOnOs;
import org.junit.jupiter.api.condition.OS;
import org.mockito.Mockito;
import org.neo4j.io.fs.watcher.DefaultFileSystemWatcher;
import org.neo4j.io.fs.watcher.FileWatchEventListener;
import org.neo4j.io.fs.watcher.resource.WatchedResource;
import org.neo4j.test.OtherThreadExecutor;
import org.neo4j.test.assertion.Assert;
import org.neo4j.test.conditions.Conditions;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;

@TestDirectoryExtension
class DefaultFileSystemWatcherTest {
    @Inject
    TestDirectory testDirectory;
    private WatchService watchServiceMock = (WatchService)Mockito.mock(WatchService.class);

    DefaultFileSystemWatcherTest() {
    }

    @Test
    void fileWatchRegistrationIsIllegal() {
        TestFileSystemWatcher watcher = this.createWatcher();
        IllegalArgumentException exception = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> watcher.watch(Path.of("notADirectory", new String[0])));
        Assertions.assertThat((String)exception.getMessage()).contains(new CharSequence[]{"Only directories can be registered to be monitored."});
    }

    @Test
    @DisabledOnOs(value={OS.WINDOWS})
    void shouldHandleMultipleSubscribersToSameFile() throws Exception {
        try (OtherThreadExecutor executor = new OtherThreadExecutor("watcher");
             DefaultFileSystemWatcher watcher = new DefaultFileSystemWatcher(FileSystems.getDefault().newWatchService());){
            AssertableFileEventListener listener = new AssertableFileEventListener();
            watcher.addFileWatchEventListener((FileWatchEventListener)listener);
            Path foo = this.testDirectory.createFile("foo");
            Path bar = this.testDirectory.createFile("bar");
            executor.executeDontWait(() -> {
                try {
                    watcher.startWatching();
                }
                catch (Exception exception) {
                    // empty catch block
                }
                return null;
            });
            WatchedResource firstWatch = watcher.watch(this.testDirectory.homePath());
            firstWatch.close();
            Assertions.assertThat((boolean)firstWatch.getWatchKey().isValid()).isFalse();
            WatchedResource secondWatch = watcher.watch(this.testDirectory.homePath());
            WatchedResource thirdWatch = watcher.watch(this.testDirectory.homePath());
            executor.waitUntilWaiting(location -> location.isAt(DefaultFileSystemWatcher.class, "startWatching"));
            this.testDirectory.getFileSystem().delete(foo);
            Assert.assertEventually((String)"Foo deleted", () -> listener.containsDeleted("foo"), (Condition)Conditions.TRUE, (long)30L, (TimeUnit)TimeUnit.SECONDS);
            secondWatch.close();
            Assertions.assertThat((boolean)thirdWatch.getWatchKey().isValid()).isTrue();
            this.testDirectory.getFileSystem().delete(bar);
            Assert.assertEventually((String)"Bar deleted", () -> listener.containsDeleted("bar"), (Condition)Conditions.TRUE, (long)30L, (TimeUnit)TimeUnit.SECONDS);
            thirdWatch.close();
            Assertions.assertThat((boolean)thirdWatch.getWatchKey().isValid()).isFalse();
        }
    }

    @Test
    void registerMultipleDirectoriesForMonitoring() throws Exception {
        try (DefaultFileSystemWatcher watcher = new DefaultFileSystemWatcher(FileSystems.getDefault().newWatchService());){
            Path directory1 = this.testDirectory.directory("test1");
            Path directory2 = this.testDirectory.directory("test2");
            WatchedResource watchedResource1 = watcher.watch(directory1);
            WatchedResource watchedResource2 = watcher.watch(directory2);
            org.junit.jupiter.api.Assertions.assertNotSame((Object)watchedResource1, (Object)watchedResource2);
        }
    }

    @Test
    void notifyListenersOnDeletion() throws InterruptedException {
        TestFileSystemWatcher watcher = this.createWatcher();
        AssertableFileEventListener listener1 = new AssertableFileEventListener();
        AssertableFileEventListener listener2 = new AssertableFileEventListener();
        watcher.addFileWatchEventListener(listener1);
        watcher.addFileWatchEventListener(listener2);
        TestWatchEvent<Path> watchEvent = new TestWatchEvent<Path>(StandardWatchEventKinds.ENTRY_DELETE, Paths.get("file1", new String[0]));
        TestWatchEvent<Path> watchEvent2 = new TestWatchEvent<Path>(StandardWatchEventKinds.ENTRY_DELETE, Paths.get("file2", new String[0]));
        TestWatchKey watchKey = new TestWatchKey(Arrays.asList(watchEvent, watchEvent2));
        this.prepareWatcher(watchKey);
        this.watch(watcher);
        listener1.assertDeleted("file1");
        listener1.assertDeleted("file2");
        listener2.assertDeleted("file1");
        listener2.assertDeleted("file2");
    }

    @Test
    void notifyListenersOnModification() throws InterruptedException {
        TestFileSystemWatcher watcher = this.createWatcher();
        AssertableFileEventListener listener1 = new AssertableFileEventListener();
        AssertableFileEventListener listener2 = new AssertableFileEventListener();
        watcher.addFileWatchEventListener(listener1);
        watcher.addFileWatchEventListener(listener2);
        TestWatchEvent<Path> watchEvent = new TestWatchEvent<Path>(StandardWatchEventKinds.ENTRY_MODIFY, Paths.get("a", new String[0]));
        TestWatchEvent<Path> watchEvent2 = new TestWatchEvent<Path>(StandardWatchEventKinds.ENTRY_MODIFY, Paths.get("b", new String[0]));
        TestWatchEvent<Path> watchEvent3 = new TestWatchEvent<Path>(StandardWatchEventKinds.ENTRY_MODIFY, Paths.get("c", new String[0]));
        TestWatchKey watchKey = new TestWatchKey(Arrays.asList(watchEvent, watchEvent2, watchEvent3));
        this.prepareWatcher(watchKey);
        this.watch(watcher);
        listener1.assertModified("a");
        listener1.assertModified("b");
        listener1.assertModified("c");
        listener2.assertModified("a");
        listener2.assertModified("b");
        listener2.assertModified("c");
    }

    @Test
    void stopWatchingAndCloseEverythingOnClosed() throws IOException {
        TestFileSystemWatcher watcher = this.createWatcher();
        watcher.close();
        ((WatchService)Mockito.verify((Object)this.watchServiceMock)).close();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)watcher.isClosed());
    }

    @Test
    void skipEmptyEvent() throws InterruptedException {
        TestFileSystemWatcher watcher = this.createWatcher();
        AssertableFileEventListener listener = new AssertableFileEventListener();
        watcher.addFileWatchEventListener(listener);
        TestWatchEvent<Object> event = new TestWatchEvent<Object>(StandardWatchEventKinds.ENTRY_MODIFY, null);
        TestWatchKey watchKey = new TestWatchKey(Arrays.asList(event));
        this.prepareWatcher(watchKey);
        this.watch(watcher);
        listener.assertNoEvents();
    }

    private void prepareWatcher(TestWatchKey watchKey) throws InterruptedException {
        Mockito.when((Object)this.watchServiceMock.take()).thenReturn((Object)watchKey).thenThrow(InterruptedException.class);
    }

    private void watch(TestFileSystemWatcher watcher) {
        try {
            watcher.startWatching();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private TestFileSystemWatcher createWatcher() {
        return new TestFileSystemWatcher(this.watchServiceMock);
    }

    private static class AssertableFileEventListener
    implements FileWatchEventListener {
        private final List<String> deletedFileNames = new ArrayList<String>();
        private final List<String> modifiedFileNames = new ArrayList<String>();

        private AssertableFileEventListener() {
        }

        public synchronized void fileDeleted(WatchKey key, String fileName) {
            this.deletedFileNames.add(fileName);
        }

        public synchronized void fileModified(WatchKey key, String fileName) {
            this.modifiedFileNames.add(fileName);
        }

        synchronized void assertNoEvents() {
            ((ListAssert)Assertions.assertThat(this.deletedFileNames).as("Should not have any deletion events", new Object[0])).isEmpty();
            ((ListAssert)Assertions.assertThat(this.modifiedFileNames).as("Should not have any modification events", new Object[0])).isEmpty();
        }

        synchronized void assertDeleted(String fileName) {
            ((ListAssert)Assertions.assertThat(this.deletedFileNames).as("Was expected to find notification about deletion.", new Object[0])).contains((Object[])new String[]{fileName});
        }

        synchronized boolean containsDeleted(String fileName) {
            return this.deletedFileNames.contains(fileName);
        }

        synchronized void assertModified(String fileName) {
            ((ListAssert)Assertions.assertThat(this.modifiedFileNames).as("Was expected to find notification about modification.", new Object[0])).contains((Object[])new String[]{fileName});
        }
    }

    private static class TestWatchEvent<T>
    implements WatchEvent {
        private WatchEvent.Kind<T> eventKind;
        private T fileName;

        TestWatchEvent(WatchEvent.Kind<T> eventKind, T fileName) {
            this.eventKind = eventKind;
            this.fileName = fileName;
        }

        public WatchEvent.Kind kind() {
            return this.eventKind;
        }

        @Override
        public int count() {
            return 0;
        }

        @Override
        public T context() {
            return this.fileName;
        }
    }

    private static class TestWatchKey
    implements WatchKey {
        private List<WatchEvent<?>> events;
        private boolean canceled;

        TestWatchKey(List<WatchEvent<?>> events) {
            this.events = events;
        }

        @Override
        public boolean isValid() {
            return false;
        }

        @Override
        public List<WatchEvent<?>> pollEvents() {
            return this.events;
        }

        @Override
        public boolean reset() {
            return false;
        }

        @Override
        public void cancel() {
            this.canceled = true;
        }

        @Override
        public Watchable watchable() {
            return null;
        }
    }

    private static class TestFileSystemWatcher
    extends DefaultFileSystemWatcher {
        private boolean closed;

        TestFileSystemWatcher(WatchService watchService) {
            super(watchService);
        }

        public void close() throws IOException {
            super.close();
            this.closed = true;
        }

        public boolean isClosed() {
            return this.closed;
        }
    }
}

