/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.pagecache;

import java.io.IOException;
import java.nio.channels.ClosedChannelException;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.DisabledOnOs;
import org.junit.jupiter.api.condition.EnabledOnOs;
import org.junit.jupiter.api.condition.OS;
import org.neo4j.internal.unsafe.UnsafeUtil;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.mem.MemoryAllocator;
import org.neo4j.io.pagecache.IOController;
import org.neo4j.io.pagecache.PageEvictionCallback;
import org.neo4j.io.pagecache.PageSwapper;
import org.neo4j.io.pagecache.PageSwapperFactory;
import org.neo4j.io.pagecache.impl.muninn.SwapperSet;
import org.neo4j.memory.LocalMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;
import org.neo4j.util.concurrent.Futures;

@TestDirectoryExtension
public abstract class PageSwapperTest {
    @Inject
    protected TestDirectory testDir;
    public static final long X = -3819410105021120785L;
    public static final long Y = 6846544296635974449L;
    public static final int Z = -16843010;
    protected static final PageEvictionCallback NO_CALLBACK = filePageId -> {};
    private static final int cachePageSize = 32;
    private final ConcurrentLinkedQueue<PageSwapper> openedSwappers = new ConcurrentLinkedQueue();
    private final MemoryAllocator mman = MemoryAllocator.createAllocator((long)ByteUnit.KibiByte.toBytes(32L), (MemoryTracker)new LocalMemoryTracker());
    private final SwapperSet swapperSet = new SwapperSet();

    protected abstract PageSwapperFactory swapperFactory(FileSystemAbstraction var1);

    protected abstract void mkdirs(Path var1) throws IOException;

    @BeforeEach
    @AfterEach
    void clearStrayInterrupts() {
        Thread.interrupted();
    }

    @AfterEach
    void closeOpenedPageSwappers() throws Exception {
        PageSwapper swapper;
        IOException exception = null;
        while ((swapper = this.openedSwappers.poll()) != null) {
            try {
                swapper.close();
            }
            catch (IOException e) {
                if (exception == null) {
                    exception = e;
                    continue;
                }
                exception.addSuppressed(e);
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    protected abstract FileSystemAbstraction getFs();

    @Test
    void readMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        this.putInt(page, 0, 1);
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Assertions.assertThat((long)this.write(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        this.putInt(page, 0, 0);
        Thread.currentThread().interrupt();
        Assertions.assertThat((long)this.read(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(1);
        Assertions.assertThat((long)this.read(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(1);
    }

    @Test
    void vectoredReadMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        this.putInt(page, 0, 1);
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Assertions.assertThat((long)this.write(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        this.putInt(page, 0, 0);
        Thread.currentThread().interrupt();
        Assertions.assertThat((long)this.read(swapper, 0L, new long[]{page}, new int[]{this.cachePageSize()}, 1)).isEqualTo(this.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(1);
        Assertions.assertThat((long)this.read(swapper, 0L, new long[]{page}, new int[]{this.cachePageSize()}, 1)).isEqualTo(this.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(1);
    }

    @Test
    void writeMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        this.putInt(page, 0, 1);
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        Assertions.assertThat((long)this.write(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        this.putInt(page, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(1);
        Assertions.assertThat((long)this.write(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        this.putInt(page, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(1);
    }

    @Test
    void vectoredWriteMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        this.putInt(page, 0, 1);
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        Assertions.assertThat((long)this.write(swapper, 0L, new long[]{page}, new int[]{this.cachePageSize()}, 1, 1)).isEqualTo(this.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        this.putInt(page, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(1);
        Assertions.assertThat((long)this.write(swapper, 0L, new long[]{page}, new int[]{this.cachePageSize()}, 1, 1)).isEqualTo(this.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        this.putInt(page, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 0, page)).isEqualTo(this.sizeOfAsLong(page));
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(1);
    }

    @Test
    void forcingMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        swapper.force();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
    }

    @Test
    void mustReopenChannelWhenReadFailsWithAsynchronousCloseException() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        long page = this.createPage();
        this.putLong(page, 0, -3819410105021120785L);
        this.putLong(page, 8, 6846544296635974449L);
        this.putInt(page, 16, -16843010);
        this.write(swapper, 0, page);
        Thread.currentThread().interrupt();
        this.read(swapper, 0, page);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        Assertions.assertThat((long)this.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)this.getLong(page, 8)).isEqualTo(6846544296635974449L);
        Assertions.assertThat((int)this.getInt(page, 16)).isEqualTo(-16843010);
        swapper.force();
    }

    @Test
    void mustReopenChannelWhenVectoredReadFailsWithAsynchronousCloseException() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        long page = this.createPage();
        this.putLong(page, 0, -3819410105021120785L);
        this.putLong(page, 8, 6846544296635974449L);
        this.putInt(page, 16, -16843010);
        this.write(swapper, 0, page);
        Thread.currentThread().interrupt();
        this.read(swapper, 0L, new long[]{page}, new int[]{this.cachePageSize()}, 1);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        Assertions.assertThat((long)this.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)this.getLong(page, 8)).isEqualTo(6846544296635974449L);
        Assertions.assertThat((int)this.getInt(page, 16)).isEqualTo(-16843010);
        swapper.force();
    }

    @Test
    void mustReopenChannelWhenWriteFailsWithAsynchronousCloseException() throws Exception {
        long page = this.createPage();
        this.putLong(page, 0, -3819410105021120785L);
        this.putLong(page, 8, 6846544296635974449L);
        this.putInt(page, 16, -16843010);
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        this.write(swapper, 0, page);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        swapper.force();
        this.clear(page);
        this.read(swapper, 0, page);
        Assertions.assertThat((long)this.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)this.getLong(page, 8)).isEqualTo(6846544296635974449L);
        Assertions.assertThat((int)this.getInt(page, 16)).isEqualTo(-16843010);
    }

    @Test
    void mustReopenChannelWhenVectoredWriteFailsWithAsynchronousCloseException() throws Exception {
        long page = this.createPage();
        this.putLong(page, 0, -3819410105021120785L);
        this.putLong(page, 8, 6846544296635974449L);
        this.putInt(page, 16, -16843010);
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        this.write(swapper, 0L, new long[]{page}, new int[]{this.cachePageSize()}, 1, 1);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        swapper.force();
        this.clear(page);
        this.read(swapper, 0, page);
        Assertions.assertThat((long)this.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)this.getLong(page, 8)).isEqualTo(6846544296635974449L);
        Assertions.assertThat((int)this.getInt(page, 16)).isEqualTo(-16843010);
    }

    @Test
    void mustReopenChannelWhenForceFailsWithAsynchronousCloseException() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        for (int i = 0; i < 10; ++i) {
            Thread.currentThread().interrupt();
            swapper.force();
            org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        }
    }

    @Test
    void readMustNotReopenExplicitlyClosedChannel() throws Exception {
        String filename = "a";
        Path file = this.file(filename);
        long page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        this.write(swapper, 0, page);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> this.read(swapper, 0, page));
    }

    @Test
    void vectoredReadMustNotReopenExplicitlyClosedChannel() throws Exception {
        String filename = "a";
        Path file = this.file(filename);
        long page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        this.write(swapper, 0, page);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> this.read(swapper, 0L, new long[]{page}, new int[]{this.cachePageSize()}, 1));
    }

    @Test
    void writeMustNotReopenExplicitlyClosedChannel() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> this.write(swapper, 0, page));
    }

    @Test
    void vectoredWriteMustNotReopenExplicitlyClosedChannel() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> this.write(swapper, 0L, new long[]{page}, new int[]{this.cachePageSize()}, 1, 1));
    }

    @Test
    void forceMustNotReopenExplicitlyClosedChannel() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> ((PageSwapper)swapper).force());
    }

    @Test
    void mustNotOverwriteDataInOtherFiles() throws Exception {
        Path fileA = this.file("a");
        Path fileB = this.file("b");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapperA = this.createSwapperAndFile(factory, fileA);
        PageSwapper swapperB = this.createSwapperAndFile(factory, fileB);
        long page = this.createPage();
        this.clear(page);
        this.putLong(page, 0, -3819410105021120785L);
        this.write(swapperA, 0, page);
        this.putLong(page, 8, 6846544296635974449L);
        this.write(swapperB, 0, page);
        this.clear(page);
        Assertions.assertThat((long)this.getLong(page, 0)).isEqualTo(0L);
        Assertions.assertThat((long)this.getLong(page, 8)).isEqualTo(0L);
        this.read(swapperA, 0, page);
        Assertions.assertThat((long)this.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)this.getLong(page, 8)).isEqualTo(0L);
    }

    @Test
    void swapperCantPreallocateWhenConfigured() throws IOException {
        Path file = this.file("notPreallocatedFile");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        try (PageSwapper swapper = this.createSwapper(factory, file, this.cachePageSize(), NO_CALLBACK, true, false, false);){
            org.junit.jupiter.api.Assertions.assertFalse((boolean)swapper.canAllocate());
        }
    }

    @Test
    void mustRunEvictionCallbackOnEviction() throws Exception {
        AtomicLong callbackFilePageId = new AtomicLong();
        PageEvictionCallback callback = callbackFilePageId::set;
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, file, this.cachePageSize(), callback, true, false, true);
        swapper.evicted(42L);
        Assertions.assertThat((long)callbackFilePageId.get()).isEqualTo(42L);
    }

    @Test
    void mustNotIssueEvictionCallbacksAfterSwapperHasBeenClosed() throws Exception {
        AtomicBoolean gotCallback = new AtomicBoolean();
        PageEvictionCallback callback = filePageId -> gotCallback.set(true);
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, file, this.cachePageSize(), callback, true, false, true);
        swapper.close();
        swapper.evicted(42L);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)gotCallback.get());
    }

    @Test
    void mustThrowExceptionIfFileDoesNotExist() {
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        org.junit.jupiter.api.Assertions.assertThrows(NoSuchFileException.class, () -> this.createSwapper(factory, this.file("does not exist"), this.cachePageSize(), NO_CALLBACK, false, false, true));
    }

    @Test
    void mustCreateNonExistingFileWithCreateFlag() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper pageSwapper = this.createSwapperAndFile(factory, this.file("does not exist"));
        long page = this.createPage();
        this.putLong(page, 0, -3819410105021120785L);
        this.write(pageSwapper, 0, page);
        this.clear(page);
        this.read(pageSwapper, 0, page);
        Assertions.assertThat((long)this.getLong(page, 0)).isEqualTo(-3819410105021120785L);
    }

    @Test
    void truncatedFilesMustBeEmpty() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(-1L);
        long page = this.createPage();
        this.putInt(page, 0, -889275714);
        this.write(swapper, 10, page);
        this.clear(page);
        this.read(swapper, 10, page);
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(-889275714);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(10L);
        swapper.close();
        swapper = this.createSwapper(factory, file, this.cachePageSize(), NO_CALLBACK, false, false, true);
        this.clear(page);
        this.read(swapper, 10, page);
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(-889275714);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(10L);
        swapper.truncate();
        this.clear(page);
        this.read(swapper, 10, page);
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(0);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(-1L);
        swapper.close();
        swapper = this.createSwapper(factory, file, this.cachePageSize(), NO_CALLBACK, false, false, true);
        this.clear(page);
        this.read(swapper, 10, page);
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(0);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(-1L);
        swapper.close();
    }

    @Test
    void positionedVectoredWriteMustFlushAllBuffersInOrder() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        long pageC = this.createPage(4);
        long pageD = this.createPage(4);
        this.putInt(pageA, 0, 2);
        this.putInt(pageB, 0, 3);
        this.putInt(pageC, 0, 4);
        this.putInt(pageD, 0, 5);
        this.write(swapper, 1L, new long[]{pageA, pageB, pageC, pageD}, new int[]{4, 4, 4, 4}, 4, 4);
        long result = this.createPage(4);
        this.read(swapper, 0, result);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(0);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 1, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(2);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 2, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(3);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 3, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(4);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 4, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(5);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 5, result)).isEqualTo(0L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(0);
    }

    @Test
    void positionedVectoredWriteMustFlushAllBuffersOfDifferentSizeInOrder() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(8);
        long pageC = this.createPage(12);
        this.putInt(pageA, 0, 2);
        this.putInt(pageB, 0, 3);
        this.putInt(pageB, 4, 4);
        this.putInt(pageC, 0, 5);
        this.putInt(pageC, 4, 6);
        this.putInt(pageC, 8, 7);
        org.junit.jupiter.api.Assertions.assertEquals((long)24L, (long)this.write(swapper, 0L, new long[]{pageA, pageB, pageC}, new int[]{4, 8, 12}, 3, 6));
        long result = this.createPage(4);
        this.read(swapper, 0, result);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(2);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 1, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(3);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 2, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(4);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 3, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(5);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 4, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(6);
        this.putInt(result, 0, 0);
        Assertions.assertThat((long)this.read(swapper, 5, result)).isEqualTo(4L);
        Assertions.assertThat((int)this.getInt(result, 0)).isEqualTo(7);
    }

    @Test
    void positionedVectoredReadMustFillAllBuffersOfDifferentSizesInOrder() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage();
        this.putInt(output, 0, 2);
        this.write(swapper, 1, output);
        this.putInt(output, 0, 3);
        this.write(swapper, 2, output);
        this.putInt(output, 0, 4);
        this.write(swapper, 3, output);
        this.putInt(output, 0, 5);
        this.write(swapper, 4, output);
        this.putInt(output, 0, 6);
        this.write(swapper, 5, output);
        this.putInt(output, 0, 7);
        this.write(swapper, 6, output);
        long pageA = this.createPage(4);
        long pageB = this.createPage(8);
        long pageC = this.createPage(12);
        Assertions.assertThat((long)this.read(swapper, 1L, new long[]{pageA, pageB, pageC}, new int[]{4, 8, 12}, 3)).isEqualTo(24L);
        Assertions.assertThat((int)this.getInt(pageA, 0)).isEqualTo(2);
        Assertions.assertThat((int)this.getInt(pageB, 0)).isEqualTo(3);
        Assertions.assertThat((int)this.getInt(pageB, 4)).isEqualTo(4);
        Assertions.assertThat((int)this.getInt(pageC, 0)).isEqualTo(5);
        Assertions.assertThat((int)this.getInt(pageC, 4)).isEqualTo(6);
        Assertions.assertThat((int)this.getInt(pageC, 8)).isEqualTo(7);
    }

    @Test
    void positionedVectoredReadMustFillAllBuffersInOrder() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage();
        this.putInt(output, 0, 2);
        this.write(swapper, 1, output);
        this.putInt(output, 0, 3);
        this.write(swapper, 2, output);
        this.putInt(output, 0, 4);
        this.write(swapper, 3, output);
        this.putInt(output, 0, 5);
        this.write(swapper, 4, output);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        long pageC = this.createPage(4);
        long pageD = this.createPage(4);
        Assertions.assertThat((long)this.read(swapper, 1L, new long[]{pageA, pageB, pageC, pageD}, new int[]{4, 4, 4, 4}, 4)).isEqualTo(16L);
        Assertions.assertThat((int)this.getInt(pageA, 0)).isEqualTo(2);
        Assertions.assertThat((int)this.getInt(pageB, 0)).isEqualTo(3);
        Assertions.assertThat((int)this.getInt(pageC, 0)).isEqualTo(4);
        Assertions.assertThat((int)this.getInt(pageD, 0)).isEqualTo(5);
    }

    @Test
    void positionedVectoredReadFromEmptyFileMustFillPagesWithZeros() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long page = this.createPage(4);
        this.putInt(page, 0, 1);
        Assertions.assertThat((long)this.read(swapper, 0L, new long[]{page}, new int[]{4}, 1)).isEqualTo(0L);
        Assertions.assertThat((int)this.getInt(page, 0)).isEqualTo(0);
    }

    @Test
    void positionedVectoredReadBeyondEndOfFileMustFillPagesWithZeros() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage(4);
        this.putInt(output, 0, -1);
        this.write(swapper, 0L, new long[]{output, output, output}, new int[]{4, 4, 4}, 3, 3);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        this.putInt(pageA, 0, -1);
        this.putInt(pageB, 0, -1);
        Assertions.assertThat((long)this.read(swapper, 3L, new long[]{pageA, pageB}, new int[]{4, 4}, 2)).isEqualTo(0L);
        Assertions.assertThat((int)this.getInt(pageA, 0)).isEqualTo(0);
        Assertions.assertThat((int)this.getInt(pageB, 0)).isEqualTo(0);
    }

    @Test
    void positionedVectoredReadBeyondEndOfFileMustFillPagesWithZerosForBuffersWithDifferentSizes() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage(4);
        this.putInt(output, 0, 42);
        this.write(swapper, 0L, new long[]{output, output, output, output}, new int[]{4, 4, 4, 4}, 4, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(8);
        long pageC = this.createPage(12);
        this.putInt(pageA, 0, -1);
        this.putInt(pageB, 0, -1);
        this.putInt(pageB, 4, -1);
        this.putInt(pageC, 0, -1);
        this.putInt(pageC, 4, -1);
        this.putInt(pageC, 8, -1);
        Assertions.assertThat((long)this.read(swapper, 0L, new long[]{pageA, pageB, pageC}, new int[]{4, 8, 12}, 3)).isEqualTo(16L);
        Assertions.assertThat((int)this.getInt(pageA, 0)).isEqualTo(42);
        Assertions.assertThat((int)this.getInt(pageB, 0)).isEqualTo(42);
        Assertions.assertThat((int)this.getInt(pageB, 4)).isEqualTo(42);
        Assertions.assertThat((int)this.getInt(pageC, 0)).isEqualTo(42);
        Assertions.assertThat((int)this.getInt(pageC, 4)).isEqualTo(0);
        Assertions.assertThat((int)this.getInt(pageC, 8)).isEqualTo(0);
    }

    @Test
    void positionedVectoredReadWhereLastPageExtendBeyondEndOfFileMustHaveRemainderZeroFilled() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage(4);
        this.putInt(output, 0, -1);
        this.write(swapper, 0L, new long[]{output, output, output, output, output}, new int[]{4, 4, 4, 4, 4}, 5, 5);
        swapper.close();
        swapper = this.createSwapper(factory, file, 8, NO_CALLBACK, false, false, true);
        long pageA = this.createPage(8);
        long pageB = this.createPage(8);
        this.putLong(pageA, 0, -3819410105021120785L);
        this.putLong(pageB, 0, 6846544296635974449L);
        Assertions.assertThat((long)this.read(swapper, 1L, new long[]{pageA, pageB}, new int[]{8, 8}, 2)).isIn(new Object[]{12L, 16L});
        Assertions.assertThat((long)this.getLong(pageA, 0)).isEqualTo(-1L);
        Assertions.assertThat((byte)this.getByte(pageB, 0)).isEqualTo((byte)-1);
        Assertions.assertThat((byte)this.getByte(pageB, 1)).isEqualTo((byte)-1);
        Assertions.assertThat((byte)this.getByte(pageB, 2)).isEqualTo((byte)-1);
        Assertions.assertThat((byte)this.getByte(pageB, 3)).isEqualTo((byte)-1);
        Assertions.assertThat((byte)this.getByte(pageB, 4)).isEqualTo((byte)0);
        Assertions.assertThat((byte)this.getByte(pageB, 5)).isEqualTo((byte)0);
        Assertions.assertThat((byte)this.getByte(pageB, 6)).isEqualTo((byte)0);
        Assertions.assertThat((byte)this.getByte(pageB, 7)).isEqualTo((byte)0);
    }

    @Test
    void positionedVectoredReadWhereSecondLastPageExtendBeyondEndOfFileMustHaveRestZeroFilled() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage(4);
        this.putInt(output, 0, 1);
        this.write(swapper, 0, output);
        this.putInt(output, 0, 2);
        this.write(swapper, 1, output);
        this.putInt(output, 0, 3);
        this.write(swapper, 2, output);
        swapper.close();
        swapper = this.createSwapper(factory, file, 8, NO_CALLBACK, false, false, true);
        long pageA = this.createPage(8);
        long pageB = this.createPage(8);
        long pageC = this.createPage(8);
        this.putInt(pageA, 0, -1);
        this.putInt(pageB, 0, -1);
        this.putInt(pageC, 0, -1);
        Assertions.assertThat((long)this.read(swapper, 0L, new long[]{pageA, pageB, pageC}, new int[]{8, 8, 8}, 3)).isIn(new Object[]{12L, 16L});
        Assertions.assertThat((int)this.getInt(pageA, 0)).isEqualTo(1);
        Assertions.assertThat((int)this.getInt(pageA, 4)).isEqualTo(2);
        Assertions.assertThat((int)this.getInt(pageB, 0)).isEqualTo(3);
        Assertions.assertThat((int)this.getInt(pageB, 4)).isEqualTo(0);
        Assertions.assertThat((long)this.getLong(pageC, 0)).isEqualTo(0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void concurrentPositionedVectoredReadsAndWritesMustNotInterfere() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        int pageCount = 100;
        int iterations = 20000;
        CountDownLatch startLatch = new CountDownLatch(1);
        long output = this.createPage(4);
        for (int i = 0; i < 100; ++i) {
            this.putInt(output, 0, i + 1);
            this.write(swapper, i, output);
        }
        Callable<Void> work = () -> {
            int i;
            ThreadLocalRandom rng = ThreadLocalRandom.current();
            int length = 10;
            int pageSize = 4;
            long[] pages = new long[length];
            int[] sizes = new int[length];
            for (i = 0; i < length; ++i) {
                pages[i] = this.createPage(pageSize);
                sizes[i] = pageSize;
            }
            startLatch.await();
            for (i = 0; i < 20000; ++i) {
                long startFilePageId = rng.nextLong(0L, 100 - pages.length);
                if (rng.nextBoolean()) {
                    long bytesRead = this.read(swapper, startFilePageId, pages, sizes, pages.length);
                    Assertions.assertThat((long)bytesRead).isEqualTo((long)pages.length * 4L);
                    for (int j = 0; j < pages.length; ++j) {
                        int expectedValue = (int)((long)(1 + j) + startFilePageId);
                        int actualValue = this.getInt(pages[j], 0);
                        Assertions.assertThat((int)actualValue).isEqualTo(expectedValue);
                    }
                    continue;
                }
                for (int j = 0; j < pages.length; ++j) {
                    int value = (int)((long)(1 + j) + startFilePageId);
                    this.putInt(pages[j], 0, value);
                }
                Assertions.assertThat((long)this.write(swapper, startFilePageId, pages, sizes, pages.length, pages.length)).isEqualTo((long)pages.length * 4L);
            }
            return null;
        };
        int threads = 8;
        ExecutorService executor = null;
        try {
            executor = Executors.newFixedThreadPool(threads, r -> {
                Thread thread = Executors.defaultThreadFactory().newThread(r);
                thread.setDaemon(true);
                return thread;
            });
            ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>(threads);
            for (int i = 0; i < threads; ++i) {
                futures.add(executor.submit(work));
            }
            startLatch.countDown();
            Futures.getAll(futures);
        }
        finally {
            if (executor != null) {
                executor.shutdown();
            }
        }
    }

    @Test
    void mustThrowNullPointerExceptionFromReadWhenPageArrayIsNull() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long page = this.createPage(4);
        int[] pageSizes = new int[]{4, 4, 4, 4};
        this.write(swapper, 0L, new long[]{page, page, page, page}, pageSizes, 4, 4);
        Assertions.assertThatThrownBy(() -> this.read(swapper, 0L, null, pageSizes, 4), (String)"vectored read with null array should have thrown", (Object[])new Object[0]).extracting(ExceptionUtils::getRootCause).isInstanceOf(NullPointerException.class);
    }

    @Test
    void mustThrowNullPointerExceptionFromWriteWhenPageArrayIsNull() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Assertions.assertThatThrownBy(() -> this.write(swapper, 0L, null, null, 4, 4), (String)"vectored write with null array should have thrown", (Object[])new Object[0]).extracting(ExceptionUtils::getRootCause).isInstanceOf(NullPointerException.class);
    }

    @Test
    void readMustThrowForNegativeFilePageIds() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> this.read(swapper, -1, this.createPage(4)));
    }

    @Test
    @DisabledOnOs(value={OS.LINUX})
    void directIOAllowedOnlyOnLinux() throws IOException {
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        Path file = this.file("file");
        IllegalArgumentException e = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.createSwapperAndFile(factory, file, true));
        Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Linux"});
    }

    @Test
    @EnabledOnOs(value={OS.LINUX})
    void doNotAllowDirectIOForPagesNotMultipleOfBlockSize() throws IOException {
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        Path file = this.file("file");
        this.checkUnsupportedPageSize(factory, file, 17);
        this.checkUnsupportedPageSize(factory, file, 115);
        this.checkUnsupportedPageSize(factory, file, 218);
        this.checkUnsupportedPageSize(factory, file, 419);
        this.checkUnsupportedPageSize(factory, file, 524);
        this.checkUnsupportedPageSize(factory, file, 1023);
        this.checkUnsupportedPageSize(factory, file, 4097);
    }

    private void checkUnsupportedPageSize(PageSwapperFactory factory, Path path, int pageSize) {
        IllegalArgumentException e = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.createSwapperAndFile(factory, path, pageSize, true));
        Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"block"});
    }

    @Test
    void writeMustThrowForNegativeFilePageIds() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> this.write(swapper, -1, this.createPage(4)));
    }

    @Test
    void vectoredReadMustThrowForNegativeFilePageIds() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> this.read(swapper, -1L, new long[]{this.createPage(4), this.createPage(4)}, new int[]{4, 4}, 2));
    }

    @Test
    void vectoredWriteMustThrowForNegativeFilePageIds() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> this.write(swapper, -1L, new long[]{this.createPage(4), this.createPage(4)}, new int[]{4, 4}, 2, 2));
    }

    @Test
    void vectoredReadMustReadNothingWhenLengthIsZero() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        this.putInt(pageA, 0, 1);
        this.putInt(pageB, 0, 2);
        long[] pages = new long[]{pageA, pageB};
        int[] pageSizes = new int[]{4, 4};
        this.write(swapper, 0L, pages, pageSizes, 2, 2);
        this.putInt(pageA, 0, 3);
        this.putInt(pageB, 0, 4);
        this.read(swapper, 0L, pages, pageSizes, 0);
        int[] expectedValues = new int[]{3, 4};
        int[] actualValues = new int[]{this.getInt(pageA, 0), this.getInt(pageB, 0)};
        Assertions.assertThat((int[])actualValues).isEqualTo((Object)expectedValues);
    }

    @Test
    void vectoredWriteMustWriteNothingWhenLengthIsZero() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        this.putInt(pageA, 0, 1);
        this.putInt(pageB, 0, 2);
        long[] pages = new long[]{pageA, pageB};
        int[] pageSizes = new int[]{4, 4};
        this.write(swapper, 0L, pages, pageSizes, 2, 2);
        this.putInt(pageA, 0, 3);
        this.putInt(pageB, 0, 4);
        this.write(swapper, 0L, pages, pageSizes, 0, 0);
        this.read(swapper, 0L, pages, pageSizes, 2);
        int[] expectedValues = new int[]{1, 2};
        int[] actualValues = new int[]{this.getInt(pageA, 0), this.getInt(pageB, 0)};
        Assertions.assertThat((int[])actualValues).isEqualTo((Object)expectedValues);
    }

    @Test
    void mustDeleteFileIfClosedWithCloseAndDelete() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        swapper.closeAndDelete();
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> this.createSwapper(factory, file, 4, NO_CALLBACK, false, false, true), (String)"should not have been able to create a page swapper for non-existing file");
    }

    protected final PageSwapperFactory createSwapperFactory(FileSystemAbstraction fileSystem) {
        return this.swapperFactory(fileSystem);
    }

    protected long createPage(int cachePageSize) {
        long address = this.mman.allocateAligned((long)(cachePageSize + 4), 1L);
        UnsafeUtil.putInt((long)address, (int)cachePageSize);
        return address + 4L;
    }

    protected void clear(long address) {
        byte b = 0;
        for (int i = 0; i < this.cachePageSize(); ++i) {
            UnsafeUtil.putByte((long)(address + (long)i), (byte)b);
        }
    }

    protected PageSwapper createSwapper(PageSwapperFactory factory, Path path, int filePageSize, PageEvictionCallback callback, boolean createIfNotExist) throws IOException {
        return this.createSwapper(factory, path, filePageSize, callback, createIfNotExist, false, true);
    }

    protected PageSwapper createSwapper(PageSwapperFactory factory, Path path, int filePageSize, PageEvictionCallback callback, boolean createIfNotExist, boolean useDirectIO, boolean preallocateStoreFiles) throws IOException {
        return this.createSwapper(factory, path, filePageSize, callback, createIfNotExist, useDirectIO, preallocateStoreFiles, IOController.DISABLED);
    }

    protected PageSwapper createSwapper(PageSwapperFactory factory, Path path, int filePageSize, PageEvictionCallback callback, boolean createIfNotExist, boolean useDirectIO, boolean preallocateStoreFiles, IOController controller) throws IOException {
        PageSwapper swapper = factory.createPageSwapper(path, filePageSize, callback, createIfNotExist, useDirectIO, preallocateStoreFiles, controller, this.swapperSet);
        this.openedSwappers.add(swapper);
        return swapper;
    }

    protected int sizeOfAsInt(long page) {
        return UnsafeUtil.getInt((long)(page - 4L));
    }

    protected void putInt(long address, int offset, int value) {
        UnsafeUtil.putInt((long)(address + (long)offset), (int)value);
    }

    protected int getInt(long address, int offset) {
        return UnsafeUtil.getInt((long)(address + (long)offset));
    }

    protected void putLong(long address, int offset, long value) {
        UnsafeUtil.putLong((long)(address + (long)offset), (long)value);
    }

    protected long getLong(long address, int offset) {
        return UnsafeUtil.getLong((long)(address + (long)offset));
    }

    protected byte getByte(long address, int offset) {
        return UnsafeUtil.getByte((long)(address + (long)offset));
    }

    private long write(PageSwapper swapper, int filePageId, long address) throws IOException {
        return swapper.write((long)filePageId, address);
    }

    private long read(PageSwapper swapper, int filePageId, long address) throws IOException {
        return swapper.read((long)filePageId, address);
    }

    private long read(PageSwapper swapper, long startFilePageId, long[] pages, int[] pageSizes, int length) throws IOException {
        if (length == 0) {
            return 0L;
        }
        return swapper.read(startFilePageId, pages, pageSizes, length);
    }

    private long write(PageSwapper swapper, long startFilePageId, long[] pages, int[] pageSizes, int length, int affectedPages) throws IOException {
        if (length == 0) {
            return 0L;
        }
        return swapper.write(startFilePageId, pages, pageSizes, length, affectedPages);
    }

    private int cachePageSize() {
        return 32;
    }

    private long createPage() {
        return this.createPage(this.cachePageSize());
    }

    private PageSwapper createSwapperAndFile(PageSwapperFactory factory, Path path) throws IOException {
        return this.createSwapperAndFile(factory, path, this.cachePageSize());
    }

    private PageSwapper createSwapperAndFile(PageSwapperFactory factory, Path path, boolean useDirectIO) throws IOException {
        return this.createSwapper(factory, path, this.cachePageSize(), NO_CALLBACK, true, useDirectIO, true);
    }

    private PageSwapper createSwapperAndFile(PageSwapperFactory factory, Path path, int filePageSize, boolean useDirectIO) throws IOException {
        return this.createSwapper(factory, path, filePageSize, NO_CALLBACK, true, useDirectIO, true);
    }

    private PageSwapper createSwapperAndFile(PageSwapperFactory factory, Path path, int filePageSize) throws IOException {
        return this.createSwapper(factory, path, filePageSize, NO_CALLBACK, true, false, true);
    }

    private Path file(String filename) throws IOException {
        Path file = this.testDir.file(filename);
        this.mkdirs(file.getParent());
        return file;
    }

    private long sizeOfAsLong(long page) {
        return this.sizeOfAsInt(page);
    }
}

