/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.pagecache.randomharness;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ObjectAssert;
import org.eclipse.collections.api.set.primitive.LongSet;
import org.eclipse.collections.impl.factory.primitive.LongSets;
import org.neo4j.io.pagecache.PageCursor;
import org.neo4j.io.pagecache.PagedFile;
import org.neo4j.io.pagecache.TinyLockManager;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.io.pagecache.impl.muninn.MuninnPageCache;
import org.neo4j.io.pagecache.randomharness.Action;
import org.neo4j.io.pagecache.randomharness.Command;
import org.neo4j.io.pagecache.randomharness.Record;
import org.neo4j.io.pagecache.randomharness.RecordFormat;

class CommandPrimer {
    private final Random rng;
    private final MuninnPageCache cache;
    private final Path[] files;
    private final Map<Path, PagedFile> fileMap;
    private final Map<Path, List<Integer>> recordsWrittenTo;
    private final List<Path> mappedFiles;
    private final Set<Path> filesTouched;
    private final int filePageCount;
    private final int filePageSize;
    private final RecordFormat recordFormat;
    private final int maxRecordCount;
    private final int recordsPerPage;
    private final TinyLockManager recordLocks;

    CommandPrimer(Random rng, MuninnPageCache cache, Path[] files, Map<Path, PagedFile> fileMap, int filePageCount, int filePageSize, RecordFormat recordFormat) {
        this.rng = rng;
        this.cache = cache;
        this.files = files;
        this.fileMap = fileMap;
        this.filePageCount = filePageCount;
        this.filePageSize = filePageSize;
        this.recordFormat = recordFormat;
        this.mappedFiles = new ArrayList<Path>();
        this.mappedFiles.addAll(fileMap.keySet());
        this.filesTouched = new HashSet<Path>();
        this.filesTouched.addAll(this.mappedFiles);
        this.recordsWrittenTo = new HashMap<Path, List<Integer>>();
        this.recordsPerPage = cache.pageSize() / recordFormat.getRecordSize();
        this.maxRecordCount = filePageCount * this.recordsPerPage;
        this.recordLocks = new TinyLockManager();
        for (Path file : files) {
            this.recordsWrittenTo.put(file, new ArrayList());
        }
    }

    public List<Path> getMappedFiles() {
        return this.mappedFiles;
    }

    public Set<Path> getFilesTouched() {
        return this.filesTouched;
    }

    public Action prime(Command command) {
        switch (command) {
            case FlushCache: {
                return this.flushCache();
            }
            case FlushFile: {
                return this.flushFile();
            }
            case MapFile: {
                return this.mapFile();
            }
            case UnmapFile: {
                return this.unmapFile();
            }
            case ReadRecord: {
                return this.readRecord();
            }
            case WriteRecord: {
                return this.writeRecord();
            }
            case ReadMulti: {
                return this.readMulti();
            }
            case WriteMulti: {
                return this.writeMulti();
            }
        }
        throw new IllegalArgumentException("Unknown command: " + String.valueOf((Object)command));
    }

    private Action flushCache() {
        return new Action(Command.FlushCache, "", new Object[0]){

            @Override
            public void perform() throws Exception {
                CommandPrimer.this.cache.flushAndForce();
            }
        };
    }

    private Action flushFile() {
        if (this.mappedFiles.size() > 0) {
            final Path file = this.mappedFiles.get(this.rng.nextInt(this.mappedFiles.size()));
            return new Action(Command.FlushFile, "[file=%s]", new Object[]{file.getFileName()}){

                @Override
                public void perform() throws Exception {
                    PagedFile pagedFile = CommandPrimer.this.fileMap.get(file);
                    if (pagedFile != null) {
                        pagedFile.flushAndForce();
                    }
                }
            };
        }
        return new Action(Command.FlushFile, "[no files mapped to flush]", new Object[0]){

            @Override
            public void perform() {
            }
        };
    }

    private Action mapFile() {
        final Path file = this.files[this.rng.nextInt(this.files.length)];
        this.mappedFiles.add(file);
        this.filesTouched.add(file);
        return new Action(Command.MapFile, "[file=%s]", new Object[]{file}){

            @Override
            public void perform() throws Exception {
                CommandPrimer.this.fileMap.put(file, CommandPrimer.this.cache.map(file, CommandPrimer.this.filePageSize, "neo4j"));
            }
        };
    }

    private Action unmapFile() {
        if (this.mappedFiles.size() > 0) {
            final Path file = this.mappedFiles.remove(this.rng.nextInt(this.mappedFiles.size()));
            return new Action(Command.UnmapFile, "[file=%s]", new Object[]{file}){

                @Override
                public void perform() {
                    CommandPrimer.this.fileMap.get(file).close();
                }
            };
        }
        return null;
    }

    private Action readRecord() {
        return this.buildReadRecord(null);
    }

    private Action writeRecord() {
        return this.buildWriteAction(null, (LongSet)LongSets.immutable.empty());
    }

    private Action readMulti() {
        int count = this.rng.nextInt(5) + 1;
        Action action = null;
        for (int i = 0; i < count; ++i) {
            action = this.buildReadRecord(action);
        }
        return action;
    }

    private Action writeMulti() {
        int count = this.rng.nextInt(5) + 1;
        Action action = null;
        for (int i = 0; i < count; ++i) {
            action = this.buildWriteAction(action, (LongSet)LongSets.immutable.empty());
        }
        return action;
    }

    private Action buildReadRecord(Action innerAction) {
        int mappedFilesCount = this.mappedFiles.size();
        if (mappedFilesCount == 0) {
            return innerAction;
        }
        Path file = this.mappedFiles.get(this.rng.nextInt(mappedFilesCount));
        List<Integer> recordsWritten = this.recordsWrittenTo.get(file);
        int recordId = recordsWritten.isEmpty() ? this.rng.nextInt(this.maxRecordCount) : recordsWritten.get(this.rng.nextInt(recordsWritten.size())).intValue();
        int pageId = recordId / this.recordsPerPage;
        int pageOffset = recordId % this.recordsPerPage * this.recordFormat.getRecordSize();
        Record expectedRecord = this.recordFormat.createRecord(file, recordId, pageId, pageOffset);
        return new ReadAction(file, recordId, pageId, pageOffset, expectedRecord, innerAction);
    }

    private Action buildWriteAction(Action innerAction, LongSet forbiddenRecordIds) {
        int recordId;
        int mappedFilesCount = this.mappedFiles.size();
        if (mappedFilesCount == 0) {
            return innerAction;
        }
        Path file = this.mappedFiles.get(this.rng.nextInt(mappedFilesCount));
        this.filesTouched.add(file);
        while (forbiddenRecordIds.contains((long)(recordId = this.rng.nextInt(this.filePageCount * this.recordsPerPage)))) {
        }
        this.recordsWrittenTo.get(file).add(recordId);
        int pageId = recordId / this.recordsPerPage;
        int pageOffset = recordId % this.recordsPerPage * this.recordFormat.getRecordSize();
        Record record = this.recordFormat.createRecord(file, recordId, pageId, pageOffset);
        return new WriteAction(file, recordId, pageId, pageOffset, record, innerAction);
    }

    private class WriteAction
    extends Action {
        private final Path file;
        private final int recordId;
        private final int pageId;
        private final int pageOffset;
        private final Record record;

        WriteAction(Path path, int recordId, int pageId, int pageOffset, Record record, Action innerAction) {
            super(Command.WriteRecord, innerAction, "[file=%s, recordId=%s, pageId=%s, pageOffset=%s, record=%s]", path, recordId, pageId, pageOffset, record);
            this.file = path;
            this.recordId = recordId;
            this.pageId = pageId;
            this.pageOffset = pageOffset;
            this.record = record;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        @Override
        public void perform() throws Exception {
            PagedFile pagedFile = CommandPrimer.this.fileMap.get(this.file);
            if (pagedFile != null) {
                PageCursor cursor;
                if (!CommandPrimer.this.recordLocks.tryLock(this.recordId)) return;
                try {
                    cursor = pagedFile.io((long)this.pageId, 2, CursorContext.NULL);
                    try {
                        if (cursor.next()) {
                            cursor.setOffset(this.pageOffset);
                            CommandPrimer.this.recordFormat.write(this.record, cursor);
                            this.performInnerAction();
                        }
                    }
                    finally {
                        if (cursor != null) {
                            cursor.close();
                        }
                    }
                }
                finally {
                    CommandPrimer.this.recordLocks.unlock(this.recordId);
                }
                try {
                    cursor = pagedFile.io((long)this.pageId, 2, CursorContext.NULL);
                    try {
                        if (!cursor.next()) return;
                        cursor.setOffset(this.pageOffset);
                        Record actualRecord = CommandPrimer.this.recordFormat.readRecord(cursor);
                        ((ObjectAssert)Assertions.assertThat((Object)actualRecord).as(this.toString(), new Object[0])).isEqualTo((Object)this.record);
                        return;
                    }
                    finally {
                        if (cursor != null) {
                            cursor.close();
                        }
                    }
                }
                finally {
                    CommandPrimer.this.recordLocks.unlock(this.recordId);
                }
            }
            this.performInnerAction();
        }
    }

    private class ReadAction
    extends Action {
        private final Path file;
        private final int pageId;
        private final int pageOffset;
        private final Record expectedRecord;

        ReadAction(Path file, int recordId, int pageId, int pageOffset, Record expectedRecord, Action innerAction) {
            super(Command.ReadRecord, innerAction, "[file=%s, recordId=%s, pageId=%s, pageOffset=%s, expectedRecord=%s]", file, recordId, pageId, pageOffset, expectedRecord);
            this.file = file;
            this.pageId = pageId;
            this.pageOffset = pageOffset;
            this.expectedRecord = expectedRecord;
        }

        @Override
        public void perform() throws Exception {
            PagedFile pagedFile = CommandPrimer.this.fileMap.get(this.file);
            if (pagedFile != null) {
                try (PageCursor cursor = pagedFile.io((long)this.pageId, 1, CursorContext.NULL);){
                    if (cursor.next()) {
                        cursor.setOffset(this.pageOffset);
                        Record actualRecord = CommandPrimer.this.recordFormat.readRecord(cursor);
                        ((ObjectAssert)Assertions.assertThat((Object)actualRecord).as(this.toString(), new Object[0])).isIn(new Object[]{this.expectedRecord, CommandPrimer.this.recordFormat.zeroRecord()});
                        this.performInnerAction();
                    }
                }
            }
        }
    }
}

