/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.pagecache;

import java.io.IOException;
import java.nio.channels.ClosedChannelException;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.DisabledOnOs;
import org.junit.jupiter.api.condition.EnabledOnOs;
import org.junit.jupiter.api.condition.OS;
import org.neo4j.internal.unsafe.UnsafeUtil;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.mem.MemoryAllocator;
import org.neo4j.io.pagecache.IOController;
import org.neo4j.io.pagecache.PageEvictionCallback;
import org.neo4j.io.pagecache.PageSwapper;
import org.neo4j.io.pagecache.PageSwapperFactory;
import org.neo4j.io.pagecache.impl.muninn.SwapperSet;
import org.neo4j.memory.LocalMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;
import org.neo4j.util.concurrent.Futures;

@TestDirectoryExtension
public abstract class PageSwapperTest {
    @Inject
    protected TestDirectory testDir;
    public static final long X = -3819410105021120785L;
    public static final long Y = 6846544296635974449L;
    public static final int Z = -16843010;
    protected static final PageEvictionCallback NO_CALLBACK = filePageId -> {};
    private static final int cachePageSize = 32;
    private final ConcurrentLinkedQueue<PageSwapper> openedSwappers = new ConcurrentLinkedQueue();
    private final MemoryAllocator mman = MemoryAllocator.createAllocator((long)ByteUnit.KibiByte.toBytes(32L), (MemoryTracker)new LocalMemoryTracker());
    private final SwapperSet swapperSet = new SwapperSet();

    protected abstract PageSwapperFactory swapperFactory(FileSystemAbstraction var1);

    protected abstract void mkdirs(Path var1) throws IOException;

    @BeforeEach
    @AfterEach
    void clearStrayInterrupts() {
        Thread.interrupted();
    }

    @AfterEach
    void closeOpenedPageSwappers() throws Exception {
        PageSwapper swapper;
        IOException exception = null;
        while ((swapper = this.openedSwappers.poll()) != null) {
            try {
                swapper.close();
            }
            catch (IOException e) {
                if (exception == null) {
                    exception = e;
                    continue;
                }
                exception.addSuppressed(e);
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    protected abstract FileSystemAbstraction getFs();

    @Test
    void readMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        PageSwapperTest.putInt(page, 0, 1);
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Assertions.assertThat((long)PageSwapperTest.write(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        PageSwapperTest.putInt(page, 0, 0);
        Thread.currentThread().interrupt();
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(1);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(1);
    }

    @Test
    void vectoredReadMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        PageSwapperTest.putInt(page, 0, 1);
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Assertions.assertThat((long)PageSwapperTest.write(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        PageSwapperTest.putInt(page, 0, 0);
        Thread.currentThread().interrupt();
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0L, new long[]{page}, new int[]{PageSwapperTest.cachePageSize()}, 1)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(1);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0L, new long[]{page}, new int[]{PageSwapperTest.cachePageSize()}, 1)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(1);
    }

    @Test
    void writeMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        PageSwapperTest.putInt(page, 0, 1);
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        Assertions.assertThat((long)PageSwapperTest.write(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        PageSwapperTest.putInt(page, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(1);
        Assertions.assertThat((long)PageSwapperTest.write(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        PageSwapperTest.putInt(page, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(1);
    }

    @Test
    void vectoredWriteMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        PageSwapperTest.putInt(page, 0, 1);
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        Assertions.assertThat((long)PageSwapperTest.write(swapper, 0L, new long[]{page}, new int[]{PageSwapperTest.cachePageSize()}, 1, 1)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        PageSwapperTest.putInt(page, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(1);
        Assertions.assertThat((long)PageSwapperTest.write(swapper, 0L, new long[]{page}, new int[]{PageSwapperTest.cachePageSize()}, 1, 1)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
        PageSwapperTest.putInt(page, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0, page)).isEqualTo(PageSwapperTest.sizeOfAsLong(page));
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(1);
    }

    @Test
    void forcingMustNotSwallowInterrupts() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        swapper.force();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
    }

    @Test
    void mustReopenChannelWhenReadFailsWithAsynchronousCloseException() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        long page = this.createPage();
        PageSwapperTest.putLong(page, 0, -3819410105021120785L);
        PageSwapperTest.putLong(page, 8, 6846544296635974449L);
        PageSwapperTest.putInt(page, 16, -16843010);
        PageSwapperTest.write(swapper, 0, page);
        Thread.currentThread().interrupt();
        PageSwapperTest.read(swapper, 0, page);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 8)).isEqualTo(6846544296635974449L);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 16)).isEqualTo(-16843010);
        swapper.force();
    }

    @Test
    void mustReopenChannelWhenVectoredReadFailsWithAsynchronousCloseException() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        long page = this.createPage();
        PageSwapperTest.putLong(page, 0, -3819410105021120785L);
        PageSwapperTest.putLong(page, 8, 6846544296635974449L);
        PageSwapperTest.putInt(page, 16, -16843010);
        PageSwapperTest.write(swapper, 0, page);
        Thread.currentThread().interrupt();
        PageSwapperTest.read(swapper, 0L, new long[]{page}, new int[]{PageSwapperTest.cachePageSize()}, 1);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 8)).isEqualTo(6846544296635974449L);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 16)).isEqualTo(-16843010);
        swapper.force();
    }

    @Test
    void mustReopenChannelWhenWriteFailsWithAsynchronousCloseException() throws Exception {
        long page = this.createPage();
        PageSwapperTest.putLong(page, 0, -3819410105021120785L);
        PageSwapperTest.putLong(page, 8, 6846544296635974449L);
        PageSwapperTest.putInt(page, 16, -16843010);
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        PageSwapperTest.write(swapper, 0, page);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        swapper.force();
        PageSwapperTest.clear(page);
        PageSwapperTest.read(swapper, 0, page);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 8)).isEqualTo(6846544296635974449L);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 16)).isEqualTo(-16843010);
    }

    @Test
    void mustReopenChannelWhenVectoredWriteFailsWithAsynchronousCloseException() throws Exception {
        long page = this.createPage();
        PageSwapperTest.putLong(page, 0, -3819410105021120785L);
        PageSwapperTest.putLong(page, 8, 6846544296635974449L);
        PageSwapperTest.putInt(page, 16, -16843010);
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        Thread.currentThread().interrupt();
        PageSwapperTest.write(swapper, 0L, new long[]{page}, new int[]{PageSwapperTest.cachePageSize()}, 1, 1);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        swapper.force();
        PageSwapperTest.clear(page);
        PageSwapperTest.read(swapper, 0, page);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 8)).isEqualTo(6846544296635974449L);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 16)).isEqualTo(-16843010);
    }

    @Test
    void mustReopenChannelWhenForceFailsWithAsynchronousCloseException() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        for (int i = 0; i < 10; ++i) {
            Thread.currentThread().interrupt();
            swapper.force();
            org.junit.jupiter.api.Assertions.assertTrue((boolean)Thread.interrupted());
        }
    }

    @Test
    void readMustNotReopenExplicitlyClosedChannel() throws Exception {
        String filename = "a";
        Path file = this.file(filename);
        long page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        PageSwapperTest.write(swapper, 0, page);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> PageSwapperTest.read(swapper, 0, page));
    }

    @Test
    void vectoredReadMustNotReopenExplicitlyClosedChannel() throws Exception {
        String filename = "a";
        Path file = this.file(filename);
        long page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        PageSwapperTest.write(swapper, 0, page);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> PageSwapperTest.read(swapper, 0L, new long[]{page}, new int[]{PageSwapperTest.cachePageSize()}, 1));
    }

    @Test
    void writeMustNotReopenExplicitlyClosedChannel() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> PageSwapperTest.write(swapper, 0, page));
    }

    @Test
    void vectoredWriteMustNotReopenExplicitlyClosedChannel() throws Exception {
        Path file = this.file("a");
        long page = this.createPage();
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> PageSwapperTest.write(swapper, 0L, new long[]{page}, new int[]{PageSwapperTest.cachePageSize()}, 1, 1));
    }

    @Test
    void forceMustNotReopenExplicitlyClosedChannel() throws Exception {
        Path file = this.file("a");
        PageSwapperFactory swapperFactory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(swapperFactory, file);
        swapper.close();
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> ((PageSwapper)swapper).force());
    }

    @Test
    void mustNotOverwriteDataInOtherFiles() throws Exception {
        Path fileA = this.file("a");
        Path fileB = this.file("b");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapperA = this.createSwapperAndFile(factory, fileA);
        PageSwapper swapperB = this.createSwapperAndFile(factory, fileB);
        long page = this.createPage();
        PageSwapperTest.clear(page);
        PageSwapperTest.putLong(page, 0, -3819410105021120785L);
        PageSwapperTest.write(swapperA, 0, page);
        PageSwapperTest.putLong(page, 8, 6846544296635974449L);
        PageSwapperTest.write(swapperB, 0, page);
        PageSwapperTest.clear(page);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 0)).isEqualTo(0L);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 8)).isEqualTo(0L);
        PageSwapperTest.read(swapperA, 0, page);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 0)).isEqualTo(-3819410105021120785L);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 8)).isEqualTo(0L);
    }

    @Test
    void swapperCantPreallocateWhenConfigured() throws IOException {
        Path file = this.file("notPreallocatedFile");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        try (PageSwapper swapper = this.createSwapper(factory, file, PageSwapperTest.cachePageSize(), NO_CALLBACK, true, false, false);){
            org.junit.jupiter.api.Assertions.assertFalse((boolean)swapper.canAllocate());
        }
    }

    @Test
    void mustRunEvictionCallbackOnEviction() throws Exception {
        AtomicLong callbackFilePageId = new AtomicLong();
        PageEvictionCallback callback = callbackFilePageId::set;
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, file, PageSwapperTest.cachePageSize(), callback, true, false, true);
        swapper.evicted(42L);
        Assertions.assertThat((long)callbackFilePageId.get()).isEqualTo(42L);
    }

    @Test
    void mustNotIssueEvictionCallbacksAfterSwapperHasBeenClosed() throws Exception {
        AtomicBoolean gotCallback = new AtomicBoolean();
        PageEvictionCallback callback = filePageId -> gotCallback.set(true);
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapper(factory, file, PageSwapperTest.cachePageSize(), callback, true, false, true);
        swapper.close();
        swapper.evicted(42L);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)gotCallback.get());
    }

    @Test
    void mustThrowExceptionIfFileDoesNotExist() {
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        org.junit.jupiter.api.Assertions.assertThrows(NoSuchFileException.class, () -> this.createSwapper(factory, this.file("does not exist"), PageSwapperTest.cachePageSize(), NO_CALLBACK, false, false, true));
    }

    @Test
    void mustCreateNonExistingFileWithCreateFlag() throws Exception {
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper pageSwapper = this.createSwapperAndFile(factory, this.file("does not exist"));
        long page = this.createPage();
        PageSwapperTest.putLong(page, 0, -3819410105021120785L);
        PageSwapperTest.write(pageSwapper, 0, page);
        PageSwapperTest.clear(page);
        PageSwapperTest.read(pageSwapper, 0, page);
        Assertions.assertThat((long)PageSwapperTest.getLong(page, 0)).isEqualTo(-3819410105021120785L);
    }

    @Test
    void truncatedFilesMustBeEmpty() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(-1L);
        long page = this.createPage();
        PageSwapperTest.putInt(page, 0, -889275714);
        PageSwapperTest.write(swapper, 10, page);
        PageSwapperTest.clear(page);
        PageSwapperTest.read(swapper, 10, page);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(-889275714);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(10L);
        swapper.close();
        swapper = this.createSwapper(factory, file, PageSwapperTest.cachePageSize(), NO_CALLBACK, false, false, true);
        PageSwapperTest.clear(page);
        PageSwapperTest.read(swapper, 10, page);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(-889275714);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(10L);
        swapper.truncate();
        PageSwapperTest.clear(page);
        PageSwapperTest.read(swapper, 10, page);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(0);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(-1L);
        swapper.close();
        swapper = this.createSwapper(factory, file, PageSwapperTest.cachePageSize(), NO_CALLBACK, false, false, true);
        PageSwapperTest.clear(page);
        PageSwapperTest.read(swapper, 10, page);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(0);
        Assertions.assertThat((long)swapper.getLastPageId()).isEqualTo(-1L);
        swapper.close();
    }

    @Test
    void positionedVectoredWriteMustFlushAllBuffersInOrder() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        long pageC = this.createPage(4);
        long pageD = this.createPage(4);
        PageSwapperTest.putInt(pageA, 0, 2);
        PageSwapperTest.putInt(pageB, 0, 3);
        PageSwapperTest.putInt(pageC, 0, 4);
        PageSwapperTest.putInt(pageD, 0, 5);
        PageSwapperTest.write(swapper, 1L, new long[]{pageA, pageB, pageC, pageD}, new int[]{4, 4, 4, 4}, 4, 4);
        long result = this.createPage(4);
        PageSwapperTest.read(swapper, 0, result);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(0);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 1, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(2);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 2, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(3);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 3, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(4);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 4, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(5);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 5, result)).isEqualTo(0L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(0);
    }

    @Test
    void positionedVectoredWriteMustFlushAllBuffersOfDifferentSizeInOrder() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(8);
        long pageC = this.createPage(12);
        PageSwapperTest.putInt(pageA, 0, 2);
        PageSwapperTest.putInt(pageB, 0, 3);
        PageSwapperTest.putInt(pageB, 4, 4);
        PageSwapperTest.putInt(pageC, 0, 5);
        PageSwapperTest.putInt(pageC, 4, 6);
        PageSwapperTest.putInt(pageC, 8, 7);
        org.junit.jupiter.api.Assertions.assertEquals((long)24L, (long)PageSwapperTest.write(swapper, 0L, new long[]{pageA, pageB, pageC}, new int[]{4, 8, 12}, 3, 6));
        long result = this.createPage(4);
        PageSwapperTest.read(swapper, 0, result);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(2);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 1, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(3);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 2, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(4);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 3, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(5);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 4, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(6);
        PageSwapperTest.putInt(result, 0, 0);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 5, result)).isEqualTo(4L);
        Assertions.assertThat((int)PageSwapperTest.getInt(result, 0)).isEqualTo(7);
    }

    @Test
    void positionedVectoredReadMustFillAllBuffersOfDifferentSizesInOrder() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage();
        PageSwapperTest.putInt(output, 0, 2);
        PageSwapperTest.write(swapper, 1, output);
        PageSwapperTest.putInt(output, 0, 3);
        PageSwapperTest.write(swapper, 2, output);
        PageSwapperTest.putInt(output, 0, 4);
        PageSwapperTest.write(swapper, 3, output);
        PageSwapperTest.putInt(output, 0, 5);
        PageSwapperTest.write(swapper, 4, output);
        PageSwapperTest.putInt(output, 0, 6);
        PageSwapperTest.write(swapper, 5, output);
        PageSwapperTest.putInt(output, 0, 7);
        PageSwapperTest.write(swapper, 6, output);
        long pageA = this.createPage(4);
        long pageB = this.createPage(8);
        long pageC = this.createPage(12);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 1L, new long[]{pageA, pageB, pageC}, new int[]{4, 8, 12}, 3)).isEqualTo(24L);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageA, 0)).isEqualTo(2);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageB, 0)).isEqualTo(3);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageB, 4)).isEqualTo(4);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageC, 0)).isEqualTo(5);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageC, 4)).isEqualTo(6);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageC, 8)).isEqualTo(7);
    }

    @Test
    void positionedVectoredReadMustFillAllBuffersInOrder() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage();
        PageSwapperTest.putInt(output, 0, 2);
        PageSwapperTest.write(swapper, 1, output);
        PageSwapperTest.putInt(output, 0, 3);
        PageSwapperTest.write(swapper, 2, output);
        PageSwapperTest.putInt(output, 0, 4);
        PageSwapperTest.write(swapper, 3, output);
        PageSwapperTest.putInt(output, 0, 5);
        PageSwapperTest.write(swapper, 4, output);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        long pageC = this.createPage(4);
        long pageD = this.createPage(4);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 1L, new long[]{pageA, pageB, pageC, pageD}, new int[]{4, 4, 4, 4}, 4)).isEqualTo(16L);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageA, 0)).isEqualTo(2);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageB, 0)).isEqualTo(3);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageC, 0)).isEqualTo(4);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageD, 0)).isEqualTo(5);
    }

    @Test
    void positionedVectoredReadFromEmptyFileMustFillPagesWithZeros() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long page = this.createPage(4);
        PageSwapperTest.putInt(page, 0, 1);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0L, new long[]{page}, new int[]{4}, 1)).isEqualTo(0L);
        Assertions.assertThat((int)PageSwapperTest.getInt(page, 0)).isEqualTo(0);
    }

    @Test
    void positionedVectoredReadBeyondEndOfFileMustFillPagesWithZeros() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage(4);
        PageSwapperTest.putInt(output, 0, -1);
        PageSwapperTest.write(swapper, 0L, new long[]{output, output, output}, new int[]{4, 4, 4}, 3, 3);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        PageSwapperTest.putInt(pageA, 0, -1);
        PageSwapperTest.putInt(pageB, 0, -1);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 3L, new long[]{pageA, pageB}, new int[]{4, 4}, 2)).isEqualTo(0L);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageA, 0)).isEqualTo(0);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageB, 0)).isEqualTo(0);
    }

    @Test
    void positionedVectoredReadBeyondEndOfFileMustFillPagesWithZerosForBuffersWithDifferentSizes() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage(4);
        PageSwapperTest.putInt(output, 0, 42);
        PageSwapperTest.write(swapper, 0L, new long[]{output, output, output, output}, new int[]{4, 4, 4, 4}, 4, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(8);
        long pageC = this.createPage(12);
        PageSwapperTest.putInt(pageA, 0, -1);
        PageSwapperTest.putInt(pageB, 0, -1);
        PageSwapperTest.putInt(pageB, 4, -1);
        PageSwapperTest.putInt(pageC, 0, -1);
        PageSwapperTest.putInt(pageC, 4, -1);
        PageSwapperTest.putInt(pageC, 8, -1);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0L, new long[]{pageA, pageB, pageC}, new int[]{4, 8, 12}, 3)).isEqualTo(16L);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageA, 0)).isEqualTo(42);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageB, 0)).isEqualTo(42);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageB, 4)).isEqualTo(42);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageC, 0)).isEqualTo(42);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageC, 4)).isEqualTo(0);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageC, 8)).isEqualTo(0);
    }

    @Test
    void positionedVectoredReadWhereLastPageExtendBeyondEndOfFileMustHaveRemainderZeroFilled() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage(4);
        PageSwapperTest.putInt(output, 0, -1);
        PageSwapperTest.write(swapper, 0L, new long[]{output, output, output, output, output}, new int[]{4, 4, 4, 4, 4}, 5, 5);
        swapper.close();
        swapper = this.createSwapper(factory, file, 8, NO_CALLBACK, false, false, true);
        long pageA = this.createPage(8);
        long pageB = this.createPage(8);
        PageSwapperTest.putLong(pageA, 0, -3819410105021120785L);
        PageSwapperTest.putLong(pageB, 0, 6846544296635974449L);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 1L, new long[]{pageA, pageB}, new int[]{8, 8}, 2)).isIn(new Object[]{12L, 16L});
        Assertions.assertThat((long)PageSwapperTest.getLong(pageA, 0)).isEqualTo(-1L);
        Assertions.assertThat((byte)PageSwapperTest.getByte(pageB, 0)).isEqualTo((byte)-1);
        Assertions.assertThat((byte)PageSwapperTest.getByte(pageB, 1)).isEqualTo((byte)-1);
        Assertions.assertThat((byte)PageSwapperTest.getByte(pageB, 2)).isEqualTo((byte)-1);
        Assertions.assertThat((byte)PageSwapperTest.getByte(pageB, 3)).isEqualTo((byte)-1);
        Assertions.assertThat((byte)PageSwapperTest.getByte(pageB, 4)).isEqualTo((byte)0);
        Assertions.assertThat((byte)PageSwapperTest.getByte(pageB, 5)).isEqualTo((byte)0);
        Assertions.assertThat((byte)PageSwapperTest.getByte(pageB, 6)).isEqualTo((byte)0);
        Assertions.assertThat((byte)PageSwapperTest.getByte(pageB, 7)).isEqualTo((byte)0);
    }

    @Test
    void positionedVectoredReadWhereSecondLastPageExtendBeyondEndOfFileMustHaveRestZeroFilled() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long output = this.createPage(4);
        PageSwapperTest.putInt(output, 0, 1);
        PageSwapperTest.write(swapper, 0, output);
        PageSwapperTest.putInt(output, 0, 2);
        PageSwapperTest.write(swapper, 1, output);
        PageSwapperTest.putInt(output, 0, 3);
        PageSwapperTest.write(swapper, 2, output);
        swapper.close();
        swapper = this.createSwapper(factory, file, 8, NO_CALLBACK, false, false, true);
        long pageA = this.createPage(8);
        long pageB = this.createPage(8);
        long pageC = this.createPage(8);
        PageSwapperTest.putInt(pageA, 0, -1);
        PageSwapperTest.putInt(pageB, 0, -1);
        PageSwapperTest.putInt(pageC, 0, -1);
        Assertions.assertThat((long)PageSwapperTest.read(swapper, 0L, new long[]{pageA, pageB, pageC}, new int[]{8, 8, 8}, 3)).isIn(new Object[]{12L, 16L});
        Assertions.assertThat((int)PageSwapperTest.getInt(pageA, 0)).isEqualTo(1);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageA, 4)).isEqualTo(2);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageB, 0)).isEqualTo(3);
        Assertions.assertThat((int)PageSwapperTest.getInt(pageB, 4)).isEqualTo(0);
        Assertions.assertThat((long)PageSwapperTest.getLong(pageC, 0)).isEqualTo(0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void concurrentPositionedVectoredReadsAndWritesMustNotInterfere() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        int pageCount = 100;
        int iterations = 20000;
        CountDownLatch startLatch = new CountDownLatch(1);
        long output = this.createPage(4);
        for (int i = 0; i < 100; ++i) {
            PageSwapperTest.putInt(output, 0, i + 1);
            PageSwapperTest.write(swapper, i, output);
        }
        Callable<Void> work = () -> {
            int i;
            ThreadLocalRandom rng = ThreadLocalRandom.current();
            int length = 10;
            int pageSize = 4;
            long[] pages = new long[length];
            int[] sizes = new int[length];
            for (i = 0; i < length; ++i) {
                pages[i] = this.createPage(pageSize);
                sizes[i] = pageSize;
            }
            startLatch.await();
            for (i = 0; i < 20000; ++i) {
                long startFilePageId = rng.nextLong(0L, 100 - pages.length);
                if (rng.nextBoolean()) {
                    long bytesRead = PageSwapperTest.read(swapper, startFilePageId, pages, sizes, pages.length);
                    Assertions.assertThat((long)bytesRead).isEqualTo((long)pages.length * 4L);
                    for (int j = 0; j < pages.length; ++j) {
                        int expectedValue = (int)((long)(1 + j) + startFilePageId);
                        int actualValue = PageSwapperTest.getInt(pages[j], 0);
                        Assertions.assertThat((int)actualValue).isEqualTo(expectedValue);
                    }
                    continue;
                }
                for (int j = 0; j < pages.length; ++j) {
                    int value = (int)((long)(1 + j) + startFilePageId);
                    PageSwapperTest.putInt(pages[j], 0, value);
                }
                Assertions.assertThat((long)PageSwapperTest.write(swapper, startFilePageId, pages, sizes, pages.length, pages.length)).isEqualTo((long)pages.length * 4L);
            }
            return null;
        };
        int threads = 8;
        ExecutorService executor = null;
        try {
            executor = Executors.newFixedThreadPool(threads, r -> {
                Thread thread = Executors.defaultThreadFactory().newThread(r);
                thread.setDaemon(true);
                return thread;
            });
            ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>(threads);
            for (int i = 0; i < threads; ++i) {
                futures.add(executor.submit(work));
            }
            startLatch.countDown();
            Futures.getAll(futures);
        }
        finally {
            if (executor != null) {
                executor.shutdown();
            }
        }
    }

    @Test
    void mustThrowNullPointerExceptionFromReadWhenPageArrayIsNull() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long page = this.createPage(4);
        int[] pageSizes = new int[]{4, 4, 4, 4};
        PageSwapperTest.write(swapper, 0L, new long[]{page, page, page, page}, pageSizes, 4, 4);
        Assertions.assertThatThrownBy(() -> PageSwapperTest.read(swapper, 0L, null, pageSizes, 4), (String)"vectored read with null array should have thrown", (Object[])new Object[0]).extracting(ExceptionUtils::getRootCause).isInstanceOf(NullPointerException.class);
    }

    @Test
    void mustThrowNullPointerExceptionFromWriteWhenPageArrayIsNull() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        Assertions.assertThatThrownBy(() -> PageSwapperTest.write(swapper, 0L, null, null, 4, 4), (String)"vectored write with null array should have thrown", (Object[])new Object[0]).extracting(ExceptionUtils::getRootCause).isInstanceOf(NullPointerException.class);
    }

    @Test
    void readMustThrowForNegativeFilePageIds() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> PageSwapperTest.read(swapper, -1, this.createPage(4)));
    }

    @Test
    @DisabledOnOs(value={OS.LINUX})
    void directIOAllowedOnlyOnLinux() throws IOException {
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        Path file = this.file("file");
        IllegalArgumentException e = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.createSwapperAndFile(factory, file, true));
        Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Linux"});
    }

    @Test
    @EnabledOnOs(value={OS.LINUX})
    void doNotAllowDirectIOForPagesNotMultipleOfBlockSize() throws IOException {
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        Path file = this.file("file");
        this.checkUnsupportedPageSize(factory, file, 17);
        this.checkUnsupportedPageSize(factory, file, 115);
        this.checkUnsupportedPageSize(factory, file, 218);
        this.checkUnsupportedPageSize(factory, file, 419);
        this.checkUnsupportedPageSize(factory, file, 524);
        this.checkUnsupportedPageSize(factory, file, 1023);
        this.checkUnsupportedPageSize(factory, file, 4097);
    }

    private void checkUnsupportedPageSize(PageSwapperFactory factory, Path path, int pageSize) {
        IllegalArgumentException e = (IllegalArgumentException)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> this.createSwapperAndFile(factory, path, pageSize, true));
        Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"block"});
    }

    @Test
    void writeMustThrowForNegativeFilePageIds() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> PageSwapperTest.write(swapper, -1, this.createPage(4)));
    }

    @Test
    void vectoredReadMustThrowForNegativeFilePageIds() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> PageSwapperTest.read(swapper, -1L, new long[]{this.createPage(4), this.createPage(4)}, new int[]{4, 4}, 2));
    }

    @Test
    void vectoredWriteMustThrowForNegativeFilePageIds() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> PageSwapperTest.write(swapper, -1L, new long[]{this.createPage(4), this.createPage(4)}, new int[]{4, 4}, 2, 2));
    }

    @Test
    void vectoredReadMustReadNothingWhenLengthIsZero() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        PageSwapperTest.putInt(pageA, 0, 1);
        PageSwapperTest.putInt(pageB, 0, 2);
        long[] pages = new long[]{pageA, pageB};
        int[] pageSizes = new int[]{4, 4};
        PageSwapperTest.write(swapper, 0L, pages, pageSizes, 2, 2);
        PageSwapperTest.putInt(pageA, 0, 3);
        PageSwapperTest.putInt(pageB, 0, 4);
        PageSwapperTest.read(swapper, 0L, pages, pageSizes, 0);
        int[] expectedValues = new int[]{3, 4};
        int[] actualValues = new int[]{PageSwapperTest.getInt(pageA, 0), PageSwapperTest.getInt(pageB, 0)};
        Assertions.assertThat((int[])actualValues).isEqualTo((Object)expectedValues);
    }

    @Test
    void vectoredWriteMustWriteNothingWhenLengthIsZero() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        long pageA = this.createPage(4);
        long pageB = this.createPage(4);
        PageSwapperTest.putInt(pageA, 0, 1);
        PageSwapperTest.putInt(pageB, 0, 2);
        long[] pages = new long[]{pageA, pageB};
        int[] pageSizes = new int[]{4, 4};
        PageSwapperTest.write(swapper, 0L, pages, pageSizes, 2, 2);
        PageSwapperTest.putInt(pageA, 0, 3);
        PageSwapperTest.putInt(pageB, 0, 4);
        PageSwapperTest.write(swapper, 0L, pages, pageSizes, 0, 0);
        PageSwapperTest.read(swapper, 0L, pages, pageSizes, 2);
        int[] expectedValues = new int[]{1, 2};
        int[] actualValues = new int[]{PageSwapperTest.getInt(pageA, 0), PageSwapperTest.getInt(pageB, 0)};
        Assertions.assertThat((int[])actualValues).isEqualTo((Object)expectedValues);
    }

    @Test
    void mustDeleteFileIfClosedWithCloseAndDelete() throws Exception {
        Path file = this.file("file");
        PageSwapperFactory factory = this.createSwapperFactory(this.getFs());
        PageSwapper swapper = this.createSwapperAndFile(factory, file, 4);
        swapper.closeAndDelete();
        org.junit.jupiter.api.Assertions.assertThrows(IOException.class, () -> this.createSwapper(factory, file, 4, NO_CALLBACK, false, false, true), (String)"should not have been able to create a page swapper for non-existing file");
    }

    protected final PageSwapperFactory createSwapperFactory(FileSystemAbstraction fileSystem) {
        return this.swapperFactory(fileSystem);
    }

    protected long createPage(int cachePageSize) {
        long address = this.mman.allocateAligned((long)(cachePageSize + 4), 1L);
        UnsafeUtil.putInt((long)address, (int)cachePageSize);
        return address + 4L;
    }

    protected static void clear(long address) {
        byte b = 0;
        for (int i = 0; i < PageSwapperTest.cachePageSize(); ++i) {
            UnsafeUtil.putByte((long)(address + (long)i), (byte)b);
        }
    }

    protected PageSwapper createSwapper(PageSwapperFactory factory, Path path, int filePageSize, PageEvictionCallback callback, boolean createIfNotExist) throws IOException {
        return this.createSwapper(factory, path, filePageSize, callback, createIfNotExist, false, true);
    }

    protected PageSwapper createSwapper(PageSwapperFactory factory, Path path, int filePageSize, PageEvictionCallback callback, boolean createIfNotExist, boolean useDirectIO, boolean preallocateStoreFiles) throws IOException {
        return this.createSwapper(factory, path, filePageSize, callback, createIfNotExist, useDirectIO, preallocateStoreFiles, IOController.DISABLED);
    }

    protected PageSwapper createSwapper(PageSwapperFactory factory, Path path, int filePageSize, PageEvictionCallback callback, boolean createIfNotExist, boolean useDirectIO, boolean preallocateStoreFiles, IOController controller) throws IOException {
        PageSwapper swapper = factory.createPageSwapper(path, filePageSize, callback, createIfNotExist, useDirectIO, preallocateStoreFiles, controller, this.swapperSet);
        this.openedSwappers.add(swapper);
        return swapper;
    }

    protected static int sizeOfAsInt(long page) {
        return UnsafeUtil.getInt((long)(page - 4L));
    }

    protected static void putInt(long address, int offset, int value) {
        UnsafeUtil.putInt((long)(address + (long)offset), (int)value);
    }

    protected static int getInt(long address, int offset) {
        return UnsafeUtil.getInt((long)(address + (long)offset));
    }

    protected static void putLong(long address, int offset, long value) {
        UnsafeUtil.putLong((long)(address + (long)offset), (long)value);
    }

    protected static long getLong(long address, int offset) {
        return UnsafeUtil.getLong((long)(address + (long)offset));
    }

    protected static byte getByte(long address, int offset) {
        return UnsafeUtil.getByte((long)(address + (long)offset));
    }

    private static long write(PageSwapper swapper, int filePageId, long address) throws IOException {
        return swapper.write((long)filePageId, address);
    }

    private static long read(PageSwapper swapper, int filePageId, long address) throws IOException {
        return swapper.read((long)filePageId, address);
    }

    private static long read(PageSwapper swapper, long startFilePageId, long[] pages, int[] pageSizes, int length) throws IOException {
        if (length == 0) {
            return 0L;
        }
        return swapper.read(startFilePageId, pages, pageSizes, length);
    }

    private static long write(PageSwapper swapper, long startFilePageId, long[] pages, int[] pageSizes, int length, int affectedPages) throws IOException {
        if (length == 0) {
            return 0L;
        }
        return swapper.write(startFilePageId, pages, pageSizes, length, affectedPages);
    }

    private static int cachePageSize() {
        return 32;
    }

    private long createPage() {
        return this.createPage(PageSwapperTest.cachePageSize());
    }

    private PageSwapper createSwapperAndFile(PageSwapperFactory factory, Path path) throws IOException {
        return this.createSwapperAndFile(factory, path, PageSwapperTest.cachePageSize());
    }

    private PageSwapper createSwapperAndFile(PageSwapperFactory factory, Path path, boolean useDirectIO) throws IOException {
        return this.createSwapper(factory, path, PageSwapperTest.cachePageSize(), NO_CALLBACK, true, useDirectIO, true);
    }

    private PageSwapper createSwapperAndFile(PageSwapperFactory factory, Path path, int filePageSize, boolean useDirectIO) throws IOException {
        return this.createSwapper(factory, path, filePageSize, NO_CALLBACK, true, useDirectIO, true);
    }

    private PageSwapper createSwapperAndFile(PageSwapperFactory factory, Path path, int filePageSize) throws IOException {
        return this.createSwapper(factory, path, filePageSize, NO_CALLBACK, true, false, true);
    }

    private Path file(String filename) throws IOException {
        Path file = this.testDir.file(filename);
        this.mkdirs(file.getParent());
        return file;
    }

    private static long sizeOfAsLong(long page) {
        return PageSwapperTest.sizeOfAsInt(page);
    }
}

