/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.schema;

import java.io.File;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.graphdb.Result;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.kernel.database.DatabaseMemoryTrackers;
import org.neo4j.kernel.impl.api.index.IndexPopulationJob;
import org.neo4j.kernel.impl.api.index.IndexingService;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogAssertions;
import org.neo4j.logging.LogProvider;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.monitoring.Monitors;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;
import org.neo4j.values.storable.RandomValues;

@TestDirectoryExtension
class IndexPopulationIT {
    @Inject
    private TestDirectory directory;
    private static GraphDatabaseAPI database;
    private static ExecutorService executorService;
    private static AssertableLogProvider logProvider;
    private static DatabaseManagementService managementService;

    IndexPopulationIT() {
    }

    @BeforeEach
    void setUp() {
        logProvider = new AssertableLogProvider(true);
        managementService = new TestDatabaseManagementServiceBuilder(this.directory.homeDir()).setInternalLogProvider((LogProvider)logProvider).build();
        database = (GraphDatabaseAPI)managementService.database("neo4j");
        executorService = Executors.newCachedThreadPool();
    }

    @AfterEach
    void tearDown() {
        executorService.shutdown();
        managementService.shutdown();
    }

    @Test
    void trackMemoryOnIndexPopulation() throws InterruptedException {
        Label nodeLabel = Label.label((String)"nodeLabel");
        String propertyName = "testProperty";
        String indexName = "testIndex";
        try (Transaction transaction = database.beginTx();){
            Node node = transaction.createNode(new Label[]{nodeLabel});
            node.setProperty(propertyName, (Object)RandomStringUtils.randomAscii((int)1024));
            transaction.commit();
        }
        Monitors monitors = (Monitors)database.getDependencyResolver().resolveDependency(Monitors.class);
        DatabaseMemoryTrackers memoryTrackers = (DatabaseMemoryTrackers)database.getDependencyResolver().resolveDependency(DatabaseMemoryTrackers.class);
        final MemoryTracker otherTracker = memoryTrackers.getOtherTracker();
        long estimatedHeapBefore = otherTracker.estimatedHeapMemory();
        long usedNativeBefore = otherTracker.usedNativeMemory();
        final AtomicLong peakUsage = new AtomicLong();
        final CountDownLatch populationJobCompleted = new CountDownLatch(1);
        monitors.addMonitorListener((Object)new IndexingService.MonitorAdapter(){

            public void populationCompleteOn(IndexDescriptor descriptor) {
                peakUsage.set(Math.max(otherTracker.usedNativeMemory(), peakUsage.get()));
            }

            public void populationJobCompleted(long peakDirectMemoryUsage) {
                populationJobCompleted.countDown();
            }
        }, new String[0]);
        try (Transaction transaction = database.beginTx();){
            transaction.schema().indexFor(nodeLabel).on(propertyName).withName(indexName).create();
            transaction.commit();
        }
        this.waitForOnlineIndexes();
        populationJobCompleted.await();
        long nativeMemoryAfterIndexCompletion = otherTracker.usedNativeMemory();
        Assertions.assertEquals((long)estimatedHeapBefore, (long)otherTracker.estimatedHeapMemory());
        Assertions.assertEquals((long)usedNativeBefore, (long)nativeMemoryAfterIndexCompletion);
        LogAssertions.assertThat((long)peakUsage.get()).isGreaterThan(nativeMemoryAfterIndexCompletion);
    }

    @Test
    void indexCreationDoNotBlockQueryExecutions() throws Exception {
        Label nodeLabel = Label.label((String)"nodeLabel");
        try (Transaction transaction = database.beginTx();){
            transaction.createNode(new Label[]{nodeLabel});
            transaction.commit();
        }
        transaction = database.beginTx();
        try {
            transaction.schema().indexFor(Label.label((String)"testLabel")).on("testProperty").create();
            Future<Number> countFuture = executorService.submit(this.countNodes());
            Assertions.assertEquals((int)1, (int)countFuture.get().intValue());
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void createIndexesFromDifferentTransactionsWithoutBlocking() throws ExecutionException, InterruptedException {
        long numberOfIndexesBeforeTest = this.countIndexes();
        Label nodeLabel = Label.label((String)"nodeLabel2");
        String testProperty = "testProperty";
        try (Transaction transaction = database.beginTx();){
            transaction.schema().indexFor(Label.label((String)"testLabel2")).on(testProperty).create();
            Future<?> creationFuture = executorService.submit(this.createIndexForLabelAndProperty(nodeLabel, testProperty));
            creationFuture.get();
            transaction.commit();
        }
        this.waitForOnlineIndexes();
        Assertions.assertEquals((long)(numberOfIndexesBeforeTest + 2L), (long)this.countIndexes());
    }

    @Test
    void indexCreationDoNotBlockWritesOnOtherLabel() throws ExecutionException, InterruptedException {
        Label markerLabel = Label.label((String)"testLabel3");
        Label nodesLabel = Label.label((String)"testLabel4");
        try (Transaction transaction = database.beginTx();){
            transaction.schema().indexFor(markerLabel).on("testProperty").create();
            Future<?> creation = executorService.submit(this.createNodeWithLabel(nodesLabel));
            creation.get();
            transaction.commit();
        }
        transaction = database.beginTx();
        try (ResourceIterator nodes = transaction.findNodes(nodesLabel);){
            Assertions.assertEquals((long)1L, (long)Iterators.count((Iterator)nodes));
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void shutdownDatabaseDuringIndexPopulations() {
        AssertableLogProvider assertableLogProvider = new AssertableLogProvider(true);
        File storeDir = this.directory.directory("shutdownDbTest", new String[0]);
        Label testLabel = Label.label((String)"testLabel");
        String propertyName = "testProperty";
        DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(storeDir).setInternalLogProvider((LogProvider)assertableLogProvider).build();
        GraphDatabaseService shutDownDb = managementService.database("neo4j");
        this.prePopulateDatabase(shutDownDb, testLabel, propertyName);
        try (Transaction transaction = shutDownDb.beginTx();){
            transaction.schema().indexFor(testLabel).on(propertyName).create();
            transaction.commit();
        }
        managementService.shutdown();
        LogAssertions.assertThat((AssertableLogProvider)assertableLogProvider).forClass(IndexPopulationJob.class).forLevel(AssertableLogProvider.Level.ERROR).doesNotHaveAnyLogs();
    }

    @Test
    void mustLogPhaseTracker() {
        Label nodeLabel = Label.label((String)"testLabel5");
        String key = "key";
        String value = "hej";
        try (Transaction transaction = database.beginTx();){
            transaction.createNode(new Label[]{nodeLabel}).setProperty(key, (Object)value);
            transaction.commit();
        }
        try (Transaction tx = database.beginTx();){
            tx.schema().indexFor(nodeLabel).on(key).create();
            tx.commit();
        }
        this.waitForOnlineIndexes();
        tx = database.beginTx();
        try {
            ResourceIterator nodes = tx.findNodes(nodeLabel, key, (Object)value);
            long nodeCount = Iterators.count((Iterator)nodes);
            Assertions.assertEquals((long)1L, (long)nodeCount, (String)"expected exactly one hit in index but was ");
            nodes.close();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        LogAssertions.assertThat((AssertableLogProvider)logProvider).forClass(IndexPopulationJob.class).forLevel(AssertableLogProvider.Level.INFO).containsMessages(new String[]{"TIME/PHASE Final:"});
    }

    private void prePopulateDatabase(GraphDatabaseService database, Label testLabel, String propertyName) {
        RandomValues randomValues = RandomValues.create();
        for (int j = 0; j < 10000; ++j) {
            try (Transaction transaction = database.beginTx();){
                Node node = transaction.createNode(new Label[]{testLabel});
                Object property = randomValues.nextValue().asObject();
                node.setProperty(propertyName, property);
                transaction.commit();
                continue;
            }
        }
    }

    private Runnable createNodeWithLabel(Label label) {
        return () -> {
            try (Transaction transaction = database.beginTx();){
                transaction.createNode(new Label[]{label});
                transaction.commit();
            }
        };
    }

    private long countIndexes() {
        try (Transaction transaction = database.beginTx();){
            long l = Iterables.count((Iterable)transaction.schema().getIndexes());
            return l;
        }
    }

    private Runnable createIndexForLabelAndProperty(Label label, String propertyKey) {
        return () -> {
            try (Transaction transaction = database.beginTx();){
                transaction.schema().indexFor(label).on(propertyKey).create();
                transaction.commit();
            }
            this.waitForOnlineIndexes();
        };
    }

    private void waitForOnlineIndexes() {
        try (Transaction transaction = database.beginTx();){
            transaction.schema().awaitIndexesOnline(1L, TimeUnit.MINUTES);
            transaction.commit();
        }
    }

    private Callable<Number> countNodes() {
        return () -> {
            try (Transaction transaction = database.beginTx();){
                Result result = transaction.execute("MATCH (n) RETURN count(n) as count");
                Map resultMap = result.next();
                Number number = (Number)resultMap.get("count");
                return number;
            }
        };
    }
}

