/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphdb;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.MultipleFoundException;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.graphdb.SchemaAcceptanceTest;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.spatial.CRS;
import org.neo4j.graphdb.spatial.Point;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.helpers.collection.MapUtil;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.test.extension.DbmsExtension;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.mockito.mock.SpatialMocks;
import org.neo4j.values.storable.CoordinateReferenceSystem;
import org.neo4j.values.storable.PointValue;
import org.neo4j.values.storable.Values;

@DbmsExtension
class IndexingAcceptanceTest {
    private static final String LONG_STRING = "a long string that has to be stored in dynamic records";
    @Inject
    private GraphDatabaseAPI db;
    private Label LABEL1;
    private Label LABEL2;
    private Label LABEL3;

    IndexingAcceptanceTest() {
    }

    @BeforeEach
    void setupLabels(TestInfo testInfo) {
        this.LABEL1 = Label.label((String)("LABEL1-" + testInfo.getDisplayName()));
        this.LABEL2 = Label.label((String)("LABEL2-" + testInfo.getDisplayName()));
        this.LABEL3 = Label.label((String)("LABEL3-" + testInfo.getDisplayName()));
    }

    @Test
    void shouldInterpretPropertyAsChangedEvenIfPropertyMovesFromOneRecordToAnother() {
        Node myNode;
        long smallValue = 10L;
        long bigValue = 0x4000000000000000L;
        try (Transaction tx = this.db.beginTx();){
            myNode = tx.createNode(new Label[]{this.LABEL1});
            myNode.setProperty("pad0", (Object)true);
            myNode.setProperty("pad1", (Object)true);
            myNode.setProperty("pad2", (Object)true);
            myNode.setProperty("key", (Object)smallValue);
            tx.commit();
        }
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"key"});
        tx = this.db.beginTx();
        try {
            tx.getNodeById(myNode.getId()).setProperty("key", (Object)bigValue);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "key", bigValue, transaction)).containsOnly((Object[])new Node[]{myNode});
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "key", smallValue, transaction)).isEmpty();
        }
    }

    @Test
    void shouldUseDynamicPropertiesToIndexANodeWhenAddedAlongsideExistingPropertiesInASeparateTransaction() {
        Node myNode;
        long id;
        try (Transaction tx = this.db.beginTx();){
            Node myNode2 = tx.createNode();
            id = myNode2.getId();
            myNode2.setProperty("key0", (Object)true);
            myNode2.setProperty("key1", (Object)true);
            tx.commit();
        }
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"key2"});
        try (Transaction tx = this.db.beginTx();){
            myNode = tx.getNodeById(id);
            myNode.addLabel(this.LABEL1);
            myNode.setProperty("key2", (Object)LONG_STRING);
            myNode.setProperty("key3", (Object)LONG_STRING);
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            myNode = transaction.getNodeById(myNode.getId());
            org.junit.jupiter.api.Assertions.assertEquals((Object)LONG_STRING, (Object)myNode.getProperty("key2"));
            org.junit.jupiter.api.Assertions.assertEquals((Object)LONG_STRING, (Object)myNode.getProperty("key3"));
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "key2", LONG_STRING, transaction)).containsOnly((Object[])new Node[]{myNode});
        }
    }

    @Test
    void searchingForNodeByPropertyShouldWorkWithoutIndex() {
        Node myNode = this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Hawking"}), this.LABEL1);
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Hawking", transaction)).containsOnly((Object[])new Node[]{myNode});
        }
    }

    @Test
    void searchingUsesIndexWhenItExists() {
        Node myNode = this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Hawking"}), this.LABEL1);
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Hawking", transaction)).containsOnly((Object[])new Node[]{myNode});
        }
    }

    @Test
    void shouldCorrectlyUpdateIndexesWhenChangingLabelsAndPropertyAtTheSameTime() {
        Node myNode = this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Hawking"}), this.LABEL1, this.LABEL2);
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL2, (String[])new String[]{"name"});
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL3, (String[])new String[]{"name"});
        try (Transaction tx = this.db.beginTx();){
            myNode = tx.getNodeById(myNode.getId());
            myNode.removeLabel(this.LABEL1);
            myNode.addLabel(this.LABEL3);
            myNode.setProperty("name", (Object)"Einstein");
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            myNode = transaction.getNodeById(myNode.getId());
            org.junit.jupiter.api.Assertions.assertEquals((Object)"Einstein", (Object)myNode.getProperty("name"));
            Assertions.assertThat((Iterable)myNode.getLabels()).containsOnly((Object[])new Label[]{this.LABEL2, this.LABEL3});
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Hawking", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Einstein", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL2, "name", "Hawking", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL2, "name", "Einstein", transaction)).containsOnly((Object[])new Node[]{myNode});
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL3, "name", "Hawking", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL3, "name", "Einstein", transaction)).containsOnly((Object[])new Node[]{myNode});
            transaction.commit();
        }
    }

    @Test
    void shouldCorrectlyUpdateIndexesWhenChangingLabelsAndPropertyMultipleTimesAllAtOnce() {
        Node myNode = this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Hawking"}), this.LABEL1, this.LABEL2);
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL2, (String[])new String[]{"name"});
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL3, (String[])new String[]{"name"});
        try (Transaction tx = this.db.beginTx();){
            myNode = tx.getNodeById(myNode.getId());
            myNode.addLabel(this.LABEL3);
            myNode.setProperty("name", (Object)"Einstein");
            myNode.removeLabel(this.LABEL1);
            myNode.setProperty("name", (Object)"Feynman");
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            myNode = transaction.getNodeById(myNode.getId());
            org.junit.jupiter.api.Assertions.assertEquals((Object)"Feynman", (Object)myNode.getProperty("name"));
            Assertions.assertThat((Iterable)myNode.getLabels()).containsOnly((Object[])new Label[]{this.LABEL2, this.LABEL3});
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Hawking", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Einstein", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Feynman", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL2, "name", "Hawking", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL2, "name", "Einstein", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL2, "name", "Feynman", transaction)).containsOnly((Object[])new Node[]{myNode});
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL3, "name", "Hawking", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL3, "name", "Einstein", transaction)).isEmpty();
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL3, "name", "Feynman", transaction)).containsOnly((Object[])new Node[]{myNode});
            transaction.commit();
        }
    }

    @Test
    void searchingByLabelAndPropertyReturnsEmptyWhenMissingLabelOrProperty() {
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Hawking", transaction)).isEmpty();
        }
    }

    @Test
    void shouldSeeIndexUpdatesWhenQueryingOutsideTransaction() {
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        Node firstNode = this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Mattias"}), this.LABEL1);
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Mattias", transaction)).containsOnly((Object[])new Node[]{firstNode});
        }
        Node secondNode = this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Taylor"}), this.LABEL1);
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.LABEL1, "name", "Taylor", transaction)).containsOnly((Object[])new Node[]{secondNode});
        }
    }

    @Test
    void createdNodeShouldShowUpWithinTransaction() {
        long sizeAfterDelete;
        long sizeBeforeDelete;
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        Node firstNode = this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Mattias"}), this.LABEL1);
        try (Transaction tx = this.db.beginTx();){
            sizeBeforeDelete = Iterators.count((Iterator)tx.findNodes(this.LABEL1, "name", (Object)"Mattias"));
            tx.getNodeById(firstNode.getId()).delete();
            sizeAfterDelete = Iterators.count((Iterator)tx.findNodes(this.LABEL1, "name", (Object)"Mattias"));
            tx.commit();
        }
        Assertions.assertThat((long)sizeBeforeDelete).isOne();
        Assertions.assertThat((long)sizeAfterDelete).isZero();
    }

    @Test
    void deletedNodeShouldShowUpWithinTransaction() {
        long sizeAfterDelete;
        long sizeBeforeDelete;
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        Node firstNode = this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Mattias"}), this.LABEL1);
        try (Transaction tx = this.db.beginTx();){
            sizeBeforeDelete = Iterators.count((Iterator)tx.findNodes(this.LABEL1, "name", (Object)"Mattias"));
            tx.getNodeById(firstNode.getId()).delete();
            sizeAfterDelete = Iterators.count((Iterator)tx.findNodes(this.LABEL1, "name", (Object)"Mattias"));
            tx.commit();
        }
        Assertions.assertThat((long)sizeBeforeDelete).isOne();
        Assertions.assertThat((long)sizeAfterDelete).isZero();
    }

    @Test
    void createdNodeShouldShowUpInIndexQuery() {
        long sizeAfterDelete;
        long sizeBeforeDelete;
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Mattias"}), this.LABEL1);
        try (Transaction transaction = this.db.beginTx();){
            sizeBeforeDelete = Iterators.count((Iterator)transaction.findNodes(this.LABEL1, "name", (Object)"Mattias"));
        }
        this.createNode((GraphDatabaseService)this.db, MapUtil.map((Object[])new Object[]{"name", "Mattias"}), this.LABEL1);
        transaction = this.db.beginTx();
        try {
            sizeAfterDelete = Iterators.count((Iterator)transaction.findNodes(this.LABEL1, "name", (Object)"Mattias"));
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
        Assertions.assertThat((long)sizeBeforeDelete).isOne();
        Assertions.assertThat((long)sizeAfterDelete).isEqualTo(2L);
    }

    @Test
    void shouldBeAbleToQuerySupportedPropertyTypes() {
        String property = "name";
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{property});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, "A String");
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, true);
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, false);
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, (byte)56);
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, Character.valueOf('z'));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, (short)12);
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, 12);
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, 12L);
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, Float.valueOf(12.0f));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, 12.0);
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, SpatialMocks.mockPoint((double)12.3, (double)45.6, (CRS)SpatialMocks.mockWGS84()));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, SpatialMocks.mockPoint((double)123.0, (double)456.0, (CRS)SpatialMocks.mockCartesian()));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, SpatialMocks.mockPoint((double)12.3, (double)45.6, (double)100.0, (CRS)SpatialMocks.mockWGS84_3D()));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, SpatialMocks.mockPoint((double)123.0, (double)456.0, (double)789.0, (CRS)SpatialMocks.mockCartesian_3D()));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.WGS84, (double[])new double[]{12.3, 45.6}));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.Cartesian, (double[])new double[]{123.0, 456.0}));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.WGS84_3D, (double[])new double[]{12.3, 45.6, 100.0}));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.Cartesian_3D, (double[])new double[]{123.0, 456.0, 789.0}));
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new String[]{"A String"});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new boolean[]{true});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Boolean[]{false});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new byte[]{56});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Byte[]{(byte)57});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new char[]{'a'});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Character[]{Character.valueOf('b')});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new short[]{12});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Short[]{(short)13});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new int[]{14});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Integer[]{15});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new long[]{16L});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Long[]{17L});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new float[]{18.0f});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Float[]{Float.valueOf(19.0f)});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new double[]{20.0});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Double[]{21.0});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Point[]{SpatialMocks.mockPoint((double)12.3, (double)45.6, (CRS)SpatialMocks.mockWGS84())});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Point[]{SpatialMocks.mockPoint((double)123.0, (double)456.0, (CRS)SpatialMocks.mockCartesian())});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Point[]{SpatialMocks.mockPoint((double)12.3, (double)45.6, (double)100.0, (CRS)SpatialMocks.mockWGS84_3D())});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new Point[]{SpatialMocks.mockPoint((double)123.0, (double)456.0, (double)789.0, (CRS)SpatialMocks.mockCartesian_3D())});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new PointValue[]{Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.WGS84, (double[])new double[]{12.3, 45.6})});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new PointValue[]{Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.Cartesian, (double[])new double[]{123.0, 456.0})});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new PointValue[]{Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.WGS84_3D, (double[])new double[]{12.3, 45.6, 100.0})});
        this.assertCanCreateAndFind((GraphDatabaseService)this.db, this.LABEL1, property, new PointValue[]{Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.Cartesian_3D, (double[])new double[]{123.0, 456.0, 789.0})});
    }

    @Test
    void shouldRetrieveMultipleNodesWithSameValueFromIndex() {
        Node node2;
        Node node1;
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        try (Transaction tx = this.db.beginTx();){
            node1 = tx.createNode(new Label[]{this.LABEL1});
            node1.setProperty("name", (Object)"Stefan");
            node2 = tx.createNode(new Label[]{this.LABEL1});
            node2.setProperty("name", (Object)"Stefan");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            ResourceIterator result = tx.findNodes(this.LABEL1, "name", (Object)"Stefan");
            org.junit.jupiter.api.Assertions.assertEquals((Object)Iterators.asSet((Object[])new Node[]{node1, node2}), (Object)Iterators.asSet((Iterator)result));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldThrowWhenMultipleResultsForSingleNode() {
        SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)this.LABEL1, (String[])new String[]{"name"});
        try (Transaction tx = this.db.beginTx();){
            Node node1 = tx.createNode(new Label[]{this.LABEL1});
            node1.setProperty("name", (Object)"Stefan");
            Node node2 = tx.createNode(new Label[]{this.LABEL1});
            node2.setProperty("name", (Object)"Stefan");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            MultipleFoundException e = (MultipleFoundException)org.junit.jupiter.api.Assertions.assertThrows(MultipleFoundException.class, () -> tx.findNode(this.LABEL1, "name", (Object)"Stefan"));
            Assertions.assertThat((Throwable)e).hasMessage(String.format("Found multiple nodes with label: '%s', property name: 'name' and property value: 'Stefan' while only one was expected.", this.LABEL1));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldAddIndexedPropertyToNodeWithDynamicLabels() {
        long nodeId;
        int indexesCount = 20;
        String labelPrefix = "foo";
        String propertyKeyPrefix = "bar";
        String propertyValuePrefix = "baz";
        for (int i = 0; i < indexesCount; ++i) {
            SchemaAcceptanceTest.createIndex((GraphDatabaseService)this.db, (Label)Label.label((String)(labelPrefix + i)), (String[])new String[]{propertyKeyPrefix + i});
        }
        try (Transaction tx = this.db.beginTx();){
            nodeId = tx.createNode().getId();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            Node node = tx.getNodeById(nodeId);
            for (int i = 0; i < indexesCount; ++i) {
                node.addLabel(Label.label((String)(labelPrefix + i)));
                node.setProperty(propertyKeyPrefix + i, (Object)(propertyValuePrefix + i));
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            for (int i = 0; i < indexesCount; ++i) {
                Label label = Label.label((String)(labelPrefix + i));
                String key = propertyKeyPrefix + i;
                String value = propertyValuePrefix + i;
                ResourceIterator nodes = tx.findNodes(label, key, (Object)value);
                org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)Iterators.count((Iterator)nodes));
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private void assertCanCreateAndFind(GraphDatabaseService db, Label label, String propertyKey, Object value) {
        Node created = this.createNode(db, MapUtil.map((Object[])new Object[]{propertyKey, value}), label);
        try (Transaction tx = db.beginTx();){
            Node found = tx.findNode(label, propertyKey, value);
            Assertions.assertThat((Object)found).isEqualTo((Object)created);
            found.delete();
            tx.commit();
        }
    }

    private List<Node> findNodesByLabelAndProperty(Label label, String propertyName, Object value, Transaction transaction) {
        return Iterators.asList((Iterator)transaction.findNodes(label, propertyName, value));
    }

    private Node createNode(GraphDatabaseService db, Map<String, Object> properties, Label ... labels) {
        try (Transaction tx = db.beginTx();){
            Node node = tx.createNode(labels);
            for (Map.Entry<String, Object> property : properties.entrySet()) {
                node.setProperty(property.getKey(), property.getValue());
            }
            tx.commit();
            Node node2 = node;
            return node2;
        }
    }
}

