/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.api.impl.schema;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.common.DependencyResolver;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.kernel.api.InternalIndexState;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.kernel.api.impl.index.storage.DirectoryFactory;
import org.neo4j.kernel.api.impl.schema.LuceneIndexProvider;
import org.neo4j.kernel.api.index.IndexDirectoryStructure;
import org.neo4j.kernel.extension.ExtensionFactory;
import org.neo4j.kernel.extension.ExtensionType;
import org.neo4j.kernel.extension.context.ExtensionContext;
import org.neo4j.kernel.impl.index.schema.AbstractIndexProviderFactory;
import org.neo4j.kernel.impl.index.schema.TokenIndexProviderFactory;
import org.neo4j.kernel.impl.transaction.log.checkpoint.CheckPointer;
import org.neo4j.kernel.impl.transaction.log.checkpoint.SimpleTriggerInfo;
import org.neo4j.kernel.impl.transaction.log.checkpoint.TriggerInfo;
import org.neo4j.kernel.impl.transaction.log.rotation.LogRotation;
import org.neo4j.kernel.impl.transaction.tracing.LogAppendEvent;
import org.neo4j.kernel.impl.transaction.tracing.LogRotateEvents;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.kernel.recovery.RecoveryExtension;
import org.neo4j.monitoring.Monitors;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.EphemeralFileSystemExtension;
import org.neo4j.test.extension.Inject;

@ExtendWith(value={EphemeralFileSystemExtension.class})
class LuceneIndexRecoveryIT {
    private static final String NUM_BANANAS_KEY = "number_of_bananas_owned";
    private static final Label myLabel = Label.label((String)"MyLabel");
    @Inject
    private EphemeralFileSystemAbstraction fs;
    private GraphDatabaseAPI db;
    private DirectoryFactory directoryFactory;
    private DatabaseManagementService managementService;

    LuceneIndexRecoveryIT() {
    }

    @BeforeEach
    void before() {
        this.directoryFactory = new DirectoryFactory.InMemoryDirectoryFactory();
    }

    @AfterEach
    void after() throws Exception {
        if (this.db != null) {
            this.managementService.shutdown();
        }
        this.directoryFactory.close();
    }

    @Test
    void addShouldBeIdempotentWhenDoingRecovery() {
        this.startDb(this.createLuceneIndexFactory());
        IndexDefinition index = this.createIndex(myLabel);
        this.waitForIndex(index);
        long nodeId = this.createNode(myLabel, "12");
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertNotNull((Object)tx.getNodeById(nodeId));
        }
        Assertions.assertEquals((int)1, (int)this.doIndexLookup(myLabel, "12").size());
        this.killDb();
        this.startDb(this.createLuceneIndexFactory());
        tx = this.db.beginTx();
        try {
            Assertions.assertNotNull((Object)tx.getNodeById(nodeId));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        Assertions.assertEquals((int)1, (int)this.doIndexLookup(myLabel, "12").size());
    }

    @Test
    void changeShouldBeIdempotentWhenDoingRecovery() throws Exception {
        this.startDb(this.createLuceneIndexFactory());
        IndexDefinition indexDefinition = this.createIndex(myLabel);
        this.waitForIndex(indexDefinition);
        long node = this.createNode(myLabel, "12");
        this.rotateLogsAndCheckPoint();
        this.updateNode(node, "13");
        this.killDb();
        this.startDb(this.createLuceneIndexFactory());
        Assertions.assertEquals((int)0, (int)this.doIndexLookup(myLabel, "12").size());
        Assertions.assertEquals((int)1, (int)this.doIndexLookup(myLabel, "13").size());
    }

    @Test
    void removeShouldBeIdempotentWhenDoingRecovery() throws Exception {
        this.startDb(this.createLuceneIndexFactory());
        IndexDefinition indexDefinition = this.createIndex(myLabel);
        this.waitForIndex(indexDefinition);
        long node = this.createNode(myLabel, "12");
        this.rotateLogsAndCheckPoint();
        this.deleteNode(node);
        this.killDb();
        this.startDb(this.createLuceneIndexFactory());
        Assertions.assertEquals((int)0, (int)this.doIndexLookup(myLabel, "12").size());
    }

    @Test
    void shouldNotAddTwiceDuringRecoveryIfCrashedDuringPopulation() {
        IndexDefinition index;
        this.startDb(this.createAlwaysInitiallyPopulatingLuceneIndexFactory());
        IndexDefinition indexDefinition = this.createIndex(myLabel);
        this.waitForIndex(indexDefinition);
        long nodeId = this.createNode(myLabel, "12");
        Assertions.assertEquals((int)1, (int)this.doIndexLookup(myLabel, "12").size());
        this.killDb();
        this.startDb(this.createAlwaysInitiallyPopulatingLuceneIndexFactory());
        try (Transaction tx = this.db.beginTx();){
            index = (IndexDefinition)tx.schema().getIndexes().iterator().next();
        }
        this.waitForIndex(index);
        tx = this.db.beginTx();
        try {
            Assertions.assertEquals((Object)"12", (Object)tx.getNodeById(nodeId).getProperty(NUM_BANANAS_KEY));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        Assertions.assertEquals((int)1, (int)this.doIndexLookup(myLabel, "12").size());
    }

    @Test
    void shouldNotUpdateTwiceDuringRecovery() {
        this.startDb(this.createLuceneIndexFactory());
        IndexDefinition indexDefinition = this.createIndex(myLabel);
        this.waitForIndex(indexDefinition);
        long nodeId = this.createNode(myLabel, "12");
        this.updateNode(nodeId, "14");
        this.killDb();
        this.startDb(this.createLuceneIndexFactory());
        Assertions.assertEquals((int)0, (int)this.doIndexLookup(myLabel, "12").size());
        Assertions.assertEquals((int)1, (int)this.doIndexLookup(myLabel, "14").size());
    }

    private void startDb(ExtensionFactory<?> indexProviderFactory) {
        if (this.db != null) {
            this.managementService.shutdown();
        }
        TestDatabaseManagementServiceBuilder factory = new TestDatabaseManagementServiceBuilder();
        factory.setFileSystem((FileSystemAbstraction)this.fs);
        factory.setExtensions(Arrays.asList(indexProviderFactory, new TokenIndexProviderFactory()));
        this.managementService = factory.impermanent().setConfig(GraphDatabaseSettings.default_schema_provider, (Object)LuceneIndexProvider.DESCRIPTOR.name()).build();
        this.db = (GraphDatabaseAPI)this.managementService.database("neo4j");
    }

    private void killDb() {
        if (this.db != null) {
            this.fs = this.fs.snapshot();
            this.managementService.shutdown();
        }
    }

    private void rotateLogsAndCheckPoint() throws IOException {
        DependencyResolver resolver = this.db.getDependencyResolver();
        ((LogRotation)resolver.resolveDependency(LogRotation.class)).rotateLogFile((LogRotateEvents)LogAppendEvent.NULL);
        ((CheckPointer)resolver.resolveDependency(CheckPointer.class)).forceCheckPoint((TriggerInfo)new SimpleTriggerInfo("test"));
    }

    private IndexDefinition createIndex(Label label) {
        try (Transaction tx = this.db.beginTx();){
            IndexDefinition definition = tx.schema().indexFor(label).on(NUM_BANANAS_KEY).create();
            tx.commit();
            IndexDefinition indexDefinition = definition;
            return indexDefinition;
        }
    }

    private void waitForIndex(IndexDefinition definition) {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().awaitIndexOnline(definition, 10L, TimeUnit.MINUTES);
            tx.commit();
        }
    }

    private Set<Node> doIndexLookup(Label myLabel, String value) {
        try (Transaction tx = this.db.beginTx();){
            ResourceIterator iter = tx.findNodes(myLabel, NUM_BANANAS_KEY, (Object)value);
            Set nodes = Iterators.asUniqueSet((Iterator)iter);
            tx.commit();
            Set set = nodes;
            return set;
        }
    }

    private long createNode(Label label, String number) {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode(new Label[]{label});
            node.setProperty(NUM_BANANAS_KEY, (Object)number);
            tx.commit();
            long l = node.getId();
            return l;
        }
    }

    private void updateNode(long nodeId, String value) {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.getNodeById(nodeId);
            node.setProperty(NUM_BANANAS_KEY, (Object)value);
            tx.commit();
        }
    }

    private void deleteNode(long node) {
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(node).delete();
            tx.commit();
        }
    }

    private ExtensionFactory<AbstractIndexProviderFactory.Dependencies> createAlwaysInitiallyPopulatingLuceneIndexFactory() {
        return new PopulatingTestLuceneIndexExtension();
    }

    private ExtensionFactory<AbstractIndexProviderFactory.Dependencies> createLuceneIndexFactory() {
        return new TestLuceneIndexExtension();
    }

    @RecoveryExtension
    private class PopulatingTestLuceneIndexExtension
    extends ExtensionFactory<AbstractIndexProviderFactory.Dependencies> {
        PopulatingTestLuceneIndexExtension() {
            super(ExtensionType.DATABASE, LuceneIndexProvider.DESCRIPTOR.getKey());
        }

        public Lifecycle newInstance(ExtensionContext context, AbstractIndexProviderFactory.Dependencies dependencies) {
            return new LuceneIndexProvider((FileSystemAbstraction)LuceneIndexRecoveryIT.this.fs, LuceneIndexRecoveryIT.this.directoryFactory, IndexDirectoryStructure.directoriesByProvider((Path)context.directory()), new Monitors(), dependencies.getConfig(), dependencies.readOnlyChecker()){

                public InternalIndexState getInitialState(IndexDescriptor descriptor, CursorContext cursorContext) {
                    return InternalIndexState.POPULATING;
                }
            };
        }
    }

    @RecoveryExtension
    private class TestLuceneIndexExtension
    extends ExtensionFactory<AbstractIndexProviderFactory.Dependencies> {
        TestLuceneIndexExtension() {
            super(ExtensionType.DATABASE, LuceneIndexProvider.DESCRIPTOR.getKey());
        }

        public Lifecycle newInstance(ExtensionContext context, AbstractIndexProviderFactory.Dependencies dependencies) {
            return new LuceneIndexProvider((FileSystemAbstraction)LuceneIndexRecoveryIT.this.fs, LuceneIndexRecoveryIT.this.directoryFactory, IndexDirectoryStructure.directoriesByProvider((Path)context.directory()), new Monitors(), dependencies.getConfig(), dependencies.readOnlyChecker());
        }
    }
}

