/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.index;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.graphdb.ConstraintViolationException;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.graphdb.schema.Schema;
import org.neo4j.index.SabotageNativeIndex;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.test.RandomSupport;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.DbmsController;
import org.neo4j.test.extension.DbmsExtension;
import org.neo4j.test.extension.ExtensionCallback;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.RandomExtension;
import org.neo4j.values.storable.CoordinateReferenceSystem;
import org.neo4j.values.storable.Values;

@DbmsExtension
@ExtendWith(value={RandomExtension.class})
public class IndexFailureOnStartupTest {
    private static final Label PERSON = Label.label((String)"Person");
    @Inject
    private RandomSupport random;
    @Inject
    private GraphDatabaseAPI db;
    @Inject
    private DbmsController controller;
    @Inject
    private FileSystemAbstraction fs;

    @Test
    void failedIndexShouldRepairAutomatically() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(PERSON).on("name").create();
            tx.commit();
        }
        this.awaitIndexesOnline(5, TimeUnit.SECONDS);
        this.createNamed(PERSON, "Johan");
        this.sabotageNativeIndexAndRestartDbms();
        this.createNamed(PERSON, "Lars");
        this.awaitIndexesOnline(5, TimeUnit.SECONDS);
        this.indexStateShouldBe(Schema.IndexState.ONLINE);
        this.assertFindsNamed(PERSON, "Lars");
    }

    @Test
    void shouldNotBeAbleToViolateConstraintWhenBackingIndexFailsToOpen() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().constraintFor(PERSON).assertPropertyIsUnique("name").create();
            tx.commit();
        }
        this.createNamed(PERSON, "Lars");
        this.sabotageNativeIndexAndRestartDbms();
        this.createNamed(PERSON, "Johan");
        org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> this.createNamed(PERSON, "Lars"));
        this.indexStateShouldBe(Schema.IndexState.ONLINE);
    }

    private void sabotageNativeIndexAndRestartDbms() {
        this.controller.restartDbms(this.db.databaseName(), builder -> {
            try {
                new SabotageNativeIndex(this.random.random()).run(this.fs, this.db.databaseLayout());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            return builder;
        });
    }

    private Path archiveFile() throws IOException {
        Path[] files;
        Path indexDir = SabotageNativeIndex.nativeIndexDirectoryStructure(this.db.databaseLayout()).rootDirectory();
        try (Stream<Path> list = Files.list(indexDir);){
            files = (Path[])list.filter(path -> Files.isRegularFile(path, new LinkOption[0]) && path.getFileName().toString().startsWith("archive-")).toArray(Path[]::new);
        }
        if (files.length == 0) {
            return null;
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)files.length);
        return files[0];
    }

    private void awaitIndexesOnline(int timeout, TimeUnit unit) {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().awaitIndexesOnline((long)timeout, unit);
            tx.commit();
        }
    }

    private void assertFindsNamed(Label label, String name) {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertNotNull((Object)tx.findNode(label, "name", (Object)name), (String)"Must be able to find node created while index was offline");
            tx.commit();
        }
    }

    private void indexStateShouldBe(Schema.IndexState value) {
        try (Transaction tx = this.db.beginTx();){
            for (IndexDefinition index : tx.schema().getIndexes()) {
                Assertions.assertThat((Comparable)tx.schema().getIndexState(index)).isEqualTo((Object)value);
            }
            tx.commit();
        }
    }

    private void createNamed(Label label, String name) {
        try (Transaction tx = this.db.beginTx();){
            tx.createNode(new Label[]{label}).setProperty("name", (Object)name);
            tx.commit();
        }
    }

    @Nested
    @DbmsExtension(configurationCallback="configure")
    class ArchiveIndex {
        ArchiveIndex() {
        }

        @ExtensionCallback
        void configure(TestDatabaseManagementServiceBuilder builder) {
            builder.setConfig(GraphDatabaseInternalSettings.archive_failed_index, (Object)true);
        }

        @Test
        void shouldArchiveFailedIndex() throws IOException {
            Node node;
            try (Transaction tx = IndexFailureOnStartupTest.this.db.beginTx();){
                node = tx.createNode(new Label[]{PERSON});
                node.setProperty("name", (Object)"Fry");
                tx.commit();
            }
            tx = IndexFailureOnStartupTest.this.db.beginTx();
            try {
                node = tx.createNode(new Label[]{PERSON});
                node.setProperty("name", (Object)Values.pointValue((CoordinateReferenceSystem)CoordinateReferenceSystem.WGS84, (double[])new double[]{1.0, 2.0}));
                tx.commit();
            }
            finally {
                if (tx != null) {
                    tx.close();
                }
            }
            tx = IndexFailureOnStartupTest.this.db.beginTx();
            try {
                tx.schema().constraintFor(PERSON).assertPropertyIsUnique("name").create();
                tx.commit();
            }
            finally {
                if (tx != null) {
                    tx.close();
                }
            }
            Assertions.assertThat((Path)IndexFailureOnStartupTest.this.archiveFile()).isNull();
            IndexFailureOnStartupTest.this.sabotageNativeIndexAndRestartDbms();
            IndexFailureOnStartupTest.this.indexStateShouldBe(Schema.IndexState.ONLINE);
            Assertions.assertThat((Path)IndexFailureOnStartupTest.this.archiveFile()).isNotNull();
        }
    }
}

