/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.schema;

import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.RandomStringUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.exceptions.KernelException;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.graphdb.Result;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexType;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.kernel.api.IndexMonitor;
import org.neo4j.internal.kernel.api.IndexQueryConstraints;
import org.neo4j.internal.kernel.api.IndexReadSession;
import org.neo4j.internal.kernel.api.NodeValueIndexCursor;
import org.neo4j.internal.kernel.api.PropertyIndexQuery;
import org.neo4j.internal.kernel.api.TokenRead;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.database.DatabaseMemoryTrackers;
import org.neo4j.kernel.impl.api.index.IndexPopulationJob;
import org.neo4j.kernel.impl.coreapi.TransactionImpl;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogAssertions;
import org.neo4j.logging.LogProvider;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.monitoring.Monitors;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;
import org.neo4j.values.storable.RandomValues;
import org.neo4j.values.storable.Values;

@TestDirectoryExtension
class IndexPopulationIT {
    @Inject
    private TestDirectory directory;
    private GraphDatabaseAPI database;
    private ExecutorService executorService;
    private AssertableLogProvider logProvider;
    private DatabaseManagementService managementService;
    private Monitors monitors;

    IndexPopulationIT() {
    }

    @BeforeEach
    void setUp() {
        this.monitors = new Monitors();
        this.logProvider = new AssertableLogProvider(true);
        this.managementService = new TestDatabaseManagementServiceBuilder(this.directory.homePath()).setInternalLogProvider((LogProvider)this.logProvider).setMonitors(this.monitors).build();
        this.database = (GraphDatabaseAPI)this.managementService.database("neo4j");
        this.executorService = Executors.newCachedThreadPool();
    }

    @AfterEach
    void tearDown() {
        this.executorService.shutdown();
        this.managementService.shutdown();
    }

    @Test
    void trackMemoryOnIndexPopulation() throws InterruptedException {
        Label nodeLabel = Label.label((String)"nodeLabel");
        String propertyName = "testProperty";
        String indexName = "testIndex";
        try (Transaction transaction = this.database.beginTx();){
            Node node = transaction.createNode(new Label[]{nodeLabel});
            node.setProperty(propertyName, (Object)RandomStringUtils.randomAscii((int)1024));
            transaction.commit();
        }
        Monitors monitors = (Monitors)this.database.getDependencyResolver().resolveDependency(Monitors.class);
        DatabaseMemoryTrackers memoryTrackers = (DatabaseMemoryTrackers)this.database.getDependencyResolver().resolveDependency(DatabaseMemoryTrackers.class);
        final MemoryTracker otherTracker = memoryTrackers.getOtherTracker();
        long estimatedHeapBefore = otherTracker.estimatedHeapMemory();
        long usedNativeBefore = otherTracker.usedNativeMemory();
        final AtomicLong peakUsage = new AtomicLong();
        final CountDownLatch populationJobCompleted = new CountDownLatch(1);
        monitors.addMonitorListener((Object)new IndexMonitor.MonitorAdapter(){

            public void populationCompleteOn(IndexDescriptor descriptor) {
                peakUsage.set(Math.max(otherTracker.usedNativeMemory(), peakUsage.get()));
            }

            public void populationJobCompleted(long peakDirectMemoryUsage) {
                populationJobCompleted.countDown();
            }
        }, new String[0]);
        try (Transaction transaction = this.database.beginTx();){
            transaction.schema().indexFor(nodeLabel).on(propertyName).withName(indexName).create();
            transaction.commit();
        }
        this.waitForOnlineIndexes();
        populationJobCompleted.await();
        long nativeMemoryAfterIndexCompletion = otherTracker.usedNativeMemory();
        org.junit.jupiter.api.Assertions.assertEquals((long)estimatedHeapBefore, (long)otherTracker.estimatedHeapMemory());
        org.junit.jupiter.api.Assertions.assertEquals((long)usedNativeBefore, (long)nativeMemoryAfterIndexCompletion);
        LogAssertions.assertThat((long)peakUsage.get()).isGreaterThan(nativeMemoryAfterIndexCompletion);
    }

    @Test
    void indexCreationDoNotBlockQueryExecutions() throws Exception {
        Label nodeLabel = Label.label((String)"nodeLabel");
        try (Transaction transaction = this.database.beginTx();){
            transaction.createNode(new Label[]{nodeLabel});
            transaction.commit();
        }
        transaction = this.database.beginTx();
        try {
            transaction.schema().indexFor(Label.label((String)"testLabel")).on("testProperty").create();
            Future<Number> countFuture = this.executorService.submit(this.countNodes());
            org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)countFuture.get().intValue());
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void createIndexesFromDifferentTransactionsWithoutBlocking() throws ExecutionException, InterruptedException {
        long numberOfIndexesBeforeTest = this.countIndexes();
        Label nodeLabel = Label.label((String)"nodeLabel2");
        String testProperty = "testProperty";
        try (Transaction transaction = this.database.beginTx();){
            transaction.schema().indexFor(Label.label((String)"testLabel2")).on(testProperty).create();
            Future<?> creationFuture = this.executorService.submit(this.createIndexForLabelAndProperty(nodeLabel, testProperty));
            creationFuture.get();
            transaction.commit();
        }
        this.waitForOnlineIndexes();
        org.junit.jupiter.api.Assertions.assertEquals((long)(numberOfIndexesBeforeTest + 2L), (long)this.countIndexes());
    }

    @Test
    void indexCreationDoNotBlockWritesOnOtherLabel() throws ExecutionException, InterruptedException {
        Label markerLabel = Label.label((String)"testLabel3");
        Label nodesLabel = Label.label((String)"testLabel4");
        try (Transaction transaction = this.database.beginTx();){
            transaction.schema().indexFor(markerLabel).on("testProperty").create();
            Future<?> creation = this.executorService.submit(this.createNodeWithLabel(nodesLabel));
            creation.get();
            transaction.commit();
        }
        transaction = this.database.beginTx();
        try (ResourceIterator nodes = transaction.findNodes(nodesLabel);){
            org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)Iterators.count((Iterator)nodes));
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void shutdownDatabaseDuringIndexPopulations() {
        AssertableLogProvider assertableLogProvider = new AssertableLogProvider(true);
        Path storeDir = this.directory.directory("shutdownDbTest");
        Label testLabel = Label.label((String)"testLabel");
        String propertyName = "testProperty";
        DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(storeDir).setInternalLogProvider((LogProvider)assertableLogProvider).build();
        GraphDatabaseService shutDownDb = managementService.database("neo4j");
        IndexPopulationIT.prePopulateDatabase(shutDownDb, testLabel, propertyName);
        try (Transaction transaction = shutDownDb.beginTx();){
            transaction.schema().indexFor(testLabel).on(propertyName).create();
            transaction.commit();
        }
        managementService.shutdown();
        LogAssertions.assertThat((AssertableLogProvider)assertableLogProvider).forClass(IndexPopulationJob.class).forLevel(AssertableLogProvider.Level.ERROR).doesNotHaveAnyLogs();
    }

    @Test
    void mustLogPhaseTracker() {
        Label nodeLabel = Label.label((String)"testLabel5");
        String key = "key";
        String value = "hej";
        try (Transaction transaction = this.database.beginTx();){
            transaction.createNode(new Label[]{nodeLabel}).setProperty(key, (Object)value);
            transaction.commit();
        }
        try (Transaction tx = this.database.beginTx();){
            tx.schema().indexFor(nodeLabel).on(key).create();
            tx.commit();
        }
        this.waitForOnlineIndexes();
        tx = this.database.beginTx();
        try {
            ResourceIterator nodes = tx.findNodes(nodeLabel, key, (Object)value);
            long nodeCount = Iterators.count((Iterator)nodes);
            org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)nodeCount, (String)"expected exactly one hit in index but was ");
            nodes.close();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).forClass(IndexPopulationJob.class).forLevel(AssertableLogProvider.Level.INFO).containsMessages(new String[]{"TIME/PHASE Final:"});
    }

    @Test
    void concurrentUpdatesPopulationOfManyIndexesOnSameSchema() throws InterruptedException, KernelException {
        Node node;
        Label nodeLabel = Label.label((String)"nodeLabel");
        String propertyName = "testProperty";
        String rangeIndex = "rangeIndex";
        String textIndex = "textIndex";
        String concurrentValue = "concurrentValue";
        CountDownLatch blockLatch = new CountDownLatch(1);
        CountDownLatch signalLatch = new CountDownLatch(1);
        this.monitors.addMonitorListener((Object)new PopulationScanCompleteBlock(rangeIndex, blockLatch, signalLatch), new String[0]);
        try (Transaction transaction = this.database.beginTx();){
            node = transaction.createNode(new Label[]{nodeLabel});
            node.setProperty(propertyName, (Object)"initialValue");
            transaction.commit();
        }
        transaction = this.database.beginTx();
        try {
            transaction.schema().indexFor(nodeLabel).on(propertyName).withIndexType(IndexType.RANGE).withName(rangeIndex).create();
            transaction.schema().indexFor(nodeLabel).on(propertyName).withIndexType(IndexType.TEXT).withName(textIndex).create();
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
        Assertions.assertThat((boolean)signalLatch.await(1L, TimeUnit.MINUTES)).isTrue();
        try (Transaction concurrentUpdater = this.database.beginTx();){
            node = concurrentUpdater.createNode(new Label[]{nodeLabel});
            node.setProperty(propertyName, (Object)concurrentValue);
            concurrentUpdater.commit();
        }
        blockLatch.countDown();
        this.waitForOnlineIndexes();
        LogAssertions.assertThat((boolean)this.nodeValueExistsInIndex(propertyName, concurrentValue, rangeIndex)).isTrue();
        LogAssertions.assertThat((boolean)this.nodeValueExistsInIndex(propertyName, concurrentValue, textIndex)).isTrue();
    }

    private boolean nodeValueExistsInIndex(String propertyName, String value, String indexName) throws KernelException {
        try (Transaction transaction = this.database.beginTx();){
            boolean bl;
            block12: {
                KernelTransaction ktx = ((TransactionImpl)transaction).kernelTransaction();
                TokenRead tokenRead = ktx.tokenRead();
                int propertyId = tokenRead.propertyKey(propertyName);
                PropertyIndexQuery.ExactPredicate query = PropertyIndexQuery.exact((int)propertyId, (Object)Values.utf8Value((byte[])value.getBytes(StandardCharsets.UTF_8)));
                IndexDescriptor index = ktx.schemaRead().indexGetForName(indexName);
                NodeValueIndexCursor cursor = ktx.cursors().allocateNodeValueIndexCursor(CursorContext.NULL, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
                try {
                    IndexReadSession indexSession = ktx.dataRead().indexReadSession(index);
                    ktx.dataRead().nodeIndexSeek(ktx.queryContext(), indexSession, cursor, IndexQueryConstraints.unconstrained(), new PropertyIndexQuery[]{query});
                    bl = cursor.next();
                    if (cursor == null) break block12;
                }
                catch (Throwable throwable) {
                    if (cursor != null) {
                        try {
                            cursor.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                cursor.close();
            }
            return bl;
        }
    }

    private static void prePopulateDatabase(GraphDatabaseService database, Label testLabel, String propertyName) {
        RandomValues randomValues = RandomValues.create();
        try (Transaction transaction = database.beginTx();){
            for (int j = 0; j < 10000; ++j) {
                Node node = transaction.createNode(new Label[]{testLabel});
                Object property = randomValues.nextValue().asObject();
                node.setProperty(propertyName, property);
            }
            transaction.commit();
        }
    }

    private Runnable createNodeWithLabel(Label label) {
        return () -> {
            try (Transaction transaction = this.database.beginTx();){
                transaction.createNode(new Label[]{label});
                transaction.commit();
            }
        };
    }

    private long countIndexes() {
        try (Transaction transaction = this.database.beginTx();){
            long l = Iterables.count((Iterable)transaction.schema().getIndexes());
            return l;
        }
    }

    private Runnable createIndexForLabelAndProperty(Label label, String propertyKey) {
        return () -> {
            try (Transaction transaction = this.database.beginTx();){
                transaction.schema().indexFor(label).on(propertyKey).create();
                transaction.commit();
            }
            this.waitForOnlineIndexes();
        };
    }

    private void waitForOnlineIndexes() {
        try (Transaction transaction = this.database.beginTx();){
            transaction.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            transaction.commit();
        }
    }

    private Callable<Number> countNodes() {
        return () -> {
            try (Transaction transaction = this.database.beginTx();){
                Result result = transaction.execute("MATCH (n) RETURN count(n) as count");
                Map resultMap = result.next();
                Number number = (Number)resultMap.get("count");
                return number;
            }
        };
    }

    private static class PopulationScanCompleteBlock
    extends IndexMonitor.MonitorAdapter {
        private final String indexName;
        private final CountDownLatch blockLatch;
        private final CountDownLatch signalLatch;

        PopulationScanCompleteBlock(String indexName, CountDownLatch blockLatch, CountDownLatch signalLatch) {
            this.indexName = indexName;
            this.blockLatch = blockLatch;
            this.signalLatch = signalLatch;
        }

        public void indexPopulationScanComplete(IndexDescriptor[] indexDescriptors) {
            if (Arrays.stream(indexDescriptors).map(IndexDescriptor::getName).anyMatch(this.indexName::equals)) {
                this.signalLatch.countDown();
                try {
                    this.blockLatch.await();
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }
}

