/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.api.impl.fulltext;

import java.util.Arrays;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.common.EntityType;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexType;
import org.neo4j.internal.schema.FulltextSchemaDescriptor;
import org.neo4j.internal.schema.IndexBehaviour;
import org.neo4j.internal.schema.IndexCapability;
import org.neo4j.internal.schema.IndexConfig;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.internal.schema.IndexPrototype;
import org.neo4j.internal.schema.IndexProviderDescriptor;
import org.neo4j.internal.schema.IndexQuery;
import org.neo4j.internal.schema.SchemaDescriptor;
import org.neo4j.internal.schema.SchemaDescriptors;
import org.neo4j.internal.schema.StorageEngineIndexingBehaviour;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.io.pagecache.tracing.cursor.DefaultPageCursorTracer;
import org.neo4j.io.pagecache.tracing.cursor.PageCursorTracer;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.api.impl.fulltext.FulltextIndexSettingsKeys;
import org.neo4j.kernel.api.impl.fulltext.LuceneFulltextTestSupport;
import org.neo4j.storageengine.api.StorageEngine;
import org.neo4j.test.extension.Inject;
import org.neo4j.util.Preconditions;
import org.neo4j.values.storable.Value;
import org.neo4j.values.storable.ValueCategory;
import org.neo4j.values.storable.Values;

class FulltextIndexTest
extends LuceneFulltextTestSupport {
    private static final String NODE_INDEX_NAME = "nodes";
    private static final String REL_INDEX_NAME = "rels";
    @Inject
    private StorageEngine storageEngine;
    private StorageEngineIndexingBehaviour indexingBehaviour;

    FulltextIndexTest() {
    }

    @BeforeEach
    void before() {
        this.indexingBehaviour = this.storageEngine.indexingBehaviour();
    }

    @Test
    void tracePageCacheAccessOnIndexQuerying() throws Exception {
        long nodeId;
        this.prepareNodeLabelPropIndex();
        long higherNodeId = -1L;
        Label label2 = Label.label((String)"label2");
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode(new Label[]{LABEL, label2});
            nodeId = node.getId();
            node.setProperty("prop", (Object)"b");
            for (int i = 0; i < 1000; ++i) {
                higherNodeId = tx.createNode().getId();
            }
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.getNodeById(nodeId).removeLabel(label2);
            tx.getNodeById(higherNodeId);
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            CursorContext cursorContext = ktx.cursorContext();
            ((DefaultPageCursorTracer)cursorContext.getCursorTracer()).setIgnoreCounterCheck(true);
            cursorContext.getCursorTracer().reportEvents();
            FulltextIndexTest.assertZeroTracer(cursorContext);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "b", nodeId);
            long pins = cursorContext.getCursorTracer().pins();
            Assertions.assertThat((long)pins).isGreaterThan(0L);
            Assertions.assertThat((long)cursorContext.getCursorTracer().hits()).isEqualTo(pins);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldFindNodeWithString() throws Exception {
        long secondID;
        long firstID;
        this.prepareNodeLabelPropIndex();
        try (Transaction tx = this.db.beginTx();){
            firstID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            secondID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "A zebroid (also zedonk, zorse, zebra mule, zonkey, and zebmule) is the offspring of any cross between a zebra and any other equine: essentially, a zebra hybrid.");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "hello", firstID);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "zebra", secondID);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "zedonk", secondID);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "cross", secondID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldRepresentPropertyChanges() throws Exception {
        long secondID;
        long firstID;
        this.prepareNodeLabelPropIndex();
        try (Transaction tx = this.db.beginTx();){
            firstID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            secondID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "A zebroid (also zedonk, zorse, zebra mule, zonkey, and zebmule) is the offspring of any cross between a zebra and any other equine: essentially, a zebra hybrid.");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            FulltextIndexTest.setNodeProp(tx, firstID, "Finally! Potato!");
            FulltextIndexTest.setNodeProp(tx, secondID, "This one is a potato farmer.");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "hello");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "zebra");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "zedonk");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "cross");
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "finally", firstID);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "farmer", secondID);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "potato", firstID, secondID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldNotFindRemovedNodes() throws Exception {
        long secondID;
        long firstID;
        this.prepareNodeLabelPropIndex();
        try (Transaction tx = this.db.beginTx();){
            firstID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            secondID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "A zebroid (also zedonk, zorse, zebra mule, zonkey, and zebmule) is the offspring of any cross between a zebra and any other equine: essentially, a zebra hybrid.");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.getNodeById(firstID).delete();
            tx.getNodeById(secondID).delete();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "hello");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "zebra");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "zedonk");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "cross");
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldNotFindRemovedProperties() throws Exception {
        long thirdID;
        long secondID;
        long firstID;
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(LABEL).on("prop").on("prop2").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction tx = this.db.beginTx();){
            firstID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            secondID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "A zebroid (also zedonk, zorse, zebra mule, zonkey, and zebmule) is the offspring of any cross between a zebra and any other equine: essentially, a zebra hybrid.");
            thirdID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            FulltextIndexTest.setNodeProp(tx, firstID, "zebra");
            FulltextIndexTest.setNodeProp(tx, secondID, "Hello. Hello again.");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            Node node = tx.getNodeById(firstID);
            Node node2 = tx.getNodeById(secondID);
            Node node3 = tx.getNodeById(thirdID);
            node.setProperty("prop", (Object)"tomtar");
            node.setProperty("prop2", (Object)"tomtar");
            node2.setProperty("prop", (Object)"tomtar");
            node2.setProperty("prop2", (Object)"Hello");
            node3.removeProperty("prop");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "hello", secondID);
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "zebra");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "zedonk");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "cross");
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldOnlyIndexIndexedProperties() throws Exception {
        long firstID;
        this.prepareNodeLabelPropIndex();
        try (Transaction tx = this.db.beginTx();){
            firstID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            FulltextIndexTest.setNodeProp(tx, firstID, "prop2", "zebra");
            Node node2 = tx.createNode(new Label[]{LABEL});
            node2.setProperty("prop2", (Object)"zebra");
            node2.setProperty("prop3", (Object)"hello");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "hello", firstID);
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "zebra");
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldSearchAcrossMultipleProperties() throws Exception {
        long thirdID;
        long secondID;
        long firstID;
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(LABEL).on("prop").on("prop2").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction tx = this.db.beginTx();){
            firstID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Tomtar tomtar oftsat i tomteutstyrsel.");
            secondID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Olof och Hans");
            FulltextIndexTest.setNodeProp(tx, secondID, "prop2", "och karl");
            Node node3 = tx.createNode(new Label[]{LABEL});
            thirdID = node3.getId();
            node3.setProperty("prop2", (Object)"Tomtar som inte tomtar ser upp till tomtar som tomtar.");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "tomtar Karl", firstID, secondID, thirdID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldOrderResultsBasedOnRelevance() throws Exception {
        long fourthID;
        long thirdID;
        long secondID;
        long firstID;
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(LABEL).on("first").on("last").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction tx = this.db.beginTx();){
            firstID = tx.createNode(new Label[]{LABEL}).getId();
            secondID = tx.createNode(new Label[]{LABEL}).getId();
            thirdID = tx.createNode(new Label[]{LABEL}).getId();
            fourthID = tx.createNode(new Label[]{LABEL}).getId();
            FulltextIndexTest.setNodeProp(tx, firstID, "first", "Full");
            FulltextIndexTest.setNodeProp(tx, firstID, "last", "Hanks");
            FulltextIndexTest.setNodeProp(tx, secondID, "first", "Tom");
            FulltextIndexTest.setNodeProp(tx, secondID, "last", "Hunk");
            FulltextIndexTest.setNodeProp(tx, thirdID, "first", "Tom");
            FulltextIndexTest.setNodeProp(tx, thirdID, "last", "Hanks");
            FulltextIndexTest.setNodeProp(tx, fourthID, "first", "Tom Hanks");
            FulltextIndexTest.setNodeProp(tx, fourthID, "last", "Tom Hanks");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsNodeIdsInOrder(ktx, NODE_INDEX_NAME, "Tom Hanks", fourthID, thirdID, firstID, secondID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldDifferentiateNodesAndRelationships() throws Exception {
        long secondRelID;
        long firstRelID;
        long secondNodeID;
        long firstNodeID;
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(LABEL).on("prop").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.schema().indexFor(RELTYPE).on("prop").withIndexType(IndexType.FULLTEXT).withName(REL_INDEX_NAME).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
            tx.schema().awaitIndexOnline(REL_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction tx = this.db.beginTx();){
            firstNodeID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            secondNodeID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "A zebroid (also zedonk, zorse, zebra mule, zonkey, and zebmule) is the offspring of any cross between a zebra and any other equine: essentially, a zebra hybrid.");
            firstRelID = FulltextIndexTest.createRelationshipIndexableByPropertyValue(tx, firstNodeID, secondNodeID, "Hello. Hello again.");
            secondRelID = FulltextIndexTest.createRelationshipIndexableByPropertyValue(tx, secondNodeID, firstNodeID, "And now, something completely different");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "hello", firstNodeID);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "zebra", secondNodeID);
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "different");
            FulltextIndexTest.assertQueryFindsIds(ktx, false, REL_INDEX_NAME, "hello", firstRelID);
            FulltextIndexTest.assertQueryFindsNothing(ktx, false, REL_INDEX_NAME, "zebra");
            FulltextIndexTest.assertQueryFindsIds(ktx, false, REL_INDEX_NAME, "different", secondRelID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldNotReturnNonMatches() throws Exception {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(LABEL).on("prop").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.schema().indexFor(RELTYPE).on("prop").withIndexType(IndexType.FULLTEXT).withName(REL_INDEX_NAME).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
            tx.schema().awaitIndexOnline(REL_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            long firstNode = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            long secondNode = FulltextIndexTest.createNodeWithProperty(tx, LABEL, "prop2", "A zebroid (also zedonk, zorse, zebra mule, zonkey, and zebmule) is the offspring of any cross between a zebra and any other equine: essentially, a zebra hybrid.");
            FulltextIndexTest.createRelationshipIndexableByPropertyValue(tx, firstNode, secondNode, "Hello. Hello again.");
            FulltextIndexTest.createRelationshipWithProperty(tx, secondNode, firstNode, "prop2", "A zebroid (also zedonk, zorse, zebra mule, zonkey, and zebmule) is the offspring of any cross between a zebra and any other equine: essentially, a zebra hybrid.");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "zebra");
            FulltextIndexTest.assertQueryFindsNothing(ktx, false, REL_INDEX_NAME, "zebra");
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldPopulateIndexWithExistingNodesAndRelationships() throws Exception {
        long secondRelID;
        long firstRelID;
        long secondNodeID;
        long firstNodeID;
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            node.createRelationshipTo(node, RELTYPE);
            node.createRelationshipTo(node, RELTYPE);
            node.createRelationshipTo(node, RELTYPE);
            firstNodeID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "Hello. Hello again.");
            secondNodeID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "This string is slightly shorter than the zebra one");
            firstRelID = FulltextIndexTest.createRelationshipIndexableByPropertyValue(tx, firstNodeID, secondNodeID, "Goodbye");
            secondRelID = FulltextIndexTest.createRelationshipIndexableByPropertyValue(tx, secondNodeID, firstNodeID, "And now, something completely different");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().indexFor(LABEL).on("prop").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.schema().indexFor(RELTYPE).on("prop").withIndexType(IndexType.FULLTEXT).withName(REL_INDEX_NAME).create();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
            tx.schema().awaitIndexOnline(REL_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "hello", firstNodeID);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "string", secondNodeID);
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "goodbye");
            FulltextIndexTest.assertQueryFindsNothing(ktx, true, NODE_INDEX_NAME, "different");
            FulltextIndexTest.assertQueryFindsNothing(ktx, false, REL_INDEX_NAME, "hello");
            FulltextIndexTest.assertQueryFindsNothing(ktx, false, REL_INDEX_NAME, "string");
            FulltextIndexTest.assertQueryFindsIds(ktx, false, REL_INDEX_NAME, "goodbye", firstRelID);
            FulltextIndexTest.assertQueryFindsIds(ktx, false, REL_INDEX_NAME, "different", secondRelID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldBeAbleToUpdateAndQueryAfterIndexChange() throws Exception {
        long fourthID;
        KernelTransaction ktx;
        long thirdID;
        long secondID;
        long firstID;
        this.prepareNodeLabelPropIndex();
        try (Transaction tx = this.db.beginTx();){
            firstID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "thing");
            secondID = tx.createNode(new Label[]{LABEL}).getId();
            FulltextIndexTest.setNodeProp(tx, secondID, "prop2", "zebra");
            thirdID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "zebra");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "thing zebra", firstID, thirdID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            tx.schema().getIndexByName(NODE_INDEX_NAME).drop();
            tx.schema().indexFor(LABEL).on("prop2").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            FulltextIndexTest.setNodeProp(tx, firstID, "prop2", "thing");
            fourthID = tx.createNode(new Label[]{LABEL}).getId();
            FulltextIndexTest.setNodeProp(tx, fourthID, "prop2", "zebra");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "thing zebra", firstID, secondID, fourthID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldBeAbleToDropAndReadIndex() throws Exception {
        long secondID;
        long firstID;
        this.prepareNodeLabelPropIndex();
        try (Transaction tx = this.db.beginTx();){
            firstID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "thing");
            secondID = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "zebra");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().getIndexByName(NODE_INDEX_NAME).drop();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.prepareNodeLabelPropIndex();
        tx = this.db.beginTx();
        try {
            KernelTransaction ktx = FulltextIndexTest.kernelTransaction(tx);
            FulltextIndexTest.assertQueryFindsIds(ktx, true, NODE_INDEX_NAME, "thing zebra", firstID, secondID);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void completeConfigurationMustInjectMissingConfigurations() throws Exception {
        int propertyKey;
        int label;
        try (Transaction tx = this.db.beginTx();){
            FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, "bla");
            tx.commit();
        }
        tx = this.getKernelTransaction();
        try {
            label = tx.tokenRead().nodeLabel(LABEL.name());
            propertyKey = tx.tokenRead().propertyKey("prop");
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        IndexConfig indexConfig = IndexConfig.with((String)FulltextIndexSettingsKeys.EVENTUALLY_CONSISTENT, (Value)Values.booleanValue((boolean)true));
        FulltextSchemaDescriptor schema = SchemaDescriptors.fulltext((EntityType)EntityType.NODE, (int[])new int[]{label}, (int[])new int[]{propertyKey});
        IndexProviderDescriptor providerDescriptor = this.indexProvider.getProviderDescriptor();
        IndexDescriptor descriptor = this.indexProvider.completeConfiguration(IndexPrototype.forSchema((SchemaDescriptor)schema, (IndexProviderDescriptor)providerDescriptor).withName("index_1").withIndexConfig(indexConfig).materialise(1L), this.indexingBehaviour);
        Assertions.assertThat((Object)descriptor.getIndexConfig().get(FulltextIndexSettingsKeys.ANALYZER)).isEqualTo((Object)Values.stringValue((String)"standard-no-stop-words"));
        Assertions.assertThat((Object)descriptor.getIndexConfig().get(FulltextIndexSettingsKeys.EVENTUALLY_CONSISTENT)).isEqualTo((Object)Values.booleanValue((boolean)true));
        Assertions.assertThat(Arrays.asList(descriptor.getCapability().behaviours())).containsExactlyInAnyOrder((Object[])new IndexBehaviour[]{IndexBehaviour.EVENTUALLY_CONSISTENT, IndexBehaviour.SKIP_AND_LIMIT});
    }

    @Test
    void completeConfigurationMustNotOverwriteExistingConfiguration() {
        IndexConfig indexConfig = IndexConfig.with((String)"A", (Value)Values.stringValue((String)"B"));
        FulltextSchemaDescriptor schema = SchemaDescriptors.fulltext((EntityType)EntityType.NODE, (int[])new int[]{1}, (int[])new int[]{1});
        IndexProviderDescriptor providerDescriptor = this.indexProvider.getProviderDescriptor();
        IndexDescriptor descriptor = this.indexProvider.completeConfiguration(IndexPrototype.forSchema((SchemaDescriptor)schema, (IndexProviderDescriptor)providerDescriptor).withName("index_1").materialise(1L), this.indexingBehaviour).withIndexConfig(indexConfig);
        org.junit.jupiter.api.Assertions.assertEquals((Object)Values.stringValue((String)"B"), (Object)descriptor.getIndexConfig().get("A"));
    }

    @Test
    void completeConfigurationMustBeIdempotent() {
        FulltextSchemaDescriptor schema = SchemaDescriptors.fulltext((EntityType)EntityType.NODE, (int[])new int[]{1}, (int[])new int[]{1});
        IndexProviderDescriptor providerDescriptor = this.indexProvider.getProviderDescriptor();
        IndexDescriptor onceCompleted = this.indexProvider.completeConfiguration(IndexPrototype.forSchema((SchemaDescriptor)schema, (IndexProviderDescriptor)providerDescriptor).withName("index_1").materialise(1L), this.indexingBehaviour);
        IndexDescriptor twiceCompleted = this.indexProvider.completeConfiguration(onceCompleted, this.indexingBehaviour);
        org.junit.jupiter.api.Assertions.assertEquals((Object)onceCompleted.getIndexConfig(), (Object)twiceCompleted.getIndexConfig());
    }

    @Test
    void mustAssignCapabilitiesToDescriptorsThatHaveNone() {
        FulltextSchemaDescriptor schema = SchemaDescriptors.fulltext((EntityType)EntityType.NODE, (int[])new int[]{1}, (int[])new int[]{1});
        IndexProviderDescriptor providerDescriptor = this.indexProvider.getProviderDescriptor();
        IndexDescriptor completed = this.indexProvider.completeConfiguration(IndexPrototype.forSchema((SchemaDescriptor)schema, (IndexProviderDescriptor)providerDescriptor).withName("index_1").materialise(1L), this.indexingBehaviour);
        org.junit.jupiter.api.Assertions.assertNotEquals((Object)IndexCapability.NO_CAPABILITY, (Object)completed.getCapability());
        completed = completed.withIndexCapability(IndexCapability.NO_CAPABILITY);
        completed = this.indexProvider.completeConfiguration(completed, this.indexingBehaviour);
        org.junit.jupiter.api.Assertions.assertNotEquals((Object)IndexCapability.NO_CAPABILITY, (Object)completed.getCapability());
    }

    @Test
    void mustNotOverwriteExistingCapabilities() {
        IndexCapability capability = new IndexCapability(){

            public boolean supportsOrdering() {
                return false;
            }

            public boolean supportsReturningValues() {
                return false;
            }

            public boolean areValueCategoriesAccepted(ValueCategory ... valueCategories) {
                Preconditions.requireNonEmpty((Object[])valueCategories);
                Preconditions.requireNoNullElements((Object[])valueCategories);
                return true;
            }

            public boolean isQuerySupported(IndexQuery.IndexQueryType queryType, ValueCategory valueCategory) {
                return true;
            }

            public double getCostMultiplier(IndexQuery.IndexQueryType ... queryTypes) {
                return 1.0;
            }

            public boolean supportPartitionedScan(IndexQuery ... queries) {
                Preconditions.requireNonEmpty((Object[])queries);
                Preconditions.requireNoNullElements((Object[])queries);
                return false;
            }
        };
        FulltextSchemaDescriptor schema = SchemaDescriptors.fulltext((EntityType)EntityType.NODE, (int[])new int[]{1}, (int[])new int[]{1});
        IndexProviderDescriptor providerDescriptor = this.indexProvider.getProviderDescriptor();
        IndexDescriptor index = IndexPrototype.forSchema((SchemaDescriptor)schema, (IndexProviderDescriptor)providerDescriptor).withName("index_1").materialise(1L).withIndexCapability(capability);
        IndexDescriptor completed = this.indexProvider.completeConfiguration(index, this.indexingBehaviour);
        org.junit.jupiter.api.Assertions.assertSame((Object)capability, (Object)completed.getCapability());
    }

    @Test
    void fulltextIndexMustNotAnswerCoreApiIndexQueries() {
        long nodeId;
        this.prepareNodeLabelPropIndex();
        try (Transaction tx = this.db.beginTx();){
            nodeId = FulltextIndexTest.createNodeIndexableByPropertyValue(tx, LABEL, 1);
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            Node node = tx.findNode(LABEL, "prop", (Object)1);
            Assertions.assertThat((long)node.getId()).isEqualTo(nodeId);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void fulltextIndexMustNotAnswerCoreApiCompositeIndexQueries() {
        long nodeId;
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(LABEL).on("prop").on("prop2").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode(new Label[]{LABEL});
            node.setProperty("prop", (Object)1);
            node.setProperty("prop2", (Object)2);
            nodeId = node.getId();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            Node node;
            try (ResourceIterator nodes = tx.findNodes(LABEL, "prop", (Object)1, "prop2", (Object)2);){
                org.junit.jupiter.api.Assertions.assertTrue((boolean)nodes.hasNext());
                node = (Node)nodes.next();
                Assertions.assertThat((long)node.getId()).isEqualTo(nodeId);
                org.junit.jupiter.api.Assertions.assertFalse((boolean)nodes.hasNext());
            }
            nodes = tx.findNodes(LABEL, "prop2", (Object)2, "prop", (Object)1);
            try {
                org.junit.jupiter.api.Assertions.assertTrue((boolean)nodes.hasNext());
                node = (Node)nodes.next();
                Assertions.assertThat((long)node.getId()).isEqualTo(nodeId);
                org.junit.jupiter.api.Assertions.assertFalse((boolean)nodes.hasNext());
            }
            finally {
                if (nodes != null) {
                    nodes.close();
                }
            }
            nodes = tx.findNodes(LABEL, "prop", (Object)1);
            try {
                org.junit.jupiter.api.Assertions.assertTrue((boolean)nodes.hasNext());
                node = (Node)nodes.next();
                Assertions.assertThat((long)node.getId()).isEqualTo(nodeId);
                org.junit.jupiter.api.Assertions.assertFalse((boolean)nodes.hasNext());
            }
            finally {
                if (nodes != null) {
                    nodes.close();
                }
            }
            nodes = tx.findNodes(LABEL, "prop2", (Object)2);
            try {
                org.junit.jupiter.api.Assertions.assertTrue((boolean)nodes.hasNext());
                node = (Node)nodes.next();
                Assertions.assertThat((long)node.getId()).isEqualTo(nodeId);
                org.junit.jupiter.api.Assertions.assertFalse((boolean)nodes.hasNext());
            }
            finally {
                if (nodes != null) {
                    nodes.close();
                }
            }
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private void prepareNodeLabelPropIndex() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(LABEL).on("prop").withIndexType(IndexType.FULLTEXT).withName(NODE_INDEX_NAME).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexOnline(NODE_INDEX_NAME, 1L, TimeUnit.MINUTES);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private static void assertZeroTracer(CursorContext cursorContext) {
        PageCursorTracer cursorTracer = cursorContext.getCursorTracer();
        Assertions.assertThat((long)cursorTracer.pins()).isZero();
        Assertions.assertThat((long)cursorTracer.unpins()).isZero();
        Assertions.assertThat((long)cursorTracer.hits()).isZero();
        Assertions.assertThat((long)cursorTracer.faults()).isZero();
    }
}

