/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.api.impl.fulltext;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.common.DependencyResolver;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.consistency.ConsistencyCheckService;
import org.neo4j.consistency.checking.ConsistencyCheckIncompleteException;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.graphdb.schema.IndexType;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.internal.schema.SchemaDescriptorSupplier;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.kernel.api.impl.fulltext.FulltextIndexProceduresUtil;
import org.neo4j.kernel.api.index.IndexUpdater;
import org.neo4j.kernel.impl.api.index.IndexProxy;
import org.neo4j.kernel.impl.api.index.IndexUpdateMode;
import org.neo4j.kernel.impl.api.index.IndexingService;
import org.neo4j.kernel.impl.coreapi.schema.IndexDefinitionImpl;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.storageengine.api.IndexEntryUpdate;
import org.neo4j.test.RandomSupport;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.RandomExtension;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;
import org.neo4j.values.storable.RandomValues;
import org.neo4j.values.storable.Value;
import org.neo4j.values.storable.Values;

@TestDirectoryExtension
@ExtendWith(value={RandomExtension.class})
class FulltextIndexConsistencyCheckIT {
    @Inject
    private TestDirectory testDirectory;
    private DatabaseLayout databaseLayout;
    @Inject
    private RandomSupport random;
    private TestDatabaseManagementServiceBuilder builder;
    private GraphDatabaseAPI database;
    private DatabaseManagementService managementService;

    FulltextIndexConsistencyCheckIT() {
    }

    @BeforeEach
    void before() {
        this.builder = new TestDatabaseManagementServiceBuilder(this.testDirectory.homePath());
    }

    @AfterEach
    void tearDown() {
        if (this.database != null) {
            this.managementService.shutdown();
        }
    }

    @Test
    void mustBeAbleToConsistencyCheckEmptyDatabaseWithWithEmptyFulltextIndexes() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithOneLabelAndOneProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.createNode(new Label[]{Label.label((String)"Label")}).setProperty("prop", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithOneLabelAndMultipleProperties() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1", "p2"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode(new Label[]{Label.label((String)"Label")});
            node.setProperty("p1", (Object)"value");
            node.setProperty("p2", (Object)"value");
            tx.createNode(new Label[]{Label.label((String)"Label")}).setProperty("p1", (Object)"value");
            tx.createNode(new Label[]{Label.label((String)"Label")}).setProperty("p2", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithMultipleLabelsAndOneProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"L1", "L2"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.createNode(new Label[]{Label.label((String)"L1"), Label.label((String)"L2")}).setProperty("prop", (Object)"value");
            tx.createNode(new Label[]{Label.label((String)"L2")}).setProperty("prop", (Object)"value");
            tx.createNode(new Label[]{Label.label((String)"L1")}).setProperty("prop", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithManyLabelsAndOneProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        String[] labels = new String[]{"L1", "L2", "L3", "L4", "L5", "L6", "L7", "L8", "L9", "L10", "L11", "L12", "L13", "L14", "L15", "L16"};
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])labels), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.createNode((Label[])Stream.of(labels).map(Label::label).toArray(Label[]::new)).setProperty("prop", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithMultipleLabelsAndMultipleProperties() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"L1", "L2"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1", "p2"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node n1 = tx.createNode(new Label[]{Label.label((String)"L1"), Label.label((String)"L2")});
            n1.setProperty("p1", (Object)"value");
            n1.setProperty("p2", (Object)"value");
            Node n2 = tx.createNode(new Label[]{Label.label((String)"L1"), Label.label((String)"L2")});
            n2.setProperty("p1", (Object)"value");
            Node n3 = tx.createNode(new Label[]{Label.label((String)"L1"), Label.label((String)"L2")});
            n3.setProperty("p2", (Object)"value");
            Node n4 = tx.createNode(new Label[]{Label.label((String)"L1")});
            n4.setProperty("p1", (Object)"value");
            n4.setProperty("p2", (Object)"value");
            Node n5 = tx.createNode(new Label[]{Label.label((String)"L1")});
            n5.setProperty("p1", (Object)"value");
            Node n6 = tx.createNode(new Label[]{Label.label((String)"L1")});
            n6.setProperty("p2", (Object)"value");
            Node n7 = tx.createNode(new Label[]{Label.label((String)"L2")});
            n7.setProperty("p1", (Object)"value");
            n7.setProperty("p2", (Object)"value");
            Node n8 = tx.createNode(new Label[]{Label.label((String)"L2")});
            n8.setProperty("p1", (Object)"value");
            Node n9 = tx.createNode(new Label[]{Label.label((String)"L2")});
            n9.setProperty("p2", (Object)"value");
            tx.createNode(new Label[]{Label.label((String)"L2")}).setProperty("p1", (Object)"value");
            tx.createNode(new Label[]{Label.label((String)"L2")}).setProperty("p2", (Object)"value");
            tx.createNode(new Label[]{Label.label((String)"L1")}).setProperty("p1", (Object)"value");
            tx.createNode(new Label[]{Label.label((String)"L1")}).setProperty("p2", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithTextArrayProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.createNode(new Label[]{Label.label((String)"Label")}).setProperty("prop", (Object)Iterators.array((Object[])new String[]{"value1", "value2"}));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithEmptyTextArrayProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.createNode(new Label[]{Label.label((String)"Label")}).setProperty("prop", (Object)new String[0]);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithEmptyTextProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.createNode(new Label[]{Label.label((String)"Label")}).setProperty("prop", (Object)"");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexWithMixOfTextAndTextArrayProperties() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.createNode(new Label[]{Label.label((String)"Label")}).setProperty("prop", (Object)"plainValue");
            tx.createNode(new Label[]{Label.label((String)"Label")}).setProperty("prop", (Object)Iterators.array((Object[])new String[]{"value1", "value2"}));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexWithOneRelationshipTypeAndOneProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RelationshipType relationshipType = RelationshipType.withName((String)"R1");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode();
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)"value");
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexWithOneRelationshipTypeAndMultipleProperties() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RelationshipType relationshipType = RelationshipType.withName((String)"R1");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1", "p2"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode();
            Relationship r1 = node.createRelationshipTo(node, relationshipType);
            r1.setProperty("p1", (Object)"value");
            r1.setProperty("p2", (Object)"value");
            Relationship r2 = node.createRelationshipTo(node, relationshipType);
            r2.setProperty("p1", (Object)"value");
            r2.setProperty("p2", (Object)"value");
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)"value");
            node.createRelationshipTo(node, relationshipType).setProperty("p2", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexWithMultipleRelationshipTypesAndOneProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RelationshipType relType1 = RelationshipType.withName((String)"R1");
        RelationshipType relType2 = RelationshipType.withName((String)"R2");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1", "R2"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node n1 = tx.createNode();
            Node n2 = tx.createNode();
            n1.createRelationshipTo(n1, relType1).setProperty("p1", (Object)"value");
            n1.createRelationshipTo(n1, relType2).setProperty("p1", (Object)"value");
            n2.createRelationshipTo(n2, relType1).setProperty("p1", (Object)"value");
            n2.createRelationshipTo(n2, relType2).setProperty("p1", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexWithMultipleRelationshipTypesAndMultipleProperties() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RelationshipType relType1 = RelationshipType.withName((String)"R1");
        RelationshipType relType2 = RelationshipType.withName((String)"R2");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1", "R2"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1", "p2"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node n1 = tx.createNode();
            Node n2 = tx.createNode();
            Relationship r1 = n1.createRelationshipTo(n1, relType1);
            r1.setProperty("p1", (Object)"value");
            r1.setProperty("p2", (Object)"value");
            Relationship r2 = n1.createRelationshipTo(n1, relType2);
            r2.setProperty("p1", (Object)"value");
            r2.setProperty("p2", (Object)"value");
            Relationship r3 = n2.createRelationshipTo(n2, relType1);
            r3.setProperty("p1", (Object)"value");
            r3.setProperty("p2", (Object)"value");
            Relationship r4 = n2.createRelationshipTo(n2, relType2);
            r4.setProperty("p1", (Object)"value");
            r4.setProperty("p2", (Object)"value");
            n1.createRelationshipTo(n2, relType1).setProperty("p1", (Object)"value");
            n1.createRelationshipTo(n2, relType2).setProperty("p1", (Object)"value");
            n1.createRelationshipTo(n2, relType1).setProperty("p2", (Object)"value");
            n1.createRelationshipTo(n2, relType2).setProperty("p2", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexWithTextArrayProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RelationshipType relationshipType = RelationshipType.withName((String)"R1");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode();
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)Iterators.array((Object[])new String[]{"value1", "value2"}));
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)Iterators.array((Object[])new String[]{"value1", "value2"}));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexWithEmptyTextArrayProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RelationshipType relationshipType = RelationshipType.withName((String)"R1");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode();
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)new String[0]);
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)new String[0]);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexWithEmptyTextProperty() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RelationshipType relationshipType = RelationshipType.withName((String)"R1");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode();
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)"");
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)"");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexWithMixOfTextAndTextArrayProperties() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RelationshipType relationshipType = RelationshipType.withName((String)"R1");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode();
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)"plainValue");
            node.createRelationshipTo(node, relationshipType).setProperty("p1", (Object)Iterators.array((Object[])new String[]{"value1", "value2"}));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeAndRelationshipIndexesAtTheSameTime() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"L1", "L2", "L3"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1", "p2"}))).close();
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1", "R2"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1", "p2"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node n1 = tx.createNode(new Label[]{Label.label((String)"L1"), Label.label((String)"L3")});
            n1.setProperty("p1", (Object)"value");
            n1.setProperty("p2", (Object)"value");
            n1.createRelationshipTo(n1, RelationshipType.withName((String)"R2")).setProperty("p1", (Object)"value");
            Node n2 = tx.createNode(new Label[]{Label.label((String)"L2")});
            n2.setProperty("p2", (Object)"value");
            Relationship r1 = n2.createRelationshipTo(n2, RelationshipType.withName((String)"R1"));
            r1.setProperty("p1", (Object)"value");
            r1.setProperty("p2", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckNodeIndexThatIsMissingNodesBecauseTheirPropertyValuesAreNotStrings() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"L1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.createNode(new Label[]{Label.label((String)"L1")}).setProperty("p1", (Object)1);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustBeAbleToConsistencyCheckRelationshipIndexThatIsMissingRelationshipsBecauseTheirPropertyValuesAreNotStrings() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"R1"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode();
            node.createRelationshipTo(node, RelationshipType.withName((String)"R1")).setProperty("p1", (Object)1);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void consistencyCheckerMustBeAbleToRunOnStoreWithFulltextIndexes() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        Label[] labels = (Label[])IntStream.range(1, 7).mapToObj(i -> Label.label((String)("LABEL" + i))).toArray(Label[]::new);
        RelationshipType[] relTypes = (RelationshipType[])IntStream.range(1, 5).mapToObj(i -> RelationshipType.withName((String)("REL" + i))).toArray(RelationshipType[]::new);
        String[] propertyKeys = (String[])IntStream.range(1, 7).mapToObj(i -> "PROP" + i).toArray(String[]::new);
        RandomValues randomValues = this.random.randomValues();
        try (Transaction tx = db.beginTx();){
            int nodeCount = 1000;
            ArrayList<Node> nodes = new ArrayList<Node>(nodeCount);
            for (int i2 = 0; i2 < nodeCount; ++i2) {
                Label[] nodeLabels = (Label[])this.random.ints((long)this.random.nextInt(labels.length), 0, labels.length).distinct().mapToObj(x -> labels[x]).toArray(Label[]::new);
                Node node = tx.createNode(nodeLabels);
                Stream.of(propertyKeys).forEach(p -> node.setProperty(p, this.random.nextBoolean() ? p : randomValues.nextValue().asObject()));
                nodes.add(node);
                int localRelCount = Math.min(nodes.size(), 5);
                this.random.ints((long)localRelCount, 0, localRelCount).distinct().mapToObj(x -> node.createRelationshipTo((Node)nodes.get(x), relTypes[this.random.nextInt(relTypes.length)])).forEach(r -> Stream.of(propertyKeys).forEach(p -> r.setProperty(p, this.random.nextBoolean() ? p : randomValues.nextValue().asObject())));
            }
            tx.commit();
        }
        tx = db.beginTx();
        try {
            int i3;
            for (i3 = 1; i3 < labels.length; ++i3) {
                tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes" + i3, FulltextIndexProceduresUtil.asNodeLabelStr((String[])((String[])Arrays.stream(labels).limit(i3).map(Label::name).toArray(String[]::new))), FulltextIndexProceduresUtil.asPropertiesStrList((String[])Arrays.copyOf(propertyKeys, i3)))).close();
            }
            for (i3 = 1; i3 < relTypes.length; ++i3) {
                tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels" + i3, FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])((String[])Arrays.stream(relTypes).limit(i3).map(RelationshipType::name).toArray(String[]::new))), FulltextIndexProceduresUtil.asPropertiesStrList((String[])Arrays.copyOf(propertyKeys, i3)))).close();
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustDiscoverNodeInStoreMissingFromIndex() throws Exception {
        long nodeId;
        IndexDescriptor indexDescriptor;
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        try (Transaction tx = db.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            indexDescriptor = FulltextIndexConsistencyCheckIT.getFulltextIndexDescriptor(tx.schema().getIndexes());
            Node node = tx.createNode(new Label[]{Label.label((String)"Label")});
            node.setProperty("prop", (Object)"value");
            nodeId = node.getId();
            tx.commit();
        }
        IndexingService indexes = FulltextIndexConsistencyCheckIT.getIndexingService((GraphDatabaseService)db);
        IndexProxy indexProxy = indexes.getIndexProxy(indexDescriptor);
        try (IndexUpdater updater = indexProxy.newUpdater(IndexUpdateMode.ONLINE, CursorContext.NULL_CONTEXT, false);){
            updater.process((IndexEntryUpdate)IndexEntryUpdate.remove((long)nodeId, (SchemaDescriptorSupplier)indexDescriptor, (Value[])new Value[]{Values.stringValue((String)"value")}));
        }
        this.managementService.shutdown();
        ConsistencyCheckService.Result result = this.checkConsistency();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)result.isSuccessful());
    }

    @Test
    void mustDiscoverNodeWithTextArrayInStoreMissingFromIndex() throws Exception {
        long nodeId;
        IndexDescriptor indexDescriptor;
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        try (Transaction tx = db.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            indexDescriptor = FulltextIndexConsistencyCheckIT.getFulltextIndexDescriptor(tx.schema().getIndexes());
            Node node = tx.createNode(new Label[]{Label.label((String)"Label")});
            node.setProperty("prop", (Object)Iterators.array((Object[])new String[]{"value1", "value2"}));
            nodeId = node.getId();
            tx.commit();
        }
        IndexingService indexes = FulltextIndexConsistencyCheckIT.getIndexingService((GraphDatabaseService)db);
        IndexProxy indexProxy = indexes.getIndexProxy(indexDescriptor);
        try (IndexUpdater updater = indexProxy.newUpdater(IndexUpdateMode.ONLINE, CursorContext.NULL_CONTEXT, false);){
            updater.process((IndexEntryUpdate)IndexEntryUpdate.remove((long)nodeId, (SchemaDescriptorSupplier)indexDescriptor, (Value[])new Value[]{Values.stringArray((String[])new String[]{"value1", "value2"})}));
        }
        this.managementService.shutdown();
        ConsistencyCheckService.Result result = this.checkConsistency();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)result.isSuccessful());
    }

    @Test
    public void shouldNotReportNodesWithoutAllPropertiesInIndex() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"L1", "L2"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1", "p2"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode(new Label[]{Label.label((String)"L1"), Label.label((String)"L2")});
            node.setProperty("p1", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    public void shouldNotReportNodesWithMorePropertiesThanInIndex() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"L1", "L2"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"p1"}))).close();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode(new Label[]{Label.label((String)"L1"), Label.label((String)"L2")});
            node.setProperty("p1", (Object)"value");
            node.setProperty("p2", (Object)"value");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.managementService.shutdown();
        FulltextIndexConsistencyCheckIT.assertIsConsistent(this.checkConsistency());
    }

    @Test
    void mustDiscoverRelationshipInStoreMissingFromIndex() throws Exception {
        long relId;
        IndexDescriptor indexDescriptor;
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"REL"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        try (Transaction tx = db.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            indexDescriptor = FulltextIndexConsistencyCheckIT.getFulltextIndexDescriptor(tx.schema().getIndexes());
            Node node = tx.createNode();
            Relationship rel = node.createRelationshipTo(node, RelationshipType.withName((String)"REL"));
            rel.setProperty("prop", (Object)"value");
            relId = rel.getId();
            tx.commit();
        }
        IndexingService indexes = FulltextIndexConsistencyCheckIT.getIndexingService((GraphDatabaseService)db);
        IndexProxy indexProxy = indexes.getIndexProxy(indexDescriptor);
        try (IndexUpdater updater = indexProxy.newUpdater(IndexUpdateMode.ONLINE, CursorContext.NULL_CONTEXT, false);){
            updater.process((IndexEntryUpdate)IndexEntryUpdate.remove((long)relId, (SchemaDescriptorSupplier)indexDescriptor, (Value[])new Value[]{Values.stringValue((String)"value")}));
        }
        this.managementService.shutdown();
        ConsistencyCheckService.Result result = this.checkConsistency();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)result.isSuccessful());
    }

    @Test
    void mustDiscoverRelationshipWithTextArrayInStoreMissingFromIndex() throws Exception {
        long relId;
        IndexDescriptor indexDescriptor;
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"REL"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        try (Transaction tx = db.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            indexDescriptor = FulltextIndexConsistencyCheckIT.getFulltextIndexDescriptor(tx.schema().getIndexes());
            Node node = tx.createNode();
            Relationship rel = node.createRelationshipTo(node, RelationshipType.withName((String)"REL"));
            rel.setProperty("prop", (Object)Iterators.array((Object[])new String[]{"value1", "value2"}));
            relId = rel.getId();
            tx.commit();
        }
        IndexingService indexes = FulltextIndexConsistencyCheckIT.getIndexingService((GraphDatabaseService)db);
        IndexProxy indexProxy = indexes.getIndexProxy(indexDescriptor);
        try (IndexUpdater updater = indexProxy.newUpdater(IndexUpdateMode.ONLINE, CursorContext.NULL_CONTEXT, false);){
            updater.process((IndexEntryUpdate)IndexEntryUpdate.remove((long)relId, (SchemaDescriptorSupplier)indexDescriptor, (Value[])new Value[]{Values.stringArray((String[])new String[]{"value1", "value2"})}));
        }
        this.managementService.shutdown();
        ConsistencyCheckService.Result result = this.checkConsistency();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)result.isSuccessful());
    }

    @Disabled(value="Turns out that this is not something that the consistency checker actually looks for, currently. The test is disabled until the consistency checker is extended with checks that will discover this sort of inconsistency.")
    @Test
    void mustDiscoverNodeInIndexWithMissingPropertyInStore() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "nodes", FulltextIndexProceduresUtil.asNodeLabelStr((String[])new String[]{"Label"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        this.managementService.shutdown();
        Path copyRoot = this.testDirectory.directory("cpy");
        this.copyStoreFiles(copyRoot);
        db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.createNode();
            node.setProperty("prop", (Object)"value");
            tx.commit();
        }
        this.managementService.shutdown();
        this.restoreStoreFiles(copyRoot);
        ConsistencyCheckService.Result result = this.checkConsistency();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)result.isSuccessful());
        Assertions.assertThat((int)result.summary().getInconsistencyCountForRecordType("INDEX")).isEqualTo(1);
    }

    @Test
    void mustDiscoverRelationshipInIndexWithMissingPropertyInStore() throws Exception {
        String relId;
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"REL"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        try (Transaction tx = db.beginTx();){
            Node node = tx.createNode();
            Relationship rel = node.createRelationshipTo(node, RelationshipType.withName((String)"REL"));
            relId = rel.getElementId();
            tx.commit();
        }
        this.managementService.shutdown();
        Path copyRoot = this.testDirectory.directory("cpy");
        this.copyStoreFiles(copyRoot);
        db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            tx.getRelationshipByElementId(relId).setProperty("prop", (Object)"value");
            tx.commit();
        }
        this.managementService.shutdown();
        this.restoreStoreFiles(copyRoot);
        ConsistencyCheckService.Result result = this.checkConsistency();
        Assertions.assertThat((boolean)result.isSuccessful()).isFalse();
        Assertions.assertThat((int)result.summary().getInconsistencyCountForRecordType("INDEX")).isEqualTo(1);
    }

    @Test
    void mustDiscoverRelationshipInIndexWithMissingRelationshipInStore() throws Exception {
        String nodeId;
        GraphDatabaseAPI db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE FULLTEXT INDEX `%s` FOR %s ON EACH %s", "rels", FulltextIndexProceduresUtil.asRelationshipTypeStr((String[])new String[]{"REL"}), FulltextIndexProceduresUtil.asPropertiesStrList((String[])new String[]{"prop"}))).close();
            tx.commit();
        }
        try (Transaction tx = db.beginTx();){
            Node node = tx.createNode();
            nodeId = node.getElementId();
            tx.commit();
        }
        this.managementService.shutdown();
        Path copyRoot = this.testDirectory.directory("cpy");
        this.copyStoreFiles(copyRoot);
        db = this.createDatabase();
        try (Transaction tx = db.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Node node = tx.getNodeByElementId(nodeId);
            Relationship rel = node.createRelationshipTo(node, RelationshipType.withName((String)"REL"));
            rel.setProperty("prop", (Object)"value");
            tx.commit();
        }
        this.managementService.shutdown();
        this.restoreStoreFiles(copyRoot);
        ConsistencyCheckService.Result result = this.checkConsistency();
        Assertions.assertThat((boolean)result.isSuccessful()).isFalse();
        Assertions.assertThat((int)result.summary().getInconsistencyCountForRecordType("INDEX")).isEqualTo(1);
    }

    private void copyStoreFiles(Path into) throws IOException {
        FileSystemAbstraction fileSystem = this.testDirectory.getFileSystem();
        for (Path storeFile : this.databaseLayout.mandatoryStoreFiles()) {
            fileSystem.copyFile(storeFile, into.resolve(storeFile.getFileName()));
        }
    }

    private void restoreStoreFiles(Path from) throws IOException {
        this.testDirectory.getFileSystem().copyRecursively(from, this.databaseLayout.databaseDirectory());
    }

    private GraphDatabaseAPI createDatabase() {
        this.managementService = this.builder.build();
        this.database = (GraphDatabaseAPI)this.managementService.database("neo4j");
        this.databaseLayout = this.database.databaseLayout();
        return this.database;
    }

    private ConsistencyCheckService.Result checkConsistency() throws ConsistencyCheckIncompleteException {
        Config config = Config.defaults((Setting)GraphDatabaseSettings.logs_directory, (Object)this.databaseLayout.databaseDirectory());
        return new ConsistencyCheckService(this.databaseLayout).with(config).runFullConsistencyCheck();
    }

    private static IndexDescriptor getFulltextIndexDescriptor(Iterable<IndexDefinition> indexes) {
        for (IndexDefinition index : indexes) {
            if (index.getIndexType() != IndexType.FULLTEXT) continue;
            return FulltextIndexConsistencyCheckIT.getIndexDescriptor(index);
        }
        return IndexDescriptor.NO_INDEX;
    }

    private static IndexDescriptor getIndexDescriptor(IndexDefinition definition) {
        return ((IndexDefinitionImpl)definition).getIndexReference();
    }

    private static IndexingService getIndexingService(GraphDatabaseService db) {
        return (IndexingService)FulltextIndexConsistencyCheckIT.getDependencyResolver(db).resolveDependency(IndexingService.class);
    }

    private static DependencyResolver getDependencyResolver(GraphDatabaseService db) {
        return ((GraphDatabaseAPI)db).getDependencyResolver();
    }

    private static void assertIsConsistent(ConsistencyCheckService.Result result) throws IOException {
        if (!result.isSuccessful()) {
            FulltextIndexConsistencyCheckIT.printReport(result);
            org.junit.jupiter.api.Assertions.fail((String)"Expected consistency check to be successful.");
        }
    }

    private static void printReport(ConsistencyCheckService.Result result) throws IOException {
        Files.readAllLines(result.reportFile()).forEach(System.err::println);
    }
}

