/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.database;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.neo4j.common.DependencyResolver;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.DatabaseShutdownException;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.TransactionFailureException;
import org.neo4j.graphdb.TransactionTerminatedException;
import org.neo4j.graphdb.TransientTransactionFailureException;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.availability.AvailabilityGuard;
import org.neo4j.kernel.availability.AvailabilityListener;
import org.neo4j.kernel.database.Database;
import org.neo4j.kernel.database.DatabaseLifeShutdownCoordinator;
import org.neo4j.kernel.database.DatabaseMonitors;
import org.neo4j.kernel.impl.api.KernelTransactions;
import org.neo4j.kernel.impl.api.ShutdownTransactionMonitor;
import org.neo4j.kernel.impl.coreapi.TransactionImpl;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.lifecycle.LifeSupport;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.InternalLogProvider;
import org.neo4j.logging.LogAssertions;
import org.neo4j.test.OtherThreadExecutor;
import org.neo4j.test.Race;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.SkipOnSpd;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;
import org.neo4j.time.Clocks;
import org.neo4j.time.FakeClock;
import org.neo4j.time.SystemNanoClock;

@TestDirectoryExtension
@SkipOnSpd
class DatabaseTransactionShutdownIT {
    private static final String UNCLEAN_SHUTDOWN_MSG = "Failed to close all transactions. Shutdown may be unclean.";
    @Inject
    TestDirectory directory;
    DatabaseManagementService dbms;
    GraphDatabaseAPI db;
    AssertableLogProvider logProvider = new AssertableLogProvider();

    DatabaseTransactionShutdownIT() {
    }

    void setUp(SystemNanoClock clock) {
        this.dbms = new TestDatabaseManagementServiceBuilder(this.directory.homePath()).setConfig(GraphDatabaseSettings.shutdown_transaction_end_timeout, (Object)Duration.ofMillis(0L)).setConfig(GraphDatabaseInternalSettings.shutdown_terminated_transaction_wait_timeout, (Object)Duration.ofMillis(1L)).setClock(clock).setInternalLogProvider((InternalLogProvider)this.logProvider).build();
        this.db = (GraphDatabaseAPI)this.dbms.database("neo4j");
    }

    @AfterEach
    void tearDown() {
        this.shutdownDbms();
    }

    @Test
    void shouldWaitForTransactionToDetectTerminationOnShutdown() throws Exception {
        FakeClock clock = Clocks.fakeClock();
        this.setUp((SystemNanoClock)clock);
        Duration waitTime = (Duration)((Config)this.db.getDependencyResolver().resolveDependency(Config.class)).get(GraphDatabaseInternalSettings.shutdown_terminated_transaction_wait_timeout);
        final CountDownLatch transactionLatch = new CountDownLatch(1);
        DatabaseMonitors databaseMonitors = (DatabaseMonitors)this.db.getDependencyResolver().resolveDependency(DatabaseMonitors.class);
        databaseMonitors.addMonitorListener((Object)new ShutdownTransactionMonitor(){

            public void awaitTerminatedTransactionClose() {
                transactionLatch.countDown();
            }
        }, new String[0]);
        try (OtherThreadExecutor executor = new OtherThreadExecutor("shouldWaitForTransactionToDetectTerminationOnShutdown");){
            Transaction tx = this.db.beginTx();
            KernelTransaction ktx = ((TransactionImpl)tx).kernelTransaction();
            tx.createNode();
            Future shutdownFuture = executor.executeDontWait(this::shutdownDbms);
            transactionLatch.await();
            Assertions.assertThat((Optional)ktx.getTerminationMark()).isNotEmpty();
            Assertions.assertThatThrownBy(() -> ((Transaction)tx).createNode()).isInstanceOf(TransactionTerminatedException.class);
            while (!shutdownFuture.isDone()) {
                clock.forward(waitTime.plusMillis(1L));
            }
            shutdownFuture.get();
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((Transaction)tx).close()).isInstanceOf(TransientTransactionFailureException.class)).hasMessageContaining("The database is not currently available to serve your request");
        }
    }

    @Test
    void shutdownCoordinatorIsTheLastComponentInDatabaseLifecycle() {
        this.setUp(Clocks.nanoClock());
        Database database = (Database)this.db.getDependencyResolver().resolveDependency(Database.class);
        LifeSupport databaseLife = database.getLife();
        Assertions.assertThat((Object)((Lifecycle)databaseLife.getLifecycleInstances().getLast())).isInstanceOf(DatabaseLifeShutdownCoordinator.class);
    }

    @Test
    void stoppingOnlyCoordinatorMakesImpossibleToStartTransactions() throws Exception {
        this.setUp(Clocks.nanoClock());
        Database database = (Database)this.db.getDependencyResolver().resolveDependency(Database.class);
        LifeSupport databaseLife = database.getLife();
        ((Lifecycle)databaseLife.getLifecycleInstances().getLast()).stop();
        Assertions.assertThatThrownBy(() -> {
            Transaction tx = this.db.beginTx();
            if (tx != null) {
                tx.close();
            }
        }).isInstanceOf(TransactionFailureException.class);
    }

    @Test
    void stoppingOnlyCoordinatorMakesOngoingTransactionTerminated() throws Exception {
        this.setUp(Clocks.nanoClock());
        Database database = (Database)this.db.getDependencyResolver().resolveDependency(Database.class);
        LifeSupport databaseLife = database.getLife();
        try (Transaction tx = this.db.beginTx();){
            TransactionImpl internalTransaction = (TransactionImpl)tx;
            KernelTransaction kernelTransaction = internalTransaction.kernelTransaction();
            ((Lifecycle)databaseLife.getLifecycleInstances().getLast()).stop();
            org.junit.jupiter.api.Assertions.assertTrue((boolean)kernelTransaction.isTerminated());
            Assertions.assertThatThrownBy(() -> ((Transaction)tx).createNode()).isInstanceOf(TransactionTerminatedException.class);
        }
    }

    @Test
    void shutdownOnlyCoordinatorMakesImpossibleToStartTransactions() throws Exception {
        this.setUp(Clocks.nanoClock());
        Database database = (Database)this.db.getDependencyResolver().resolveDependency(Database.class);
        LifeSupport databaseLife = database.getLife();
        ((Lifecycle)databaseLife.getLifecycleInstances().getLast()).shutdown();
        Assertions.assertThatThrownBy(() -> {
            Transaction tx = this.db.beginTx();
            if (tx != null) {
                tx.close();
            }
        }).isInstanceOf(DatabaseShutdownException.class);
    }

    @Test
    void shouldWaitForTransactionToDetectTerminationAndCloseOnShutdown() throws Exception {
        this.setUp((SystemNanoClock)Clocks.fakeClock());
        final CountDownLatch transactionLatch = new CountDownLatch(1);
        DatabaseMonitors databaseMonitors = (DatabaseMonitors)this.db.getDependencyResolver().resolveDependency(DatabaseMonitors.class);
        databaseMonitors.addMonitorListener((Object)new ShutdownTransactionMonitor(){

            public void awaitTerminatedTransactionClose() {
                transactionLatch.countDown();
            }
        }, new String[0]);
        try (OtherThreadExecutor executor = new OtherThreadExecutor("waitForTransactionToDetectTerminationAndCloseOnShutdown");){
            Transaction tx = this.db.beginTx();
            KernelTransaction ktx = ((TransactionImpl)tx).kernelTransaction();
            tx.createNode();
            Future shutdownFuture = executor.executeDontWait(this::shutdownDbms);
            transactionLatch.await();
            Assertions.assertThat((Optional)ktx.getTerminationMark()).isNotEmpty();
            tx.close();
            shutdownFuture.get();
        }
    }

    @Test
    void shouldNotAllowNewTransactionsAfterUnavailable() throws Exception {
        this.setUp(Clocks.nanoClock());
        DependencyResolver dep = this.db.getDependencyResolver();
        final KernelTransactions ktxs = (KernelTransactions)dep.resolveDependency(KernelTransactions.class);
        ((AvailabilityGuard)dep.resolveDependency(AvailabilityGuard.class)).addListener(new AvailabilityListener(){

            public void unavailable() {
                ktxs.unblockNewTransactions();
            }
        });
        try (OtherThreadExecutor executor = new OtherThreadExecutor("notAllowNewTransactionsAfterUnavailable");){
            ktxs.blockNewTransactions();
            Future future = executor.executeDontWait(() -> {
                try {
                    Transaction tx = this.db.beginTx();
                    if (tx != null) {
                        tx.close();
                    }
                }
                catch (RuntimeException e) {
                    return e;
                }
                return null;
            });
            executor.waitUntilWaiting(details -> details.isAt(KernelTransactions.class, "newKernelTransaction"));
            this.shutdownDbms();
            Assertions.assertThat((Throwable)((RuntimeException)future.get())).isInstanceOf(DatabaseShutdownException.class);
        }
    }

    @Test
    void transactionCloseReleaseIdsOnShutdown() {
        this.setUp(Clocks.nanoClock());
        try (Transaction transaction = this.db.beginTx();){
            transaction.createNode();
            transaction.createNode();
            transaction.createNode();
            this.shutdownDbms();
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((Transaction)transaction).close()).isInstanceOf(TransientTransactionFailureException.class)).hasMessageContaining("The database is not currently available to serve your request").rootCause().isInstanceOf(DatabaseShutdownException.class);
        }
    }

    @Test
    void shouldLogUncleanShutdownOnLeakedTransaction() {
        this.setUp(Clocks.nanoClock());
        try (Transaction leakedTx = this.db.beginTx();){
            this.dbms.shutdown();
            LogAssertions.assertThat((AssertableLogProvider)this.logProvider).forClass(KernelTransactions.class).forLevel(AssertableLogProvider.Level.WARN).containsMessages(new String[]{UNCLEAN_SHUTDOWN_MSG});
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((Transaction)leakedTx).close()).isInstanceOf(TransientTransactionFailureException.class)).hasMessageContaining("The database is not currently available to serve your request");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void shouldNotLeakedTransactionsOnShutdownRace() throws Throwable {
        this.setUp(Clocks.nanoClock());
        DependencyResolver dep = this.db.getDependencyResolver();
        KernelTransactions ktxs = (KernelTransactions)dep.resolveDependency(KernelTransactions.class);
        Race race = new Race();
        int threads = 10;
        CountDownLatch latch = new CountDownLatch(threads);
        AtomicBoolean done = new AtomicBoolean(false);
        race.addContestants(threads, () -> {
            latch.countDown();
            while (!done.get()) {
                try {
                    Transaction tx = this.db.beginTx();
                    if (tx == null) continue;
                    tx.close();
                }
                catch (DatabaseShutdownException | TransactionFailureException | TransientTransactionFailureException throwable) {}
            }
        });
        Race.Async future = null;
        try {
            future = race.goAsync();
            latch.await();
            this.dbms.shutdown();
            Assertions.assertThat((boolean)ktxs.haveActiveTransaction()).isFalse();
            LogAssertions.assertThat((AssertableLogProvider)this.logProvider).forClass(Database.class).forLevel(AssertableLogProvider.Level.WARN).doesNotContainMessage(UNCLEAN_SHUTDOWN_MSG);
        }
        finally {
            done.set(true);
            if (future != null) {
                future.await(1L, TimeUnit.MINUTES);
            }
        }
    }

    private Void shutdownDbms() {
        if (this.dbms != null) {
            this.dbms.shutdown();
            this.dbms = null;
        }
        return null;
    }
}

