/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphdb;

import java.io.IOException;
import java.nio.file.Path;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.AbstractCollectionAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.common.EntityType;
import org.neo4j.exceptions.KernelException;
import org.neo4j.graphdb.ConstraintViolationException;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.ResourceIterator;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.graphdb.schema.IndexType;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.kernel.api.IndexQueryConstraints;
import org.neo4j.internal.kernel.api.IndexReadSession;
import org.neo4j.internal.kernel.api.InternalIndexState;
import org.neo4j.internal.kernel.api.PropertyIndexQuery;
import org.neo4j.internal.kernel.api.RelationshipValueIndexCursor;
import org.neo4j.internal.kernel.api.exceptions.schema.IndexNotFoundKernelException;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.io.IOUtils;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.impl.api.index.IndexProxy;
import org.neo4j.kernel.impl.api.index.IndexingService;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.kernel.impl.coreapi.schema.IndexDefinitionImpl;
import org.neo4j.kernel.impl.transaction.log.CheckpointInfo;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.test.RandomSupport;
import org.neo4j.test.TestLabels;
import org.neo4j.test.extension.DbmsController;
import org.neo4j.test.extension.DbmsExtension;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.RandomExtension;

@DbmsExtension
@ExtendWith(value={RandomExtension.class})
class RelationshipTypeIndexIT {
    private static final RelationshipType REL_TYPE = RelationshipType.withName((String)"REL_TYPE");
    private static final RelationshipType OTHER_REL_TYPE = RelationshipType.withName((String)"OTHER_REL_TYPE");
    private static final String PROPERTY = "prop";
    private static final String PROPERTY_VALUE = "value";
    @Inject
    GraphDatabaseService db;
    @Inject
    DbmsController dbmsController;
    @Inject
    FileSystemAbstraction fs;
    @Inject
    RandomSupport random;

    RelationshipTypeIndexIT() {
    }

    @Test
    void shouldSeeAddedRelationship() throws IndexNotFoundKernelException {
        HashSet<String> expectedIds = new HashSet<String>();
        this.createRelationshipInTx(expectedIds);
        this.assertContainsRelationships(expectedIds);
    }

    @Test
    void shouldNotSeeRemovedRelationship() throws IndexNotFoundKernelException {
        String nodeId;
        Node node;
        HashSet<String> expectedIds = new HashSet<String>();
        try (Transaction tx = this.db.beginTx();){
            node = tx.createNode(new Label[]{TestLabels.LABEL_ONE});
            nodeId = node.getElementId();
            node.createRelationshipTo(tx.createNode(), REL_TYPE);
            expectedIds.add(RelationshipTypeIndexIT.createRelationship(tx).getElementId());
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            node = tx.getNodeByElementId(nodeId);
            Iterables.forEach((Iterable)node.getRelationships(), Entity::delete);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.assertContainsRelationships(expectedIds);
    }

    @Test
    void shouldRebuildIfMissingDuringStartup() throws IndexNotFoundKernelException, IOException {
        HashSet<String> expectedIds = new HashSet<String>();
        this.createRelationshipInTx(expectedIds);
        ResourceIterator<Path> files = this.getRelationshipTypeIndexFiles();
        this.dbmsController.restartDbms(builder -> {
            files.forEachRemaining(IOUtils.uncheckedConsumer(file -> this.fs.deleteFile(file)));
            return builder;
        });
        this.awaitIndexesOnline();
        this.assertContainsRelationships(expectedIds);
    }

    @Test
    void shouldPopulateIndex() throws KernelException {
        int numberOfRelationships = 10;
        try (Transaction tx = this.db.beginTx();){
            for (int i = 0; i < numberOfRelationships; ++i) {
                RelationshipTypeIndexIT.createRelationship(tx);
            }
            tx.commit();
        }
        String indexName = this.createFulltextRelationshipIndex();
        org.junit.jupiter.api.Assertions.assertEquals((int)numberOfRelationships, (int)this.countRelationshipsInFulltextIndex(indexName));
    }

    @Test
    void shouldBeRecovered() throws IndexNotFoundKernelException {
        HashSet<String> expectedIds = new HashSet<String>();
        this.createRelationshipInTx(expectedIds);
        this.dbmsController.restartDbms(builder -> {
            this.removeLastCheckpointRecordFromLastLogFile();
            return builder;
        });
        this.awaitIndexesOnline();
        this.assertContainsRelationships(expectedIds);
    }

    @Test
    void shouldRecoverIndex() throws KernelException {
        int numberOfRelationships = 10;
        try (Transaction tx = this.db.beginTx();){
            for (int i = 0; i < numberOfRelationships; ++i) {
                RelationshipTypeIndexIT.createRelationship(tx);
            }
            tx.commit();
        }
        String indexName = this.createFulltextRelationshipIndex();
        this.dbmsController.restartDbms(builder -> {
            this.removeLastCheckpointRecordFromLastLogFile();
            return builder;
        });
        this.awaitIndexesOnline();
        org.junit.jupiter.api.Assertions.assertEquals((int)numberOfRelationships, (int)this.countRelationshipsInFulltextIndex(indexName));
    }

    @Test
    void shouldCorrectlyValidateRelationshipPropertyExistenceConstraint() {
        try (Transaction tx = this.db.beginTx();){
            int invalidRelationship = this.random.nextInt(100);
            for (int i = 0; i < 100; ++i) {
                if (i == invalidRelationship) {
                    tx.createNode().createRelationshipTo(tx.createNode(), REL_TYPE);
                    continue;
                }
                RelationshipTypeIndexIT.createRelationship(tx);
                RelationshipTypeIndexIT.createRelationship(tx, OTHER_REL_TYPE);
            }
            tx.commit();
        }
        org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> {
            try (Transaction tx = this.db.beginTx();){
                tx.schema().constraintFor(REL_TYPE).assertPropertyExists(PROPERTY).create();
                tx.commit();
            }
        });
    }

    private ResourceIterator<Path> getRelationshipTypeIndexFiles() throws IndexNotFoundKernelException, IOException {
        return this.getIndexProxy().snapshotFiles();
    }

    private static Relationship createRelationship(Transaction tx) {
        return RelationshipTypeIndexIT.createRelationship(tx, REL_TYPE);
    }

    private static Relationship createRelationship(Transaction tx, RelationshipType type) {
        Relationship relationship = tx.createNode().createRelationshipTo(tx.createNode(), type);
        relationship.setProperty(PROPERTY, (Object)PROPERTY_VALUE);
        return relationship;
    }

    private void createRelationshipInTx(Set<String> expectedIds) {
        try (Transaction tx = this.db.beginTx();){
            Relationship relationship = RelationshipTypeIndexIT.createRelationship(tx);
            expectedIds.add(relationship.getElementId());
            tx.commit();
        }
    }

    IndexDescriptor findTokenIndex() {
        try (Transaction tx = this.db.beginTx();){
            for (IndexDefinition indexDef : tx.schema().getIndexes()) {
                IndexDescriptor index = ((IndexDefinitionImpl)indexDef).getIndexReference();
                if (!index.schema().isAnyTokenSchemaDescriptor() || index.schema().entityType() != EntityType.RELATIONSHIP || index.getIndexType() != org.neo4j.internal.schema.IndexType.LOOKUP) continue;
                IndexDescriptor indexDescriptor = index;
                return indexDescriptor;
            }
        }
        org.junit.jupiter.api.Assertions.fail((String)"Didn't find expected token index");
        return null;
    }

    private IndexProxy getIndexProxy() throws IndexNotFoundKernelException {
        IndexingService indexingService = (IndexingService)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(IndexingService.class);
        return indexingService.getIndexProxy(this.findTokenIndex());
    }

    private void assertContainsRelationships(Set<String> expectedIds) throws IndexNotFoundKernelException {
        Assertions.assertThat((Comparable)this.getIndexProxy().getState()).isEqualTo((Object)InternalIndexState.ONLINE);
        try (Transaction tx = this.db.beginTx();
             ResourceIterator relationships = tx.findRelationships(REL_TYPE);){
            HashSet<String> actualIds = new HashSet<String>();
            while (relationships.hasNext()) {
                Relationship relationship = (Relationship)relationships.next();
                actualIds.add(relationship.getElementId());
            }
            ((AbstractCollectionAssert)Assertions.assertThat(actualIds).as("contains expected relationships", new Object[0])).isEqualTo(expectedIds);
        }
    }

    private int countRelationshipsInFulltextIndex(String indexName) throws KernelException {
        int relationshipsInIndex;
        try (Transaction transaction = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)transaction).kernelTransaction();
            IndexDescriptor index = ktx.schemaRead().indexGetForName(indexName);
            IndexReadSession indexReadSession = ktx.dataRead().indexReadSession(index);
            relationshipsInIndex = 0;
            try (RelationshipValueIndexCursor cursor = ktx.cursors().allocateRelationshipValueIndexCursor(ktx.cursorContext(), ktx.memoryTracker());){
                ktx.dataRead().relationshipIndexSeek(ktx.queryContext(), indexReadSession, cursor, IndexQueryConstraints.unconstrained(), new PropertyIndexQuery[]{PropertyIndexQuery.fulltextSearch((String)"*")});
                while (cursor.next()) {
                    ++relationshipsInIndex;
                }
            }
        }
        return relationshipsInIndex;
    }

    private String createFulltextRelationshipIndex() {
        String indexName;
        try (Transaction transaction = this.db.beginTx();){
            indexName = transaction.schema().indexFor(REL_TYPE).on(PROPERTY).withIndexType(IndexType.FULLTEXT).create().getName();
            transaction.commit();
        }
        this.awaitIndexesOnline();
        return indexName;
    }

    private void awaitIndexesOnline() {
        try (Transaction transaction = this.db.beginTx();){
            transaction.schema().awaitIndexesOnline(10L, TimeUnit.MINUTES);
            transaction.commit();
        }
    }

    private void removeLastCheckpointRecordFromLastLogFile() {
        block8: {
            try {
                LogFiles logFiles = this.buildLogFiles();
                Optional latestCheckpoint = logFiles.getCheckpointFile().findLatestCheckpoint();
                if (!latestCheckpoint.isPresent()) break block8;
                try (StoreChannel storeChannel = this.fs.write(logFiles.getCheckpointFile().getCurrentFile());){
                    storeChannel.truncate(((CheckpointInfo)latestCheckpoint.get()).checkpointEntryPosition().getByteOffset());
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private LogFiles buildLogFiles() throws IOException {
        DatabaseLayout databaseLayout = ((GraphDatabaseAPI)this.db).databaseLayout();
        return LogFilesBuilder.logFilesBasedOnlyBuilder((Path)databaseLayout.getTransactionLogsDirectory(), (FileSystemAbstraction)this.fs).build();
    }
}

