/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphdb;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.Condition;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.function.Executable;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.StringSearchMode;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexType;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.kernel.api.IndexMonitor;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.kernel.api.index.IndexSample;
import org.neo4j.kernel.impl.api.index.stats.IndexStatisticsStore;
import org.neo4j.kernel.impl.coreapi.schema.IndexDefinitionImpl;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.monitoring.Monitors;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.assertion.Assert;
import org.neo4j.test.conditions.Conditions;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.Neo4jLayoutExtension;

@Neo4jLayoutExtension
public class TextIndexIT {
    @Inject
    protected DatabaseLayout databaseLayout;
    private DatabaseManagementService dbms;

    @AfterEach
    void tearDown() {
        if (this.dbms != null) {
            this.dbms.shutdown();
        }
    }

    @Test
    void shouldNotAllowTextIndexCreationForMultipleTokens() {
        this.dbms = new TestDatabaseManagementServiceBuilder(this.databaseLayout).build();
        GraphDatabaseService db = this.dbms.database("neo4j");
        RelationshipType[] relations = new RelationshipType[]{RelationshipType.withName((String)"FRIEND"), RelationshipType.withName((String)"FROM")};
        Label[] labels = new Label[]{Label.label((String)"PERSON"), Label.label((String)"EMPLOYEE")};
        try (Transaction tx = db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(labels).on("name").withIndexType(IndexType.TEXT).create());
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(relations).on("name").withIndexType(IndexType.TEXT).create());
        }
    }

    @Test
    void shouldRejectIndexCreationWithCompositeKeys() {
        this.dbms = new TestDatabaseManagementServiceBuilder(this.databaseLayout).build();
        GraphDatabaseService db = this.dbms.database("neo4j");
        RelationshipType rel = RelationshipType.withName((String)"FRIEND");
        Label label = Label.label((String)"PERSON");
        try (Transaction tx = db.beginTx();){
            this.assertUnsupported(() -> tx.schema().indexFor(label).on("key1").on("key2").withIndexType(IndexType.TEXT).create());
            this.assertUnsupported(() -> tx.schema().indexFor(rel).on("key1").on("key2").withIndexType(IndexType.TEXT).create());
        }
    }

    private void assertUnsupported(Executable executable) {
        String message = ((UnsupportedOperationException)org.junit.jupiter.api.Assertions.assertThrows(UnsupportedOperationException.class, (Executable)executable)).getMessage();
        Assertions.assertThat((String)message).isEqualTo("Composite indexes are not supported for TEXT index type.");
    }

    @Test
    void shouldCreateAndDropIndexWithCypher() {
        String nodeIndex = "some_node_text_index";
        String relationshipIndex = "some_rel_text_index";
        Label person = Label.label((String)"PERSON");
        RelationshipType relation = RelationshipType.withName((String)"FRIEND");
        this.dbms = new TestDatabaseManagementServiceBuilder(this.databaseLayout).build();
        GraphDatabaseService db = this.dbms.database("neo4j");
        try (Transaction tx = db.beginTx();){
            tx.execute(String.format("CREATE TEXT INDEX %s FOR (p:PERSON) ON (p.name)", nodeIndex));
            tx.execute(String.format("CREATE TEXT INDEX %s FOR ()-[r:FRIEND]-() ON (r.since)", relationshipIndex));
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(5L, TimeUnit.MINUTES);
            Assertions.assertThat((Object)tx.schema().getIndexByName(nodeIndex)).isNotNull();
            Assertions.assertThat((Object)tx.schema().getIndexByName(relationshipIndex)).isNotNull();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = db.beginTx();
        try {
            tx.execute(String.format("DROP INDEX %s", nodeIndex));
            tx.execute(String.format("DROP INDEX %s", relationshipIndex));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = db.beginTx();
        try {
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().getIndexByName(nodeIndex));
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().getIndexByName(relationshipIndex));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldCreateAndDropTextIndexes() {
        String nodeIndex = "some_node_text_index";
        String relationshipIndex = "some_rel_text_index";
        Label person = Label.label((String)"PERSON");
        RelationshipType relation = RelationshipType.withName((String)"FRIEND");
        this.dbms = new TestDatabaseManagementServiceBuilder(this.databaseLayout).build();
        GraphDatabaseService db = this.dbms.database("neo4j");
        try (Transaction tx = db.beginTx();){
            tx.schema().indexFor(person).on("name").withName(nodeIndex).withIndexType(IndexType.TEXT).create();
            tx.schema().indexFor(relation).on("name").withName(relationshipIndex).withIndexType(IndexType.TEXT).create();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(5L, TimeUnit.MINUTES);
            Assertions.assertThat((Object)tx.schema().getIndexByName(nodeIndex)).isNotNull();
            Assertions.assertThat((Object)tx.schema().getIndexByName(relationshipIndex)).isNotNull();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = db.beginTx();
        try {
            tx.schema().getIndexByName(nodeIndex).drop();
            tx.schema().getIndexByName(relationshipIndex).drop();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = db.beginTx();
        try {
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().getIndexByName(nodeIndex));
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().getIndexByName(relationshipIndex));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldFindNodesUsingTextIndex() {
        Label person = Label.label((String)"PERSON");
        IndexAccessMonitor monitor = new IndexAccessMonitor();
        this.dbms = new TestDatabaseManagementServiceBuilder(this.databaseLayout).setMonitors(monitor.monitors()).build();
        GraphDatabaseService db = this.dbms.database("neo4j");
        try (Transaction tx = db.beginTx();){
            tx.schema().indexFor(person).on("name").withIndexType(IndexType.TEXT).create();
            tx.schema().indexFor(person).on("name").withIndexType(IndexType.RANGE).create();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.createNode(new Label[]{person}).setProperty("name", (Object)"David Smith Adams");
            tx.createNode(new Label[]{person}).setProperty("name", (Object)"Smith Evans");
            tx.createNode(new Label[]{person}).setProperty("name", (Object)"Smith James");
            tx.createNode(new Label[]{person}).setProperty("name", (Object)"Luke Smith");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        monitor.reset();
        tx = db.beginTx();
        try {
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findNodes(person, "name", "Smith", StringSearchMode.CONTAINS))).isEqualTo(4L);
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findNodes(person, "name", "Unknown", StringSearchMode.CONTAINS))).isEqualTo(0L);
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findNodes(person, "name", "Smith", StringSearchMode.PREFIX))).isEqualTo(2L);
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findNodes(person, "name", "Smith", StringSearchMode.SUFFIX))).isEqualTo(1L);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        Assertions.assertThat((int)monitor.accessed(org.neo4j.internal.schema.IndexType.TEXT)).isEqualTo(4);
        Assertions.assertThat((int)monitor.accessed(org.neo4j.internal.schema.IndexType.RANGE)).isEqualTo(0);
    }

    @Test
    void shouldFindRelationshipsUsingTextIndex() {
        Label person = Label.label((String)"PERSON");
        RelationshipType relation = RelationshipType.withName((String)"FRIEND");
        IndexAccessMonitor monitor = new IndexAccessMonitor();
        this.dbms = new TestDatabaseManagementServiceBuilder(this.databaseLayout).setMonitors(monitor.monitors()).build();
        GraphDatabaseService db = this.dbms.database("neo4j");
        try (Transaction tx = db.beginTx();){
            tx.schema().indexFor(relation).on("since").withIndexType(IndexType.TEXT).create();
            tx.schema().indexFor(relation).on("since").withIndexType(IndexType.RANGE).create();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.createNode(new Label[]{person}).createRelationshipTo(tx.createNode(new Label[]{person}), relation).setProperty("since", (Object)"two years");
            tx.createNode(new Label[]{person}).createRelationshipTo(tx.createNode(new Label[]{person}), relation).setProperty("since", (Object)"five years, two months");
            tx.createNode(new Label[]{person}).createRelationshipTo(tx.createNode(new Label[]{person}), relation).setProperty("since", (Object)"three months");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        monitor.reset();
        tx = db.beginTx();
        try {
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findRelationships(relation, "since", "years", StringSearchMode.CONTAINS))).isEqualTo(2L);
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findRelationships(relation, "since", "unknown", StringSearchMode.CONTAINS))).isEqualTo(0L);
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findRelationships(relation, "since", "five", StringSearchMode.PREFIX))).isEqualTo(1L);
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findRelationships(relation, "since", "months", StringSearchMode.SUFFIX))).isEqualTo(2L);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        Assertions.assertThat((int)monitor.accessed(org.neo4j.internal.schema.IndexType.TEXT)).isEqualTo(4);
        Assertions.assertThat((int)monitor.accessed(org.neo4j.internal.schema.IndexType.RANGE)).isEqualTo(0);
    }

    @Test
    void shouldRecoverIndexUpdatesAfterCrash() {
        Label person = Label.label((String)"PERSON");
        EphemeralFileSystemAbstraction fs = new EphemeralFileSystemAbstraction();
        this.dbms = this.startDbms((FileSystemAbstraction)fs, new Monitors());
        GraphDatabaseService db = this.dbms.database("neo4j");
        try (Transaction tx = db.beginTx();){
            tx.schema().indexFor(person).on("name").withIndexType(IndexType.TEXT).create();
            tx.commit();
        }
        tx = db.beginTx();
        try {
            tx.createNode(new Label[]{person}).setProperty("name", (Object)"David Smith Adams");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        EphemeralFileSystemAbstraction crashedFs = fs.snapshot();
        this.dbms.shutdown();
        IndexAccessMonitor monitor = new IndexAccessMonitor();
        this.dbms = this.startDbms((FileSystemAbstraction)crashedFs, monitor.monitors());
        db = this.dbms.database("neo4j");
        try (Transaction tx = db.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
            Assertions.assertThat((long)Iterators.count((Iterator)tx.findNodes(person, "name", "Smith", StringSearchMode.CONTAINS))).isEqualTo(1L);
            Assertions.assertThat((int)monitor.accessed(org.neo4j.internal.schema.IndexType.TEXT)).isEqualTo(1);
        }
    }

    @Test
    void shouldSampleIndex() {
        Label person = Label.label((String)"PERSON");
        IndexAccessMonitor monitor = new IndexAccessMonitor();
        this.dbms = new TestDatabaseManagementServiceBuilder(this.databaseLayout).setMonitors(monitor.monitors()).build();
        GraphDatabaseAPI db = (GraphDatabaseAPI)this.dbms.database("neo4j");
        try (Transaction tx = db.beginTx();){
            for (int i = 0; i < 5; ++i) {
                tx.createNode(new Label[]{person}).setProperty("name", (Object)("name" + i));
            }
            tx.commit();
        }
        this.createTextIndex(db, person, "idx_name");
        tx = db.beginTx();
        try {
            tx.createNode(new Label[]{person}).setProperty("name", (Object)"new node");
            tx.findNode(person, "name", (Object)"name0").setProperty("name", (Object)"updated name");
            tx.findNode(person, "name", (Object)"name1").removeProperty("name");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        Assert.assertEventually(() -> this.indexSample(db, "idx_name"), (Condition)Conditions.condition(sample -> sample.indexSize() == 5L && sample.sampleSize() >= 5L && sample.uniqueValues() >= 5L), (long)1L, (TimeUnit)TimeUnit.MINUTES);
    }

    @Test
    void shouldNotIndexNonTextProperties() {
        Node node;
        Label person = Label.label((String)"PERSON");
        IndexAccessMonitor monitor = new IndexAccessMonitor();
        this.dbms = new TestDatabaseManagementServiceBuilder(this.databaseLayout).setMonitors(monitor.monitors()).build();
        GraphDatabaseAPI db = (GraphDatabaseAPI)this.dbms.database("neo4j");
        this.createTextIndex(db, person, "idx_name");
        try (Transaction tx = db.beginTx();){
            node = tx.createNode(new Label[]{person});
            node.setProperty("name", (Object)42);
            tx.commit();
        }
        tx = db.beginTx();
        try {
            Assertions.assertThat((Object)tx.findNode(person, "name", (Object)42)).isEqualTo((Object)node);
            Assertions.assertThat((int)monitor.accessed(org.neo4j.internal.schema.IndexType.TEXT)).isEqualTo(0);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private void createTextIndex(GraphDatabaseAPI db, Label person, String indexName) {
        try (Transaction tx = db.beginTx();){
            tx.schema().indexFor(person).on("name").withIndexType(IndexType.TEXT).withName(indexName).create();
            tx.commit();
        }
    }

    private DatabaseManagementService startDbms(FileSystemAbstraction fs, Monitors monitors) {
        return new TestDatabaseManagementServiceBuilder(this.databaseLayout).setFileSystem(fs).setMonitors(monitors).build();
    }

    private IndexSample indexSample(GraphDatabaseAPI db, String indexName) {
        try (Transaction tx = db.beginTx();){
            IndexDefinitionImpl index = (IndexDefinitionImpl)tx.schema().getIndexByName(indexName);
            IndexSample indexSample = ((IndexStatisticsStore)db.getDependencyResolver().resolveDependency(IndexStatisticsStore.class)).indexSample(index.getIndexReference().getId());
            return indexSample;
        }
    }

    static class IndexAccessMonitor
    extends IndexMonitor.MonitorAdapter {
        private final Map<org.neo4j.internal.schema.IndexType, Integer> counts = new HashMap<org.neo4j.internal.schema.IndexType, Integer>();

        IndexAccessMonitor() {
        }

        public void queried(IndexDescriptor descriptor) {
            this.counts.putIfAbsent(descriptor.getIndexType(), 0);
            this.counts.computeIfPresent(descriptor.getIndexType(), (type, value) -> value + 1);
        }

        public int accessed(org.neo4j.internal.schema.IndexType type) {
            return this.counts.getOrDefault(type, 0);
        }

        Monitors monitors() {
            Monitors monitors = new Monitors();
            monitors.addMonitorListener((Object)this, new String[0]);
            return monitors;
        }

        void reset() {
            this.counts.clear();
        }
    }
}

