/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.core;

import java.util.Arrays;
import java.util.HashSet;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.graphdb.ConstraintViolationException;
import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.ResourceIterable;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.helpers.collection.PrefetchingIterator;
import org.neo4j.kernel.impl.AbstractNeo4jTestCase;
import org.neo4j.kernel.impl.MyRelTypes;

class TestLoopRelationships
extends AbstractNeo4jTestCase {
    TestLoopRelationships() {
    }

    @Test
    void canCreateRelationshipBetweenTwoNodesWithLoopsThenDeleteOneOfTheNodesAndItsRelationships() {
        String sourceId = this.createNode();
        String targetId = this.createNode();
        try (Transaction transaction = this.getGraphDb().beginTx();){
            Node source = transaction.getNodeByElementId(sourceId);
            Node target = transaction.getNodeByElementId(targetId);
            source.createRelationshipTo(source, (RelationshipType)MyRelTypes.TEST);
            target.createRelationshipTo(target, (RelationshipType)MyRelTypes.TEST);
            source.createRelationshipTo(target, (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        transaction = this.getGraphDb().beginTx();
        try {
            Node target = transaction.getNodeByElementId(targetId);
            Iterables.forEach((Iterable)target.getRelationships(), Entity::delete);
            target.delete();
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void canDeleteNodeAfterDeletingItsRelationshipsIfThoseRelationshipsIncludeLoops() {
        String nodeId = this.createNode();
        try (Transaction transaction = this.getGraphDb().beginTx();){
            Node node = transaction.getNodeByElementId(nodeId);
            TestLoopRelationships.txCreateLoop(node);
            TestLoopRelationships.txCreateRel(transaction, node);
            TestLoopRelationships.txCreateLoop(node);
            Iterables.forEach((Iterable)node.getRelationships(), Entity::delete);
            node.delete();
            transaction.commit();
        }
    }

    private static void txCreateRel(Transaction transaction, Node node) {
        node.createRelationshipTo(transaction.createNode(), (RelationshipType)MyRelTypes.TEST);
    }

    private static void txCreateLoop(Node node) {
        node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
    }

    @Test
    void canAddLoopRelationship() {
        Node node;
        String nodeId = this.createNode();
        try (Transaction transaction = this.getGraphDb().beginTx();){
            node = transaction.getNodeByElementId(nodeId);
            node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        transaction = this.getGraphDb().beginTx();
        try {
            node = transaction.getNodeByElementId(nodeId);
            for (Direction dir : Direction.values()) {
                int count = 0;
                try (ResourceIterable relationships = node.getRelationships(dir);){
                    for (Relationship rel : relationships) {
                        org.junit.jupiter.api.Assertions.assertEquals((Object)node, (Object)rel.getStartNode(), (String)"start node");
                        org.junit.jupiter.api.Assertions.assertEquals((Object)node, (Object)rel.getEndNode(), (String)"end node");
                        org.junit.jupiter.api.Assertions.assertEquals((Object)node, (Object)rel.getOtherNode(node), (String)"other node");
                        ++count;
                    }
                }
                org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)count, (String)(dir.name() + " relationship count"));
            }
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void canAddManyLoopRelationships() {
        this.testAddManyLoopRelationships(2);
        this.testAddManyLoopRelationships(3);
        this.testAddManyLoopRelationships(5);
    }

    private void testAddManyLoopRelationships(int count) {
        for (boolean[] loop : TestLoopRelationships.permutations(count)) {
            Relationship[] relationships = new Relationship[count];
            String rootId = this.createNode();
            try (Transaction transaction = this.getGraphDb().beginTx();){
                Node root = transaction.getNodeByElementId(rootId);
                for (int i = 0; i < count; ++i) {
                    relationships[i] = loop[i] ? root.createRelationshipTo(root, (RelationshipType)MyRelTypes.TEST) : root.createRelationshipTo(transaction.createNode(), (RelationshipType)MyRelTypes.TEST);
                }
                transaction.commit();
            }
            this.verifyRelationships(Arrays.toString(loop), rootId, loop, relationships);
        }
    }

    @Test
    void canAddLoopRelationshipAndOtherRelationships() {
        this.testAddLoopRelationshipAndOtherRelationships(2);
        this.testAddLoopRelationshipAndOtherRelationships(3);
        this.testAddLoopRelationshipAndOtherRelationships(5);
    }

    private void testAddLoopRelationshipAndOtherRelationships(int size) {
        for (int i = 0; i < size; ++i) {
            String root = this.createNode();
            Relationship[] relationships = this.createRelationships(size, i, root);
            this.verifyRelationships(String.format("loop on %s of %s", i, size), root, i, relationships);
        }
    }

    @Test
    void canAddAndRemoveLoopRelationshipAndOtherRelationships() {
        this.testAddAndRemoveLoopRelationshipAndOtherRelationships(2);
        this.testAddAndRemoveLoopRelationshipAndOtherRelationships(3);
        this.testAddAndRemoveLoopRelationshipAndOtherRelationships(5);
    }

    @Test
    void getSingleRelationshipOnNodeWithOneLoopOnly() {
        Relationship singleRelationship;
        Node node;
        String nodeId = this.createNode();
        try (Transaction transaction = this.getGraphDb().beginTx();){
            node = transaction.getNodeByElementId(nodeId);
            singleRelationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.OUTGOING));
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.BOTH));
            transaction.commit();
        }
        transaction = this.getGraphDb().beginTx();
        try {
            node = transaction.getNodeByElementId(nodeId);
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.OUTGOING));
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.BOTH));
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void cannotDeleteNodeWithLoopStillAttached() {
        Node node;
        GraphDatabaseService db = this.getGraphDb();
        try (Transaction tx = db.beginTx();){
            node = tx.createNode();
            node.createRelationshipTo(node, RelationshipType.withName((String)"MAYOR_OF"));
            tx.commit();
        }
        try (Transaction transaction = this.getGraphDb().beginTx();){
            transaction.getNodeById(node.getId()).delete();
            ConstraintViolationException e = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((Transaction)transaction).commit());
            Assertions.assertThat((String)e.getMessage()).isEqualTo("Cannot delete node<" + node.getId() + ">, because it still has relationships. To delete this node, you must first delete its relationships.");
        }
    }

    @Test
    void getOtherNodeFunctionsCorrectly() {
        String relationship;
        String nodeId = this.createNode();
        try (Transaction transaction = this.getGraphDb().beginTx();){
            Node node = transaction.getNodeByElementId(nodeId);
            relationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST).getElementId();
            transaction.commit();
        }
        for (int i = 0; i < 2; ++i) {
            try (Transaction transaction = this.getGraphDb().beginTx();){
                Node node = transaction.getNodeByElementId(nodeId);
                Relationship rel = transaction.getRelationshipByElementId(relationship);
                org.junit.jupiter.api.Assertions.assertEquals((Object)node, (Object)rel.getOtherNode(node));
                org.junit.jupiter.api.Assertions.assertEquals(Arrays.asList(node, node), Arrays.asList(rel.getNodes()));
                org.junit.jupiter.api.Assertions.assertThrows(NotFoundException.class, () -> rel.getOtherNode(transaction.createNode()));
                transaction.commit();
                continue;
            }
        }
    }

    @Test
    void getNewlyCreatedLoopRelationshipFromCache() {
        Relationship relationship;
        Node node;
        String nodeId = this.createNode();
        try (Transaction transaction = this.getGraphDb().beginTx();){
            Node node2 = transaction.getNodeByElementId(nodeId);
            node2.createRelationshipTo(transaction.createNode(), (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        try (Transaction transaction = this.getGraphDb().beginTx();){
            node = transaction.getNodeByElementId(nodeId);
            relationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        transaction = this.getGraphDb().beginTx();
        try {
            node = transaction.getNodeByElementId(nodeId);
            org.junit.jupiter.api.Assertions.assertEquals((Object)relationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    private void testAddAndRemoveLoopRelationshipAndOtherRelationships(int size) {
        for (boolean[] delete : TestLoopRelationships.permutations(size)) {
            for (int i = 0; i < size; ++i) {
                String root = this.createNode();
                Relationship[] relationships = this.createRelationships(size, i, root);
                try (Transaction transaction = this.getGraphDb().beginTx();){
                    for (int j = 0; j < size; ++j) {
                        if (!delete[j]) continue;
                        transaction.getRelationshipById(relationships[j].getId()).delete();
                        relationships[j] = null;
                    }
                    transaction.commit();
                }
                this.verifyRelationships(String.format("loop on %s of %s, delete %s", i, size, Arrays.toString(delete)), root, i, relationships);
            }
        }
    }

    private static Iterable<boolean[]> permutations(final int size) {
        final int max = 1 << size;
        return () -> new PrefetchingIterator<boolean[]>(){
            int pos;

            protected boolean[] fetchNextOrNull() {
                if (this.pos < max) {
                    int cur = this.pos++;
                    boolean[] result = new boolean[size];
                    for (int i = 0; i < size; ++i) {
                        result[i] = (cur & 1) == 1;
                        cur >>= 1;
                    }
                    return result;
                }
                return null;
            }
        };
    }

    private Relationship[] createRelationships(int count, int loop, String root) {
        String[] nodes = new String[count];
        for (int i = 0; i < count; ++i) {
            nodes[i] = loop == i ? root : this.createNode();
        }
        Relationship[] relationships = new Relationship[count];
        try (Transaction transaction = this.getGraphDb().beginTx();){
            Node node = transaction.getNodeByElementId(root);
            for (int i = 0; i < count; ++i) {
                relationships[i] = node.createRelationshipTo(transaction.getNodeByElementId(nodes[i]), (RelationshipType)MyRelTypes.TEST);
            }
            transaction.commit();
        }
        return relationships;
    }

    private void verifyRelationships(String message, String root, int loop, Relationship ... relationships) {
        boolean[] loops = new boolean[relationships.length];
        for (int i = 0; i < relationships.length; ++i) {
            loops[i] = i == loop;
        }
        this.verifyRelationships(message, root, loops, relationships);
    }

    private void verifyRelationships(String message, String node, boolean[] loop, Relationship ... relationships) {
        try (Transaction transaction = this.getGraphDb().beginTx();){
            Node root = transaction.getNodeByElementId(node);
            for (Direction dir : Direction.values()) {
                HashSet<Relationship> expected = new HashSet<Relationship>();
                for (int i = 0; i < relationships.length; ++i) {
                    if (relationships[i] == null || dir == Direction.INCOMING && !loop[i]) continue;
                    expected.add(relationships[i]);
                }
                try (ResourceIterable rootRels = root.getRelationships(dir);){
                    for (Relationship rel : rootRels) {
                        org.junit.jupiter.api.Assertions.assertTrue((boolean)expected.remove(rel), (String)(message + ": unexpected relationship: " + String.valueOf(rel)));
                    }
                }
                org.junit.jupiter.api.Assertions.assertTrue((boolean)expected.isEmpty(), (String)(message + ": expected relationships not seen " + String.valueOf(expected)));
            }
            transaction.commit();
        }
    }
}

