/*
 * Decompiled with CFR 0.152.
 */
package recovery;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.mockfs.EphemeralFileSystemAbstraction;
import org.neo4j.helpers.collection.Iterators;
import org.neo4j.internal.kernel.api.Kernel;
import org.neo4j.internal.kernel.api.Transaction;
import org.neo4j.internal.kernel.api.security.LoginContext;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.pagecache.IOLimiter;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.kernel.api.index.IndexProvider;
import org.neo4j.kernel.impl.api.index.inmemory.InMemoryIndexProvider;
import org.neo4j.kernel.impl.api.index.inmemory.InMemoryIndexProviderFactory;
import org.neo4j.kernel.impl.core.TokenHolder;
import org.neo4j.kernel.impl.core.TokenHolders;
import org.neo4j.kernel.impl.transaction.log.checkpoint.CheckPointer;
import org.neo4j.kernel.impl.transaction.log.checkpoint.SimpleTriggerInfo;
import org.neo4j.kernel.impl.transaction.log.checkpoint.TriggerInfo;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.storageengine.api.StorageEngine;
import org.neo4j.test.TestGraphDatabaseFactory;
import org.neo4j.test.rule.fs.EphemeralFileSystemRule;

@RunWith(value=Parameterized.class)
public class TestRecoveryScenarios {
    @Rule
    public final EphemeralFileSystemRule fsRule = new EphemeralFileSystemRule();
    private final Label label = Label.label((String)"label");
    private GraphDatabaseAPI db;
    private final InMemoryIndexProvider indexProvider = new InMemoryIndexProvider(100);
    private final FlushStrategy flush;

    @Test
    public void shouldRecoverTransactionWhereNodeIsDeletedInTheFuture() throws Exception {
        Node node = this.createNodeWithProperty("key", "value", this.label);
        this.checkPoint();
        this.setProperty(node, "other-key", 1);
        this.deleteNode(node);
        this.flush.flush(this.db);
        this.crashAndRestart(this.indexProvider);
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            node = this.db.getNodeById(node.getId());
            tx.success();
            Assert.fail((String)"Should not exist");
        }
        catch (NotFoundException e) {
            Assert.assertEquals((Object)("Node " + node.getId() + " not found"), (Object)e.getMessage());
        }
    }

    @Test
    public void shouldRecoverTransactionWherePropertyIsRemovedInTheFuture() throws Exception {
        this.createIndex(this.label, "key");
        Node node = this.createNodeWithProperty("key", "value", new Label[0]);
        this.checkPoint();
        this.addLabel(node, this.label);
        InMemoryIndexProvider outdatedIndexProvider = this.indexProvider.snapshot();
        this.removeProperty(node, "key");
        this.flush.flush(this.db);
        this.crashAndRestart(outdatedIndexProvider);
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            Assert.assertEquals((String)"Updates not propagated correctly during recovery", Collections.emptyList(), (Object)Iterators.asList((Iterator)this.db.findNodes(this.label, "key", (Object)"value")));
            tx.success();
        }
    }

    @Test
    public void shouldRecoverTransactionWhereManyLabelsAreRemovedInTheFuture() throws Exception {
        Node node;
        this.createIndex(this.label, "key");
        Label[] labels = new Label[16];
        for (int i = 0; i < labels.length; ++i) {
            labels[i] = Label.label((String)("Label" + Integer.toHexString(i)));
        }
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            node = this.db.createNode(labels);
            node.addLabel(this.label);
            tx.success();
        }
        this.checkPoint();
        InMemoryIndexProvider outdatedIndexProvider = this.indexProvider.snapshot();
        this.setProperty(node, "key", "value");
        this.removeLabels(node, labels);
        this.flush.flush(this.db);
        this.crashAndRestart(outdatedIndexProvider);
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            Assert.assertEquals((Object)node, (Object)this.db.findNode(this.label, "key", (Object)"value"));
            tx.success();
        }
    }

    @Test
    public void shouldRecoverCounts() throws Exception {
        Node node = this.createNode(this.label);
        this.checkPoint();
        this.deleteNode(node);
        this.crashAndRestart(this.indexProvider);
        try (Transaction tx = ((Kernel)this.db.getDependencyResolver().resolveDependency(Kernel.class)).beginTransaction(Transaction.Type.explicit, LoginContext.AUTH_DISABLED);){
            Assert.assertEquals((long)0L, (long)tx.dataRead().countsForNode(-1));
            TokenHolder holder = ((TokenHolders)this.db.getDependencyResolver().resolveDependency(TokenHolders.class)).labelTokens();
            int labelId = holder.getIdByName(this.label.name());
            Assert.assertEquals((long)0L, (long)tx.dataRead().countsForNode(labelId));
            tx.success();
        }
    }

    private void removeLabels(Node node, Label ... labels) {
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            for (Label label : labels) {
                node.removeLabel(label);
            }
            tx.success();
        }
    }

    private void removeProperty(Node node, String key) {
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            node.removeProperty(key);
            tx.success();
        }
    }

    private void addLabel(Node node, Label label) {
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            node.addLabel(label);
            tx.success();
        }
    }

    private Node createNode(Label ... labels) {
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            Node node = this.db.createNode(labels);
            tx.success();
            Node node2 = node;
            return node2;
        }
    }

    private Node createNodeWithProperty(String key, String value, Label ... labels) {
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            Node node = this.db.createNode(labels);
            node.setProperty(key, (Object)value);
            tx.success();
            Node node2 = node;
            return node2;
        }
    }

    private void createIndex(Label label, String key) {
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            this.db.schema().indexFor(label).on(key).create();
            tx.success();
        }
        tx = this.db.beginTx();
        var4_4 = null;
        try {
            this.db.schema().awaitIndexesOnline(10L, TimeUnit.SECONDS);
            tx.success();
        }
        catch (Throwable throwable) {
            var4_4 = throwable;
            throw throwable;
        }
        finally {
            if (tx != null) {
                if (var4_4 != null) {
                    try {
                        tx.close();
                    }
                    catch (Throwable throwable) {
                        var4_4.addSuppressed(throwable);
                    }
                } else {
                    tx.close();
                }
            }
        }
    }

    @Parameterized.Parameters(name="{0}")
    public static List<Object[]> flushStrategy() {
        ArrayList<Object[]> parameters = new ArrayList<Object[]>();
        for (FlushStrategy flushStrategy : FlushStrategy.values()) {
            parameters.add(flushStrategy.parameters);
        }
        return parameters;
    }

    public TestRecoveryScenarios(FlushStrategy flush) {
        this.flush = flush;
    }

    @Before
    public void before() {
        this.db = (GraphDatabaseAPI)this.databaseFactory(this.fsRule.get(), this.indexProvider).newImpermanentDatabase();
    }

    private TestGraphDatabaseFactory databaseFactory(FileSystemAbstraction fs, InMemoryIndexProvider indexProvider) {
        return new TestGraphDatabaseFactory().setFileSystem(fs).setKernelExtensions(Arrays.asList(new InMemoryIndexProviderFactory((IndexProvider)indexProvider)));
    }

    @After
    public void after() {
        this.db.shutdown();
    }

    private void checkPoint() throws IOException {
        ((CheckPointer)this.db.getDependencyResolver().resolveDependency(CheckPointer.class)).forceCheckPoint((TriggerInfo)new SimpleTriggerInfo("test"));
    }

    private void deleteNode(Node node) {
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            node.delete();
            tx.success();
        }
    }

    private void setProperty(Node node, String key, Object value) {
        try (org.neo4j.graphdb.Transaction tx = this.db.beginTx();){
            node.setProperty(key, value);
            tx.success();
        }
    }

    private void crashAndRestart(InMemoryIndexProvider indexProvider) throws Exception {
        GraphDatabaseAPI db1 = this.db;
        EphemeralFileSystemAbstraction uncleanFs = this.fsRule.snapshot(() -> ((GraphDatabaseService)db1).shutdown());
        this.db = (GraphDatabaseAPI)this.databaseFactory((FileSystemAbstraction)uncleanFs, indexProvider).newImpermanentDatabase();
    }

    public static enum FlushStrategy {
        FORCE_EVERYTHING{

            @Override
            void flush(GraphDatabaseAPI db) {
                IOLimiter limiter = IOLimiter.unlimited();
                ((StorageEngine)db.getDependencyResolver().resolveDependency(StorageEngine.class)).flushAndForce(limiter);
            }
        }
        ,
        FLUSH_PAGE_CACHE{

            @Override
            void flush(GraphDatabaseAPI db) throws IOException {
                ((PageCache)db.getDependencyResolver().resolveDependency(PageCache.class)).flushAndForce();
            }
        };

        final Object[] parameters = new Object[]{this};

        abstract void flush(GraphDatabaseAPI var1) throws IOException;
    }
}

