/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.api.index;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.lang3.StringUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.neo4j.graphdb.DependencyResolver;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.helpers.collection.Iterables;
import org.neo4j.internal.kernel.api.IndexOrder;
import org.neo4j.internal.kernel.api.IndexQuery;
import org.neo4j.internal.kernel.api.IndexReference;
import org.neo4j.internal.kernel.api.NodeValueIndexCursor;
import org.neo4j.internal.kernel.api.exceptions.KernelException;
import org.neo4j.internal.kernel.api.exceptions.schema.IndexNotFoundKernelException;
import org.neo4j.internal.kernel.api.schema.SchemaDescriptor;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.api.Statement;
import org.neo4j.kernel.api.schema.LabelSchemaDescriptor;
import org.neo4j.kernel.api.schema.SchemaDescriptorFactory;
import org.neo4j.kernel.impl.api.index.BatchingMultipleIndexPopulator;
import org.neo4j.kernel.impl.api.index.IndexingService;
import org.neo4j.kernel.impl.api.index.MultipleIndexPopulator;
import org.neo4j.kernel.impl.api.index.UpdatesTracker;
import org.neo4j.kernel.impl.core.ThreadToStatementContextBridge;
import org.neo4j.kernel.impl.storageengine.impl.recordstorage.RecordStorageEngine;
import org.neo4j.kernel.impl.store.NeoStores;
import org.neo4j.kernel.impl.store.RecordStore;
import org.neo4j.kernel.impl.store.SchemaStorage;
import org.neo4j.kernel.impl.store.counts.CountsTracker;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.monitoring.Monitors;
import org.neo4j.register.Register;
import org.neo4j.register.Registers;
import org.neo4j.storageengine.api.schema.IndexDescriptor;
import org.neo4j.storageengine.api.schema.StoreIndexDescriptor;
import org.neo4j.test.Barrier;
import org.neo4j.test.rule.DatabaseRule;
import org.neo4j.test.rule.EmbeddedDatabaseRule;
import org.neo4j.test.rule.RandomRule;
import org.neo4j.util.FeatureToggles;
import org.neo4j.values.storable.Values;

@RunWith(value=Parameterized.class)
public class IndexStatisticsTest {
    private static final double UNIQUE_NAMES = 10.0;
    private static final String[] NAMES = new String[]{"Andres", "Davide", "Jakub", "Chris", "Tobias", "Stefan", "Petra", "Rickard", "Mattias", "Emil", "Chris", "Chris"};
    private static final int CREATION_MULTIPLIER = Integer.getInteger(IndexStatisticsTest.class.getName() + ".creationMultiplier", 1000);
    private static final String PERSON_LABEL = "Person";
    private static final String NAME_PROPERTY = "name";
    @Parameterized.Parameter
    public boolean multiThreadedPopulationEnabled;
    @Rule
    public final DatabaseRule dbRule = new EmbeddedDatabaseRule().withSetting(GraphDatabaseSettings.index_background_sampling_enabled, "false").startLazily();
    @Rule
    public final RandomRule random = new RandomRule();
    private GraphDatabaseService db;
    private ThreadToStatementContextBridge bridge;
    private final IndexOnlineMonitor indexOnlineMonitor = new IndexOnlineMonitor();

    @Parameterized.Parameters(name="multiThreadedIndexPopulationEnabled = {0}")
    public static Object[] multiThreadedIndexPopulationEnabledValues() {
        return new Object[]{true, false};
    }

    @Before
    public void before() {
        this.dbRule.withSetting(GraphDatabaseSettings.multi_threaded_schema_index_population_enabled, this.multiThreadedPopulationEnabled + "");
        int batchSize = this.random.nextInt(1, 5);
        FeatureToggles.set(MultipleIndexPopulator.class, (String)"queue_threshold", (Object)batchSize);
        FeatureToggles.set(BatchingMultipleIndexPopulator.class, (String)"queue_threshold", (Object)batchSize);
        GraphDatabaseAPI graphDatabaseAPI = this.dbRule.getGraphDatabaseAPI();
        this.db = graphDatabaseAPI;
        DependencyResolver dependencyResolver = graphDatabaseAPI.getDependencyResolver();
        this.bridge = (ThreadToStatementContextBridge)dependencyResolver.resolveDependency(ThreadToStatementContextBridge.class);
        ((Monitors)graphDatabaseAPI.getDependencyResolver().resolveDependency(Monitors.class)).addMonitorListener((Object)this.indexOnlineMonitor, new String[0]);
    }

    @After
    public void tearDown() {
        FeatureToggles.clear(MultipleIndexPopulator.class, (String)"queue_threshold");
        FeatureToggles.clear(BatchingMultipleIndexPopulator.class, (String)"queue_threshold");
    }

    @Test
    public void shouldProvideIndexStatisticsForDataCreatedWhenPopulationBeforeTheIndexIsOnline() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexReference index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        Assert.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)0.0);
        Assert.assertEquals((long)4L, (long)this.indexSize(index));
        Assert.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldNotSeeDataCreatedAfterPopulation() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        IndexReference index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.createSomePersons();
        Assert.assertEquals((double)1.0, (double)this.indexSelectivity(index), (double)0.0);
        Assert.assertEquals((long)0L, (long)this.indexSize(index));
        Assert.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsForDataSeenDuringPopulationAndIgnoreDataCreatedAfterPopulation() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexReference index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.createSomePersons();
        Assert.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)0.0);
        Assert.assertEquals((long)4L, (long)this.indexSize(index));
        Assert.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldRemoveIndexStatisticsAfterIndexIsDeleted() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexReference index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        SchemaStorage storage = new SchemaStorage((RecordStore)this.neoStores().getSchemaStore());
        long indexId = storage.indexGetForSchema((IndexDescriptor)index).getId();
        this.dropIndex(index);
        try {
            this.indexSelectivity(index);
            Assert.fail((String)"Expected IndexNotFoundKernelException to be thrown");
        }
        catch (IndexNotFoundKernelException e) {
            Register.DoubleLongRegister actual = this.getTracker().indexSample(indexId, Registers.newDoubleLongRegister());
            this.assertDoubleLongEquals(0L, 0L, actual);
        }
        Register.DoubleLongRegister actual = this.getTracker().indexUpdatesAndSize(indexId, Registers.newDoubleLongRegister());
        this.assertDoubleLongEquals(0L, 0L, actual);
    }

    @Test
    public void shouldProvideIndexSelectivityWhenThereAreManyDuplicates() throws Exception {
        this.indexOnlineMonitor.initialize(0);
        int created = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER).length;
        IndexReference index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        double expectedSelectivity = 10.0 / (double)created;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(created, this.indexSize(index));
        Assert.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        int initialNodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER).length;
        IndexReference index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreations(CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        IndexStatisticsTest.assertCorrectIndexUpdates(updatesTracker.createdAfterPopulation(), this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndDeletions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexReference index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsAndDeletions(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChanges() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexReference index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsAndUpdates(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.createdAfterPopulation() + updatesTracker.updatedAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChangesAndDeletions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexReference index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsDeletionsAndUpdates(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation() + updatesTracker.updatedAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldWorkWhileHavingHeavyConcurrentUpdates() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        int threads = 5;
        this.indexOnlineMonitor.initialize(threads);
        ExecutorService executorService = Executors.newFixedThreadPool(threads);
        IndexReference index = this.createPersonNameIndex();
        ArrayList<Callable<UpdatesTracker>> jobs = new ArrayList<Callable<UpdatesTracker>>(threads);
        for (int i = 0; i < threads; ++i) {
            jobs.add(() -> this.executeCreationsDeletionsAndUpdates(nodes, CREATION_MULTIPLIER));
        }
        List futures = executorService.invokeAll(jobs);
        UpdatesTracker result = new UpdatesTracker();
        result.notifyPopulationCompleted();
        for (Future future : futures) {
            result.add((UpdatesTracker)future.get());
        }
        this.awaitIndexesOnline();
        executorService.shutdown();
        Assert.assertTrue((boolean)executorService.awaitTermination(1L, TimeUnit.MINUTES));
        this.assertIndexedNodesMatchesStoreNodes();
        int seenWhilePopulating = initialNodes + result.createdDuringPopulation() - result.deletedDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize("Tracker had " + result, seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = result.deletedAfterPopulation() + result.createdAfterPopulation() + result.updatedAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexUpdates("Tracker had " + result, expectedIndexUpdates, this.indexUpdates(index));
    }

    private void assertIndexedNodesMatchesStoreNodes() throws Exception {
        int nodesInStore = 0;
        Label label = Label.label((String)PERSON_LABEL);
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((ThreadToStatementContextBridge)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(ThreadToStatementContextBridge.class)).getKernelTransactionBoundToThisThread(true);
            ArrayList<String> mismatches = new ArrayList<String>();
            int labelId = ktx.tokenRead().nodeLabel(PERSON_LABEL);
            int propertyKeyId = ktx.tokenRead().propertyKey(NAME_PROPERTY);
            IndexReference index = ktx.schemaRead().index(labelId, new int[]{propertyKeyId});
            try (NodeValueIndexCursor cursor = ktx.cursors().allocateNodeValueIndexCursor();){
                for (Node node : Iterables.filter(n -> n.hasLabel(label) && n.hasProperty(NAME_PROPERTY), (Iterable)this.db.getAllNodes())) {
                    ++nodesInStore;
                    String name = (String)node.getProperty(NAME_PROPERTY);
                    ktx.dataRead().nodeIndexSeek(index, cursor, IndexOrder.NONE, false, new IndexQuery[]{IndexQuery.exact((int)propertyKeyId, (Object)name)});
                    boolean found = false;
                    while (cursor.next()) {
                        long indexedNode = cursor.nodeReference();
                        if (indexedNode != node.getId()) continue;
                        if (found) {
                            mismatches.add("Index has multiple entries for " + name + " and " + indexedNode);
                        }
                        found = true;
                    }
                    if (found) continue;
                    mismatches.add("Index is missing entry for " + name);
                }
                if (!mismatches.isEmpty()) {
                    Assert.fail((String)StringUtils.join((Object[])mismatches.toArray(), (String)String.format("%n", new Object[0])));
                }
                ktx.dataRead().nodeIndexSeek(index, cursor, IndexOrder.NONE, false, new IndexQuery[]{IndexQuery.exists((int)propertyKeyId)});
                int nodesInIndex = 0;
                while (cursor.next()) {
                    ++nodesInIndex;
                }
                Assert.assertEquals((long)nodesInStore, (long)nodesInIndex);
            }
        }
    }

    private void deleteNode(long nodeId) {
        try (Transaction tx = this.db.beginTx();){
            this.db.getNodeById(nodeId).delete();
            tx.success();
        }
    }

    private boolean changeName(long nodeId, Object newValue) {
        boolean changeIndexedNode = false;
        try (Transaction tx = this.db.beginTx();){
            Node node = this.db.getNodeById(nodeId);
            Object oldValue = node.getProperty(NAME_PROPERTY);
            if (!oldValue.equals(newValue)) {
                changeIndexedNode = true;
            }
            node.setProperty(NAME_PROPERTY, newValue);
            tx.success();
        }
        return changeIndexedNode;
    }

    private int createNamedPeople(long[] nodes, int offset) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = this.bridge.getKernelTransactionBoundToThisThread(true);
            for (String name : NAMES) {
                long nodeId = this.createPersonNode(ktx, name);
                if (nodes == null) continue;
                nodes[offset++] = nodeId;
            }
            tx.success();
        }
        return NAMES.length;
    }

    private long[] repeatCreateNamedPeopleFor(int totalNumberOfPeople) throws Exception {
        long[] nodes = new long[totalNumberOfPeople];
        int threads = 100;
        int peoplePerThread = totalNumberOfPeople / 100;
        ExecutorService service = Executors.newFixedThreadPool(100);
        AtomicReference exception = new AtomicReference();
        ArrayList<Callable<Void>> jobs = new ArrayList<Callable<Void>>(100);
        int i = 0;
        while (i < 100) {
            int finalI = i++;
            jobs.add(() -> {
                for (int offset = finalI * peoplePerThread; offset < (finalI + 1) * peoplePerThread; offset += this.createNamedPeople(nodes, offset)) {
                    try {
                        continue;
                    }
                    catch (KernelException e) {
                        exception.compareAndSet(null, e);
                        throw new RuntimeException(e);
                    }
                }
                return null;
            });
        }
        for (Future job : service.invokeAll(jobs)) {
            job.get();
        }
        service.awaitTermination(1L, TimeUnit.SECONDS);
        service.shutdown();
        Exception ex = (Exception)exception.get();
        if (ex != null) {
            throw ex;
        }
        return nodes;
    }

    private void dropIndex(IndexReference index) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = this.bridge.getKernelTransactionBoundToThisThread(true);
            try (Statement ignore = ktx.acquireStatement();){
                ktx.schemaWrite().indexDrop(index);
            }
            tx.success();
        }
    }

    private long indexSize(IndexReference reference) throws KernelException {
        return ((IndexingService)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(IndexingService.class)).indexUpdatesAndSize(reference.schema()).readSecond();
    }

    private long indexUpdates(IndexReference reference) throws KernelException {
        return ((IndexingService)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(IndexingService.class)).indexUpdatesAndSize(reference.schema()).readFirst();
    }

    private double indexSelectivity(IndexReference reference) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            double selectivity = this.getSelectivity(reference);
            tx.success();
            double d = selectivity;
            return d;
        }
    }

    private double getSelectivity(IndexReference reference) throws IndexNotFoundKernelException {
        return this.bridge.getKernelTransactionBoundToThisThread(true).schemaRead().indexUniqueValuesSelectivity(reference);
    }

    private CountsTracker getTracker() {
        return ((RecordStorageEngine)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(RecordStorageEngine.class)).testAccessNeoStores().getCounts();
    }

    private void createSomePersons() throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = this.bridge.getKernelTransactionBoundToThisThread(true);
            this.createPersonNode(ktx, "Davide");
            this.createPersonNode(ktx, "Stefan");
            this.createPersonNode(ktx, "John");
            this.createPersonNode(ktx, "John");
            tx.success();
        }
    }

    private long createPersonNode(KernelTransaction ktx, Object value) throws KernelException {
        int labelId = ktx.tokenWrite().labelGetOrCreateForName(PERSON_LABEL);
        int propertyKeyId = ktx.tokenWrite().propertyKeyGetOrCreateForName(NAME_PROPERTY);
        long nodeId = ktx.dataWrite().nodeCreate();
        ktx.dataWrite().nodeAddLabel(nodeId, labelId);
        ktx.dataWrite().nodeSetProperty(nodeId, propertyKeyId, Values.of((Object)value));
        return nodeId;
    }

    private IndexReference createPersonNameIndex() throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            IndexReference index;
            KernelTransaction ktx = this.bridge.getKernelTransactionBoundToThisThread(true);
            try (Statement ignore = ktx.acquireStatement();){
                int labelId = ktx.tokenWrite().labelGetOrCreateForName(PERSON_LABEL);
                int propertyKeyId = ktx.tokenWrite().propertyKeyGetOrCreateForName(NAME_PROPERTY);
                LabelSchemaDescriptor descriptor = SchemaDescriptorFactory.forLabel((int)labelId, (int[])new int[]{propertyKeyId});
                index = ktx.schemaWrite().indexCreate((SchemaDescriptor)descriptor);
            }
            tx.success();
            IndexReference indexReference = index;
            return indexReference;
        }
    }

    private NeoStores neoStores() {
        return ((RecordStorageEngine)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(RecordStorageEngine.class)).testAccessNeoStores();
    }

    private void awaitIndexesOnline() {
        try (Transaction ignored = this.db.beginTx();){
            this.db.schema().awaitIndexesOnline(3L, TimeUnit.MINUTES);
        }
    }

    private UpdatesTracker executeCreations(int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(null, numberOfCreations, false, false);
    }

    private UpdatesTracker executeCreationsAndDeletions(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, true, false);
    }

    private UpdatesTracker executeCreationsAndUpdates(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, false, true);
    }

    private UpdatesTracker executeCreationsDeletionsAndUpdates(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, true, true);
    }

    private UpdatesTracker internalExecuteCreationsDeletionsAndUpdates(long[] nodes, int numberOfCreations, boolean allowDeletions, boolean allowUpdates) throws KernelException, InterruptedException {
        if (this.random.nextBoolean()) {
            this.indexOnlineMonitor.startSignal.await();
        }
        ThreadLocalRandom random = ThreadLocalRandom.current();
        UpdatesTracker updatesTracker = new UpdatesTracker();
        int offset = 0;
        while (updatesTracker.created() < numberOfCreations) {
            int created = this.createNamedPeople(nodes, offset);
            offset += created;
            updatesTracker.increaseCreated(created);
            this.notifyIfPopulationCompleted(updatesTracker);
            if (allowDeletions && updatesTracker.created() % 24 == 0) {
                long nodeId = nodes[((Random)random).nextInt(nodes.length)];
                try {
                    this.deleteNode(nodeId);
                    updatesTracker.increaseDeleted(1);
                }
                catch (NotFoundException notFoundException) {
                    // empty catch block
                }
                this.notifyIfPopulationCompleted(updatesTracker);
            }
            if (!allowUpdates || updatesTracker.created() % 24 != 0) continue;
            int randomIndex = ((Random)random).nextInt(nodes.length);
            try {
                if (this.changeName(nodes[randomIndex], NAMES[((Random)random).nextInt(NAMES.length)])) {
                    updatesTracker.increaseUpdated(1);
                }
            }
            catch (NotFoundException notFoundException) {
                // empty catch block
            }
            this.notifyIfPopulationCompleted(updatesTracker);
        }
        this.notifyPopulationCompleted(updatesTracker);
        return updatesTracker;
    }

    private void notifyPopulationCompleted(UpdatesTracker updatesTracker) {
        this.indexOnlineMonitor.updatesDone();
        updatesTracker.notifyPopulationCompleted();
    }

    private void notifyIfPopulationCompleted(UpdatesTracker updatesTracker) {
        if (this.isCompletedPopulation(updatesTracker)) {
            this.notifyPopulationCompleted(updatesTracker);
        }
    }

    private boolean isCompletedPopulation(UpdatesTracker updatesTracker) {
        return !updatesTracker.isPopulationCompleted() && this.indexOnlineMonitor.isIndexOnline();
    }

    private void assertDoubleLongEquals(long expectedUniqueValue, long expectedSampledSize, Register.DoubleLongRegister register) {
        Assert.assertEquals((long)expectedUniqueValue, (long)register.readFirst());
        Assert.assertEquals((long)expectedSampledSize, (long)register.readSecond());
    }

    private static void assertCorrectIndexSize(long expected, long actual) {
        IndexStatisticsTest.assertCorrectIndexSize("", expected, actual);
    }

    private static void assertCorrectIndexSize(String info, long expected, long actual) {
        String message = String.format("Expected number of entries to not differ (expected: %d actual: %d) %s", expected, actual, info);
        Assert.assertEquals((String)message, (long)0L, (long)Math.abs(expected - actual));
    }

    private static void assertCorrectIndexUpdates(long expected, long actual) {
        IndexStatisticsTest.assertCorrectIndexUpdates("", expected, actual);
    }

    private static void assertCorrectIndexUpdates(String info, long expected, long actual) {
        String message = String.format("Expected number of index updates to not differ (expected: %d actual: %d). %s", expected, actual, info);
        Assert.assertEquals((String)message, (long)0L, (long)Math.abs(expected - actual));
    }

    private static void assertCorrectIndexSelectivity(double expected, double actual) {
        String message = String.format("Expected number of entries to not differ (expected: %f actual: %f)", expected, actual);
        Assert.assertEquals((String)message, (double)expected, (double)actual, (double)0.0);
    }

    private static class IndexOnlineMonitor
    extends IndexingService.MonitorAdapter {
        private CountDownLatch updateTrackerCompletionLatch;
        private final CountDownLatch startSignal = new CountDownLatch(1);
        private volatile boolean isOnline;
        private Barrier.Control barrier;

        private IndexOnlineMonitor() {
        }

        void initialize(int numberOfUpdateTrackers) {
            this.updateTrackerCompletionLatch = new CountDownLatch(numberOfUpdateTrackers);
            if (numberOfUpdateTrackers > 0) {
                this.barrier = new Barrier.Control();
            }
        }

        void updatesDone() {
            this.updateTrackerCompletionLatch.countDown();
            try {
                this.updateTrackerCompletionLatch.await();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            if (this.barrier != null) {
                this.barrier.reached();
            }
        }

        public void indexPopulationScanStarting() {
            this.startSignal.countDown();
        }

        public void indexPopulationScanComplete() {
            this.isOnline = true;
            if (this.barrier != null) {
                try {
                    this.barrier.await();
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        public void populationCompleteOn(StoreIndexDescriptor descriptor) {
            if (this.barrier != null) {
                this.barrier.release();
            }
        }

        boolean isIndexOnline() {
            return this.isOnline;
        }
    }
}

