/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.store.watch;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.SystemUtils;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.neo4j.graphdb.DependencyResolver;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.index.impl.lucene.explicit.LuceneDataSource;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.FileUtils;
import org.neo4j.io.fs.watcher.FileWatchEventListener;
import org.neo4j.io.fs.watcher.FileWatcher;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.kernel.impl.transaction.log.checkpoint.CheckPointer;
import org.neo4j.kernel.impl.transaction.log.checkpoint.SimpleTriggerInfo;
import org.neo4j.kernel.impl.transaction.log.checkpoint.TriggerInfo;
import org.neo4j.kernel.impl.util.watcher.DefaultFileDeletionEventListener;
import org.neo4j.kernel.impl.util.watcher.FileSystemWatcherService;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogProvider;
import org.neo4j.test.TestGraphDatabaseFactory;
import org.neo4j.test.rule.TestDirectory;

public class FileWatchIT {
    private static final long TEST_TIMEOUT = 600000L;
    @Rule
    public final TestDirectory testDirectory = TestDirectory.testDirectory();
    private File storeDir;
    private AssertableLogProvider logProvider;
    private GraphDatabaseService database;

    @Before
    public void setUp() {
        this.storeDir = this.testDirectory.storeDir();
        this.logProvider = new AssertableLogProvider();
        this.database = new TestGraphDatabaseFactory().setInternalLogProvider((LogProvider)this.logProvider).newEmbeddedDatabase(this.storeDir);
    }

    @After
    public void tearDown() {
        FileWatchIT.shutdownDatabaseSilently(this.database);
    }

    @Test(timeout=600000L)
    public void notifyAboutStoreFileDeletion() throws Exception {
        Assume.assumeFalse((boolean)SystemUtils.IS_OS_WINDOWS);
        String fileName = this.testDirectory.databaseLayout().metadataStore().getName();
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkpointer = FileWatchIT.getCheckpointer(this.database);
        DeletionLatchEventListener deletionListener = new DeletionLatchEventListener(fileName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)deletionListener);
        do {
            FileWatchIT.createNode(this.database);
            FileWatchIT.forceCheckpoint(checkpointer);
        } while (!deletionListener.awaitModificationNotification());
        FileWatchIT.deleteFile(this.testDirectory.storeDir(), fileName);
        deletionListener.awaitDeletionNotification();
        this.logProvider.assertContainsMessageContaining("'" + fileName + "' which belongs to the store was deleted while database was running.");
    }

    @Test(timeout=600000L)
    public void notifyWhenFileWatchingFailToStart() {
        AssertableLogProvider logProvider = new AssertableLogProvider(true);
        GraphDatabaseService db = null;
        try {
            db = new TestGraphDatabaseFactory().setInternalLogProvider((LogProvider)logProvider).setFileSystem((FileSystemAbstraction)new NonWatchableFileSystemAbstraction()).newEmbeddedDatabase(this.testDirectory.storeDir("failed-start-db"));
            logProvider.assertContainsMessageContaining("Can not create file watcher for current file system. File monitoring capabilities for store files will be disabled.");
        }
        catch (Throwable throwable) {
            FileWatchIT.shutdownDatabaseSilently(db);
            throw throwable;
        }
        FileWatchIT.shutdownDatabaseSilently(db);
    }

    @Test(timeout=600000L)
    public void notifyAboutExplicitIndexFolderRemoval() throws InterruptedException, IOException {
        String monitoredDirectory = FileWatchIT.getExplicitIndexDirectory(this.testDirectory.databaseLayout());
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkPointer = FileWatchIT.getCheckpointer(this.database);
        DeletionLatchEventListener deletionListener = new DeletionLatchEventListener(monitoredDirectory);
        String metadataStore = this.testDirectory.databaseLayout().metadataStore().getName();
        ModificationEventListener modificationEventListener = new ModificationEventListener(metadataStore);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)deletionListener);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)modificationEventListener);
        do {
            FileWatchIT.createNode(this.database);
            FileWatchIT.forceCheckpoint(checkPointer);
        } while (!modificationEventListener.awaitModificationNotification());
        FileWatchIT.deleteStoreDirectory(this.storeDir, monitoredDirectory);
        deletionListener.awaitDeletionNotification();
        this.logProvider.assertContainsMessageContaining("'" + monitoredDirectory + "' which belongs to the store was deleted while database was running.");
    }

    @Test(timeout=600000L)
    public void doNotNotifyAboutLuceneIndexFilesDeletion() throws InterruptedException, IOException {
        DependencyResolver dependencyResolver = ((GraphDatabaseAPI)this.database).getDependencyResolver();
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkPointer = (CheckPointer)dependencyResolver.resolveDependency(CheckPointer.class);
        String propertyStoreName = this.testDirectory.databaseLayout().propertyStore().getName();
        AccumulativeDeletionEventListener accumulativeListener = new AccumulativeDeletionEventListener();
        ModificationEventListener modificationListener = new ModificationEventListener(propertyStoreName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)modificationListener);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)accumulativeListener);
        String labelName = "labelName";
        String propertyName = "propertyName";
        Label testLabel = Label.label((String)labelName);
        FileWatchIT.createIndexes(this.database, propertyName, testLabel);
        do {
            FileWatchIT.createNode(this.database, propertyName, testLabel);
            FileWatchIT.forceCheckpoint(checkPointer);
        } while (!modificationListener.awaitModificationNotification());
        fileWatcher.removeFileWatchEventListener((FileWatchEventListener)modificationListener);
        ModificationEventListener afterRemovalListener = new ModificationEventListener(propertyStoreName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)afterRemovalListener);
        FileWatchIT.dropAllIndexes(this.database);
        do {
            FileWatchIT.createNode(this.database, propertyName, testLabel);
            FileWatchIT.forceCheckpoint(checkPointer);
        } while (!afterRemovalListener.awaitModificationNotification());
        accumulativeListener.assertDoesNotHaveAnyDeletions();
    }

    @Test(timeout=600000L)
    public void doNotMonitorTransactionLogFiles() throws InterruptedException, IOException {
        Assume.assumeFalse((boolean)SystemUtils.IS_OS_WINDOWS);
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkpointer = FileWatchIT.getCheckpointer(this.database);
        String metadataStore = this.testDirectory.databaseLayout().metadataStore().getName();
        ModificationEventListener modificationEventListener = new ModificationEventListener(metadataStore);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)modificationEventListener);
        do {
            FileWatchIT.createNode(this.database);
            FileWatchIT.forceCheckpoint(checkpointer);
        } while (!modificationEventListener.awaitModificationNotification());
        String fileName = "neostore.transaction.db.0";
        DeletionLatchEventListener deletionListener = new DeletionLatchEventListener(fileName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)deletionListener);
        FileWatchIT.deleteFile(this.testDirectory.storeDir(), fileName);
        deletionListener.awaitDeletionNotification();
        AssertableLogProvider.LogMatcher logMatcher = AssertableLogProvider.inLog(DefaultFileDeletionEventListener.class).info(Matchers.containsString((String)fileName));
        this.logProvider.assertNone(logMatcher);
    }

    @Test(timeout=600000L)
    public void notifyWhenWholeStoreDirectoryRemoved() throws IOException, InterruptedException {
        Assume.assumeFalse((boolean)SystemUtils.IS_OS_WINDOWS);
        String fileName = this.testDirectory.databaseLayout().metadataStore().getName();
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkpointer = FileWatchIT.getCheckpointer(this.database);
        ModificationEventListener modificationListener = new ModificationEventListener(fileName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)modificationListener);
        do {
            FileWatchIT.createNode(this.database);
            FileWatchIT.forceCheckpoint(checkpointer);
        } while (!modificationListener.awaitModificationNotification());
        fileWatcher.removeFileWatchEventListener((FileWatchEventListener)modificationListener);
        String storeDirectoryName = this.testDirectory.databaseLayout().databaseDirectory().getName();
        DeletionLatchEventListener eventListener = new DeletionLatchEventListener(storeDirectoryName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)eventListener);
        FileUtils.deleteRecursively((File)this.testDirectory.databaseLayout().databaseDirectory());
        eventListener.awaitDeletionNotification();
        this.logProvider.assertContainsMessageContaining("'" + storeDirectoryName + "' which belongs to the store was deleted while database was running.");
    }

    @Test(timeout=600000L)
    public void shouldLogWhenDisabled() {
        AssertableLogProvider logProvider = new AssertableLogProvider(true);
        GraphDatabaseService db = null;
        try {
            db = new TestGraphDatabaseFactory().setInternalLogProvider((LogProvider)logProvider).setFileSystem((FileSystemAbstraction)new NonWatchableFileSystemAbstraction()).newEmbeddedDatabaseBuilder(this.testDirectory.directory("failed-start-db")).setConfig(GraphDatabaseSettings.filewatcher_enabled, "false").newGraphDatabase();
            logProvider.assertContainsMessageContaining("File watcher disabled by configuration.");
        }
        catch (Throwable throwable) {
            FileWatchIT.shutdownDatabaseSilently(db);
            throw throwable;
        }
        FileWatchIT.shutdownDatabaseSilently(db);
    }

    private static void shutdownDatabaseSilently(GraphDatabaseService databaseService) {
        if (databaseService != null) {
            try {
                databaseService.shutdown();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    private static void dropAllIndexes(GraphDatabaseService database) {
        try (Transaction transaction = database.beginTx();){
            for (IndexDefinition definition : database.schema().getIndexes()) {
                definition.drop();
            }
            transaction.success();
        }
    }

    private static void createIndexes(GraphDatabaseService database, String propertyName, Label testLabel) {
        try (Transaction transaction = database.beginTx();){
            database.schema().indexFor(testLabel).on(propertyName).create();
            transaction.success();
        }
        var4_4 = null;
        try (Transaction ignored = database.beginTx();){
            database.schema().awaitIndexesOnline(1L, TimeUnit.MINUTES);
        }
        catch (Throwable throwable) {
            var4_4 = throwable;
            throw throwable;
        }
    }

    private static void forceCheckpoint(CheckPointer checkPointer) throws IOException {
        checkPointer.forceCheckPoint((TriggerInfo)new SimpleTriggerInfo("testForceCheckPoint"));
    }

    private static String getExplicitIndexDirectory(DatabaseLayout databaseLayout) {
        File schemaIndexDirectory = LuceneDataSource.getLuceneIndexStoreDirectory((DatabaseLayout)databaseLayout);
        Path relativeIndexPath = databaseLayout.databaseDirectory().toPath().relativize(schemaIndexDirectory.toPath());
        return relativeIndexPath.getName(0).toString();
    }

    private static void createNode(GraphDatabaseService database, String propertyName, Label testLabel) {
        try (Transaction transaction = database.beginTx();){
            Node node = database.createNode(new Label[]{testLabel});
            node.setProperty(propertyName, (Object)"value");
            transaction.success();
        }
    }

    private static CheckPointer getCheckpointer(GraphDatabaseService database) {
        return (CheckPointer)((GraphDatabaseAPI)database).getDependencyResolver().resolveDependency(CheckPointer.class);
    }

    private static FileWatcher getFileWatcher(GraphDatabaseService database) {
        DependencyResolver dependencyResolver = ((GraphDatabaseAPI)database).getDependencyResolver();
        return ((FileSystemWatcherService)dependencyResolver.resolveDependency(FileSystemWatcherService.class)).getFileWatcher();
    }

    private static void deleteFile(File storeDir, String fileName) {
        File metadataStore = new File(storeDir, fileName);
        FileUtils.deleteFile((File)metadataStore);
    }

    private static void deleteStoreDirectory(File storeDir, String directoryName) throws IOException {
        File directory = new File(storeDir, directoryName);
        FileUtils.deleteRecursively((File)directory);
    }

    private static void createNode(GraphDatabaseService database) {
        try (Transaction transaction = database.beginTx();){
            database.createNode();
            transaction.success();
        }
    }

    private static class DeletionLatchEventListener
    extends ModificationEventListener {
        private final CountDownLatch deletionLatch = new CountDownLatch(1);

        DeletionLatchEventListener(String expectedFileName) {
            super(expectedFileName);
        }

        public void fileDeleted(String fileName) {
            if (fileName.endsWith(this.expectedFileName)) {
                this.deletionLatch.countDown();
            }
        }

        void awaitDeletionNotification() throws InterruptedException {
            this.deletionLatch.await();
        }
    }

    private static class ModificationEventListener
    implements FileWatchEventListener {
        final String expectedFileName;
        private final CountDownLatch modificationLatch = new CountDownLatch(1);

        ModificationEventListener(String expectedFileName) {
            this.expectedFileName = expectedFileName;
        }

        public void fileModified(String fileName) {
            if (this.expectedFileName.equals(fileName)) {
                this.modificationLatch.countDown();
            }
        }

        boolean awaitModificationNotification() throws InterruptedException {
            return this.modificationLatch.await(1L, TimeUnit.SECONDS);
        }
    }

    private static class AccumulativeDeletionEventListener
    implements FileWatchEventListener {
        private final List<String> deletedFiles = new ArrayList<String>();

        private AccumulativeDeletionEventListener() {
        }

        public void fileDeleted(String fileName) {
            this.deletedFiles.add(fileName);
        }

        void assertDoesNotHaveAnyDeletions() {
            Assert.assertThat((String)"Should not have any deletions registered", this.deletedFiles, (Matcher)Matchers.empty());
        }
    }

    private static class NonWatchableFileSystemAbstraction
    extends DefaultFileSystemAbstraction {
        private NonWatchableFileSystemAbstraction() {
        }

        public FileWatcher fileWatcher() throws IOException {
            throw new IOException("You can't watch me!");
        }
    }
}

