/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.newapi;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.collections.impl.set.mutable.primitive.LongHashSet;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.exceptions.KernelException;
import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.kernel.api.NodeCursor;
import org.neo4j.internal.kernel.api.PropertyCursor;
import org.neo4j.internal.kernel.api.Read;
import org.neo4j.internal.kernel.api.RelationshipGroupCursor;
import org.neo4j.internal.kernel.api.RelationshipScanCursor;
import org.neo4j.internal.kernel.api.RelationshipTraversalCursor;
import org.neo4j.internal.kernel.api.Write;
import org.neo4j.internal.kernel.api.exceptions.EntityNotFoundException;
import org.neo4j.internal.kernel.api.security.AccessMode;
import org.neo4j.internal.kernel.api.security.AuthSubject;
import org.neo4j.internal.kernel.api.security.LoginContext;
import org.neo4j.internal.kernel.api.security.SecurityContext;
import org.neo4j.internal.kernel.api.security.TestAccessMode;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.impl.newapi.KernelAPIWriteTestBase;
import org.neo4j.kernel.impl.newapi.KernelAPIWriteTestSupport;
import org.neo4j.kernel.impl.newapi.RelationshipTestSupport;
import org.neo4j.values.storable.Value;
import org.neo4j.values.storable.ValueGroup;
import org.neo4j.values.storable.Values;

public abstract class RelationshipTransactionStateTestBase<G extends KernelAPIWriteTestSupport>
extends KernelAPIWriteTestBase<G> {
    @Test
    void shouldSeeSingleRelationshipInTransaction() throws Exception {
        int label;
        long n2;
        long n1;
        try (KernelTransaction tx = this.beginTransaction();){
            n1 = tx.dataWrite().nodeCreate();
            n2 = tx.dataWrite().nodeCreate();
            long decoyNode = tx.dataWrite().nodeCreate();
            label = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            tx.dataWrite().relationshipCreate(n2, label, decoyNode);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            long r = tx.dataWrite().relationshipCreate(n1, label, n2);
            try (RelationshipScanCursor relationship = tx.cursors().allocateRelationshipScanCursor();){
                tx.dataRead().singleRelationship(r, relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should find relationship");
                Assertions.assertEquals((int)label, (int)relationship.type());
                Assertions.assertEquals((long)n1, (long)relationship.sourceNodeReference());
                Assertions.assertEquals((long)n2, (long)relationship.targetNodeReference());
                Assertions.assertEquals((long)r, (long)relationship.relationshipReference());
                Assertions.assertFalse((boolean)relationship.next(), (String)"should only find one relationship");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldNotSeeSingleRelationshipWhichWasDeletedInTransaction() throws Exception {
        long r;
        try (KernelTransaction tx = this.beginTransaction();){
            long n1 = tx.dataWrite().nodeCreate();
            long n2 = tx.dataWrite().nodeCreate();
            int label = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            long decoyNode = tx.dataWrite().nodeCreate();
            tx.dataWrite().relationshipCreate(n2, label, decoyNode);
            r = tx.dataWrite().relationshipCreate(n1, label, n2);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            Assertions.assertTrue((boolean)tx.dataWrite().relationshipDelete(r), (String)"should delete relationship");
            try (RelationshipScanCursor relationship = tx.cursors().allocateRelationshipScanCursor();){
                tx.dataRead().singleRelationship(r, relationship);
                Assertions.assertFalse((boolean)relationship.next(), (String)"should not find relationship");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldScanRelationshipInTransaction() throws Exception {
        int type;
        long n2;
        long n1;
        int nRelationshipsInStore = 10;
        try (KernelTransaction tx = this.beginTransaction();){
            n1 = tx.dataWrite().nodeCreate();
            n2 = tx.dataWrite().nodeCreate();
            type = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            this.relateNTimes(10, type, n1, n2, tx);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            long r = tx.dataWrite().relationshipCreate(n1, type, n2);
            try (RelationshipScanCursor relationship = tx.cursors().allocateRelationshipScanCursor();){
                tx.dataRead().allRelationshipsScan(relationship);
                this.assertCountRelationships(relationship, 11, n1, type, n2);
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldNotScanRelationshipWhichWasDeletedInTransaction() throws Exception {
        long r;
        int type;
        long n2;
        long n1;
        int nRelationshipsInStore = 11;
        try (KernelTransaction tx = this.beginTransaction();){
            n1 = tx.dataWrite().nodeCreate();
            n2 = tx.dataWrite().nodeCreate();
            type = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            this.relateNTimes(5, type, n1, n2, tx);
            r = tx.dataWrite().relationshipCreate(n1, type, n2);
            this.relateNTimes(5, type, n1, n2, tx);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            Assertions.assertTrue((boolean)tx.dataWrite().relationshipDelete(r), (String)"should delete relationship");
            try (RelationshipScanCursor relationship = tx.cursors().allocateRelationshipScanCursor();){
                tx.dataRead().allRelationshipsScan(relationship);
                this.assertCountRelationships(relationship, 10, n1, type, n2);
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldSeeRelationshipInTransaction() throws Exception {
        long n2;
        long n1;
        try (KernelTransaction tx = this.beginTransaction();){
            n1 = tx.dataWrite().nodeCreate();
            n2 = tx.dataWrite().nodeCreate();
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            int label = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            long r = tx.dataWrite().relationshipCreate(n1, label, n2);
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor relationship = tx.cursors().allocateRelationshipTraversalCursor();){
                tx.dataRead().singleNode(n1, node);
                Assertions.assertTrue((boolean)node.next(), (String)"should access node");
                node.allRelationships(relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should find relationship");
                Assertions.assertEquals((long)r, (long)relationship.relationshipReference());
                Assertions.assertFalse((boolean)relationship.next(), (String)"should only find one relationship");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldNotSeeRelationshipDeletedInTransaction() throws Exception {
        long r;
        long n1;
        try (KernelTransaction tx = this.beginTransaction();){
            n1 = tx.dataWrite().nodeCreate();
            long n2 = tx.dataWrite().nodeCreate();
            int label = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            r = tx.dataWrite().relationshipCreate(n1, label, n2);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            tx.dataWrite().relationshipDelete(r);
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor relationship = tx.cursors().allocateRelationshipTraversalCursor();){
                tx.dataRead().singleNode(n1, node);
                Assertions.assertTrue((boolean)node.next(), (String)"should access node");
                node.allRelationships(relationship);
                Assertions.assertFalse((boolean)relationship.next(), (String)"should not find relationship");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldSeeRelationshipInTransactionBeforeCursorInitialization() throws Exception {
        long n2;
        long n1;
        try (KernelTransaction tx = this.beginTransaction();){
            n1 = tx.dataWrite().nodeCreate();
            n2 = tx.dataWrite().nodeCreate();
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            int label = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            long r = tx.dataWrite().relationshipCreate(n1, label, n2);
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor relationship = tx.cursors().allocateRelationshipTraversalCursor();){
                tx.dataRead().singleNode(n1, node);
                Assertions.assertTrue((boolean)node.next(), (String)"should access node");
                node.allRelationships(relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should find relationship");
                Assertions.assertEquals((long)r, (long)relationship.relationshipReference());
                tx.dataWrite().relationshipCreate(n1, label, n2);
                Assertions.assertFalse((boolean)relationship.next(), (String)"should not find relationship added after cursor init");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldTraverseSparseNodeWithoutGroups() throws Exception {
        this.traverseWithoutGroups(RelationshipTestSupport.sparse(graphDb), false);
    }

    @Test
    void shouldTraverseDenseNodeWithoutGroups() throws Exception {
        this.traverseWithoutGroups(RelationshipTestSupport.dense(graphDb), false);
    }

    @Test
    void shouldTraverseSparseNodeWithoutGroupsWithDetachedReferences() throws Exception {
        this.traverseWithoutGroups(RelationshipTestSupport.sparse(graphDb), true);
    }

    @Test
    void shouldTraverseDenseNodeWithoutGroupsWithDetachedReferences() throws Exception {
        this.traverseWithoutGroups(RelationshipTestSupport.dense(graphDb), true);
    }

    @Test
    void shouldTraverseSparseNodeViaGroups() throws Exception {
        this.traverseViaGroups(RelationshipTestSupport.sparse(graphDb), false);
    }

    @Test
    void shouldTraverseDenseNodeViaGroups() throws Exception {
        this.traverseViaGroups(RelationshipTestSupport.dense(graphDb), false);
    }

    @Test
    void shouldTraverseSparseNodeViaGroupsWithDetachedReferences() throws Exception {
        this.traverseViaGroups(RelationshipTestSupport.sparse(graphDb), true);
    }

    @Test
    void shouldTraverseDenseNodeViaGroupsWithDetachedReferences() throws Exception {
        this.traverseViaGroups(RelationshipTestSupport.dense(graphDb), true);
    }

    @Test
    void shouldSeeNewRelationshipPropertyInTransaction() throws Exception {
        try (KernelTransaction tx = this.beginTransaction();){
            String propKey1 = "prop1";
            String propKey2 = "prop2";
            long n1 = tx.dataWrite().nodeCreate();
            long n2 = tx.dataWrite().nodeCreate();
            int label = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            long r = tx.dataWrite().relationshipCreate(n1, label, n2);
            int prop1 = tx.token().propertyKeyGetOrCreateForName(propKey1);
            int prop2 = tx.token().propertyKeyGetOrCreateForName(propKey2);
            Assertions.assertEquals((Object)tx.dataWrite().relationshipSetProperty(r, prop1, (Value)Values.stringValue((String)"hello")), (Object)Values.NO_VALUE);
            Assertions.assertEquals((Object)tx.dataWrite().relationshipSetProperty(r, prop2, (Value)Values.stringValue((String)"world")), (Object)Values.NO_VALUE);
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor relationship = tx.cursors().allocateRelationshipTraversalCursor();
                 PropertyCursor property = tx.cursors().allocatePropertyCursor();){
                tx.dataRead().singleNode(n1, node);
                Assertions.assertTrue((boolean)node.next(), (String)"should access node");
                node.allRelationships(relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should access relationship");
                relationship.properties(property);
                while (property.next()) {
                    if (property.propertyKey() == prop1) {
                        Assertions.assertEquals((Object)property.propertyValue(), (Object)Values.stringValue((String)"hello"));
                        continue;
                    }
                    if (property.propertyKey() == prop2) {
                        Assertions.assertEquals((Object)property.propertyValue(), (Object)Values.stringValue((String)"world"));
                        continue;
                    }
                    Assertions.fail((String)(property.propertyKey() + " was not the property key you were looking for"));
                }
                Assertions.assertFalse((boolean)relationship.next(), (String)"should only find one relationship");
            }
            tx.commit();
        }
    }

    @Test
    void shouldSeeAddedPropertyFromExistingRelationshipWithoutPropertiesInTransaction() throws Exception {
        long relationshipId;
        String propKey = "prop1";
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            relationshipId = write.relationshipCreate(write.nodeCreate(), tx.tokenWrite().relationshipTypeGetOrCreateForName("R"), write.nodeCreate());
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            int propToken = tx.token().propertyKeyGetOrCreateForName(propKey);
            Assertions.assertEquals((Object)tx.dataWrite().relationshipSetProperty(relationshipId, propToken, (Value)Values.stringValue((String)"hello")), (Object)Values.NO_VALUE);
            try (RelationshipScanCursor relationship = tx.cursors().allocateRelationshipScanCursor();
                 PropertyCursor property = tx.cursors().allocatePropertyCursor();){
                tx.dataRead().singleRelationship(relationshipId, relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should access relationship");
                relationship.properties(property);
                Assertions.assertTrue((boolean)property.next());
                Assertions.assertEquals((int)propToken, (int)property.propertyKey());
                Assertions.assertEquals((Object)property.propertyValue(), (Object)Values.stringValue((String)"hello"));
                Assertions.assertFalse((boolean)property.next(), (String)"should only find one properties");
                Assertions.assertFalse((boolean)relationship.next(), (String)"should only find one relationship");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction transaction = graphDb.beginTx();){
            MatcherAssert.assertThat((Object)transaction.getRelationshipById(relationshipId).getProperty(propKey), (Matcher)Matchers.equalTo((Object)"hello"));
        }
    }

    @Test
    void shouldSeeAddedPropertyFromExistingRelationshipWithPropertiesInTransaction() throws Exception {
        RelationshipScanCursor relationship;
        int propToken1;
        long relationshipId;
        String propKey1 = "prop1";
        String propKey2 = "prop2";
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            relationshipId = write.relationshipCreate(write.nodeCreate(), tx.tokenWrite().relationshipTypeGetOrCreateForName("R"), write.nodeCreate());
            propToken1 = tx.token().propertyKeyGetOrCreateForName(propKey1);
            Assertions.assertEquals((Object)write.relationshipSetProperty(relationshipId, propToken1, (Value)Values.stringValue((String)"hello")), (Object)Values.NO_VALUE);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            int propToken2 = tx.token().propertyKeyGetOrCreateForName(propKey2);
            Assertions.assertEquals((Object)tx.dataWrite().relationshipSetProperty(relationshipId, propToken2, (Value)Values.stringValue((String)"world")), (Object)Values.NO_VALUE);
            relationship = tx.cursors().allocateRelationshipScanCursor();
            try (PropertyCursor property = tx.cursors().allocatePropertyCursor();){
                tx.dataRead().singleRelationship(relationshipId, relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should access relationship");
                relationship.properties(property);
                while (property.next()) {
                    if (property.propertyKey() == propToken1) {
                        Assertions.assertEquals((Object)property.propertyValue(), (Object)Values.stringValue((String)"hello"));
                        continue;
                    }
                    if (property.propertyKey() == propToken2) {
                        Assertions.assertEquals((Object)property.propertyValue(), (Object)Values.stringValue((String)"world"));
                        continue;
                    }
                    Assertions.fail((String)(property.propertyKey() + " was not the property you were looking for"));
                }
                Assertions.assertFalse((boolean)relationship.next(), (String)"should only find one relationship");
            }
            finally {
                if (relationship != null) {
                    relationship.close();
                }
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction transaction = graphDb.beginTx();){
            relationship = transaction.getRelationshipById(relationshipId);
            MatcherAssert.assertThat((Object)relationship.getProperty(propKey1), (Matcher)Matchers.equalTo((Object)"hello"));
            MatcherAssert.assertThat((Object)relationship.getProperty(propKey2), (Matcher)Matchers.equalTo((Object)"world"));
        }
    }

    @Test
    void shouldSeeUpdatedPropertyFromExistingRelationshipWithPropertiesInTransaction() throws Exception {
        int propToken;
        long relationshipId;
        String propKey = "prop1";
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            relationshipId = write.relationshipCreate(write.nodeCreate(), tx.tokenWrite().relationshipTypeGetOrCreateForName("R"), write.nodeCreate());
            propToken = tx.token().propertyKeyGetOrCreateForName(propKey);
            Assertions.assertEquals((Object)write.relationshipSetProperty(relationshipId, propToken, (Value)Values.stringValue((String)"hello")), (Object)Values.NO_VALUE);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            Assertions.assertEquals((Object)tx.dataWrite().relationshipSetProperty(relationshipId, propToken, (Value)Values.stringValue((String)"world")), (Object)Values.stringValue((String)"hello"));
            try (RelationshipScanCursor relationship = tx.cursors().allocateRelationshipScanCursor();
                 PropertyCursor property = tx.cursors().allocatePropertyCursor();){
                tx.dataRead().singleRelationship(relationshipId, relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should access relationship");
                relationship.properties(property);
                Assertions.assertTrue((boolean)property.next());
                Assertions.assertEquals((int)propToken, (int)property.propertyKey());
                Assertions.assertEquals((Object)property.propertyValue(), (Object)Values.stringValue((String)"world"));
                Assertions.assertFalse((boolean)property.next(), (String)"should only find one property");
                Assertions.assertFalse((boolean)relationship.next(), (String)"should only find one relationship");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction transaction = graphDb.beginTx();){
            MatcherAssert.assertThat((Object)transaction.getRelationshipById(relationshipId).getProperty(propKey), (Matcher)Matchers.equalTo((Object)"world"));
        }
    }

    @Test
    void shouldNotSeeRemovedPropertyInTransaction() throws Exception {
        int propToken;
        long relationshipId;
        String propKey = "prop1";
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            relationshipId = write.relationshipCreate(write.nodeCreate(), tx.tokenWrite().relationshipTypeGetOrCreateForName("R"), write.nodeCreate());
            propToken = tx.token().propertyKeyGetOrCreateForName(propKey);
            Assertions.assertEquals((Object)write.relationshipSetProperty(relationshipId, propToken, (Value)Values.stringValue((String)"hello")), (Object)Values.NO_VALUE);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            Assertions.assertEquals((Object)tx.dataWrite().relationshipRemoveProperty(relationshipId, propToken), (Object)Values.stringValue((String)"hello"));
            try (RelationshipScanCursor relationship = tx.cursors().allocateRelationshipScanCursor();
                 PropertyCursor property = tx.cursors().allocatePropertyCursor();){
                tx.dataRead().singleRelationship(relationshipId, relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should access relationship");
                relationship.properties(property);
                Assertions.assertFalse((boolean)property.next(), (String)"should not find any properties");
                Assertions.assertFalse((boolean)relationship.next(), (String)"should only find one relationship");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction transaction = graphDb.beginTx();){
            Assertions.assertFalse((boolean)transaction.getRelationshipById(relationshipId).hasProperty(propKey));
        }
    }

    @Test
    void shouldSeeRemovedThenAddedPropertyInTransaction() throws Exception {
        int propToken;
        long relationshipId;
        String propKey = "prop1";
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            relationshipId = write.relationshipCreate(write.nodeCreate(), tx.tokenWrite().relationshipTypeGetOrCreateForName("R"), write.nodeCreate());
            propToken = tx.token().propertyKeyGetOrCreateForName(propKey);
            Assertions.assertEquals((Object)write.relationshipSetProperty(relationshipId, propToken, (Value)Values.stringValue((String)"hello")), (Object)Values.NO_VALUE);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            Assertions.assertEquals((Object)tx.dataWrite().relationshipRemoveProperty(relationshipId, propToken), (Object)Values.stringValue((String)"hello"));
            Assertions.assertEquals((Object)tx.dataWrite().relationshipSetProperty(relationshipId, propToken, (Value)Values.stringValue((String)"world")), (Object)Values.NO_VALUE);
            try (RelationshipScanCursor relationship = tx.cursors().allocateRelationshipScanCursor();
                 PropertyCursor property = tx.cursors().allocatePropertyCursor();){
                tx.dataRead().singleRelationship(relationshipId, relationship);
                Assertions.assertTrue((boolean)relationship.next(), (String)"should access relationship");
                relationship.properties(property);
                Assertions.assertTrue((boolean)property.next());
                Assertions.assertEquals((int)propToken, (int)property.propertyKey());
                Assertions.assertEquals((Object)property.propertyValue(), (Object)Values.stringValue((String)"world"));
                Assertions.assertFalse((boolean)property.next(), (String)"should not find any properties");
                Assertions.assertFalse((boolean)relationship.next(), (String)"should only find one relationship");
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        try (Transaction transaction = graphDb.beginTx();){
            MatcherAssert.assertThat((Object)transaction.getRelationshipById(relationshipId).getProperty(propKey), (Matcher)Matchers.equalTo((Object)"world"));
        }
    }

    @Test
    void shouldCountFromTxState() throws Exception {
        this.assertCount(100, RelationshipDirection.OUT, group -> {
            Assertions.assertEquals((int)101, (int)group.outgoingCount());
            Assertions.assertEquals((int)0, (int)group.incomingCount());
            Assertions.assertEquals((int)0, (int)group.loopCount());
            Assertions.assertEquals((int)101, (int)group.totalCount());
        });
        this.assertCount(1, RelationshipDirection.OUT, group -> {
            Assertions.assertEquals((int)2, (int)group.outgoingCount());
            Assertions.assertEquals((int)0, (int)group.incomingCount());
            Assertions.assertEquals((int)0, (int)group.loopCount());
            Assertions.assertEquals((int)2, (int)group.totalCount());
        });
        this.assertCount(100, RelationshipDirection.IN, group -> {
            Assertions.assertEquals((int)0, (int)group.outgoingCount());
            Assertions.assertEquals((int)101, (int)group.incomingCount());
            Assertions.assertEquals((int)0, (int)group.outgoingCount());
            Assertions.assertEquals((int)101, (int)group.totalCount());
        });
        this.assertCount(1, RelationshipDirection.IN, group -> {
            Assertions.assertEquals((int)0, (int)group.outgoingCount());
            Assertions.assertEquals((int)2, (int)group.incomingCount());
            Assertions.assertEquals((int)0, (int)group.loopCount());
            Assertions.assertEquals((int)2, (int)group.totalCount());
        });
        this.assertCount(100, RelationshipDirection.LOOP, group -> {
            Assertions.assertEquals((int)0, (int)group.incomingCount());
            Assertions.assertEquals((int)0, (int)group.outgoingCount());
            Assertions.assertEquals((int)101, (int)group.loopCount());
            Assertions.assertEquals((int)101, (int)group.totalCount());
        });
        this.assertCount(1, RelationshipDirection.LOOP, group -> {
            Assertions.assertEquals((int)0, (int)group.outgoingCount());
            Assertions.assertEquals((int)0, (int)group.incomingCount());
            Assertions.assertEquals((int)2, (int)group.loopCount());
            Assertions.assertEquals((int)2, (int)group.totalCount());
        });
    }

    @Test
    void groupCursorShouldSeeNewTypes() throws Exception {
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            long start = write.nodeCreate();
            int outgoing = tx.tokenWrite().relationshipTypeGetOrCreateForName("OUT");
            int incoming = tx.tokenWrite().relationshipTypeGetOrCreateForName("IN");
            int looping = tx.tokenWrite().relationshipTypeGetOrCreateForName("LOOP");
            long out = write.relationshipCreate(start, outgoing, write.nodeCreate());
            long in1 = write.relationshipCreate(write.nodeCreate(), incoming, start);
            long in2 = write.relationshipCreate(write.nodeCreate(), incoming, start);
            long loop = write.relationshipCreate(start, looping, start);
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor traversal = tx.cursors().allocateRelationshipTraversalCursor();
                 RelationshipGroupCursor group = tx.cursors().allocateRelationshipGroupCursor();){
                Read read = tx.dataRead();
                read.singleNode(start, node);
                Assertions.assertTrue((boolean)node.next());
                node.relationships(group);
                while (group.next()) {
                    int t = group.type();
                    if (t == outgoing) {
                        Assertions.assertEquals((int)1, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        this.assertRelationships(RelationshipDirection.OUT, group, traversal, out);
                        this.assertNoRelationships(RelationshipDirection.IN, group, traversal);
                        this.assertNoRelationships(RelationshipDirection.LOOP, group, traversal);
                        continue;
                    }
                    if (t == incoming) {
                        Assertions.assertEquals((int)0, (int)group.outgoingCount());
                        Assertions.assertEquals((int)2, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        this.assertRelationships(RelationshipDirection.IN, group, traversal, in1, in2);
                        this.assertNoRelationships(RelationshipDirection.OUT, group, traversal);
                        this.assertNoRelationships(RelationshipDirection.LOOP, group, traversal);
                        continue;
                    }
                    if (t == looping) {
                        Assertions.assertEquals((int)0, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)1, (int)group.loopCount());
                        this.assertRelationships(RelationshipDirection.LOOP, group, traversal, loop);
                        this.assertNoRelationships(RelationshipDirection.OUT, group, traversal);
                        this.assertNoRelationships(RelationshipDirection.IN, group, traversal);
                        continue;
                    }
                    Assertions.fail((String)(t + "  is not the type you're looking for "));
                }
            }
        }
    }

    @Test
    void groupCursorShouldAddToCountFromTxState() throws Exception {
        long existingRelationship;
        int type;
        long start;
        Write write;
        try (KernelTransaction tx = this.beginTransaction();){
            write = tx.dataWrite();
            start = write.nodeCreate();
            type = tx.tokenWrite().relationshipTypeGetOrCreateForName("OUT");
            existingRelationship = write.relationshipCreate(start, type, write.nodeCreate());
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            write = tx.dataWrite();
            long newRelationship = write.relationshipCreate(start, type, write.nodeCreate());
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor traversal = tx.cursors().allocateRelationshipTraversalCursor();
                 RelationshipGroupCursor group = tx.cursors().allocateRelationshipGroupCursor();){
                Read read = tx.dataRead();
                read.singleNode(start, node);
                Assertions.assertTrue((boolean)node.next());
                node.relationships(group);
                Assertions.assertTrue((boolean)group.next());
                Assertions.assertEquals((int)type, (int)group.type());
                Assertions.assertEquals((int)2, (int)group.outgoingCount());
                Assertions.assertEquals((int)0, (int)group.incomingCount());
                Assertions.assertEquals((int)0, (int)group.loopCount());
                this.assertRelationships(RelationshipDirection.OUT, group, traversal, newRelationship, existingRelationship);
                Assertions.assertFalse((boolean)group.next());
            }
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void groupCursorShouldSeeBothOldAndNewRelationshipsFromSparseNode() throws Exception {
        long existingRelationship;
        int one;
        long start;
        Write write;
        try (KernelTransaction tx = this.beginTransaction();){
            write = tx.dataWrite();
            start = write.nodeCreate();
            one = tx.tokenWrite().relationshipTypeGetOrCreateForName("ONE");
            existingRelationship = write.relationshipCreate(start, one, write.nodeCreate());
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            write = tx.dataWrite();
            int two = tx.tokenWrite().relationshipTypeGetOrCreateForName("TWO");
            long newRelationship = write.relationshipCreate(start, two, write.nodeCreate());
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor traversal = tx.cursors().allocateRelationshipTraversalCursor();
                 RelationshipGroupCursor group = tx.cursors().allocateRelationshipGroupCursor();){
                Read read = tx.dataRead();
                read.singleNode(start, node);
                Assertions.assertTrue((boolean)node.next());
                Assertions.assertFalse((boolean)node.isDense());
                node.relationships(group);
                while (group.next()) {
                    int t = group.type();
                    if (t == one) {
                        Assertions.assertEquals((int)1, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        this.assertRelationships(RelationshipDirection.OUT, group, traversal, existingRelationship);
                        continue;
                    }
                    if (t == two) {
                        Assertions.assertEquals((int)1, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        this.assertRelationships(RelationshipDirection.OUT, group, traversal, newRelationship);
                        continue;
                    }
                    Assertions.fail((String)(t + "  is not the type you're looking for "));
                }
            }
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void groupCursorShouldSeeBothOldAndNewRelationshipsFromDenseNode() throws Exception {
        int bulk;
        long existingRelationship;
        int one;
        long start;
        Write write;
        try (KernelTransaction tx = this.beginTransaction();){
            write = tx.dataWrite();
            start = write.nodeCreate();
            one = tx.tokenWrite().relationshipTypeGetOrCreateForName("ONE");
            existingRelationship = write.relationshipCreate(start, one, write.nodeCreate());
            bulk = tx.tokenWrite().relationshipTypeGetOrCreateForName("BULK");
            for (int i = 0; i < 100; ++i) {
                write.relationshipCreate(start, bulk, write.nodeCreate());
            }
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            write = tx.dataWrite();
            int two = tx.tokenWrite().relationshipTypeGetOrCreateForName("TWO");
            long newRelationship = write.relationshipCreate(start, two, write.nodeCreate());
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor traversal = tx.cursors().allocateRelationshipTraversalCursor();
                 RelationshipGroupCursor group = tx.cursors().allocateRelationshipGroupCursor();){
                Read read = tx.dataRead();
                read.singleNode(start, node);
                Assertions.assertTrue((boolean)node.next());
                Assertions.assertTrue((boolean)node.isDense());
                node.relationships(group);
                while (group.next()) {
                    int t = group.type();
                    if (t == one) {
                        Assertions.assertEquals((int)1, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        this.assertRelationships(RelationshipDirection.OUT, group, traversal, existingRelationship);
                        continue;
                    }
                    if (t == two) {
                        Assertions.assertEquals((int)1, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        this.assertRelationships(RelationshipDirection.OUT, group, traversal, newRelationship);
                        continue;
                    }
                    if (t == bulk) {
                        Assertions.assertEquals((int)100, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        continue;
                    }
                    Assertions.fail((String)(t + "  is not the type you're looking for "));
                }
            }
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void groupCursorShouldNewRelationshipBetweenAlreadyConnectedSparseNodes() throws Exception {
        long existingRelationship;
        int type;
        long start;
        Write write;
        try (KernelTransaction tx = this.beginTransaction();){
            write = tx.dataWrite();
            start = write.nodeCreate();
            long end = write.nodeCreate();
            type = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            existingRelationship = write.relationshipCreate(start, type, end);
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            write = tx.dataWrite();
            long newRelationship = write.relationshipCreate(start, type, write.nodeCreate());
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor traversal = tx.cursors().allocateRelationshipTraversalCursor();
                 RelationshipGroupCursor group = tx.cursors().allocateRelationshipGroupCursor();){
                Read read = tx.dataRead();
                read.singleNode(start, node);
                Assertions.assertTrue((boolean)node.next());
                Assertions.assertFalse((boolean)node.isDense());
                node.relationships(group);
                Assertions.assertTrue((boolean)group.next());
                Assertions.assertEquals((int)type, (int)group.type());
                Assertions.assertEquals((int)2, (int)group.outgoingCount());
                Assertions.assertEquals((int)0, (int)group.incomingCount());
                Assertions.assertEquals((int)0, (int)group.loopCount());
                this.assertRelationships(RelationshipDirection.OUT, group, traversal, newRelationship, existingRelationship);
                Assertions.assertFalse((boolean)group.next());
            }
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void groupCursorShouldNewRelationshipBetweenAlreadyConnectedDenseNodes() throws Exception {
        int bulk;
        long existingRelationship;
        int type;
        long start;
        Write write;
        try (KernelTransaction tx = this.beginTransaction();){
            write = tx.dataWrite();
            start = write.nodeCreate();
            long end = write.nodeCreate();
            type = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            existingRelationship = write.relationshipCreate(start, type, end);
            bulk = tx.tokenWrite().relationshipTypeGetOrCreateForName("BULK");
            for (int i = 0; i < 100; ++i) {
                write.relationshipCreate(start, bulk, write.nodeCreate());
            }
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            write = tx.dataWrite();
            long newRelationship = write.relationshipCreate(start, type, write.nodeCreate());
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor traversal = tx.cursors().allocateRelationshipTraversalCursor();
                 RelationshipGroupCursor group = tx.cursors().allocateRelationshipGroupCursor();){
                Read read = tx.dataRead();
                read.singleNode(start, node);
                Assertions.assertTrue((boolean)node.next());
                Assertions.assertTrue((boolean)node.isDense());
                node.relationships(group);
                while (group.next()) {
                    int t = group.type();
                    if (t == type) {
                        Assertions.assertEquals((int)2, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        this.assertRelationships(RelationshipDirection.OUT, group, traversal, existingRelationship, newRelationship);
                        continue;
                    }
                    if (t == bulk) {
                        Assertions.assertEquals((int)bulk, (int)group.type());
                        Assertions.assertEquals((int)100, (int)group.outgoingCount());
                        Assertions.assertEquals((int)0, (int)group.incomingCount());
                        Assertions.assertEquals((int)0, (int)group.loopCount());
                        continue;
                    }
                    Assertions.fail((String)(t + "  is not the type you're looking for "));
                }
            }
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldCountNewRelationships() throws Exception {
        int relationship;
        Write write;
        try (KernelTransaction tx = this.beginTransaction();){
            write = tx.dataWrite();
            relationship = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            write.relationshipCreate(write.nodeCreate(), relationship, write.nodeCreate());
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            write = tx.dataWrite();
            write.relationshipCreate(write.nodeCreate(), relationship, write.nodeCreate());
            long countsTxState = tx.dataRead().countsForRelationship(-1, relationship, -1);
            long countsNoTxState = tx.dataRead().countsForRelationshipWithoutTxState(-1, relationship, -1);
            Assertions.assertEquals((long)2L, (long)countsTxState);
            Assertions.assertEquals((long)1L, (long)countsNoTxState);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldNotCountRemovedRelationships() throws Exception {
        long relationship;
        int relationshipId;
        Write write;
        try (KernelTransaction tx = this.beginTransaction();){
            write = tx.dataWrite();
            relationshipId = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            relationship = write.relationshipCreate(write.nodeCreate(), relationshipId, write.nodeCreate());
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            write = tx.dataWrite();
            write.relationshipDelete(relationship);
            long countsTxState = tx.dataRead().countsForRelationship(-1, relationshipId, -1);
            long countsNoTxState = tx.dataRead().countsForRelationshipWithoutTxState(-1, relationshipId, -1);
            Assertions.assertEquals((long)0L, (long)countsTxState);
            Assertions.assertEquals((long)1L, (long)countsNoTxState);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldCountNewRelationshipsRestrictedUser() throws Exception {
        int relationship;
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            relationship = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            write.relationshipCreate(write.nodeCreate(), relationship, write.nodeCreate());
            tx.commit();
        }
        SecurityContext loginContext = new SecurityContext(AuthSubject.AUTH_DISABLED, (AccessMode)new TestAccessMode(true, false, true, false));
        try (KernelTransaction tx = this.beginTransaction((LoginContext)loginContext);){
            Write write = tx.dataWrite();
            write.relationshipCreate(write.nodeCreate(), relationship, write.nodeCreate());
            long countsTxState = tx.dataRead().countsForRelationship(-1, relationship, -1);
            long countsNoTxState = tx.dataRead().countsForRelationshipWithoutTxState(-1, relationship, -1);
            Assertions.assertEquals((long)2L, (long)countsTxState);
            Assertions.assertEquals((long)1L, (long)countsNoTxState);
        }
    }

    @Test
    void shouldNotCountRemovedRelationshipsRestrictedUser() throws Exception {
        long relationship;
        int relationshipId;
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            relationshipId = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            relationship = write.relationshipCreate(write.nodeCreate(), relationshipId, write.nodeCreate());
            tx.commit();
        }
        SecurityContext loginContext = new SecurityContext(AuthSubject.AUTH_DISABLED, (AccessMode)new TestAccessMode(true, false, true, false));
        try (KernelTransaction tx = this.beginTransaction((LoginContext)loginContext);){
            Write write = tx.dataWrite();
            write.relationshipDelete(relationship);
            long countsTxState = tx.dataRead().countsForRelationship(-1, relationshipId, -1);
            long countsNoTxState = tx.dataRead().countsForRelationshipWithoutTxState(-1, relationshipId, -1);
            Assertions.assertEquals((long)0L, (long)countsTxState);
            Assertions.assertEquals((long)1L, (long)countsNoTxState);
        }
    }

    private void assertRelationships(RelationshipDirection direction, RelationshipGroupCursor group, RelationshipTraversalCursor traversal, long ... relationships) {
        switch (direction) {
            case OUT: {
                group.outgoing(traversal);
                break;
            }
            case IN: {
                group.incoming(traversal);
                break;
            }
            case LOOP: {
                group.loops(traversal);
                break;
            }
            default: {
                throw new AssertionError((Object)"Where is your god now!");
            }
        }
        LongHashSet set = LongHashSet.newSetWith((long[])relationships);
        for (long relationship : relationships) {
            Assertions.assertTrue((boolean)traversal.next());
            Assertions.assertTrue((boolean)set.contains(traversal.relationshipReference()));
            set.remove(traversal.relationshipReference());
        }
        Assertions.assertTrue((boolean)set.isEmpty());
        Assertions.assertFalse((boolean)traversal.next());
    }

    private void assertNoRelationships(RelationshipDirection direction, RelationshipGroupCursor group, RelationshipTraversalCursor traversal) {
        switch (direction) {
            case OUT: {
                group.outgoing(traversal);
                Assertions.assertFalse((boolean)traversal.next());
                break;
            }
            case IN: {
                group.incoming(traversal);
                Assertions.assertFalse((boolean)traversal.next());
                break;
            }
            case LOOP: {
                group.loops(traversal);
                Assertions.assertFalse((boolean)traversal.next());
                break;
            }
            default: {
                throw new AssertionError((Object)"Where is your god now!");
            }
        }
    }

    private void traverseWithoutGroups(RelationshipTestSupport.StartNode start, boolean detached) throws Exception {
        try (KernelTransaction tx = this.beginTransaction();){
            Map<String, Integer> expectedCounts = this.modifyStartNodeRelationships(start, tx);
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipTraversalCursor relationship = tx.cursors().allocateRelationshipTraversalCursor();){
                tx.dataRead().singleNode(start.id, node);
                Assertions.assertTrue((boolean)node.next(), (String)"access node");
                if (detached) {
                    tx.dataRead().relationships(start.id, node.allRelationshipsReference(), relationship);
                } else {
                    node.allRelationships(relationship);
                }
                Map<String, Integer> counts = RelationshipTestSupport.count(tx, relationship);
                RelationshipTestSupport.assertCounts(expectedCounts, counts);
            }
            tx.rollback();
        }
    }

    private void traverseViaGroups(RelationshipTestSupport.StartNode start, boolean detached) throws Exception {
        try (KernelTransaction tx = this.beginTransaction();){
            Read read = tx.dataRead();
            Map<String, Integer> expectedCounts = this.modifyStartNodeRelationships(start, tx);
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipGroupCursor group = tx.cursors().allocateRelationshipGroupCursor();
                 RelationshipTraversalCursor relationship = tx.cursors().allocateRelationshipTraversalCursor();){
                read.singleNode(start.id, node);
                Assertions.assertTrue((boolean)node.next(), (String)"access node");
                if (detached) {
                    read.relationshipGroups(start.id, node.relationshipGroupReference(), group);
                } else {
                    node.relationships(group);
                }
                while (group.next()) {
                    if (detached) {
                        read.relationships(start.id, group.outgoingReference(), relationship);
                    } else {
                        group.outgoing(relationship);
                    }
                    RelationshipTestSupport.assertCount(tx, relationship, expectedCounts, group.type(), Direction.OUTGOING);
                    if (detached) {
                        read.relationships(start.id, group.incomingReference(), relationship);
                    } else {
                        group.incoming(relationship);
                    }
                    RelationshipTestSupport.assertCount(tx, relationship, expectedCounts, group.type(), Direction.INCOMING);
                    if (detached) {
                        read.relationships(start.id, group.loopsReference(), relationship);
                    } else {
                        group.loops(relationship);
                    }
                    RelationshipTestSupport.assertCount(tx, relationship, expectedCounts, group.type(), Direction.BOTH);
                }
            }
        }
    }

    private Map<String, Integer> modifyStartNodeRelationships(RelationshipTestSupport.StartNode start, KernelTransaction tx) throws KernelException {
        HashMap<String, Integer> expectedCounts = new HashMap<String, Integer>();
        for (Map.Entry<String, List<RelationshipTestSupport.StartRelationship>> kv : start.relationships.entrySet()) {
            List<RelationshipTestSupport.StartRelationship> rs = kv.getValue();
            RelationshipTestSupport.StartRelationship head = rs.get(0);
            int type = tx.token().relationshipType(head.type.name());
            switch (head.direction) {
                case INCOMING: {
                    tx.dataWrite().relationshipCreate(tx.dataWrite().nodeCreate(), type, start.id);
                    tx.dataWrite().relationshipCreate(tx.dataWrite().nodeCreate(), type, start.id);
                    break;
                }
                case OUTGOING: {
                    tx.dataWrite().relationshipCreate(start.id, type, tx.dataWrite().nodeCreate());
                    tx.dataWrite().relationshipCreate(start.id, type, tx.dataWrite().nodeCreate());
                    break;
                }
                case BOTH: {
                    tx.dataWrite().relationshipCreate(start.id, type, start.id);
                    tx.dataWrite().relationshipCreate(start.id, type, start.id);
                    break;
                }
                default: {
                    throw new IllegalStateException("Oh ye be cursed, foul checkstyle!");
                }
            }
            tx.dataWrite().relationshipDelete(head.id);
            expectedCounts.put(kv.getKey(), rs.size() + 1);
        }
        String newTypeName = "NEW";
        int newType = tx.token().relationshipTypeGetOrCreateForName(newTypeName);
        tx.dataWrite().relationshipCreate(tx.dataWrite().nodeCreate(), newType, start.id);
        tx.dataWrite().relationshipCreate(start.id, newType, tx.dataWrite().nodeCreate());
        tx.dataWrite().relationshipCreate(start.id, newType, start.id);
        expectedCounts.put(RelationshipTestSupport.computeKey(newTypeName, Direction.OUTGOING), 1);
        expectedCounts.put(RelationshipTestSupport.computeKey(newTypeName, Direction.INCOMING), 1);
        expectedCounts.put(RelationshipTestSupport.computeKey(newTypeName, Direction.BOTH), 1);
        return expectedCounts;
    }

    @Test
    void hasPropertiesShouldSeeNewlyCreatedProperties() throws Exception {
        long relationship;
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            int token = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            relationship = write.relationshipCreate(write.nodeCreate(), token, write.nodeCreate());
            tx.commit();
        }
        tx = this.beginTransaction();
        try (RelationshipScanCursor cursor = tx.cursors().allocateRelationshipScanCursor();){
            tx.dataRead().singleRelationship(relationship, cursor);
            Assertions.assertTrue((boolean)cursor.next());
            Assertions.assertFalse((boolean)this.hasProperties(cursor, tx));
            tx.dataWrite().relationshipSetProperty(relationship, tx.tokenWrite().propertyKeyGetOrCreateForName("prop"), (Value)Values.stringValue((String)"foo"));
            Assertions.assertTrue((boolean)this.hasProperties(cursor, tx));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void hasPropertiesShouldSeeNewlyCreatedPropertiesOnNewlyCreatedRelationship() throws Exception {
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            int token = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            long relationship = write.relationshipCreate(write.nodeCreate(), token, write.nodeCreate());
            try (RelationshipScanCursor cursor = tx.cursors().allocateRelationshipScanCursor();){
                tx.dataRead().singleRelationship(relationship, cursor);
                Assertions.assertTrue((boolean)cursor.next());
                Assertions.assertFalse((boolean)this.hasProperties(cursor, tx));
                tx.dataWrite().relationshipSetProperty(relationship, tx.tokenWrite().propertyKeyGetOrCreateForName("prop"), (Value)Values.stringValue((String)"foo"));
                Assertions.assertTrue((boolean)this.hasProperties(cursor, tx));
            }
        }
    }

    @Test
    void hasPropertiesShouldSeeNewlyRemovedProperties() throws Exception {
        int prop3;
        int prop2;
        int prop1;
        long relationship;
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            int token = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            relationship = write.relationshipCreate(write.nodeCreate(), token, write.nodeCreate());
            prop1 = tx.tokenWrite().propertyKeyGetOrCreateForName("prop1");
            prop2 = tx.tokenWrite().propertyKeyGetOrCreateForName("prop2");
            prop3 = tx.tokenWrite().propertyKeyGetOrCreateForName("prop3");
            tx.dataWrite().relationshipSetProperty(relationship, prop1, (Value)Values.longValue((long)1L));
            tx.dataWrite().relationshipSetProperty(relationship, prop2, (Value)Values.longValue((long)2L));
            tx.dataWrite().relationshipSetProperty(relationship, prop3, (Value)Values.longValue((long)3L));
            tx.commit();
        }
        tx = this.beginTransaction();
        try (RelationshipScanCursor cursor = tx.cursors().allocateRelationshipScanCursor();){
            tx.dataRead().singleRelationship(relationship, cursor);
            Assertions.assertTrue((boolean)cursor.next());
            Assertions.assertTrue((boolean)this.hasProperties(cursor, tx));
            tx.dataWrite().relationshipRemoveProperty(relationship, prop1);
            Assertions.assertTrue((boolean)this.hasProperties(cursor, tx));
            tx.dataWrite().relationshipRemoveProperty(relationship, prop2);
            Assertions.assertTrue((boolean)this.hasProperties(cursor, tx));
            tx.dataWrite().relationshipRemoveProperty(relationship, prop3);
            Assertions.assertFalse((boolean)this.hasProperties(cursor, tx));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void propertyTypeShouldBeTxStateAware() throws Exception {
        long relationship;
        try (KernelTransaction tx = this.beginTransaction();){
            Write write = tx.dataWrite();
            int token = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            relationship = write.relationshipCreate(write.nodeCreate(), token, write.nodeCreate());
            tx.commit();
        }
        tx = this.beginTransaction();
        try (RelationshipScanCursor relationships = tx.cursors().allocateRelationshipScanCursor();
             PropertyCursor properties = tx.cursors().allocatePropertyCursor();){
            tx.dataRead().singleRelationship(relationship, relationships);
            Assertions.assertTrue((boolean)relationships.next());
            Assertions.assertFalse((boolean)this.hasProperties(relationships, tx));
            int prop = tx.tokenWrite().propertyKeyGetOrCreateForName("prop");
            tx.dataWrite().relationshipSetProperty(relationship, prop, (Value)Values.stringValue((String)"foo"));
            relationships.properties(properties);
            Assertions.assertTrue((boolean)properties.next());
            MatcherAssert.assertThat((Object)properties.propertyType(), (Matcher)Matchers.equalTo((Object)ValueGroup.TEXT));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private boolean hasProperties(RelationshipScanCursor cursor, KernelTransaction tx) {
        try (PropertyCursor propertyCursor = tx.cursors().allocatePropertyCursor();){
            cursor.properties(propertyCursor);
            boolean bl = propertyCursor.next();
            return bl;
        }
    }

    private void relateNTimes(int nRelationshipsInStore, int type, long n1, long n2, KernelTransaction tx) throws KernelException {
        for (int i = 0; i < nRelationshipsInStore; ++i) {
            tx.dataWrite().relationshipCreate(n1, type, n2);
        }
    }

    private void assertCountRelationships(RelationshipScanCursor relationship, int expectedCount, long sourceNode, int type, long targetNode) {
        int count = 0;
        while (relationship.next()) {
            Assertions.assertEquals((long)sourceNode, (long)relationship.sourceNodeReference());
            Assertions.assertEquals((int)type, (int)relationship.type());
            Assertions.assertEquals((long)targetNode, (long)relationship.targetNodeReference());
            ++count;
        }
        Assertions.assertEquals((int)expectedCount, (int)count);
    }

    private void assertCount(int count, RelationshipDirection direction, Consumer<RelationshipGroupCursor> asserter) throws Exception {
        int type;
        long start;
        Write write;
        try (KernelTransaction tx = this.beginTransaction();){
            write = tx.dataWrite();
            start = write.nodeCreate();
            type = tx.tokenWrite().relationshipTypeGetOrCreateForName("R");
            for (int i = 0; i < count; ++i) {
                this.createRelationship(direction, start, type, write);
            }
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            write = tx.dataWrite();
            this.createRelationship(direction, start, type, write);
            try (NodeCursor node = tx.cursors().allocateNodeCursor();
                 RelationshipGroupCursor group = tx.cursors().allocateRelationshipGroupCursor();){
                Read read = tx.dataRead();
                read.singleNode(start, node);
                Assertions.assertTrue((boolean)node.next());
                node.relationships(group);
                Assertions.assertTrue((boolean)group.next());
                asserter.accept(group);
            }
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private void createRelationship(RelationshipDirection direction, long start, int type, Write write) throws EntityNotFoundException {
        switch (direction) {
            case OUT: {
                write.relationshipCreate(start, type, write.nodeCreate());
                break;
            }
            case IN: {
                write.relationshipCreate(write.nodeCreate(), type, start);
                break;
            }
            case LOOP: {
                write.relationshipCreate(start, type, start);
                break;
            }
            default: {
                throw new IllegalStateException("Checkstyle, you win again!");
            }
        }
    }

    static enum RelationshipDirection {
        OUT,
        IN,
        LOOP;

    }
}

