/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.File;
import java.io.IOException;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Transaction;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.kernel.recovery.RecoveryRequiredChecker;
import org.neo4j.storageengine.api.StorageEngineFactory;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.Neo4jLayoutExtension;
import org.neo4j.test.extension.pagecache.PageCacheSupportExtension;
import org.neo4j.test.rule.TestDirectory;

@Neo4jLayoutExtension
class RecoveryRequiredCheckerTest {
    @RegisterExtension
    static PageCacheSupportExtension pageCacheExtension = new PageCacheSupportExtension();
    @Inject
    private TestDirectory testDirectory;
    @Inject
    private FileSystemAbstraction fileSystem;
    @Inject
    private DatabaseLayout databaseLayout;
    private File storeDir;
    private final StorageEngineFactory storageEngineFactory = StorageEngineFactory.selectStorageEngine();

    RecoveryRequiredCheckerTest() {
    }

    @BeforeEach
    void setup() {
        this.storeDir = this.testDirectory.homeDir();
    }

    @Test
    void shouldNotWantToRecoverIntactStore() throws Exception {
        this.startStopAndCreateDefaultData();
        PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);
        RecoveryRequiredChecker recoverer = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
        MatcherAssert.assertThat((Object)recoverer.isRecoveryRequiredAt(this.databaseLayout), (Matcher)CoreMatchers.is((Object)false));
    }

    @Test
    void shouldWantToRecoverBrokenStore() throws Exception {
        try (EphemeralFileSystemAbstraction ephemeralFs = this.createAndCrashWithDefaultConfig();){
            PageCache pageCache = pageCacheExtension.getPageCache((FileSystemAbstraction)ephemeralFs);
            RecoveryRequiredChecker recoverer = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig((FileSystemAbstraction)ephemeralFs, pageCache, this.storageEngineFactory);
            MatcherAssert.assertThat((Object)recoverer.isRecoveryRequiredAt(this.databaseLayout), (Matcher)CoreMatchers.is((Object)true));
        }
    }

    @Test
    void shouldBeAbleToRecoverBrokenStore() throws Exception {
        try (EphemeralFileSystemAbstraction ephemeralFs = this.createAndCrashWithDefaultConfig();){
            PageCache pageCache = pageCacheExtension.getPageCache((FileSystemAbstraction)ephemeralFs);
            RecoveryRequiredChecker recoverer = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig((FileSystemAbstraction)ephemeralFs, pageCache, this.storageEngineFactory);
            MatcherAssert.assertThat((Object)recoverer.isRecoveryRequiredAt(this.databaseLayout), (Matcher)CoreMatchers.is((Object)true));
            RecoveryRequiredCheckerTest.startStopDatabase((FileSystemAbstraction)ephemeralFs, this.storeDir);
            MatcherAssert.assertThat((Object)recoverer.isRecoveryRequiredAt(this.databaseLayout), (Matcher)CoreMatchers.is((Object)false));
        }
    }

    @Test
    void shouldBeAbleToRecoverBrokenStoreWithLogsInSeparateAbsoluteLocation() throws Exception {
        File customTransactionLogsLocation = this.testDirectory.directory("transactions", new String[0]);
        Config config = Config.newBuilder().set(GraphDatabaseSettings.neo4j_home, (Object)this.testDirectory.homeDir().toPath()).set(GraphDatabaseSettings.transaction_logs_root_path, (Object)customTransactionLogsLocation.toPath().toAbsolutePath()).build();
        this.recoverBrokenStoreWithConfig(config);
    }

    @Test
    void shouldNotWantToRecoverEmptyStore() throws Exception {
        DatabaseLayout databaseLayout = DatabaseLayout.ofFlat((File)this.testDirectory.directory("dir-without-store", new String[0]));
        PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);
        RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(databaseLayout));
    }

    @Test
    void shouldWantToRecoverStoreWithoutOneIdFile() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertAllIdFilesExist();
        PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);
        RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
        this.fileSystem.deleteFileOrThrow(this.databaseLayout.idNodeStore());
        Assertions.assertTrue((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
    }

    @Test
    void shouldWantToRecoverStoreWithoutAllIdFiles() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertAllIdFilesExist();
        PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);
        RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
        for (File idFile : this.databaseLayout.idFiles()) {
            this.fileSystem.deleteFileOrThrow(idFile);
        }
        Assertions.assertTrue((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
    }

    @Test
    void recoveryRequiredWhenAnyStoreFileIsMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);
        RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
        this.fileSystem.deleteFileOrThrow(this.databaseLayout.nodeStore());
        Assertions.assertTrue((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
    }

    @Test
    void recoveryRequiredWhenSeveralStoreFileAreMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);
        RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
        this.fileSystem.deleteFileOrThrow(this.databaseLayout.relationshipStore());
        this.fileSystem.deleteFileOrThrow(this.databaseLayout.propertyStore());
        this.fileSystem.deleteFileOrThrow(this.databaseLayout.relationshipTypeTokenStore());
        Assertions.assertTrue((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
    }

    @Test
    void recoveryNotRequiredWhenCountStoreIsMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);
        RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
        this.fileSystem.deleteFileOrThrow(this.databaseLayout.countStore());
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
    }

    @Test
    void recoveryNotRequiredWhenIndexStatisticStoreIsMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);
        RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
        this.fileSystem.deleteFileOrThrow(this.databaseLayout.indexStatisticsStore());
        Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout));
    }

    private void recoverBrokenStoreWithConfig(Config config) throws IOException {
        try (EphemeralFileSystemAbstraction ephemeralFs = RecoveryRequiredCheckerTest.createSomeDataAndCrash(this.storeDir, config);){
            PageCache pageCache = pageCacheExtension.getPageCache((FileSystemAbstraction)ephemeralFs);
            RecoveryRequiredChecker recoveryChecker = RecoveryRequiredCheckerTest.getRecoveryChecker((FileSystemAbstraction)ephemeralFs, pageCache, this.storageEngineFactory, config);
            MatcherAssert.assertThat((Object)recoveryChecker.isRecoveryRequiredAt(DatabaseLayout.of((Config)config)), (Matcher)CoreMatchers.is((Object)true));
            DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(this.storeDir).setFileSystem((FileSystemAbstraction)ephemeralFs).setConfig(config).build();
            managementService.shutdown();
            MatcherAssert.assertThat((Object)recoveryChecker.isRecoveryRequiredAt(this.databaseLayout), (Matcher)CoreMatchers.is((Object)false));
        }
    }

    private EphemeralFileSystemAbstraction createAndCrashWithDefaultConfig() throws IOException {
        return RecoveryRequiredCheckerTest.createSomeDataAndCrash(this.storeDir, Config.defaults());
    }

    private void assertAllIdFilesExist() {
        for (File idFile : this.databaseLayout.idFiles()) {
            Assertions.assertTrue((boolean)this.fileSystem.fileExists(idFile), (String)("ID file " + idFile + " does not exist"));
        }
    }

    private void assertStoreFilesExist() {
        for (File file : this.databaseLayout.storeFiles()) {
            Assertions.assertTrue((boolean)this.fileSystem.fileExists(file), (String)("Store file " + file + " does not exist"));
        }
    }

    private static RecoveryRequiredChecker getRecoveryCheckerWithDefaultConfig(FileSystemAbstraction fileSystem, PageCache pageCache, StorageEngineFactory storageEngineFactory) {
        return RecoveryRequiredCheckerTest.getRecoveryChecker(fileSystem, pageCache, storageEngineFactory, Config.defaults());
    }

    private static RecoveryRequiredChecker getRecoveryChecker(FileSystemAbstraction fileSystem, PageCache pageCache, StorageEngineFactory storageEngineFactory, Config config) {
        return new RecoveryRequiredChecker(fileSystem, pageCache, config, storageEngineFactory);
    }

    private static EphemeralFileSystemAbstraction createSomeDataAndCrash(File store, Config config) throws IOException {
        try (EphemeralFileSystemAbstraction ephemeralFs = new EphemeralFileSystemAbstraction();){
            DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(store).setFileSystem((FileSystemAbstraction)ephemeralFs).setConfig(config).build();
            GraphDatabaseService db = managementService.database("neo4j");
            try (Transaction tx = db.beginTx();){
                tx.createNode();
                tx.commit();
            }
            EphemeralFileSystemAbstraction snapshot = ephemeralFs.snapshot();
            managementService.shutdown();
            EphemeralFileSystemAbstraction ephemeralFileSystemAbstraction = snapshot;
            return ephemeralFileSystemAbstraction;
        }
    }

    private static DatabaseManagementService startDatabase(FileSystemAbstraction fileSystem, File storeDir) {
        return new TestDatabaseManagementServiceBuilder(storeDir).setFileSystem(fileSystem).build();
    }

    private static void startStopDatabase(FileSystemAbstraction fileSystem, File storeDir) {
        DatabaseManagementService managementService = RecoveryRequiredCheckerTest.startDatabase(fileSystem, storeDir);
        managementService.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void startStopAndCreateDefaultData() {
        DatabaseManagementService managementService = RecoveryRequiredCheckerTest.startDatabase(this.fileSystem, this.storeDir);
        try {
            GraphDatabaseService database = managementService.database("neo4j");
            try (Transaction transaction = database.beginTx();){
                transaction.createNode();
                transaction.commit();
            }
        }
        finally {
            managementService.shutdown();
        }
    }
}

