/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.newapi;

import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.exceptions.KernelException;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.kernel.api.NodeCursor;
import org.neo4j.internal.kernel.api.PropertyCursor;
import org.neo4j.internal.kernel.api.exceptions.EntityNotFoundException;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.impl.newapi.KernelAPIWriteTestBase;
import org.neo4j.kernel.impl.newapi.KernelAPIWriteTestSupport;
import org.neo4j.values.storable.TextValue;
import org.neo4j.values.storable.Value;
import org.neo4j.values.storable.Values;

public abstract class NodeWriteTestBase<G extends KernelAPIWriteTestSupport>
extends KernelAPIWriteTestBase<G> {
    private static final String propertyKey = "prop";
    private static final String labelName = "Town";

    @Test
    void shouldCreateNode() throws Exception {
        long node;
        try (KernelTransaction tx = this.beginTransaction();){
            node = tx.dataWrite().nodeCreate();
            tx.commit();
        }
        tx = graphDb.beginTx();
        try {
            Assertions.assertEquals((long)node, (long)tx.getNodeById(node).getId());
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldRollbackOnFailure() throws Exception {
        long node;
        try (KernelTransaction tx = this.beginTransaction();){
            node = tx.dataWrite().nodeCreate();
            tx.rollback();
        }
        try {
            tx = graphDb.beginTx();
            try {
                tx.getNodeById(node);
                Assertions.fail((String)"There should be no node");
            }
            finally {
                if (tx != null) {
                    tx.close();
                }
            }
        }
        catch (NotFoundException notFoundException) {
            // empty catch block
        }
    }

    @Test
    void shouldRemoveNode() throws Exception {
        long node = this.createNode();
        try (KernelTransaction tx = this.beginTransaction();){
            tx.dataWrite().nodeDelete(node);
            tx.commit();
        }
        tx = graphDb.beginTx();
        try {
            try {
                tx.getNodeById(node);
                Assertions.fail((String)"Did not remove node");
            }
            catch (NotFoundException notFoundException) {
                // empty catch block
            }
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldNotRemoveNodeThatDoesNotExist() throws Exception {
        long node = 0L;
        try (KernelTransaction tx = this.beginTransaction();){
            Assertions.assertFalse((boolean)tx.dataWrite().nodeDelete(node));
            tx.rollback();
        }
        tx = this.beginTransaction();
        try {
            Assertions.assertFalse((boolean)tx.dataWrite().nodeDelete(node));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldAddLabelNode() throws Exception {
        long node = this.createNode();
        try (KernelTransaction tx = this.beginTransaction();){
            int labelId = tx.token().labelGetOrCreateForName(labelName);
            Assertions.assertTrue((boolean)tx.dataWrite().nodeAddLabel(node, labelId));
            tx.commit();
        }
        this.assertLabels(node, labelName);
    }

    @Test
    void shouldAddLabelNodeOnce() throws Exception {
        long node = this.createNodeWithLabel(labelName);
        try (KernelTransaction tx = this.beginTransaction();){
            int labelId = tx.token().labelGetOrCreateForName(labelName);
            Assertions.assertFalse((boolean)tx.dataWrite().nodeAddLabel(node, labelId));
            tx.commit();
        }
        this.assertLabels(node, labelName);
    }

    @Test
    void shouldRemoveLabel() throws Exception {
        long nodeId = this.createNodeWithLabel(labelName);
        try (KernelTransaction tx = this.beginTransaction();){
            int labelId = tx.token().labelGetOrCreateForName(labelName);
            Assertions.assertTrue((boolean)tx.dataWrite().nodeRemoveLabel(nodeId, labelId));
            tx.commit();
        }
        this.assertNoLabels(nodeId);
    }

    @Test
    void shouldNotAddLabelToNonExistingNode() throws Exception {
        long node = 1337L;
        try (KernelTransaction tx = this.beginTransaction();){
            int labelId = tx.token().labelGetOrCreateForName(labelName);
            Assertions.assertThrows(KernelException.class, () -> tx.dataWrite().nodeAddLabel(node, labelId));
        }
    }

    @Test
    void shouldRemoveLabelOnce() throws Exception {
        int labelId;
        long nodeId = this.createNodeWithLabel(labelName);
        try (KernelTransaction tx = this.beginTransaction();){
            labelId = tx.token().labelGetOrCreateForName(labelName);
            Assertions.assertTrue((boolean)tx.dataWrite().nodeRemoveLabel(nodeId, labelId));
            tx.commit();
        }
        tx = this.beginTransaction();
        try {
            labelId = tx.token().labelGetOrCreateForName(labelName);
            Assertions.assertFalse((boolean)tx.dataWrite().nodeRemoveLabel(nodeId, labelId));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        this.assertNoLabels(nodeId);
    }

    @Test
    void shouldAddPropertyToNode() throws Exception {
        long node = this.createNode();
        try (KernelTransaction tx = this.beginTransaction();){
            int token = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeSetProperty(node, token, (Value)Values.stringValue((String)"hello")), (Matcher)Matchers.equalTo((Object)Values.NO_VALUE));
            tx.commit();
        }
        this.assertProperty(node, propertyKey, "hello");
    }

    @Test
    void shouldRollbackSetNodeProperty() throws Exception {
        long node = this.createNode();
        try (KernelTransaction tx = this.beginTransaction();){
            int token = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeSetProperty(node, token, (Value)Values.stringValue((String)"hello")), (Matcher)Matchers.equalTo((Object)Values.NO_VALUE));
            tx.rollback();
        }
        this.assertNoProperty(node, propertyKey);
    }

    @Test
    void shouldThrowWhenSettingPropertyOnDeletedNode() throws Exception {
        long node = this.createNode();
        this.deleteNode(node);
        try (KernelTransaction tx = this.beginTransaction();){
            int token = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            tx.dataWrite().nodeSetProperty(node, token, (Value)Values.stringValue((String)"hello"));
            Assertions.fail((String)"Expected EntityNotFoundException");
        }
        catch (EntityNotFoundException entityNotFoundException) {
            // empty catch block
        }
    }

    @Test
    void shouldUpdatePropertyToNode() throws Exception {
        long node = this.createNodeWithProperty(propertyKey, 42);
        try (KernelTransaction tx = this.beginTransaction();){
            int token = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeSetProperty(node, token, (Value)Values.stringValue((String)"hello")), (Matcher)Matchers.equalTo((Object)Values.intValue((int)42)));
            tx.commit();
        }
        this.assertProperty(node, propertyKey, "hello");
    }

    @Test
    void shouldRemovePropertyFromNode() throws Exception {
        long node = this.createNodeWithProperty(propertyKey, 42);
        try (KernelTransaction tx = this.beginTransaction();){
            int token = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeRemoveProperty(node, token), (Matcher)Matchers.equalTo((Object)Values.intValue((int)42)));
            tx.commit();
        }
        this.assertNoProperty(node, propertyKey);
    }

    @Test
    void shouldRemoveNonExistingPropertyFromNode() throws Exception {
        long node = this.createNode();
        try (KernelTransaction tx = this.beginTransaction();){
            int token = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeRemoveProperty(node, token), (Matcher)Matchers.equalTo((Object)Values.NO_VALUE));
            tx.commit();
        }
        this.assertNoProperty(node, propertyKey);
    }

    @Test
    void shouldRemovePropertyFromNodeTwice() throws Exception {
        long node = this.createNodeWithProperty(propertyKey, 42);
        try (KernelTransaction tx = this.beginTransaction();){
            int token = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeRemoveProperty(node, token), (Matcher)Matchers.equalTo((Object)Values.intValue((int)42)));
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeRemoveProperty(node, token), (Matcher)Matchers.equalTo((Object)Values.NO_VALUE));
            tx.commit();
        }
        this.assertNoProperty(node, propertyKey);
    }

    @Test
    void shouldUpdatePropertyToNodeInTransaction() throws Exception {
        long node = this.createNode();
        try (KernelTransaction tx = this.beginTransaction();){
            int token = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeSetProperty(node, token, (Value)Values.stringValue((String)"hello")), (Matcher)Matchers.equalTo((Object)Values.NO_VALUE));
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeSetProperty(node, token, (Value)Values.stringValue((String)"world")), (Matcher)Matchers.equalTo((Object)Values.stringValue((String)"hello")));
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeSetProperty(node, token, (Value)Values.intValue((int)1337)), (Matcher)Matchers.equalTo((Object)Values.stringValue((String)"world")));
            tx.commit();
        }
        this.assertProperty(node, propertyKey, 1337);
    }

    @Test
    void shouldRemoveReSetAndTwiceRemovePropertyOnNode() throws Exception {
        long node = this.createNodeWithProperty(propertyKey, "bar");
        try (KernelTransaction tx = this.beginTransaction();){
            int prop = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            tx.dataWrite().nodeRemoveProperty(node, prop);
            tx.dataWrite().nodeSetProperty(node, prop, Values.of((Object)"bar"));
            tx.dataWrite().nodeRemoveProperty(node, prop);
            tx.dataWrite().nodeRemoveProperty(node, prop);
            tx.commit();
        }
        this.assertNoProperty(node, propertyKey);
    }

    @Test
    void shouldNotWriteWhenSettingPropertyToSameValue() throws Exception {
        TextValue theValue = Values.stringValue((String)"The Value");
        long nodeId = this.createNodeWithProperty(propertyKey, theValue.asObject());
        KernelTransaction tx = this.beginTransaction();
        int property = tx.token().propertyKeyGetOrCreateForName(propertyKey);
        MatcherAssert.assertThat((Object)tx.dataWrite().nodeSetProperty(nodeId, property, (Value)theValue), (Matcher)Matchers.equalTo((Object)theValue));
        MatcherAssert.assertThat((Object)tx.commit(), (Matcher)Matchers.equalTo((Object)0L));
    }

    @Test
    void shouldSetAndReadLargeByteArrayPropertyToNode() throws Exception {
        int prop;
        long node = this.createNode();
        Value largeByteArray = Values.of((Object)new byte[100000]);
        try (KernelTransaction tx = this.beginTransaction();){
            prop = tx.token().propertyKeyGetOrCreateForName(propertyKey);
            MatcherAssert.assertThat((Object)tx.dataWrite().nodeSetProperty(node, prop, largeByteArray), (Matcher)Matchers.equalTo((Object)Values.NO_VALUE));
            tx.commit();
        }
        tx = this.beginTransaction();
        try (NodeCursor nodeCursor = tx.cursors().allocateNodeCursor();
             PropertyCursor propertyCursor = tx.cursors().allocatePropertyCursor();){
            tx.dataRead().singleNode(node, nodeCursor);
            Assertions.assertTrue((boolean)nodeCursor.next());
            nodeCursor.properties(propertyCursor);
            Assertions.assertTrue((boolean)propertyCursor.next());
            Assertions.assertEquals((int)propertyCursor.propertyKey(), (int)prop);
            MatcherAssert.assertThat((Object)propertyCursor.propertyValue(), (Matcher)Matchers.equalTo((Object)largeByteArray));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private long createNode() {
        long node;
        try (Transaction ctx = graphDb.beginTx();){
            node = ctx.createNode().getId();
            ctx.commit();
        }
        return node;
    }

    private void deleteNode(long node) {
        try (Transaction tx = graphDb.beginTx();){
            tx.getNodeById(node).delete();
            tx.commit();
        }
    }

    private long createNodeWithLabel(String labelName) {
        long node;
        try (Transaction ctx = graphDb.beginTx();){
            node = ctx.createNode(new Label[]{Label.label((String)labelName)}).getId();
            ctx.commit();
        }
        return node;
    }

    private long createNodeWithProperty(String propertyKey, Object value) {
        Node node;
        try (Transaction ctx = graphDb.beginTx();){
            node = ctx.createNode();
            node.setProperty(propertyKey, value);
            ctx.commit();
        }
        return node.getId();
    }

    private void assertNoLabels(long nodeId) {
        try (Transaction tx = graphDb.beginTx();){
            MatcherAssert.assertThat((Object)tx.getNodeById(nodeId).getLabels(), (Matcher)Matchers.equalTo((Object)Iterables.empty()));
        }
    }

    private void assertLabels(long nodeId, String label) {
        try (Transaction tx = graphDb.beginTx();){
            MatcherAssert.assertThat((Object)tx.getNodeById(nodeId).getLabels(), (Matcher)Matchers.containsInAnyOrder((Object[])new Label[]{Label.label((String)label)}));
        }
    }

    private void assertNoProperty(long node, String propertyKey) {
        try (Transaction tx = graphDb.beginTx();){
            Assertions.assertFalse((boolean)tx.getNodeById(node).hasProperty(propertyKey));
        }
    }

    private void assertProperty(long node, String propertyKey, Object value) {
        try (Transaction tx = graphDb.beginTx();){
            MatcherAssert.assertThat((Object)tx.getNodeById(node).getProperty(propertyKey), (Matcher)Matchers.equalTo((Object)value));
        }
    }
}

