/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.File;
import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.DatabaseStateService;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.dbms.database.DatabaseStartAbortedException;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreFileChannel;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.kernel.availability.CompositeDatabaseAvailabilityGuard;
import org.neo4j.kernel.extension.ExtensionFactory;
import org.neo4j.kernel.extension.context.ExtensionContext;
import org.neo4j.kernel.impl.store.MetaDataStore;
import org.neo4j.kernel.impl.storemigration.LegacyTransactionLogsLocator;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.ReadableClosablePositionAwareChecksumChannel;
import org.neo4j.kernel.impl.transaction.log.ReadableLogChannel;
import org.neo4j.kernel.impl.transaction.log.entry.CheckPoint;
import org.neo4j.kernel.impl.transaction.log.entry.LogEntry;
import org.neo4j.kernel.impl.transaction.log.entry.VersionAwareLogEntryReader;
import org.neo4j.kernel.impl.transaction.log.files.LogFile;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.kernel.lifecycle.LifecycleAdapter;
import org.neo4j.kernel.recovery.Recovery;
import org.neo4j.kernel.recovery.RecoveryMonitor;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogProvider;
import org.neo4j.monitoring.Monitors;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.Neo4jLayoutExtension;
import org.neo4j.test.extension.pagecache.PageCacheExtension;

@PageCacheExtension
@Neo4jLayoutExtension
class RecoveryIT {
    private static final int TEN_KB = (int)ByteUnit.kibiBytes((long)10L);
    @Inject
    private DefaultFileSystemAbstraction fileSystem;
    @Inject
    private PageCache pageCache;
    @Inject
    private Neo4jLayout neo4jLayout;
    @Inject
    private DatabaseLayout databaseLayout;
    private TestDatabaseManagementServiceBuilder builder;
    private DatabaseManagementService managementService;

    RecoveryIT() {
    }

    @Test
    void recoveryRequiredOnDatabaseWithoutCorrectCheckpoints() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        this.managementService.shutdown();
        this.removeLastCheckpointRecordFromLastLogFile();
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
    }

    @Test
    void recoveryNotRequiredWhenDatabaseNotFound() throws Exception {
        DatabaseLayout absentDatabase = this.neo4jLayout.databaseLayout("absent");
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)absentDatabase, (Config)Config.defaults()));
    }

    @Test
    void recoverEmptyDatabase() throws Exception {
        this.createDatabase();
        this.managementService.shutdown();
        this.removeLastCheckpointRecordFromLastLogFile();
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithNodes() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfNodes = 10;
        for (int i = 0; i < numberOfNodes; ++i) {
            this.createSingleNode((GraphDatabaseService)database);
        }
        this.managementService.shutdown();
        this.removeLastCheckpointRecordFromLastLogFile();
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try (Transaction tx = recoveredDatabase.beginTx();){
            Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)tx.getAllNodes()));
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithNodesAndRelationshipsAndRelationshipTypes() throws Exception {
        Transaction transaction;
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfRelationships = 10;
        int numberOfNodes = numberOfRelationships * 2;
        for (int i = 0; i < numberOfRelationships; ++i) {
            transaction = database.beginTx();
            try {
                Node start = transaction.createNode();
                Node stop = transaction.createNode();
                start.createRelationshipTo(stop, RelationshipType.withName((String)String.valueOf(i)));
                transaction.commit();
                continue;
            }
            finally {
                if (transaction != null) {
                    transaction.close();
                }
            }
        }
        this.managementService.shutdown();
        this.removeLastCheckpointRecordFromLastLogFile();
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try {
            transaction = recoveredDatabase.beginTx();
            try {
                Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)transaction.getAllNodes()));
                Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationships()));
                Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationshipTypesInUse()));
            }
            finally {
                if (transaction != null) {
                    transaction.close();
                }
            }
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithProperties() throws Exception {
        Transaction transaction;
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfRelationships = 10;
        int numberOfNodes = numberOfRelationships * 2;
        for (int i = 0; i < numberOfRelationships; ++i) {
            transaction = database.beginTx();
            try {
                Node start = transaction.createNode();
                Node stop = transaction.createNode();
                start.setProperty("start" + i, (Object)i);
                stop.setProperty("stop" + i, (Object)i);
                start.createRelationshipTo(stop, RelationshipType.withName((String)String.valueOf(i)));
                transaction.commit();
                continue;
            }
            finally {
                if (transaction != null) {
                    transaction.close();
                }
            }
        }
        this.managementService.shutdown();
        this.removeLastCheckpointRecordFromLastLogFile();
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try {
            transaction = recoveredDatabase.beginTx();
            try {
                Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)transaction.getAllNodes()));
                Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationships()));
                Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationshipTypesInUse()));
                Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)transaction.getAllPropertyKeys()));
            }
            finally {
                if (transaction != null) {
                    transaction.close();
                }
            }
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithIndex() throws Exception {
        long numberOfPropertyKeys;
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfRelationships = 10;
        int numberOfNodes = numberOfRelationships * 2;
        String startProperty = "start";
        String stopProperty = "stop";
        Label startMarker = Label.label((String)"start");
        Label stopMarker = Label.label((String)"stop");
        try (Transaction transaction = database.beginTx();){
            transaction.schema().indexFor(startMarker).on(startProperty).create();
            transaction.schema().constraintFor(stopMarker).assertPropertyIsUnique(stopProperty).create();
            transaction.commit();
        }
        transaction = database.beginTx();
        try {
            transaction.schema().awaitIndexesOnline(1L, TimeUnit.MINUTES);
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
        for (int i = 0; i < numberOfRelationships; ++i) {
            try (Transaction transaction = database.beginTx();){
                Node start = transaction.createNode(new Label[]{startMarker});
                Node stop = transaction.createNode(new Label[]{stopMarker});
                start.setProperty(startProperty, (Object)i);
                stop.setProperty(stopProperty, (Object)i);
                start.createRelationshipTo(stop, RelationshipType.withName((String)String.valueOf(i)));
                transaction.commit();
                continue;
            }
        }
        try (Transaction transaction = database.beginTx();){
            numberOfPropertyKeys = Iterables.count((Iterable)transaction.getAllPropertyKeys());
        }
        this.managementService.shutdown();
        this.removeLastCheckpointRecordFromLastLogFile();
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try (Transaction transaction = recoveredDatabase.beginTx();){
            Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)transaction.getAllNodes()));
            Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationships()));
            Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationshipTypesInUse()));
            Assertions.assertEquals((long)numberOfPropertyKeys, (long)Iterables.count((Iterable)transaction.getAllPropertyKeys()));
        }
        finally {
            this.managementService.shutdown();
        }
    }

    @Test
    void recoverDatabaseWithFirstTransactionLogFileWithoutShutdownCheckpoint() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        this.managementService.shutdown();
        Assertions.assertEquals((int)1, (int)this.countCheckPointsInTransactionLogs());
        this.removeLastCheckpointRecordFromLastLogFile();
        Assertions.assertEquals((int)0, (int)this.countCheckPointsInTransactionLogs());
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        this.startStopDatabase();
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void failToStartDatabaseWithRemovedTransactionLogs() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        this.managementService.shutdown();
        this.removeTransactionLogs();
        GraphDatabaseAPI restartedDb = this.createDatabase();
        try {
            DatabaseStateService dbStateService = (DatabaseStateService)restartedDb.getDependencyResolver().resolveDependency(DatabaseStateService.class);
            Optional failure = dbStateService.causeOfFailure(restartedDb.databaseId());
            Assertions.assertTrue((boolean)failure.isPresent());
            MatcherAssert.assertThat((Object)ExceptionUtils.getRootCause((Throwable)((Throwable)failure.get())).getMessage(), (Matcher)Matchers.containsString((String)"Transaction logs are missing and recovery is not possible."));
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void failToStartDatabaseWithTransactionLogsInLegacyLocation() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        this.managementService.shutdown();
        File[] txLogFiles = this.buildLogFiles().logFiles();
        File databasesDirectory = this.databaseLayout.getNeo4jLayout().databasesDirectory();
        DatabaseLayout legacyLayout = Neo4jLayout.ofFlat((File)databasesDirectory).databaseLayout(this.databaseLayout.getDatabaseName());
        LegacyTransactionLogsLocator logsLocator = new LegacyTransactionLogsLocator(Config.defaults(), legacyLayout);
        File transactionLogsDirectory = logsLocator.getTransactionLogsDirectory();
        Assertions.assertNotNull((Object)txLogFiles);
        Assertions.assertTrue((txLogFiles.length > 0 ? 1 : 0) != 0);
        for (File logFile : txLogFiles) {
            this.fileSystem.moveToDirectory(logFile, transactionLogsDirectory);
        }
        AssertableLogProvider logProvider = new AssertableLogProvider();
        this.builder.setInternalLogProvider((LogProvider)logProvider);
        GraphDatabaseAPI restartedDb = this.createDatabase();
        try {
            DatabaseStateService dbStateService = (DatabaseStateService)restartedDb.getDependencyResolver().resolveDependency(DatabaseStateService.class);
            Optional failure = dbStateService.causeOfFailure(restartedDb.databaseId());
            Assertions.assertTrue((boolean)failure.isPresent());
            MatcherAssert.assertThat((Object)ExceptionUtils.getRootCause((Throwable)((Throwable)failure.get())).getMessage(), (Matcher)Matchers.containsString((String)"Transaction logs are missing and recovery is not possible."));
            AssertableLogProvider.MessageMatcher messageMatcher = logProvider.formattedMessageMatcher();
            messageMatcher.assertContains(txLogFiles[0].getName());
        }
        finally {
            this.managementService.shutdown();
        }
    }

    @Test
    void startDatabaseWithRemovedSingleTransactionLogFile() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        PageCache pageCache = this.getDatabasePageCache(database);
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        Assertions.assertEquals((long)-1L, (long)MetaDataStore.getRecord((PageCache)pageCache, (File)database.databaseLayout().metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.LAST_MISSING_STORE_FILES_RECOVERY_TIMESTAMP));
        this.managementService.shutdown();
        this.removeTransactionLogs();
        this.startStopDatabaseWithForcedRecovery();
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        this.verifyRecoveryTimestampPresent(database);
    }

    @Test
    void startDatabaseWithRemovedMultipleTransactionLogFiles() throws Exception {
        DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(GraphDatabaseSettings.logical_log_rotation_threshold, (Object)ByteUnit.mebiBytes((long)1L)).build();
        GraphDatabaseService database = managementService.database("neo4j");
        while (this.countTransactionLogFiles() < 5) {
            RecoveryIT.generateSomeData(database);
        }
        managementService.shutdown();
        this.removeTransactionLogs();
        this.startStopDatabaseWithForcedRecovery();
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
    }

    @Test
    void killAndStartDatabaseAfterTransactionLogsRemoval() throws Exception {
        DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(GraphDatabaseSettings.logical_log_rotation_threshold, (Object)ByteUnit.mebiBytes((long)1L)).build();
        GraphDatabaseService database = managementService.database("neo4j");
        while (this.countTransactionLogFiles() < 5) {
            RecoveryIT.generateSomeData(database);
        }
        managementService.shutdown();
        this.removeTransactionLogs();
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Assertions.assertEquals((int)0, (int)this.countTransactionLogFiles());
        DatabaseManagementService forcedRecoveryManagementService = this.forcedRecoveryManagement();
        GraphDatabaseService service = forcedRecoveryManagementService.database("neo4j");
        this.createSingleNode(service);
        forcedRecoveryManagementService.shutdown();
        Assertions.assertEquals((int)1, (int)this.countTransactionLogFiles());
        Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        this.removeLastCheckpointRecordFromLastLogFile();
        this.startStopDatabase();
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Assertions.assertEquals((int)3, (int)this.countCheckPointsInTransactionLogs());
    }

    @Test
    void killAndStartDatabaseAfterTransactionLogsRemovalWithSeveralFilesWithoutCheckpoint() throws Exception {
        DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(GraphDatabaseSettings.logical_log_rotation_threshold, (Object)ByteUnit.mebiBytes((long)1L)).build();
        GraphDatabaseService database = managementService.database("neo4j");
        while (this.countTransactionLogFiles() < 5) {
            RecoveryIT.generateSomeData(database);
        }
        managementService.shutdown();
        this.removeHighestLogFile();
        Assertions.assertEquals((int)4, (int)this.countTransactionLogFiles());
        Assertions.assertEquals((int)0, (int)this.countCheckPointsInTransactionLogs());
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        this.startStopDatabase();
        Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        this.removeLastCheckpointRecordFromLastLogFile();
        this.removeLastCheckpointRecordFromLastLogFile();
        this.startStopDatabase();
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
    }

    @Test
    void startDatabaseAfterTransactionLogsRemovalAndKillAfterRecovery() throws Exception {
        DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(GraphDatabaseSettings.logical_log_rotation_threshold, (Object)ByteUnit.mebiBytes((long)1L)).build();
        GraphDatabaseService database = managementService.database("neo4j");
        while (this.countTransactionLogFiles() < 5) {
            RecoveryIT.generateSomeData(database);
        }
        managementService.shutdown();
        this.removeHighestLogFile();
        Assertions.assertEquals((int)4, (int)this.countTransactionLogFiles());
        Assertions.assertEquals((int)0, (int)this.countCheckPointsInTransactionLogs());
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        this.startStopDatabase();
        Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        this.removeLastCheckpointRecordFromLastLogFile();
        this.startStopDatabase();
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        this.removeLastCheckpointRecordFromLastLogFile();
        GraphDatabaseAPI service = this.createDatabase();
        this.createSingleNode((GraphDatabaseService)service);
        this.managementService.shutdown();
        this.removeLastCheckpointRecordFromLastLogFile();
        this.startStopDatabase();
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Assertions.assertEquals((int)3, (int)this.countCheckPointsInTransactionLogs());
    }

    @Test
    void recoverDatabaseWithoutOneIdFile() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        this.fileSystem.deleteFileOrThrow(layout.idRelationshipStore());
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)layout, (Config)Config.defaults()));
        Recovery.performRecovery((FileSystemAbstraction)this.fileSystem, (PageCache)this.pageCache, (Config)Config.defaults(), (DatabaseLayout)layout);
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)layout, (Config)Config.defaults()));
        Assertions.assertTrue((boolean)this.fileSystem.fileExists(layout.idRelationshipStore()));
    }

    @Test
    void recoverDatabaseWithoutIdFiles() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        for (File idFile : layout.idFiles()) {
            this.fileSystem.deleteFileOrThrow(idFile);
        }
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)layout, (Config)Config.defaults()));
        Recovery.performRecovery((FileSystemAbstraction)this.fileSystem, (PageCache)this.pageCache, (Config)Config.defaults(), (DatabaseLayout)layout);
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)layout, (Config)Config.defaults()));
        for (File idFile : layout.idFiles()) {
            Assertions.assertTrue((boolean)this.fileSystem.fileExists(idFile));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void cancelRecoveryInTheMiddle() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        this.removeLastCheckpointRecordFromLastLogFile();
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)layout, (Config)Config.defaults()));
        Monitors monitors = new Monitors();
        RecoveryMonitor recoveryMonitor = new RecoveryMonitor(){

            public void reverseStoreRecoveryCompleted(long lowestRecoveredTxId) {
                GlobalGuardConsumer.globalGuard.stop();
            }
        };
        monitors.addMonitorListener((Object)recoveryMonitor, new String[0]);
        DatabaseManagementService service = new TestDatabaseManagementServiceBuilder(layout.getNeo4jLayout()).addExtension((ExtensionFactory)new GlobalGuardConsumerTestExtensionFactory()).setMonitors(monitors).build();
        try {
            Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> service.database("neo4j").beginTx());
            MatcherAssert.assertThat((Object)ExceptionUtils.getRootCause((Throwable)e), (Matcher)Matchers.instanceOf(DatabaseStartAbortedException.class));
        }
        finally {
            service.shutdown();
        }
    }

    private void createSingleNode(GraphDatabaseService service) {
        try (Transaction transaction = service.beginTx();){
            transaction.createNode();
            transaction.commit();
        }
    }

    private void startStopDatabase() {
        this.createDatabase();
        this.managementService.shutdown();
    }

    private void recoverDatabase() throws Exception {
        Assertions.assertTrue((boolean)Recovery.isRecoveryRequired((DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
        Recovery.performRecovery((DatabaseLayout)this.databaseLayout);
        Assertions.assertFalse((boolean)Recovery.isRecoveryRequired((DatabaseLayout)this.databaseLayout, (Config)Config.defaults()));
    }

    private int countCheckPointsInTransactionLogs() throws IOException {
        int checkpointCounter = 0;
        LogFiles logFiles = this.buildLogFiles();
        LogFile transactionLogFile = logFiles.getLogFile();
        VersionAwareLogEntryReader entryReader = new VersionAwareLogEntryReader();
        LogPosition startPosition = logFiles.extractHeader(logFiles.getHighestLogVersion()).getStartPosition();
        try (ReadableLogChannel reader = transactionLogFile.getReader(startPosition);){
            LogEntry logEntry;
            do {
                if (!((logEntry = entryReader.readLogEntry((ReadableClosablePositionAwareChecksumChannel)reader)) instanceof CheckPoint)) continue;
                ++checkpointCounter;
            } while (logEntry != null);
        }
        return checkpointCounter;
    }

    private LogFiles buildLogFiles() throws IOException {
        return LogFilesBuilder.logFilesBasedOnlyBuilder((File)this.databaseLayout.getTransactionLogsDirectory(), (FileSystemAbstraction)this.fileSystem).build();
    }

    private void removeTransactionLogs() throws IOException {
        File[] txLogFiles;
        LogFiles logFiles = this.buildLogFiles();
        for (File logFile : txLogFiles = logFiles.logFiles()) {
            this.fileSystem.deleteFile(logFile);
        }
    }

    private void removeHighestLogFile() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        long highestLogVersion = logFiles.getHighestLogVersion();
        this.removeFileByVersion(logFiles, highestLogVersion);
    }

    private void removeFileByVersion(LogFiles logFiles, long version) {
        File versionFile = logFiles.getLogFileForVersion(version);
        Assertions.assertNotNull((Object)versionFile);
        this.fileSystem.deleteFile(versionFile);
    }

    private int countTransactionLogFiles() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        return logFiles.logFiles().length;
    }

    private void removeLastCheckpointRecordFromLastLogFile() throws IOException {
        LogPosition checkpointPosition = null;
        LogFiles logFiles = this.buildLogFiles();
        LogFile transactionLogFile = logFiles.getLogFile();
        VersionAwareLogEntryReader entryReader = new VersionAwareLogEntryReader();
        LogPosition startPosition = logFiles.extractHeader(logFiles.getHighestLogVersion()).getStartPosition();
        try (ReadableLogChannel reader = transactionLogFile.getReader(startPosition);){
            LogEntry logEntry;
            do {
                if (!((logEntry = entryReader.readLogEntry((ReadableClosablePositionAwareChecksumChannel)reader)) instanceof CheckPoint)) continue;
                checkpointPosition = ((CheckPoint)logEntry).getLogPosition();
            } while (logEntry != null);
        }
        if (checkpointPosition != null) {
            try (StoreFileChannel storeChannel = this.fileSystem.write(logFiles.getHighestLogFile());){
                storeChannel.truncate(checkpointPosition.getByteOffset());
            }
        }
    }

    private static void generateSomeData(GraphDatabaseService database) {
        for (int i = 0; i < 10; ++i) {
            try (Transaction transaction = database.beginTx();){
                Node node1 = transaction.createNode();
                Node node2 = transaction.createNode();
                node1.createRelationshipTo(node2, RelationshipType.withName((String)("Type" + i)));
                node2.setProperty("a", (Object)RandomStringUtils.randomAlphanumeric((int)TEN_KB));
                transaction.commit();
                continue;
            }
        }
    }

    private GraphDatabaseAPI createDatabase() {
        this.createBuilder();
        this.managementService = this.builder.build();
        return (GraphDatabaseAPI)this.managementService.database("neo4j");
    }

    private void createBuilder() {
        if (this.builder == null) {
            this.builder = new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(GraphDatabaseSettings.logical_log_rotation_threshold, (Object)((Long)GraphDatabaseSettings.logical_log_rotation_threshold.defaultValue()));
        }
    }

    private void startStopDatabaseWithForcedRecovery() {
        DatabaseManagementService forcedRecoveryManagementService = this.forcedRecoveryManagement();
        forcedRecoveryManagementService.shutdown();
    }

    private DatabaseManagementService forcedRecoveryManagement() {
        return new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(GraphDatabaseSettings.fail_on_missing_files, (Object)false).build();
    }

    private PageCache getDatabasePageCache(GraphDatabaseAPI databaseAPI) {
        return (PageCache)databaseAPI.getDependencyResolver().resolveDependency(PageCache.class);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyRecoveryTimestampPresent(GraphDatabaseAPI databaseAPI) throws IOException {
        GraphDatabaseAPI restartedDatabase = this.createDatabase();
        try {
            PageCache restartedCache = this.getDatabasePageCache(restartedDatabase);
            MatcherAssert.assertThat((Object)MetaDataStore.getRecord((PageCache)restartedCache, (File)databaseAPI.databaseLayout().metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.LAST_MISSING_STORE_FILES_RECOVERY_TIMESTAMP), (Matcher)Matchers.greaterThan((Comparable)Long.valueOf(0L)));
        }
        finally {
            this.managementService.shutdown();
        }
    }

    private static class GlobalGuardConsumer
    extends LifecycleAdapter {
        private static CompositeDatabaseAvailabilityGuard globalGuard;

        GlobalGuardConsumer(GlobalGuardConsumerTestExtensionFactory.Dependencies dependencies) {
            globalGuard = dependencies.globalGuard();
        }
    }

    private static class GlobalGuardConsumerTestExtensionFactory
    extends ExtensionFactory<Dependencies> {
        GlobalGuardConsumerTestExtensionFactory() {
            super("globalGuardConsumer");
        }

        public Lifecycle newInstance(ExtensionContext context, Dependencies dependencies) {
            return new GlobalGuardConsumer(dependencies);
        }

        static interface Dependencies {
            public CompositeDatabaseAvailabilityGuard globalGuard();
        }
    }
}

