/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphdb;

import java.lang.reflect.Executable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.RepeatedTest;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.EnumSource;
import org.neo4j.function.ThrowingFunction;
import org.neo4j.graphdb.ConstraintViolationException;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.SchemaAcceptanceTestBase;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.ConstraintCreator;
import org.neo4j.graphdb.schema.ConstraintDefinition;
import org.neo4j.graphdb.schema.ConstraintType;
import org.neo4j.graphdb.schema.IndexCreator;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.graphdb.schema.IndexSetting;
import org.neo4j.graphdb.schema.IndexSettingImpl;
import org.neo4j.graphdb.schema.IndexType;
import org.neo4j.graphdb.schema.Schema;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.schema.IndexProviderDescriptor;
import org.neo4j.kernel.api.exceptions.schema.AlreadyConstrainedException;
import org.neo4j.kernel.api.exceptions.schema.AlreadyIndexedException;
import org.neo4j.kernel.api.exceptions.schema.ConstraintWithNameAlreadyExistsException;
import org.neo4j.kernel.api.exceptions.schema.EquivalentSchemaRuleAlreadyExistsException;
import org.neo4j.kernel.api.exceptions.schema.IndexWithNameAlreadyExistsException;
import org.neo4j.kernel.impl.coreapi.schema.IndexDefinitionImpl;
import org.neo4j.kernel.impl.index.schema.FulltextIndexProviderFactory;
import org.neo4j.test.extension.ImpermanentDbmsExtension;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.actors.Actor;
import org.neo4j.test.extension.actors.ActorsExtension;
import org.neo4j.util.concurrent.BinaryLatch;

@ImpermanentDbmsExtension
class SchemaAcceptanceTest
extends SchemaAcceptanceTestBase {
    @Inject
    private GraphDatabaseService db;
    private final Label label = Label.label((String)"MY_LABEL");
    private final Label otherLabel = Label.label((String)"MY_OTHER_LABEL");
    private final RelationshipType relType = RelationshipType.withName((String)"MY_REL_TYPE");
    private final RelationshipType otherRelType = RelationshipType.withName((String)"MY_OTHER_REL_TYPE");
    private final String propertyKey = "my_property_key";
    private final String secondPropertyKey = "my_second_property_key";

    SchemaAcceptanceTest() {
    }

    @Test
    void addingAnIndexingRuleShouldSucceed() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertThat(this.getIndexes(tx, this.label)).containsOnly((Object[])new IndexDefinition[]{index});
        }
    }

    @Test
    void addingACompositeIndexingRuleShouldSucceed() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key", "my_second_property_key");
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertThat(this.getIndexes(tx, this.label)).containsOnly((Object[])new IndexDefinition[]{index});
        }
    }

    @Test
    void addingNamedIndexRuleShouldSucceed() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, "MyIndex", this.label, "my_property_key");
        Assertions.assertThat((String)index.getName()).isEqualTo("MyIndex");
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.getIndexes(transaction, this.label)).containsOnly((Object[])new IndexDefinition[]{index});
        }
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfEquivalentIndexExist(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.indexFor(this.label).on("my_property_key").withName("name").create(), schema1 -> schema1.indexFor(this.label).on("my_property_key").withName("name").create(), ConstraintViolationException.class);
        Class<EquivalentSchemaRuleAlreadyExistsException> expectedCause = EquivalentSchemaRuleAlreadyExistsException.class;
        String expectedMessage = "An equivalent index already exists, 'Index( id=1, name='name', type='GENERAL BTREE', schema=(:MY_LABEL {my_property_key}), indexProvider='native-btree-1.0' )'.";
        this.assertExpectedException(exception, expectedCause, expectedMessage);
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfEquivalentUniquenessConstraintExist(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("name").create(), schema1 -> schema1.constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("name").create(), ConstraintViolationException.class);
        Class<EquivalentSchemaRuleAlreadyExistsException> expectedCause = EquivalentSchemaRuleAlreadyExistsException.class;
        this.assertExpectedException(exception, expectedCause, "An equivalent constraint already exists, 'Constraint( ", "name='name', type='UNIQUENESS', schema=(:MY_LABEL {my_property_key}), ownedIndex=1 )'.");
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfSchemaAlreadyIndexedWhenCreatingIndex(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.indexFor(this.label).on("my_property_key").withName("name").create(), schema1 -> schema1.indexFor(this.label).on("my_property_key").withName("otherName").create(), ConstraintViolationException.class);
        Class<AlreadyIndexedException> expectedCause = AlreadyIndexedException.class;
        String expectedMessage = "There already exists an index (:MY_LABEL {my_property_key}).";
        this.assertExpectedException(exception, expectedCause, expectedMessage);
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfSchemaAlreadyIndexedWhenCreatingUniquenessConstraint(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.indexFor(this.label).on("my_property_key").withName("name").create(), schema1 -> schema1.constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("otherName").create(), ConstraintViolationException.class);
        Class<AlreadyIndexedException> expectedCause = AlreadyIndexedException.class;
        String expectedMessage = "There already exists an index (:MY_LABEL {my_property_key}). A constraint cannot be created until the index has been dropped.";
        this.assertExpectedException(exception, expectedCause, expectedMessage);
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfSchemaAlreadyUniquenessConstrainedWhenCreatingIndex(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("name").create(), schema1 -> schema1.indexFor(this.label).on("my_property_key").withName("otherName").create(), ConstraintViolationException.class);
        Class<AlreadyConstrainedException> expectedCause = AlreadyConstrainedException.class;
        String expectedMessage = "There is a uniqueness constraint on (:MY_LABEL {my_property_key}), so an index is already created that matches this.";
        this.assertExpectedException(exception, expectedCause, expectedMessage);
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfSchemaAlreadyUniquenessConstrainedWhenCreatingUniquenessConstraint(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("name").create(), schema1 -> schema1.constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("otherName").create(), ConstraintViolationException.class);
        Class<AlreadyConstrainedException> expectedCause = AlreadyConstrainedException.class;
        this.assertExpectedException(exception, expectedCause, "Constraint already exists: Constraint( ", "name='name', type='UNIQUENESS', schema=(:MY_LABEL {my_property_key}), ownedIndex=1 )");
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfIndexWithNameExistsWhenCreatingIndex(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.indexFor(this.label).on("my_property_key").withName("name").create(), schema1 -> schema1.indexFor(this.label).on("my_second_property_key").withName("name").create(), ConstraintViolationException.class);
        Class<IndexWithNameAlreadyExistsException> expectedCause = IndexWithNameAlreadyExistsException.class;
        String expectedMessage = "There already exists an index called 'name'.";
        this.assertExpectedException(exception, expectedCause, expectedMessage);
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfIndexWithNameExistsWhenCreatingUniquenessConstraint(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.indexFor(this.label).on("my_property_key").withName("name").create(), schema1 -> schema1.constraintFor(this.label).assertPropertyIsUnique("my_second_property_key").withName("name").create(), ConstraintViolationException.class);
        Class<IndexWithNameAlreadyExistsException> expectedCause = IndexWithNameAlreadyExistsException.class;
        String expectedMessage = "There already exists an index called 'name'.";
        this.assertExpectedException(exception, expectedCause, expectedMessage);
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfConstraintWithNameExistsWhenCreatingIndex(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("name").create(), schema1 -> schema1.indexFor(this.label).on("my_second_property_key").withName("name").create(), ConstraintViolationException.class);
        Class<ConstraintWithNameAlreadyExistsException> expectedCause = ConstraintWithNameAlreadyExistsException.class;
        String expectedMessage = "There already exists a constraint called 'name'.";
        this.assertExpectedException(exception, expectedCause, expectedMessage);
    }

    @ParameterizedTest
    @EnumSource(value=SchemaAcceptanceTestBase.SchemaTxStrategy.class)
    void shouldThrowIfConstraintWithNameExistsWhenCreatingUniquenessConstraint(SchemaAcceptanceTestBase.SchemaTxStrategy txStrategy) {
        ConstraintViolationException exception = txStrategy.execute(this.db, schema -> schema.constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("name").create(), schema1 -> schema1.constraintFor(this.label).assertPropertyIsUnique("my_second_property_key").withName("name").create(), ConstraintViolationException.class);
        Class<ConstraintWithNameAlreadyExistsException> expectedCause = ConstraintWithNameAlreadyExistsException.class;
        String expectedMessage = "There already exists a constraint called 'name'.";
        this.assertExpectedException(exception, expectedCause, expectedMessage);
    }

    @Test
    void shouldThrowConstraintViolationIfAskedToCreateCompoundConstraint() {
        try (Transaction tx = this.db.beginTx();){
            Schema schema = tx.schema();
            schema.constraintFor(this.label).assertPropertyIsUnique("my_property_key").assertPropertyIsUnique("other_property").create();
            tx.commit();
            org.junit.jupiter.api.Assertions.fail((String)"Should not be able to create constraint on multiple propertyKey keys");
        }
        catch (UnsupportedOperationException e) {
            Assertions.assertThat((Throwable)e).hasMessageContaining("can only create one unique constraint");
        }
    }

    @Test
    void droppingExistingIndexRuleShouldSucceed() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        this.dropIndex(index);
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertThat(this.getIndexes(tx, this.label)).isEmpty();
        }
    }

    @Test
    void droppingNonExistingIndexShouldGiveHelpfulExceptionInSameTransaction() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            index = tx.schema().getIndexByName(index.getName());
            index.drop();
            try {
                index.drop();
                org.junit.jupiter.api.Assertions.fail((String)"Should not be able to drop index twice");
            }
            catch (ConstraintViolationException e) {
                Assertions.assertThat((Throwable)e).hasMessageContaining("Unable to drop index: Index does not exist: Index( id=1, name='index_a0d2924', type='GENERAL BTREE', schema=(:MY_LABEL {my_property_key}), indexProvider='native-btree-1.0' )");
            }
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            Assertions.assertThat(this.getIndexes(tx, this.label)).doesNotContain((Object[])new IndexDefinition[]{index});
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void droppingNonExistingIndexShouldGiveHelpfulExceptionInSeparateTransactions() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        this.dropIndex(index);
        try {
            this.dropIndex(index);
            org.junit.jupiter.api.Assertions.fail((String)"Should not be able to drop index twice");
        }
        catch (Exception e) {
            Assertions.assertThat((Throwable)e).hasMessageContaining("No index found with the name 'index_a0d2924'.");
        }
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertThat(this.getIndexes(tx, this.label)).doesNotContain((Object[])new IndexDefinition[]{index});
        }
    }

    @Test
    void awaitingIndexComingOnlineWorks() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            tx.schema().awaitIndexOnline(index, 1L, TimeUnit.MINUTES);
            org.junit.jupiter.api.Assertions.assertEquals((Object)Schema.IndexState.ONLINE, (Object)tx.schema().getIndexState(index));
        }
    }

    @Test
    void awaitingIndexComingOnlineByNameWorks() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, "my_index", this.label, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            tx.schema().awaitIndexOnline("my_index", 1L, TimeUnit.MINUTES);
            org.junit.jupiter.api.Assertions.assertEquals((Object)Schema.IndexState.ONLINE, (Object)tx.schema().getIndexState(index));
        }
    }

    @Test
    void awaitingAllIndexesComingOnlineWorks() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        SchemaAcceptanceTest.createIndex(this.db, this.label, "other_property");
        SchemaAcceptanceTest.waitForIndex(this.db, index);
        try (Transaction tx = this.db.beginTx();){
            tx.schema().awaitIndexesOnline(1L, TimeUnit.MINUTES);
            org.junit.jupiter.api.Assertions.assertEquals((Object)Schema.IndexState.ONLINE, (Object)tx.schema().getIndexState(index));
        }
    }

    @Test
    void shouldPopulateIndex() {
        Node node = this.createNode(this.db, "my_property_key", "Neo", this.label);
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        SchemaAcceptanceTest.waitForIndex(this.db, index);
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.label, "my_property_key", "Neo", transaction)).containsOnly((Object[])new Node[]{node});
        }
    }

    @Test
    void recreatingDroppedIndexMustProduceNewDefinition() {
        Node node = this.createNode(this.db, "my_property_key", "Neo", this.label);
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        SchemaAcceptanceTest.waitForIndex(this.db, index);
        this.dropIndex(index);
        IndexDefinition newIndex = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        try {
            SchemaAcceptanceTest.waitForIndex(this.db, index);
        }
        catch (NotFoundException e) {
            Assertions.assertThat((Throwable)e).hasMessageContaining("No index was found");
        }
        SchemaAcceptanceTest.waitForIndex(this.db, newIndex);
        try (Transaction transaction = this.db.beginTx();){
            Assertions.assertThat(this.getIndexes(transaction, this.label)).contains((Object[])new IndexDefinition[]{index});
            Assertions.assertThat(this.findNodesByLabelAndProperty(this.label, "my_property_key", "Neo", transaction)).contains((Object[])new Node[]{node});
            transaction.commit();
        }
    }

    private List<Node> findNodesByLabelAndProperty(Label label, String propertyKey, String value, Transaction transaction) {
        return Iterators.asList((Iterator)transaction.findNodes(label, propertyKey, (Object)value));
    }

    @Test
    void shouldCreateUniquenessConstraint() {
        ConstraintDefinition constraint = this.createUniquenessConstraint(this.label, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            constraint = tx.schema().getConstraintByName(constraint.getName());
            org.junit.jupiter.api.Assertions.assertEquals((Object)ConstraintType.UNIQUENESS, (Object)constraint.getConstraintType());
            org.junit.jupiter.api.Assertions.assertEquals((Object)this.label.name(), (Object)constraint.getLabel().name());
            org.junit.jupiter.api.Assertions.assertEquals((Object)Iterators.asSet((Object[])new String[]{"my_property_key"}), (Object)Iterables.asSet((Iterable)constraint.getPropertyKeys()));
            org.junit.jupiter.api.Assertions.assertEquals((Object)"constraint_c8a3b28f", (Object)constraint.getName());
            tx.commit();
        }
    }

    @Test
    void shouldCreateNamedUniquenessConstraint() {
        ConstraintDefinition constraint = this.createUniquenessConstraint("MyConstraint", this.label, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            constraint = tx.schema().getConstraintByName(constraint.getName());
            org.junit.jupiter.api.Assertions.assertEquals((Object)ConstraintType.UNIQUENESS, (Object)constraint.getConstraintType());
            org.junit.jupiter.api.Assertions.assertEquals((Object)this.label.name(), (Object)constraint.getLabel().name());
            org.junit.jupiter.api.Assertions.assertEquals((Object)Iterators.asSet((Object[])new String[]{"my_property_key"}), (Object)Iterables.asSet((Iterable)constraint.getPropertyKeys()));
            org.junit.jupiter.api.Assertions.assertEquals((Object)"MyConstraint", (Object)constraint.getName());
            tx.commit();
        }
    }

    @Test
    void shouldGetConstraintByName() {
        ConstraintDefinition expectedConstraint = this.createUniquenessConstraint("MyConstraint", this.label, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            ConstraintDefinition actualConstraint = tx.schema().getConstraintByName("MyConstraint");
            Assertions.assertThat((Object)actualConstraint).isEqualTo((Object)expectedConstraint);
            tx.commit();
        }
    }

    @Test
    void shouldListAddedConstraintsByLabel() {
        ConstraintDefinition constraint1 = this.createUniquenessConstraint(this.label, "my_property_key");
        this.createUniquenessConstraint(this.otherLabel, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertThat(this.getConstraints(tx, this.label)).containsOnly((Object[])new ConstraintDefinition[]{constraint1});
        }
    }

    private Iterable<ConstraintDefinition> getConstraints(Transaction tx, Label label) {
        return tx.schema().getConstraints(label);
    }

    @Test
    void shouldListAddedConstraints() {
        ConstraintDefinition constraint1 = this.createUniquenessConstraint(this.label, "my_property_key");
        ConstraintDefinition constraint2 = this.createUniquenessConstraint(this.otherLabel, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertThat((Iterable)tx.schema().getConstraints()).containsOnly((Object[])new ConstraintDefinition[]{constraint1, constraint2});
        }
    }

    @Test
    void shouldDropUniquenessConstraint() {
        ConstraintDefinition constraint = this.createUniquenessConstraint(this.label, "my_property_key");
        this.dropConstraint(this.db, constraint);
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertThat(this.getConstraints(tx, this.label)).isEmpty();
        }
    }

    @Test
    void addingConstraintWhenIndexAlreadyExistsGivesNiceError() {
        SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        try {
            this.createUniquenessConstraint(this.label, "my_property_key");
            org.junit.jupiter.api.Assertions.fail((String)"Expected exception to be thrown");
        }
        catch (ConstraintViolationException e) {
            org.junit.jupiter.api.Assertions.assertEquals((Object)"There already exists an index (:MY_LABEL {my_property_key}). A constraint cannot be created until the index has been dropped.", (Object)e.getMessage());
        }
    }

    @Test
    void addingUniquenessConstraintWhenDuplicateDataExistsGivesNiceError() {
        try (Transaction transaction = this.db.beginTx();){
            transaction.createNode(new Label[]{this.label}).setProperty("my_property_key", (Object)"value1");
            transaction.createNode(new Label[]{this.label}).setProperty("my_property_key", (Object)"value1");
            transaction.commit();
        }
        try {
            this.createUniquenessConstraint(this.label, "my_property_key");
            org.junit.jupiter.api.Assertions.fail((String)"Expected exception to be thrown");
        }
        catch (ConstraintViolationException e) {
            Assertions.assertThat((Throwable)e).hasMessageContaining("Unable to create Constraint( name='constraint_c8a3b28f', type='UNIQUENESS', schema=(:MY_LABEL {my_property_key}) )");
        }
    }

    @Test
    void addedUncommittedIndexesShouldBeVisibleWithinTheTransaction() {
        IndexDefinition indexA = SchemaAcceptanceTest.createIndex(this.db, this.label, "a");
        this.createUniquenessConstraint(this.label, "b");
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertThat((long)Iterables.count((Iterable)tx.schema().getIndexes(this.label))).isEqualTo(2L);
            IndexDefinition indexC = tx.schema().indexFor(this.label).on("c").create();
            Assertions.assertThat((long)Iterables.count((Iterable)tx.schema().getIndexes(this.label))).isEqualTo(3L);
            Assertions.assertThat((Comparable)tx.schema().getIndexState(indexA)).isEqualTo((Object)Schema.IndexState.ONLINE);
            Assertions.assertThat((Comparable)tx.schema().getIndexState(indexC)).isEqualTo((Object)Schema.IndexState.POPULATING);
            Assertions.assertThat((float)tx.schema().getIndexPopulationProgress(indexA).getCompletedPercentage()).isGreaterThan(0.0f);
            Assertions.assertThat((float)tx.schema().getIndexPopulationProgress(indexC).getCompletedPercentage()).isGreaterThanOrEqualTo(0.0f);
        }
    }

    @Test
    void indexNamesMustBeUnique() {
        SchemaAcceptanceTest.createIndex(this.db, "MyIndex", this.label, "my_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> SchemaAcceptanceTest.createIndex(this.db, "MyIndex", this.label, "my_second_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining("MyIndex");
    }

    @Test
    void indexNamesMustBeUniqueEvenWhenGenerated() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> SchemaAcceptanceTest.createIndex(this.db, index.getName(), this.otherLabel, "my_second_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining(index.getName());
    }

    @Test
    void indexNamesMustBeUniqueEvenWhenGenerated2() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, "index_a0d2924", this.otherLabel, "my_second_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining(index.getName());
    }

    @Test
    void constraintNamesMustBeUnique() {
        this.createUniquenessConstraint("MyConstraint", this.label, "my_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> this.createUniquenessConstraint("MyConstraint", this.label, "my_second_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining("MyConstraint");
    }

    @Test
    void cannotCreateConstraintWithSameNameAsExistingIndex() {
        SchemaAcceptanceTest.createIndex(this.db, "MySchema", this.label, "my_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> this.createUniquenessConstraint("MySchema", this.label, "my_second_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining("MySchema");
    }

    @Test
    void cannotCreateIndexWithSameNameAsExistingIndexWithGeneratedName() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> SchemaAcceptanceTest.createIndex(this.db, index.getName(), this.otherLabel, "my_second_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining(index.getName());
    }

    @Test
    void cannotCreateConstraintWithSameNameAsExistingIndexWithGeneratedName() {
        IndexDefinition index = SchemaAcceptanceTest.createIndex(this.db, this.label, "my_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> this.createUniquenessConstraint(index.getName(), this.otherLabel, "my_second_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining(index.getName());
    }

    @Test
    void cannotCreateIndexWithSameNameAsExistingConstraint() {
        this.createUniquenessConstraint("MySchema", this.label, "my_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> SchemaAcceptanceTest.createIndex(this.db, "MySchema", this.label, "my_second_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining("MySchema");
    }

    @Test
    void cannotCreateIndexWithSameNameAsExistingConstraintWithGeneratedName() {
        ConstraintDefinition constraint = this.createUniquenessConstraint(this.label, "my_property_key");
        ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> SchemaAcceptanceTest.createIndex(this.db, constraint.getName(), this.label, "my_second_property_key"));
        Assertions.assertThat((Throwable)exception).hasMessageContaining(constraint.getName());
    }

    @Test
    void uniquenessConstraintIndexesMustBeNamedAfterTheirConstraints() {
        this.createUniquenessConstraint("MySchema", this.label, "my_property_key");
        try (Transaction tx = this.db.beginTx();){
            IndexDefinition index = tx.schema().getIndexByName("MySchema");
            org.junit.jupiter.api.Assertions.assertTrue((boolean)index.isConstraintIndex());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)index.isNodeIndex());
            org.junit.jupiter.api.Assertions.assertEquals((Object)"MySchema", (Object)index.getName());
            tx.commit();
        }
    }

    @Test
    void indexNamesInTransactionStateMustBeUnique() {
        try (Transaction tx = this.db.beginTx();){
            String indexName = "MyIndex";
            tx.schema().indexFor(this.label).on("my_property_key").withName("MyIndex").create();
            IndexCreator creator = tx.schema().indexFor(this.otherLabel).on("my_second_property_key").withName("MyIndex");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((IndexCreator)creator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining(SchemaAcceptanceTest.alreadyExistsIndexMessage("MyIndex"));
            tx.commit();
        }
    }

    @Test
    void indexNamesInTransactionStateMustBeUniqueEvenWhenGenerated() {
        try (Transaction tx = this.db.beginTx();){
            IndexDefinition index = tx.schema().indexFor(this.label).on("my_property_key").create();
            IndexCreator creator = tx.schema().indexFor(this.otherLabel).on("my_second_property_key").withName(index.getName());
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((IndexCreator)creator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining(SchemaAcceptanceTest.alreadyExistsIndexMessage(index.getName()));
            tx.commit();
        }
    }

    @Test
    void indexNamesInTransactionStateMustBeUniqueEvenWhenGenerated2() {
        try (Transaction tx = this.db.beginTx();){
            IndexDefinition index = tx.schema().indexFor(this.otherLabel).on("my_second_property_key").withName("index_a0d2924").create();
            IndexCreator creator = tx.schema().indexFor(this.label).on("my_property_key");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((IndexCreator)creator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining(SchemaAcceptanceTest.alreadyExistsIndexMessage(index.getName()));
            tx.commit();
        }
    }

    @Test
    void constraintNamesInTransactionStateMustBeUnique() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("MyConstraint").create();
            ConstraintCreator creator = tx.schema().constraintFor(this.otherLabel).assertPropertyIsUnique("my_second_property_key").withName("MyConstraint");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((ConstraintCreator)creator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining(SchemaAcceptanceTest.thereAlreadyExistsConstraintMessage("MyConstraint"));
            tx.commit();
        }
    }

    @Test
    void constraintNamesInTransactionStateMustBeUniqueEvenWhenGenerated() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintDefinition constraint = tx.schema().constraintFor(this.label).assertPropertyIsUnique("my_property_key").create();
            ConstraintCreator creator = tx.schema().constraintFor(this.otherLabel).assertPropertyIsUnique("my_second_property_key").withName(constraint.getName());
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((ConstraintCreator)creator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining(SchemaAcceptanceTest.thereAlreadyExistsConstraintMessage(constraint.getName()));
            tx.commit();
        }
    }

    @Test
    void constraintNamesInTransactionStateMustBeUniqueEvenWhenGenerated2() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintDefinition constraint = tx.schema().constraintFor(this.otherLabel).assertPropertyIsUnique("my_second_property_key").withName("constraint_c8a3b28f").create();
            ConstraintCreator creator = tx.schema().constraintFor(this.label).assertPropertyIsUnique("my_property_key");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((ConstraintCreator)creator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining(SchemaAcceptanceTest.thereAlreadyExistsConstraintMessage(constraint.getName()));
            tx.commit();
        }
    }

    @Test
    void constraintAndIndexNamesInTransactionStateMustBeUnique() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().constraintFor(this.label).assertPropertyIsUnique("my_property_key").withName("MySchema").create();
            IndexCreator creator = tx.schema().indexFor(this.otherLabel).on("my_second_property_key").withName("MySchema");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((IndexCreator)creator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining("MySchema");
            tx.commit();
        }
    }

    @Test
    void indexAndConstraintNamesInTransactionStateMustBeUnique() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(this.label).on("my_property_key").withName("MySchema").create();
            ConstraintCreator creator = tx.schema().constraintFor(this.otherLabel).assertPropertyIsUnique("my_second_property_key").withName("MySchema");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((ConstraintCreator)creator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining("MySchema");
            tx.commit();
        }
    }

    @Test
    void nodeKeyConstraintsMustNotAvailableInCommunityEdition() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator constraintCreator = tx.schema().constraintFor(this.label).assertPropertyIsNodeKey("my_property_key");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((ConstraintCreator)constraintCreator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining("Enterprise Edition");
            tx.commit();
        }
    }

    @Test
    void propertyExistenceConstraintsMustNotBeAvailableInCommunityEdition() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator constraintCreator = tx.schema().constraintFor(this.label).assertPropertyExists("my_property_key");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((ConstraintCreator)constraintCreator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining("Enterprise Edition");
            tx.commit();
        }
    }

    @Test
    void propertyExistenceConstraintsOnRelationshipMustNotBeAvailableInCommunityEdition() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator constraintCreator = tx.schema().constraintFor(this.relType).assertPropertyExists("my_property_key");
            ConstraintViolationException exception = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((ConstraintCreator)constraintCreator).create());
            Assertions.assertThat((Throwable)exception).hasMessageContaining("Enterprise Edition");
            tx.commit();
        }
    }

    @Test
    void indexNamesCannotContainBackTicks() {
        try (Transaction tx = this.db.beginTx();){
            IndexCreator creator = tx.schema().indexFor(this.label).withName("a`b").on("my_property_key");
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((IndexCreator)creator).create());
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            Assertions.assertThat((long)Iterables.count((Iterable)tx.schema().getIndexes())).isZero();
            Assertions.assertThat((long)Iterables.count((Iterable)tx.schema().getConstraints())).isZero();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void constraintNamesCannotContainBackTicks() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator creator = tx.schema().constraintFor(this.label).withName("a`b").assertPropertyIsUnique("my_property_key");
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((ConstraintCreator)creator).create());
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            Assertions.assertThat((long)Iterables.count((Iterable)tx.schema().getIndexes())).isZero();
            Assertions.assertThat((long)Iterables.count((Iterable)tx.schema().getConstraints())).isZero();
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void mustCreateFullTextIndexBySettingIndexType() {
        try (Transaction tx = this.db.beginTx();){
            IndexCreator creator = tx.schema().indexFor(this.label).on("my_property_key").withIndexType(IndexType.FULLTEXT);
            IndexDefinition definition = creator.create();
            org.junit.jupiter.api.Assertions.assertEquals((Object)IndexType.FULLTEXT, (Object)definition.getIndexType());
            IndexProviderDescriptor provider = ((IndexDefinitionImpl)definition).getIndexReference().getIndexProvider();
            org.junit.jupiter.api.Assertions.assertEquals((Object)provider, (Object)FulltextIndexProviderFactory.DESCRIPTOR);
            tx.commit();
        }
    }

    @Test
    void mustBeAbleToGetIndexConfig() {
        Map config;
        IndexDefinition index;
        try (Transaction tx = this.db.beginTx();){
            index = tx.schema().indexFor(this.label).on("my_property_key").withName("my_index").create();
            config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertNotNull((Object)config);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)config.containsKey(IndexSettingImpl.SPATIAL_CARTESIAN_MIN));
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            index = tx.schema().getIndexByName("my_index");
            config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertNotNull((Object)config);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)config.containsKey(IndexSettingImpl.SPATIAL_CARTESIAN_MIN));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void mustBeAbleToGetFullTextIndexConfig() {
        Map config;
        IndexDefinition index;
        try (Transaction tx = this.db.beginTx();){
            index = tx.schema().indexFor(this.label).withName("my_index").on("my_property_key").withIndexType(IndexType.FULLTEXT).create();
            config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertNotNull((Object)config);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)config.containsKey(IndexSettingImpl.FULLTEXT_ANALYZER));
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            index = tx.schema().getIndexByName("my_index");
            config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertNotNull((Object)config);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)config.containsKey(IndexSettingImpl.FULLTEXT_ANALYZER));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void mustBeAbleToSetFullTextIndexConfig() {
        Map config;
        IndexDefinition index;
        try (Transaction tx = this.db.beginTx();){
            index = tx.schema().indexFor(this.label).withName("my_index").on("my_property_key").withIndexType(IndexType.FULLTEXT).withIndexConfiguration(Map.of(IndexSettingImpl.FULLTEXT_ANALYZER, "swedish", IndexSettingImpl.FULLTEXT_EVENTUALLY_CONSISTENT, true)).create();
            config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertEquals((Object)"swedish", config.get(IndexSettingImpl.FULLTEXT_ANALYZER));
            org.junit.jupiter.api.Assertions.assertEquals((Object)true, config.get(IndexSettingImpl.FULLTEXT_EVENTUALLY_CONSISTENT));
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            index = tx.schema().getIndexByName("my_index");
            config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertEquals((Object)"swedish", config.get(IndexSettingImpl.FULLTEXT_ANALYZER));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void mustBeAbleToSetBtreeIndexConfig() {
        Map config;
        IndexDefinition index;
        try (Transaction tx = this.db.beginTx();){
            index = tx.schema().indexFor(this.label).withName("my_index").on("my_property_key").withIndexConfiguration(Map.of(IndexSettingImpl.SPATIAL_CARTESIAN_MAX, new double[]{200.0, 200.0}, IndexSettingImpl.SPATIAL_WGS84_MIN, new double[]{-90.0, -90.0})).create();
            config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertArrayEquals((double[])new double[]{200.0, 200.0}, (double[])((double[])config.get(IndexSettingImpl.SPATIAL_CARTESIAN_MAX)));
            org.junit.jupiter.api.Assertions.assertArrayEquals((double[])new double[]{-90.0, -90.0}, (double[])((double[])config.get(IndexSettingImpl.SPATIAL_WGS84_MIN)));
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            index = tx.schema().getIndexByName("my_index");
            config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertArrayEquals((double[])new double[]{200.0, 200.0}, (double[])((double[])config.get(IndexSettingImpl.SPATIAL_CARTESIAN_MAX)));
            org.junit.jupiter.api.Assertions.assertArrayEquals((double[])new double[]{-90.0, -90.0}, (double[])((double[])config.get(IndexSettingImpl.SPATIAL_WGS84_MIN)));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void indexConfigurationExample() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(Label.label((String)"Email")).on("from").on("to").on("cc").on("bcc").withName("email-addresses").withIndexType(IndexType.FULLTEXT).withIndexConfiguration(Map.of(IndexSetting.fulltext_Analyzer(), "email")).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            IndexDefinition index = tx.schema().getIndexByName("email-addresses");
            Assertions.assertThat((Iterable)index.getPropertyKeys()).contains((Object[])new String[]{"from", "to", "cc", "bcc"});
            Assertions.assertThat(index.getIndexConfiguration().get(IndexSetting.fulltext_Analyzer())).isEqualTo((Object)"email");
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void indexSettingValuesMustHaveCorrectType() {
        try (Transaction tx = this.db.beginTx();){
            IndexCreator creator = tx.schema().indexFor(this.label).withName("my_index").on("my_property_key");
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexType(IndexType.FULLTEXT).withIndexConfiguration(Map.of(IndexSettingImpl.FULLTEXT_ANALYZER, 1)).create());
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexType(IndexType.FULLTEXT).withIndexConfiguration(Map.of(IndexSettingImpl.FULLTEXT_ANALYZER, true)).create());
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexType(IndexType.FULLTEXT).withIndexConfiguration(Map.of(IndexSettingImpl.FULLTEXT_EVENTUALLY_CONSISTENT, "true")).create());
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexType(IndexType.FULLTEXT).withIndexConfiguration(Map.of(IndexSettingImpl.FULLTEXT_EVENTUALLY_CONSISTENT, 1)).create());
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexConfiguration(Map.of(IndexSettingImpl.SPATIAL_CARTESIAN_MAX, "1")).create());
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexConfiguration(Map.of(IndexSettingImpl.SPATIAL_CARTESIAN_MAX, 1)).create());
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexConfiguration(Map.of(IndexSettingImpl.SPATIAL_CARTESIAN_MAX, 1.0)).create());
            tx.commit();
        }
    }

    @Test
    void indexSettingsWithNonsensicalValuesMustBeRejected() {
        try (Transaction tx = this.db.beginTx();){
            IndexCreator creator = tx.schema().indexFor(this.label).withName("my_index").on("my_property_key");
            Exception e = (Exception)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexType(IndexType.FULLTEXT).withIndexConfiguration(Map.of(IndexSettingImpl.FULLTEXT_ANALYZER, "analyzer that does not exist")).create());
            Assertions.assertThat((Throwable)e).hasMessageContaining("'analyzer that does not exist'");
            e = (Exception)org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> creator.withIndexConfiguration(Map.of(IndexSettingImpl.SPATIAL_CARTESIAN_MAX, new double[]{100.0, 10.0, 1.0})).create());
            Assertions.assertThat((Throwable)e).hasMessageContaining("Invalid spatial index settings");
            tx.commit();
        }
    }

    @Test
    void creatingFullTextIndexOnMultipleLabelsMustBePossible() {
        IndexDefinition index;
        try (Transaction tx = this.db.beginTx();){
            index = tx.schema().indexFor(new Label[]{this.label, this.otherLabel}).on("my_property_key").withIndexType(IndexType.FULLTEXT).withName("index").create();
            Assertions.assertThat((Iterable)index.getLabels()).contains((Object[])new Label[]{this.label, this.otherLabel});
            org.junit.jupiter.api.Assertions.assertTrue((boolean)index.isMultiTokenIndex());
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            index = tx.schema().getIndexByName("index");
            ArrayList labelNames = new ArrayList();
            index.getLabels().forEach(label -> labelNames.add(label.name()));
            Assertions.assertThat(labelNames).contains((Object[])new String[]{this.label.name(), this.otherLabel.name()});
            org.junit.jupiter.api.Assertions.assertTrue((boolean)index.isMultiTokenIndex());
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void mustThrowWhenCreatingBtreeIndexWithZeroLabels() {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(new Label[0]).on("my_property_key").create());
            tx.commit();
        }
    }

    @Test
    void mustThrowWhenCreatingBtreeIndexWithMoreThanOneLabel() {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(new Label[]{this.label, this.otherLabel}).on("my_property_key").create());
            tx.commit();
        }
    }

    @Test
    void mustThrowWhenCreatingFullTextIndexWithZeroLabels() {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(new Label[0]).on("my_property_key").withIndexType(IndexType.FULLTEXT).create());
            tx.commit();
        }
    }

    @Test
    void creatingFullTextIndexOnRelationshipTypeMustBePossible() {
        IndexDefinition index;
        try (Transaction tx = this.db.beginTx();){
            index = tx.schema().indexFor(this.relType).on("my_property_key").withIndexType(IndexType.FULLTEXT).withName("index").create();
            org.junit.jupiter.api.Assertions.assertTrue((boolean)index.isRelationshipIndex());
            Assertions.assertThat((Iterable)index.getRelationshipTypes()).contains((Object[])new RelationshipType[]{this.relType});
            Assertions.assertThat((Comparable)index.getIndexType()).isEqualTo((Object)IndexType.FULLTEXT);
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            index = tx.schema().getIndexByName("index");
            org.junit.jupiter.api.Assertions.assertTrue((boolean)index.isRelationshipIndex());
            Assertions.assertThat((Iterable)index.getRelationshipTypes()).contains((Object[])new RelationshipType[]{this.relType});
            Assertions.assertThat((Comparable)index.getIndexType()).isEqualTo((Object)IndexType.FULLTEXT);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void creatingMultiTokenFullTextIndexOnRelationshipTypesMustBePossible() {
        IndexDefinition index;
        try (Transaction tx = this.db.beginTx();){
            index = tx.schema().indexFor(new RelationshipType[]{this.relType, this.otherRelType}).on("my_property_key").withIndexType(IndexType.FULLTEXT).withName("index").create();
            org.junit.jupiter.api.Assertions.assertTrue((boolean)index.isRelationshipIndex());
            Assertions.assertThat((Iterable)index.getRelationshipTypes()).contains((Object[])new RelationshipType[]{this.relType, this.otherRelType});
            Assertions.assertThat((Comparable)index.getIndexType()).isEqualTo((Object)IndexType.FULLTEXT);
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            index = tx.schema().getIndexByName("index");
            org.junit.jupiter.api.Assertions.assertTrue((boolean)index.isRelationshipIndex());
            Assertions.assertThat((Iterable)index.getRelationshipTypes()).contains((Object[])new RelationshipType[]{this.relType, this.otherRelType});
            Assertions.assertThat((Comparable)index.getIndexType()).isEqualTo((Object)IndexType.FULLTEXT);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void mustThrowWhenCreatingFullTextIndexOnZeroRelationshipTypes() {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(new RelationshipType[0]).on("my_property_key").withIndexType(IndexType.FULLTEXT).create());
            tx.commit();
        }
    }

    @Test
    void mustThrowWhenCreatingBtreeIndexOnRelationshipType() {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(this.relType).on("my_property_key").create());
            tx.commit();
        }
    }

    @Test
    void mustThrowWhenCreatingBtreeIndexOnZeroRelationshipTypes() {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(new RelationshipType[0]).on("my_property_key").create());
            tx.commit();
        }
    }

    @Test
    void mustThrowWhenCreatingBtreeIndexOnMultipleRelationshipTypes() {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> tx.schema().indexFor(new RelationshipType[]{this.relType, this.otherRelType}).on("my_property_key").create());
            tx.commit();
        }
    }

    @Test
    void uniquenessConstraintIndexesAreBtreeIndexTypeByDefault() {
        String name;
        try (Transaction tx = this.db.beginTx();){
            ConstraintDefinition constraint = tx.schema().constraintFor(this.label).assertPropertyIsUnique("my_property_key").create();
            name = constraint.getName();
            IndexDefinition index = tx.schema().getIndexByName(name);
            Assertions.assertThat((Comparable)index.getIndexType()).isEqualTo((Object)IndexType.BTREE);
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            IndexDefinition index = tx.schema().getIndexByName(name);
            Assertions.assertThat((Comparable)index.getIndexType()).isEqualTo((Object)IndexType.BTREE);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void mustBeAbleToCreateUniquenessConstraintWithBtreeIndexType() {
        String name;
        try (Transaction tx = this.db.beginTx();){
            ConstraintDefinition constraint = tx.schema().constraintFor(this.label).assertPropertyIsUnique("my_property_key").withIndexType(IndexType.BTREE).create();
            name = constraint.getName();
            IndexDefinition index = tx.schema().getIndexByName(name);
            Assertions.assertThat((Comparable)index.getIndexType()).isEqualTo((Object)IndexType.BTREE);
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            IndexDefinition index = tx.schema().getIndexByName(name);
            Assertions.assertThat((Comparable)index.getIndexType()).isEqualTo((Object)IndexType.BTREE);
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void creatingUniquenessConstraintWithFullTextIndexTypeMustThrow() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator creator = tx.schema().constraintFor(this.label).assertPropertyIsUnique("my_property_key").withIndexType(IndexType.FULLTEXT);
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((ConstraintCreator)creator).create());
        }
    }

    @Test
    void creatingNodePropertyExistenceConstraintMustThrowWhenGivenIndexType() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator creator = tx.schema().constraintFor(this.label).assertPropertyExists("my_property_key").withIndexType(IndexType.BTREE);
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((ConstraintCreator)creator).create());
            tx.commit();
        }
    }

    @Test
    void creatingRelationshipPropertyExistenceConstraintsMustThrowWhenGivenIndexType() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator creator = tx.schema().constraintFor(this.relType).assertPropertyExists("my_property_key").withIndexType(IndexType.BTREE);
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((ConstraintCreator)creator).create());
            tx.commit();
        }
    }

    @Test
    void mustBeAbleToSpecifyIndexConfigurationForUniquenessConstraint() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintDefinition constraint = tx.schema().constraintFor(this.label).withName("my constraint").assertPropertyIsUnique("my_property_key").withIndexConfiguration(Map.of(IndexSettingImpl.SPATIAL_CARTESIAN_MAX, new double[]{200.0, 200.0}, IndexSettingImpl.SPATIAL_WGS84_MIN, new double[]{-90.0, -90.0})).create();
            IndexDefinition index = tx.schema().getIndexByName(constraint.getName());
            Map config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertArrayEquals((double[])new double[]{200.0, 200.0}, (double[])((double[])config.get(IndexSettingImpl.SPATIAL_CARTESIAN_MAX)));
            org.junit.jupiter.api.Assertions.assertArrayEquals((double[])new double[]{-90.0, -90.0}, (double[])((double[])config.get(IndexSettingImpl.SPATIAL_WGS84_MIN)));
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            IndexDefinition index = tx.schema().getIndexByName("my constraint");
            Map config = index.getIndexConfiguration();
            org.junit.jupiter.api.Assertions.assertArrayEquals((double[])new double[]{200.0, 200.0}, (double[])((double[])config.get(IndexSettingImpl.SPATIAL_CARTESIAN_MAX)));
            org.junit.jupiter.api.Assertions.assertArrayEquals((double[])new double[]{-90.0, -90.0}, (double[])((double[])config.get(IndexSettingImpl.SPATIAL_WGS84_MIN)));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void creatingNodePropertyExistenceConstraintMustThrowWhenGivenIndexConfiguration() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator creator = tx.schema().constraintFor(this.label).withIndexConfiguration(Map.of()).assertPropertyExists("my_property_key");
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((ConstraintCreator)creator).create());
            tx.commit();
        }
    }

    @Test
    void creatingRelationshipPropertyExistenceConstraintMustThrowWhenGivenIndexConfiguration() {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator creator = tx.schema().constraintFor(this.relType).withIndexConfiguration(Map.of()).assertPropertyExists("my_property_key");
            org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> ((ConstraintCreator)creator).create());
            tx.commit();
        }
    }

    private static String alreadyExistsIndexMessage(String indexName) {
        return "There already exists an index called '" + indexName + "'";
    }

    private static String thereAlreadyExistsConstraintMessage(String constraintName) {
        return "There already exists a constraint called '" + constraintName + "'.";
    }

    private void dropConstraint(GraphDatabaseService db, ConstraintDefinition constraint) {
        try (Transaction tx = db.beginTx();){
            tx.schema().getConstraintByName(constraint.getName()).drop();
            tx.commit();
        }
    }

    private ConstraintDefinition createUniquenessConstraint(Label label, String prop) {
        return this.createUniquenessConstraint(null, label, prop);
    }

    private ConstraintDefinition createUniquenessConstraint(String name, Label label, String prop) {
        try (Transaction tx = this.db.beginTx();){
            ConstraintCreator creator = tx.schema().constraintFor(label);
            creator = creator.assertPropertyIsUnique(prop).withName(name);
            ConstraintDefinition constraint = creator.create();
            tx.commit();
            ConstraintDefinition constraintDefinition = constraint;
            return constraintDefinition;
        }
    }

    private void dropIndex(IndexDefinition index) {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().getIndexByName(index.getName()).drop();
            tx.commit();
        }
    }

    private Node createNode(GraphDatabaseService db, String key, Object value, Label label) {
        try (Transaction tx = db.beginTx();){
            Node node = tx.createNode(new Label[]{label});
            node.setProperty(key, value);
            tx.commit();
            Node node2 = node;
            return node2;
        }
    }

    public static IndexDefinition createIndex(GraphDatabaseService db, Label label, String ... properties) {
        return SchemaAcceptanceTest.createIndex(db, null, label, properties);
    }

    public static IndexDefinition createIndex(GraphDatabaseService db, String name, Label label, String ... properties) {
        IndexDefinition indexDef = SchemaAcceptanceTest.createIndexNoWait(db, name, label, properties);
        SchemaAcceptanceTest.waitForIndex(db, indexDef);
        return indexDef;
    }

    public static IndexDefinition createIndexNoWait(GraphDatabaseService db, String name, Label label, String ... properties) {
        IndexDefinition indexDef;
        try (Transaction tx = db.beginTx();){
            IndexCreator indexCreator = tx.schema().indexFor(label);
            for (String property : properties) {
                indexCreator = indexCreator.on(property);
            }
            if (name != null) {
                indexCreator = indexCreator.withName(name);
            }
            indexDef = indexCreator.create();
            tx.commit();
        }
        return indexDef;
    }

    public static void waitForIndex(GraphDatabaseService beansAPI, IndexDefinition indexDef) {
        try (Transaction tx = beansAPI.beginTx();){
            tx.schema().awaitIndexOnline(indexDef, 10L, TimeUnit.MINUTES);
        }
    }

    private Iterable<IndexDefinition> getIndexes(Transaction tx, Label label) {
        return tx.schema().getIndexes(label);
    }

    @Nested
    @ActorsExtension
    class SchemaConcurrency {
        @Inject
        Actor first;
        @Inject
        Actor second;
        BinaryLatch startLatch;

        SchemaConcurrency() {
        }

        @BeforeEach
        void setUp() {
            this.startLatch = new BinaryLatch();
        }

        @RepeatedTest(value=20)
        void cannotCreateIndexesWithTheSameNameInConcurrentTransactions() throws Exception {
            String indexName = "MyIndex";
            Future firstFuture = this.first.submit(this.schemaTransaction((ThrowingFunction<Transaction, Object, Exception>)((ThrowingFunction)tx -> tx.schema().indexFor(SchemaAcceptanceTest.this.label).on("my_property_key").withName(indexName))));
            Future secondFuture = this.second.submit(this.schemaTransaction((ThrowingFunction<Transaction, Object, Exception>)((ThrowingFunction)tx -> tx.schema().indexFor(SchemaAcceptanceTest.this.otherLabel).on("my_second_property_key").withName(indexName))));
            this.raceTransactions(firstFuture, secondFuture);
            this.assertOneSuccessAndOneFailure(firstFuture, secondFuture);
        }

        @RepeatedTest(value=20)
        void cannotCreateConstraintsWithTheSameNameInConcurrentTransactions() throws Exception {
            String constraintName = "MyConstraint";
            Future firstFuture = this.first.submit(this.schemaTransaction((ThrowingFunction<Transaction, Object, Exception>)((ThrowingFunction)tx -> tx.schema().constraintFor(SchemaAcceptanceTest.this.label).assertPropertyIsUnique("my_property_key").withName(constraintName))));
            Future secondFuture = this.second.submit(this.schemaTransaction((ThrowingFunction<Transaction, Object, Exception>)((ThrowingFunction)tx -> tx.schema().constraintFor(SchemaAcceptanceTest.this.otherLabel).assertPropertyIsUnique("my_second_property_key").withName(constraintName))));
            this.raceTransactions(firstFuture, secondFuture);
            this.assertOneSuccessAndOneFailure(firstFuture, secondFuture);
        }

        @RepeatedTest(value=20)
        void cannotCreateIndexesAndConstraintsWithTheSameNameInConcurrentTransactions() throws Exception {
            String schemaName = "MySchema";
            Future firstFuture = this.first.submit(this.schemaTransaction((ThrowingFunction<Transaction, Object, Exception>)((ThrowingFunction)tx -> tx.schema().constraintFor(SchemaAcceptanceTest.this.label).assertPropertyIsUnique("my_property_key").withName(schemaName))));
            Future secondFuture = this.second.submit(this.schemaTransaction((ThrowingFunction<Transaction, Object, Exception>)((ThrowingFunction)tx -> tx.schema().indexFor(SchemaAcceptanceTest.this.otherLabel).on("my_second_property_key").withName(schemaName))));
            this.raceTransactions(firstFuture, secondFuture);
            this.assertOneSuccessAndOneFailure(firstFuture, secondFuture);
        }

        @Test
        void droppingConstraintMustLockNameForIndexCreate() throws Exception {
            String schemaName = "MySchema";
            SchemaAcceptanceTest.this.createUniquenessConstraint(schemaName, SchemaAcceptanceTest.this.label, "my_property_key");
            try (Transaction tx = SchemaAcceptanceTest.this.db.beginTx();){
                tx.schema().awaitIndexesOnline(1L, TimeUnit.MINUTES);
                tx.commit();
            }
            BinaryLatch afterFirstDropsConstraint = new BinaryLatch();
            BinaryLatch pauseFirst = new BinaryLatch();
            BinaryLatch beforeSecondCreatesIndex = new BinaryLatch();
            Future firstFuture = this.first.submit(() -> {
                try (Transaction tx = SchemaAcceptanceTest.this.db.beginTx();){
                    tx.schema().getConstraintByName(schemaName).drop();
                    afterFirstDropsConstraint.release();
                    pauseFirst.await();
                    tx.commit();
                }
            });
            Future secondFuture = this.second.submit(() -> {
                afterFirstDropsConstraint.await();
                try (Transaction tx = SchemaAcceptanceTest.this.db.beginTx();){
                    beforeSecondCreatesIndex.release();
                    IndexCreator indexCreator = tx.schema().indexFor(SchemaAcceptanceTest.this.otherLabel).on("my_second_property_key").withName(schemaName);
                    indexCreator.create();
                    tx.commit();
                }
            });
            this.first.untilWaitingIn((Executable)BinaryLatch.class.getMethod("await", new Class[0]));
            beforeSecondCreatesIndex.await();
            this.second.untilWaitingIn((Executable)Object.class.getMethod("wait", Long.TYPE));
            this.second.untilWaiting();
            pauseFirst.release();
            firstFuture.get();
            secondFuture.get();
            try (Transaction tx = SchemaAcceptanceTest.this.db.beginTx();){
                org.junit.jupiter.api.Assertions.assertFalse((boolean)tx.schema().getConstraints().iterator().hasNext());
                Iterator indexes = tx.schema().getIndexes().iterator();
                org.junit.jupiter.api.Assertions.assertTrue((boolean)indexes.hasNext());
                org.junit.jupiter.api.Assertions.assertEquals((Object)((IndexDefinition)indexes.next()).getName(), (Object)schemaName);
                org.junit.jupiter.api.Assertions.assertFalse((boolean)indexes.hasNext());
                tx.commit();
            }
        }

        @RepeatedTest(value=10)
        void awaitIndexesMustNotThrowOnConcurrentlyDroppedIndexes() throws Exception {
            AtomicBoolean stop = new AtomicBoolean();
            ConcurrentLinkedQueue<IndexDefinition> indexes = new ConcurrentLinkedQueue<IndexDefinition>();
            try (Transaction tx = SchemaAcceptanceTest.this.db.beginTx();){
                for (int i = 0; i < 50; ++i) {
                    indexes.add(tx.schema().indexFor(Label.label((String)("Label_" + i))).on("propl_" + i).create());
                }
                tx.commit();
            }
            Future firstFuture = this.first.submit(() -> {
                this.startLatch.await();
                while (!stop.get()) {
                    Transaction tx = SchemaAcceptanceTest.this.db.beginTx();
                    try {
                        try {
                            tx.schema().awaitIndexesOnline(20L, TimeUnit.MINUTES);
                        }
                        catch (Exception e) {
                            stop.set(true);
                            indexes.clear();
                            throw e;
                        }
                        tx.commit();
                    }
                    finally {
                        if (tx == null) continue;
                        tx.close();
                    }
                }
            });
            Future secondFuture = this.second.submit(() -> {
                this.startLatch.await();
                try {
                    IndexDefinition index;
                    while ((index = (IndexDefinition)indexes.poll()) != null) {
                        Transaction tx = SchemaAcceptanceTest.this.db.beginTx();
                        try {
                            Thread.sleep(1L);
                            tx.schema().getIndexByName(index.getName()).drop();
                            tx.commit();
                        }
                        finally {
                            if (tx == null) continue;
                            tx.close();
                        }
                    }
                    return null;
                }
                finally {
                    stop.set(true);
                }
            });
            this.raceTransactions(firstFuture, secondFuture);
            firstFuture.get();
            secondFuture.get();
        }

        private Callable<Void> schemaTransaction(ThrowingFunction<Transaction, Object, Exception> action) {
            return () -> {
                try (Transaction tx = SchemaAcceptanceTest.this.db.beginTx();){
                    Object creator = action.apply((Object)tx);
                    this.startLatch.await();
                    if (creator instanceof IndexCreator) {
                        ((IndexCreator)creator).create();
                    } else if (creator instanceof ConstraintCreator) {
                        ((ConstraintCreator)creator).create();
                    } else {
                        org.junit.jupiter.api.Assertions.fail((String)("Don't know how to create from " + creator));
                    }
                    tx.commit();
                }
                return null;
            };
        }

        private void raceTransactions(Future<Void> firstFuture, Future<Void> secondFuture) throws InterruptedException, NoSuchMethodException {
            this.first.untilWaitingIn((Executable)BinaryLatch.class.getMethod("await", new Class[0]));
            this.second.untilWaitingIn((Executable)BinaryLatch.class.getMethod("await", new Class[0]));
            this.startLatch.release();
            while (!firstFuture.isDone() || !secondFuture.isDone()) {
                Thread.onSpinWait();
            }
        }

        private void assertOneSuccessAndOneFailure(Future<Void> firstFuture, Future<Void> secondFuture) throws InterruptedException {
            Throwable firstThrowable = this.getException(firstFuture);
            Throwable secondThrowable = this.getException(secondFuture);
            if (firstThrowable == null && secondThrowable == null) {
                org.junit.jupiter.api.Assertions.fail((String)"Both transactions completed successfully, when one of them should have thrown.");
            }
            if (firstThrowable == null) {
                Assertions.assertThat((Throwable)secondThrowable).isInstanceOf(ConstraintViolationException.class);
            }
            if (secondThrowable == null) {
                Assertions.assertThat((Throwable)firstThrowable).isInstanceOf(ConstraintViolationException.class);
            }
        }

        private Throwable getException(Future<Void> future) throws InterruptedException {
            try {
                future.get();
                return null;
            }
            catch (ExecutionException e) {
                return e.getCause();
            }
        }
    }
}

