/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.database;

import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.RandomStringUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.EnumSource;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.recordstorage.RecordStorageEngine;
import org.neo4j.internal.recordstorage.RecordStorageReader;
import org.neo4j.io.pagecache.tracing.cursor.PageCursorTracer;
import org.neo4j.kernel.database.Database;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.test.extension.DbmsExtension;
import org.neo4j.test.extension.Inject;
import org.neo4j.token.TokenHolders;

@DbmsExtension
class TruncateDatabaseIT {
    @Inject
    private GraphDatabaseAPI databaseAPI;
    @Inject
    private Database database;

    TruncateDatabaseIT() {
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    void executeTransactionsOnTrucatedDatabase(DatabaseTruncator truncator) {
        this.createTenNodes();
        truncator.truncate(this.database);
        this.createTenNodes();
        org.junit.jupiter.api.Assertions.assertEquals((long)10L, (long)this.countNodes());
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    void truncateDeleteAllNodes(DatabaseTruncator truncator) {
        this.createTenNodes();
        org.junit.jupiter.api.Assertions.assertEquals((long)10L, (long)this.countNodes());
        truncator.truncate(this.database);
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)this.countNodes());
        try (Transaction transaction = this.databaseAPI.beginTx();){
            org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)transaction.createNode().getId());
        }
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    void truncateDeleteAllRelationships(DatabaseTruncator truncator) {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            for (int i = 0; i < 10; ++i) {
                Node nodeA = transaction.createNode();
                Node nodeB = transaction.createNode();
                nodeA.createRelationshipTo(nodeB, RelationshipType.withName((String)"any"));
            }
            transaction.commit();
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)10L, (long)this.countRelationships());
        truncator.truncate(this.database);
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)this.countRelationships());
        transaction = this.databaseAPI.beginTx();
        try {
            Node newA = transaction.createNode();
            Node newB = transaction.createNode();
            org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)newA.createRelationshipTo(newB, RelationshipType.withName((String)"new")).getId());
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    void truncateKeepsRelationshipTypes(DatabaseTruncator truncator) {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            for (int i = 0; i < 10; ++i) {
                Node nodeA = transaction.createNode();
                Node nodeB = transaction.createNode();
                nodeA.createRelationshipTo(nodeB, RelationshipType.withName((String)("any" + i)));
            }
            transaction.commit();
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)10L, (long)this.countRelationshipTypes());
        truncator.truncate(this.database);
        org.junit.jupiter.api.Assertions.assertEquals((long)10L, (long)this.countRelationshipTypes());
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    void truncateKeepsLabels(DatabaseTruncator truncator) {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            for (int i = 0; i < 10; ++i) {
                transaction.createNode(new Label[]{Label.label((String)("start" + i))});
                transaction.createNode(new Label[]{Label.label((String)("finish" + i))});
            }
            transaction.commit();
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)20L, (long)this.countLabels());
        truncator.truncate(this.database);
        org.junit.jupiter.api.Assertions.assertEquals((long)20L, (long)this.countLabels());
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    void truncateKeepsPropertyKeys(DatabaseTruncator truncator) {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            for (int i = 0; i < 10; ++i) {
                Node node = transaction.createNode();
                node.setProperty("a" + i, (Object)RandomStringUtils.random((int)10));
                node.setProperty("b" + i, (Object)RandomStringUtils.random((int)10));
            }
            transaction.commit();
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)20L, (long)this.countPropertyKeys());
        truncator.truncate(this.database);
        org.junit.jupiter.api.Assertions.assertEquals((long)20L, (long)this.countPropertyKeys());
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    @Disabled
    void truncateKeepsIndexDefinitions(DatabaseTruncator truncator) {
        Label indexLabel = Label.label((String)"indexLabel");
        String indexedProperty = "indexedProperty";
        String indexName = "truncatedIndex";
        try (Transaction transaction = this.databaseAPI.beginTx();){
            transaction.schema().indexFor(indexLabel).on(indexedProperty).withName(indexName).create();
            transaction.commit();
        }
        this.awaitIndexes();
        transaction = this.databaseAPI.beginTx();
        try {
            for (int i = 0; i < 10; ++i) {
                Node node = transaction.createNode(new Label[]{indexLabel});
                node.setProperty(indexedProperty, (Object)RandomStringUtils.random((int)10));
            }
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
        truncator.truncate(this.database);
        transaction = this.databaseAPI.beginTx();
        try {
            IndexDefinition indexDefinition = transaction.schema().getIndexByName(indexName);
            org.junit.jupiter.api.Assertions.assertEquals((Object)indexLabel, indexDefinition.getLabels().iterator().next());
            org.junit.jupiter.api.Assertions.assertEquals((Object)indexedProperty, indexDefinition.getPropertyKeys().iterator().next());
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    void truncateDeleteAllTransactionLogs(DatabaseTruncator truncator) {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            for (int i = 0; i < 10; ++i) {
                Node node = transaction.createNode();
                node.setProperty("a" + i, (Object)RandomStringUtils.random((int)10));
                node.setProperty("b" + i, (Object)RandomStringUtils.random((int)10));
            }
            transaction.commit();
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)20L, (long)this.countPropertyKeys());
        long lastEntryId = this.getLogFiles().getLogFileInformation().getLastEntryId();
        Assertions.assertThat((long)lastEntryId).isGreaterThanOrEqualTo(10L);
        truncator.truncate(this.database);
        long truncatedLastEntryId = this.getLogFiles().getLogFileInformation().getLastEntryId();
        Assertions.assertThat((long)truncatedLastEntryId).isEqualTo(1L);
    }

    @ParameterizedTest
    @EnumSource(value=TruncationTypes.class)
    void truncateDeleteCountsStoreData(DatabaseTruncator truncator) {
        Label human = Label.label((String)"human");
        RelationshipType relationshipType = RelationshipType.withName((String)"relationship");
        try (Transaction transaction = this.databaseAPI.beginTx();){
            for (int i = 0; i < 10; ++i) {
                Node nodeA = transaction.createNode(new Label[]{human});
                Node nodeB = transaction.createNode(new Label[]{human});
                nodeA.setProperty("a" + i, (Object)RandomStringUtils.random((int)10));
                nodeB.setProperty("b" + i, (Object)RandomStringUtils.random((int)10));
                nodeA.createRelationshipTo(nodeB, relationshipType);
            }
            transaction.commit();
        }
        TokenHolders tokenHolders = this.database.getTokenHolders();
        int labelId = tokenHolders.labelTokens().getIdByName(human.name());
        int typeId = tokenHolders.relationshipTypeTokens().getIdByName(relationshipType.name());
        try (RecordStorageReader reader = this.getRecordStorageEngine().newReader();){
            org.junit.jupiter.api.Assertions.assertEquals((long)20L, (long)reader.countsForNode(labelId, PageCursorTracer.NULL));
            org.junit.jupiter.api.Assertions.assertEquals((long)10L, (long)reader.countsForRelationship(-1, typeId, -1, PageCursorTracer.NULL));
        }
        truncator.truncate(this.database);
        reader = this.getRecordStorageEngine().newReader();
        try {
            org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)reader.countsForNode(labelId, PageCursorTracer.NULL));
            org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)reader.countsForRelationship(-1, typeId, -1, PageCursorTracer.NULL));
        }
        finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    private RecordStorageEngine getRecordStorageEngine() {
        return (RecordStorageEngine)this.database.getDependencyResolver().resolveDependency(RecordStorageEngine.class);
    }

    private LogFiles getLogFiles() {
        return (LogFiles)this.database.getDependencyResolver().resolveDependency(LogFiles.class);
    }

    private void awaitIndexes() {
        try (Transaction tx = this.databaseAPI.beginTx();){
            tx.schema().awaitIndexesOnline(10L, TimeUnit.MINUTES);
        }
    }

    private long countNodes() {
        try (Transaction tx = this.databaseAPI.beginTx();){
            long l = Iterables.count((Iterable)tx.getAllNodes());
            return l;
        }
    }

    private long countRelationships() {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            long l = Iterables.count((Iterable)transaction.getAllRelationships());
            return l;
        }
    }

    private long countRelationshipTypes() {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            long l = Iterables.count((Iterable)transaction.getAllRelationshipTypes());
            return l;
        }
    }

    private long countLabels() {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            long l = Iterables.count((Iterable)transaction.getAllLabels());
            return l;
        }
    }

    private long countPropertyKeys() {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            long l = Iterables.count((Iterable)transaction.getAllPropertyKeys());
            return l;
        }
    }

    private void createTenNodes() {
        try (Transaction transaction = this.databaseAPI.beginTx();){
            for (int i = 0; i < 10; ++i) {
                transaction.createNode();
            }
            transaction.commit();
        }
    }

    private static enum TruncationTypes implements DatabaseTruncator
    {
        RUNNING_DATABASE_TRUNCATION{

            @Override
            public void truncate(Database database) {
                database.truncate();
            }
        }
        ,
        STOPPED_DATABASE_TRUNCATION{

            @Override
            public void truncate(Database database) {
                database.stop();
                database.truncate();
                database.start();
            }
        };

    }

    private static interface DatabaseTruncator {
        public void truncate(Database var1);
    }
}

