/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.counts;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.kernel.api.TokenRead;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.test.Barrier;
import org.neo4j.test.extension.ImpermanentDbmsExtension;
import org.neo4j.test.extension.Inject;

@ImpermanentDbmsExtension
class RelationshipCountsTest {
    @Inject
    private GraphDatabaseAPI db;
    private final ExecutorService executor = Executors.newSingleThreadExecutor();

    RelationshipCountsTest() {
    }

    @AfterEach
    void tearDown() {
        this.executor.shutdown();
    }

    @Test
    void shouldReportNumberOfRelationshipsInAnEmptyGraph() {
        long relationshipCount = this.numberOfRelationships();
        Assertions.assertEquals((long)0L, (long)relationshipCount);
    }

    @Test
    void shouldReportTotalNumberOfRelationships() {
        long during;
        long before = this.numberOfRelationships();
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            during = this.countsForRelationship(tx, null, null, null);
            tx.commit();
        }
        long after = this.numberOfRelationships();
        Assertions.assertEquals((long)0L, (long)before);
        Assertions.assertEquals((long)3L, (long)during);
        Assertions.assertEquals((long)3L, (long)after);
    }

    @Test
    void shouldAccountForDeletedRelationships() {
        long during;
        Relationship rel;
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            rel = node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            tx.commit();
        }
        long before = this.numberOfRelationships();
        try (Transaction tx = this.db.beginTx();){
            tx.getRelationshipById(rel.getId()).delete();
            during = this.countsForRelationship(tx, null, null, null);
            tx.commit();
        }
        long after = this.numberOfRelationships();
        Assertions.assertEquals((long)3L, (long)before);
        Assertions.assertEquals((long)2L, (long)during);
        Assertions.assertEquals((long)2L, (long)after);
    }

    @Test
    void shouldNotCountRelationshipsCreatedInOtherTransaction() throws Exception {
        Barrier.Control barrier = new Barrier.Control();
        long before = this.numberOfRelationships();
        Future<Long> tx = this.executor.submit(() -> {
            try (Transaction txn = this.db.beginTx();){
                Node node = txn.createNode();
                node.createRelationshipTo(txn.createNode(), RelationshipType.withName((String)"KNOWS"));
                node.createRelationshipTo(txn.createNode(), RelationshipType.withName((String)"KNOWS"));
                long whatThisThreadSees = this.countsForRelationship(txn, null, null, null);
                barrier.reached();
                txn.commit();
                Long l = whatThisThreadSees;
                return l;
            }
        });
        barrier.await();
        long during = this.numberOfRelationships();
        barrier.release();
        long whatOtherThreadSees = tx.get();
        long after = this.numberOfRelationships();
        Assertions.assertEquals((long)0L, (long)before);
        Assertions.assertEquals((long)0L, (long)during);
        Assertions.assertEquals((long)2L, (long)after);
        Assertions.assertEquals((long)after, (long)whatOtherThreadSees);
    }

    @Test
    void shouldNotCountRelationshipsDeletedInOtherTransaction() throws Exception {
        Relationship rel;
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            rel = node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"KNOWS"));
            tx.commit();
        }
        Barrier.Control barrier = new Barrier.Control();
        long before = this.numberOfRelationships();
        Future<Long> tx = this.executor.submit(() -> {
            try (Transaction txn = this.db.beginTx();){
                txn.getRelationshipById(rel.getId()).delete();
                long whatThisThreadSees = this.countsForRelationship(txn, null, null, null);
                barrier.reached();
                txn.commit();
                Long l = whatThisThreadSees;
                return l;
            }
        });
        barrier.await();
        long during = this.numberOfRelationships();
        barrier.release();
        long whatOtherThreadSees = tx.get();
        long after = this.numberOfRelationships();
        Assertions.assertEquals((long)3L, (long)before);
        Assertions.assertEquals((long)3L, (long)during);
        Assertions.assertEquals((long)2L, (long)after);
        Assertions.assertEquals((long)after, (long)whatOtherThreadSees);
    }

    @Test
    void shouldCountRelationshipsByType() {
        try (Transaction tx = this.db.beginTx();){
            tx.createNode().createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"FOO"));
            tx.createNode().createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"FOO"));
            tx.createNode().createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"BAR"));
            tx.createNode().createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"BAR"));
            tx.createNode().createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"BAR"));
            tx.createNode().createRelationshipTo(tx.createNode(), RelationshipType.withName((String)"BAZ"));
            tx.commit();
        }
        long total = this.numberOfRelationships();
        long foo = this.numberOfRelationships(RelationshipType.withName((String)"FOO"));
        long bar = this.numberOfRelationships(RelationshipType.withName((String)"BAR"));
        long baz = this.numberOfRelationships(RelationshipType.withName((String)"BAZ"));
        long qux = this.numberOfRelationships(RelationshipType.withName((String)"QUX"));
        Assertions.assertEquals((long)2L, (long)foo);
        Assertions.assertEquals((long)3L, (long)bar);
        Assertions.assertEquals((long)1L, (long)baz);
        Assertions.assertEquals((long)0L, (long)qux);
        Assertions.assertEquals((long)6L, (long)total);
    }

    @Test
    void shouldUpdateRelationshipWithLabelCountsWhenDeletingNodeWithRelationship() {
        Node foo;
        try (Transaction tx = this.db.beginTx();){
            foo = tx.createNode(new Label[]{Label.label((String)"Foo")});
            Node bar = tx.createNode(new Label[]{Label.label((String)"Bar")});
            foo.createRelationshipTo(bar, RelationshipType.withName((String)"BAZ"));
            tx.commit();
        }
        long before = this.numberOfRelationshipsMatching(Label.label((String)"Foo"), RelationshipType.withName((String)"BAZ"), null);
        try (Transaction tx = this.db.beginTx();){
            foo = tx.getNodeById(foo.getId());
            for (Relationship relationship : foo.getRelationships()) {
                relationship.delete();
            }
            foo.delete();
            tx.commit();
        }
        long after = this.numberOfRelationshipsMatching(Label.label((String)"Foo"), RelationshipType.withName((String)"BAZ"), null);
        Assertions.assertEquals((long)(before - 1L), (long)after);
    }

    @Test
    void shouldUpdateRelationshipWithLabelCountsWhenDeletingNodesWithRelationships() {
        int i;
        int numberOfNodes = 2;
        Node nodes = new Node[numberOfNodes];
        try (Transaction tx = this.db.beginTx();){
            for (int i2 = 0; i2 < numberOfNodes; ++i2) {
                Node foo = tx.createNode(new Label[]{Label.label((String)("Foo" + i2))});
                foo.addLabel(Label.label((String)"Common"));
                Node bar = tx.createNode(new Label[]{Label.label((String)("Bar" + i2))});
                foo.createRelationshipTo(bar, RelationshipType.withName((String)("BAZ" + i2)));
                nodes[i2] = foo;
            }
            tx.commit();
        }
        long[] beforeCommon = new long[numberOfNodes];
        long[] before = new long[numberOfNodes];
        for (int i3 = 0; i3 < numberOfNodes; ++i3) {
            beforeCommon[i3] = this.numberOfRelationshipsMatching(Label.label((String)"Common"), RelationshipType.withName((String)("BAZ" + i3)), null);
            before[i3] = this.numberOfRelationshipsMatching(Label.label((String)("Foo" + i3)), RelationshipType.withName((String)("BAZ" + i3)), null);
        }
        try (Transaction tx = this.db.beginTx();){
            for (Node node : nodes) {
                node = tx.getNodeById(node.getId());
                for (Relationship relationship : node.getRelationships()) {
                    relationship.delete();
                }
                node.delete();
            }
            tx.commit();
        }
        long[] afterCommon = new long[numberOfNodes];
        long[] after = new long[numberOfNodes];
        for (i = 0; i < numberOfNodes; ++i) {
            afterCommon[i] = this.numberOfRelationshipsMatching(Label.label((String)"Common"), RelationshipType.withName((String)("BAZ" + i)), null);
            after[i] = this.numberOfRelationshipsMatching(Label.label((String)("Foo" + i)), RelationshipType.withName((String)("BAZ" + i)), null);
        }
        for (i = 0; i < numberOfNodes; ++i) {
            Assertions.assertEquals((long)(beforeCommon[i] - 1L), (long)afterCommon[i]);
            Assertions.assertEquals((long)(before[i] - 1L), (long)after[i]);
        }
    }

    @Test
    void shouldUpdateRelationshipWithLabelCountsWhenRemovingLabelAndDeletingRelationship() {
        Node foo;
        try (Transaction tx = this.db.beginTx();){
            foo = tx.createNode(new Label[]{Label.label((String)"Foo")});
            Node bar = tx.createNode(new Label[]{Label.label((String)"Bar")});
            foo.createRelationshipTo(bar, RelationshipType.withName((String)"BAZ"));
            tx.commit();
        }
        long before = this.numberOfRelationshipsMatching(Label.label((String)"Foo"), RelationshipType.withName((String)"BAZ"), null);
        try (Transaction tx = this.db.beginTx();){
            foo = tx.getNodeById(foo.getId());
            for (Relationship relationship : foo.getRelationships()) {
                relationship.delete();
            }
            foo.removeLabel(Label.label((String)"Foo"));
            tx.commit();
        }
        long after = this.numberOfRelationshipsMatching(Label.label((String)"Foo"), RelationshipType.withName((String)"BAZ"), null);
        Assertions.assertEquals((long)(before - 1L), (long)after);
    }

    private long numberOfRelationships(RelationshipType type) {
        return this.numberOfRelationshipsMatching(null, type, null);
    }

    private long numberOfRelationships() {
        return this.numberOfRelationshipsMatching(null, null, null);
    }

    private long numberOfRelationshipsMatching(Label lhs, RelationshipType type, Label rhs) {
        try (Transaction tx = this.db.beginTx();){
            long nodeCount = this.countsForRelationship(tx, lhs, type, rhs);
            tx.commit();
            long l = nodeCount;
            return l;
        }
    }

    private long countsForRelationship(Transaction tx, Label start, RelationshipType type, Label end) {
        int endId;
        int typeId;
        int startId;
        KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
        TokenRead tokenRead = ktx.tokenRead();
        if (start == null) {
            startId = -1;
        } else {
            startId = tokenRead.nodeLabel(start.name());
            if (-1 == startId) {
                return 0L;
            }
        }
        if (type == null) {
            typeId = -1;
        } else {
            typeId = tokenRead.relationshipType(type.name());
            if (-1 == typeId) {
                return 0L;
            }
        }
        if (end == null) {
            endId = -1;
        } else {
            endId = tokenRead.nodeLabel(end.name());
            if (-1 == endId) {
                return 0L;
            }
        }
        return ktx.dataRead().countsForRelationship(startId, typeId, endId);
    }
}

