/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.api.index;

import java.io.IOException;
import java.lang.invoke.CallSite;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.exceptions.KernelException;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.kernel.api.IndexQuery;
import org.neo4j.internal.kernel.api.IndexQueryConstraints;
import org.neo4j.internal.kernel.api.IndexReadSession;
import org.neo4j.internal.kernel.api.NodeValueIndexCursor;
import org.neo4j.internal.kernel.api.exceptions.schema.IndexNotFoundKernelException;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.internal.schema.LabelSchemaDescriptor;
import org.neo4j.internal.schema.SchemaDescriptor;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.api.index.IndexSample;
import org.neo4j.kernel.impl.api.index.IndexingService;
import org.neo4j.kernel.impl.api.index.MultipleIndexPopulator;
import org.neo4j.kernel.impl.api.index.UpdatesTracker;
import org.neo4j.kernel.impl.api.index.stats.IndexStatisticsStore;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.monitoring.Monitors;
import org.neo4j.test.Barrier;
import org.neo4j.test.rule.DbmsRule;
import org.neo4j.test.rule.EmbeddedDbmsRule;
import org.neo4j.test.rule.RandomRule;
import org.neo4j.util.FeatureToggles;
import org.neo4j.util.concurrent.Futures;
import org.neo4j.values.storable.Values;

public class IndexStatisticsTest {
    private static final double UNIQUE_NAMES = 10.0;
    private static final String[] NAMES = new String[]{"Andres", "Davide", "Jakub", "Chris", "Tobias", "Stefan", "Petra", "Rickard", "Mattias", "Emil", "Chris", "Chris"};
    private static final int CREATION_MULTIPLIER = Integer.getInteger(IndexStatisticsTest.class.getName() + ".creationMultiplier", 1000);
    private static final String PERSON_LABEL = "Person";
    private static final String NAME_PROPERTY = "name";
    @Rule
    public final DbmsRule db = new EmbeddedDbmsRule().withSetting(GraphDatabaseSettings.index_background_sampling_enabled, (Object)false).startLazily();
    @Rule
    public final RandomRule random = new RandomRule();
    private final IndexOnlineMonitor indexOnlineMonitor = new IndexOnlineMonitor();

    @Before
    public void before() {
        int batchSize = this.random.nextInt(1, 5);
        FeatureToggles.set(MultipleIndexPopulator.class, (String)"queue_threshold", (Object)batchSize);
        FeatureToggles.set(MultipleIndexPopulator.class, (String)"print_debug", (Object)true);
        ((Monitors)this.db.getGraphDatabaseAPI().getDependencyResolver().resolveDependency(Monitors.class)).addMonitorListener((Object)this.indexOnlineMonitor, new String[0]);
    }

    @After
    public void tearDown() {
        FeatureToggles.clear(MultipleIndexPopulator.class, (String)"queue_threshold");
        FeatureToggles.clear(MultipleIndexPopulator.class, (String)"print_debug");
    }

    @Test
    public void shouldProvideIndexStatisticsForDataCreatedWhenPopulationBeforeTheIndexIsOnline() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        Assert.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)0.0);
        Assert.assertEquals((long)4L, (long)this.indexSize(index));
        Assert.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldUpdateIndexStatisticsForDataCreatedAfterCleanRestart() throws KernelException, IOException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        long indexUpdatesBeforeRestart = this.indexUpdates(index);
        this.db.restartDatabase();
        this.createSomePersons();
        Assertions.assertThat((long)this.indexUpdates(index)).isGreaterThan(indexUpdatesBeforeRestart);
    }

    @Test
    public void shouldNotSeeDataCreatedAfterPopulation() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.createSomePersons();
        Assert.assertEquals((double)1.0, (double)this.indexSelectivity(index), (double)0.0);
        Assert.assertEquals((long)0L, (long)this.indexSize(index));
        Assert.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsForDataSeenDuringPopulationAndIgnoreDataCreatedAfterPopulation() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.createSomePersons();
        Assert.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)0.0);
        Assert.assertEquals((long)4L, (long)this.indexSize(index));
        Assert.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldRemoveIndexStatisticsAfterIndexIsDeleted() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.dropIndex(index);
        try {
            this.indexSelectivity(index);
            Assert.fail((String)"Expected IndexNotFoundKernelException to be thrown");
        }
        catch (IndexNotFoundKernelException e) {
            IndexSample sample = this.getIndexingStatisticsStore().indexSample(index.getId());
            Assert.assertEquals((long)0L, (long)sample.uniqueValues());
            Assert.assertEquals((long)0L, (long)sample.sampleSize());
        }
        IndexSample indexSample = this.getIndexingStatisticsStore().indexSample(index.getId());
        Assert.assertEquals((long)0L, (long)indexSample.indexSize());
        Assert.assertEquals((long)0L, (long)indexSample.updates());
    }

    @Test
    public void shouldProvideIndexSelectivityWhenThereAreManyDuplicates() throws Exception {
        this.indexOnlineMonitor.initialize(0);
        int created = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER).length;
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.assertCorrectIndexSelectivity(index, created);
        this.assertCorrectIndexSize(created, this.indexSize(index));
        Assert.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        int initialNodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER).length;
        IndexDescriptor index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreations(CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedUpdates = updatesTracker.createdAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndDeletions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexDescriptor index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsAndDeletions(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChanges() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexDescriptor index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsAndUpdates(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.createdAfterPopulation() + updatesTracker.updatedAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChangesAndDeletions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexDescriptor index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsDeletionsAndUpdates(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation() + updatesTracker.updatedAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldWorkWhileHavingHeavyConcurrentUpdates() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        int threads = 5;
        this.indexOnlineMonitor.initialize(threads);
        ExecutorService executorService = Executors.newFixedThreadPool(threads);
        IndexDescriptor index = this.createPersonNameIndex();
        ArrayList<Callable<UpdatesTracker>> jobs = new ArrayList<Callable<UpdatesTracker>>(threads);
        for (int i = 0; i < threads; ++i) {
            jobs.add(() -> this.executeCreationsDeletionsAndUpdates(nodes, CREATION_MULTIPLIER));
        }
        List futures = executorService.invokeAll(jobs);
        UpdatesTracker result = new UpdatesTracker();
        result.notifyPopulationCompleted();
        for (UpdatesTracker jobTracker : Futures.getAllResults(futures)) {
            result.add(jobTracker);
        }
        this.awaitIndexesOnline();
        executorService.shutdown();
        Assert.assertTrue((boolean)executorService.awaitTermination(1L, TimeUnit.MINUTES));
        this.assertIndexedNodesMatchesStoreNodes(index);
        int seenWhilePopulating = initialNodes + result.createdDuringPopulation() - result.deletedDuringPopulation();
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize("Tracker had " + result, seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = result.deletedAfterPopulation() + result.createdAfterPopulation() + result.updatedAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        IndexStatisticsTest.assertCorrectIndexUpdates("Tracker had " + result, expectedIndexUpdates, this.indexUpdates(index));
    }

    private void assertIndexedNodesMatchesStoreNodes(IndexDescriptor index) throws Exception {
        int nodesInStore = 0;
        Label label = Label.label((String)PERSON_LABEL);
        try (Transaction transaction = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)transaction).kernelTransaction();
            ArrayList<CallSite> mismatches = new ArrayList<CallSite>();
            int propertyKeyId = ktx.tokenRead().propertyKey(NAME_PROPERTY);
            IndexReadSession indexSession = ktx.dataRead().indexReadSession(index);
            try (NodeValueIndexCursor cursor = ktx.cursors().allocateNodeValueIndexCursor(ktx.pageCursorTracer(), ktx.memoryTracker());){
                for (Node node : Iterables.filter(n -> n.hasLabel(label) && n.hasProperty(NAME_PROPERTY), (Iterable)transaction.getAllNodes())) {
                    ++nodesInStore;
                    String name = (String)node.getProperty(NAME_PROPERTY);
                    ktx.dataRead().nodeIndexSeek(indexSession, cursor, IndexQueryConstraints.unconstrained(), new IndexQuery[]{IndexQuery.exact((int)propertyKeyId, (Object)name)});
                    boolean found = false;
                    while (cursor.next()) {
                        long indexedNode = cursor.nodeReference();
                        if (indexedNode != node.getId()) continue;
                        if (found) {
                            mismatches.add((CallSite)((Object)("Index has multiple entries for " + name + " and " + indexedNode)));
                        }
                        found = true;
                    }
                    if (found) continue;
                    mismatches.add((CallSite)((Object)("Index is missing entry for " + name + " " + node)));
                }
                if (!mismatches.isEmpty()) {
                    Assert.fail((String)String.join((CharSequence)String.format("%n", new Object[0]), mismatches));
                }
                ktx.dataRead().nodeIndexSeek(indexSession, cursor, IndexQueryConstraints.unconstrained(), new IndexQuery[]{IndexQuery.exists((int)propertyKeyId)});
                int nodesInIndex = 0;
                while (cursor.next()) {
                    ++nodesInIndex;
                }
                Assert.assertEquals((long)nodesInStore, (long)nodesInIndex);
            }
        }
    }

    private void deleteNode(long nodeId) {
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(nodeId).delete();
            tx.commit();
        }
    }

    private boolean changeName(long nodeId, Object newValue) {
        boolean changeIndexedNode = false;
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.getNodeById(nodeId);
            Object oldValue = node.getProperty(NAME_PROPERTY);
            if (!oldValue.equals(newValue)) {
                changeIndexedNode = true;
            }
            node.setProperty(NAME_PROPERTY, newValue);
            tx.commit();
        }
        return changeIndexedNode;
    }

    private int createNamedPeople(long[] nodes, int offset) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            for (String name : NAMES) {
                long nodeId = this.createPersonNode(ktx, name);
                if (nodes == null) continue;
                nodes[offset++] = nodeId;
            }
            tx.commit();
        }
        return NAMES.length;
    }

    private long[] repeatCreateNamedPeopleFor(int totalNumberOfPeople) throws Exception {
        long[] nodes = new long[totalNumberOfPeople];
        int threads = 100;
        int peoplePerThread = totalNumberOfPeople / 100;
        ExecutorService service = Executors.newFixedThreadPool(100);
        AtomicReference exception = new AtomicReference();
        ArrayList<Callable<Void>> jobs = new ArrayList<Callable<Void>>(100);
        int i = 0;
        while (i < 100) {
            int finalI = i++;
            jobs.add(() -> {
                for (int offset = finalI * peoplePerThread; offset < (finalI + 1) * peoplePerThread; offset += this.createNamedPeople(nodes, offset)) {
                    try {
                        continue;
                    }
                    catch (KernelException e) {
                        exception.compareAndSet(null, e);
                        throw new RuntimeException(e);
                    }
                }
                return null;
            });
        }
        Futures.getAllResults(service.invokeAll(jobs));
        service.awaitTermination(1L, TimeUnit.SECONDS);
        service.shutdown();
        Exception ex = (Exception)exception.get();
        if (ex != null) {
            throw ex;
        }
        return nodes;
    }

    private void dropIndex(IndexDescriptor index) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            ktx.schemaWrite().indexDrop(index);
            tx.commit();
        }
    }

    private long indexSize(IndexDescriptor reference) {
        return this.resolveDependency(IndexStatisticsStore.class).indexSample(reference.getId()).indexSize();
    }

    private long indexUpdates(IndexDescriptor reference) {
        return this.resolveDependency(IndexStatisticsStore.class).indexSample(reference.getId()).updates();
    }

    private double indexSelectivity(IndexDescriptor reference) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            double selectivity = this.getSelectivity(tx, reference);
            tx.commit();
            double d = selectivity;
            return d;
        }
    }

    private double getSelectivity(Transaction tx, IndexDescriptor reference) throws IndexNotFoundKernelException {
        return ((InternalTransaction)tx).kernelTransaction().schemaRead().indexUniqueValuesSelectivity(reference);
    }

    private IndexStatisticsStore getIndexingStatisticsStore() {
        return this.resolveDependency(IndexStatisticsStore.class);
    }

    private void createSomePersons() throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            this.createPersonNode(ktx, "Davide");
            this.createPersonNode(ktx, "Stefan");
            this.createPersonNode(ktx, "John");
            this.createPersonNode(ktx, "John");
            tx.commit();
        }
    }

    private long createPersonNode(KernelTransaction ktx, Object value) throws KernelException {
        int labelId = ktx.tokenWrite().labelGetOrCreateForName(PERSON_LABEL);
        int propertyKeyId = ktx.tokenWrite().propertyKeyGetOrCreateForName(NAME_PROPERTY);
        long nodeId = ktx.dataWrite().nodeCreate();
        ktx.dataWrite().nodeAddLabel(nodeId, labelId);
        ktx.dataWrite().nodeSetProperty(nodeId, propertyKeyId, Values.of((Object)value));
        return nodeId;
    }

    private IndexDescriptor createPersonNameIndex() throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            int labelId = ktx.tokenWrite().labelGetOrCreateForName(PERSON_LABEL);
            int propertyKeyId = ktx.tokenWrite().propertyKeyGetOrCreateForName(NAME_PROPERTY);
            LabelSchemaDescriptor schema = SchemaDescriptor.forLabel((int)labelId, (int[])new int[]{propertyKeyId});
            IndexDescriptor index = ktx.schemaWrite().indexCreate((SchemaDescriptor)schema, "my index");
            tx.commit();
            IndexDescriptor indexDescriptor = index;
            return indexDescriptor;
        }
    }

    private <T> T resolveDependency(Class<T> clazz) {
        return (T)this.db.getDependencyResolver().resolveDependency(clazz);
    }

    private void awaitIndexesOnline() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().awaitIndexesOnline(3L, TimeUnit.MINUTES);
        }
    }

    private UpdatesTracker executeCreations(int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(null, numberOfCreations, false, false);
    }

    private UpdatesTracker executeCreationsAndDeletions(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, true, false);
    }

    private UpdatesTracker executeCreationsAndUpdates(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, false, true);
    }

    private UpdatesTracker executeCreationsDeletionsAndUpdates(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, true, true);
    }

    private UpdatesTracker internalExecuteCreationsDeletionsAndUpdates(long[] nodes, int numberOfCreations, boolean allowDeletions, boolean allowUpdates) throws KernelException, InterruptedException {
        if (this.random.nextBoolean()) {
            this.indexOnlineMonitor.startSignal.await();
        }
        ThreadLocalRandom random = ThreadLocalRandom.current();
        UpdatesTracker updatesTracker = new UpdatesTracker();
        int offset = 0;
        while (updatesTracker.created() < numberOfCreations) {
            int created = this.createNamedPeople(nodes, offset);
            offset += created;
            updatesTracker.increaseCreated(created);
            this.notifyIfPopulationCompleted(updatesTracker);
            if (allowDeletions && updatesTracker.created() % 24 == 0) {
                long nodeId = nodes[((Random)random).nextInt(nodes.length)];
                try {
                    this.deleteNode(nodeId);
                    updatesTracker.increaseDeleted(1);
                }
                catch (NotFoundException notFoundException) {
                    // empty catch block
                }
                this.notifyIfPopulationCompleted(updatesTracker);
            }
            if (!allowUpdates || updatesTracker.created() % 24 != 0) continue;
            int randomIndex = ((Random)random).nextInt(nodes.length);
            try {
                if (this.changeName(nodes[randomIndex], NAMES[((Random)random).nextInt(NAMES.length)])) {
                    updatesTracker.increaseUpdated(1);
                }
            }
            catch (NotFoundException notFoundException) {
                // empty catch block
            }
            this.notifyIfPopulationCompleted(updatesTracker);
        }
        this.notifyPopulationCompleted(updatesTracker);
        return updatesTracker;
    }

    private void notifyPopulationCompleted(UpdatesTracker updatesTracker) {
        this.indexOnlineMonitor.updatesDone();
        updatesTracker.notifyPopulationCompleted();
    }

    private void notifyIfPopulationCompleted(UpdatesTracker updatesTracker) {
        if (this.isCompletedPopulation(updatesTracker)) {
            this.notifyPopulationCompleted(updatesTracker);
        }
    }

    private boolean isCompletedPopulation(UpdatesTracker updatesTracker) {
        return !updatesTracker.isPopulationCompleted() && this.indexOnlineMonitor.isIndexOnline();
    }

    private void assertCorrectIndexSize(long expected, long actual) {
        this.assertCorrectIndexSize("", expected, actual);
    }

    private void assertCorrectIndexSize(String info, long expected, long actual) {
        long updatesAfterCompletion = this.indexOnlineMonitor.indexSampleOnCompletion.updates();
        String message = String.format("Expected number of entries to not differ (expected: %d actual: %d) %s", expected, actual, info);
        ((AbstractLongAssert)Assertions.assertThat((long)Math.abs(expected - actual)).withFailMessage(message, new Object[0])).isLessThanOrEqualTo(updatesAfterCompletion);
    }

    private static void assertCorrectIndexUpdates(long expected, long actual) {
        IndexStatisticsTest.assertCorrectIndexUpdates("", expected, actual);
    }

    private static void assertCorrectIndexUpdates(String info, long expected, long actual) {
        String message = String.format("Expected number of index updates to not differ (expected: %d actual: %d). %s", expected, actual, info);
        Assert.assertEquals((String)message, (long)0L, (long)Math.abs(expected - actual));
    }

    private void assertCorrectIndexSelectivity(IndexDescriptor index, long numberOfEntries) throws KernelException {
        double expected = 10.0 / (double)numberOfEntries;
        double actual = this.indexSelectivity(index);
        double maxDelta = Double.max(1.0E-4, (double)this.indexOnlineMonitor.indexSampleOnCompletion.updates() / (double)numberOfEntries);
        String message = String.format("Expected number of entries to not differ (expected: %f actual: %f)", expected, actual);
        Assert.assertEquals((String)message, (double)expected, (double)actual, (double)maxDelta);
    }

    private class IndexOnlineMonitor
    extends IndexingService.MonitorAdapter {
        private CountDownLatch updateTrackerCompletionLatch;
        private final CountDownLatch startSignal = new CountDownLatch(1);
        private volatile boolean isOnline;
        private Barrier.Control barrier;
        private IndexSample indexSampleOnCompletion;

        private IndexOnlineMonitor() {
        }

        void initialize(int numberOfUpdateTrackers) {
            this.updateTrackerCompletionLatch = new CountDownLatch(numberOfUpdateTrackers);
            if (numberOfUpdateTrackers > 0) {
                this.barrier = new Barrier.Control();
            }
        }

        void updatesDone() {
            this.updateTrackerCompletionLatch.countDown();
            try {
                this.updateTrackerCompletionLatch.await();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            if (this.barrier != null) {
                this.barrier.reached();
            }
        }

        public void indexPopulationScanStarting() {
            this.startSignal.countDown();
        }

        public void indexPopulationScanComplete() {
            this.isOnline = true;
            if (this.barrier != null) {
                try {
                    this.barrier.await();
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        public void populationCompleteOn(IndexDescriptor descriptor) {
            this.indexSampleOnCompletion = IndexStatisticsTest.this.getIndexingStatisticsStore().indexSample(descriptor.getId());
            if (this.barrier != null) {
                this.barrier.release();
            }
        }

        boolean isIndexOnline() {
            return this.isOnline;
        }
    }
}

