/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.database;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.assertj.core.api.AssertionsForInterfaceTypes;
import org.eclipse.collections.api.set.ImmutableSet;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.neo4j.collection.Dependencies;
import org.neo4j.common.DependencyResolver;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.pagecache.DelegatingPageCache;
import org.neo4j.io.pagecache.DelegatingPagedFile;
import org.neo4j.io.pagecache.IOLimiter;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.PagedFile;
import org.neo4j.io.pagecache.tracing.cursor.context.VersionContextSupplier;
import org.neo4j.kernel.availability.DatabaseAvailabilityGuard;
import org.neo4j.kernel.database.Database;
import org.neo4j.kernel.database.DatabaseFileHelper;
import org.neo4j.kernel.lifecycle.LifeSupport;
import org.neo4j.kernel.lifecycle.LifecycleAdapter;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogAssertions;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.internal.DatabaseLogService;
import org.neo4j.logging.internal.LogService;
import org.neo4j.memory.GlobalMemoryGroupTracker;
import org.neo4j.memory.MemoryGroup;
import org.neo4j.memory.MemoryPools;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.monitoring.DatabaseHealth;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.DbmsExtension;
import org.neo4j.test.extension.ExtensionCallback;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.pagecache.PageCacheSupportExtension;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;

@TestDirectoryExtension
@DbmsExtension(configurationCallback="configure")
class DatabaseIT {
    @RegisterExtension
    static PageCacheSupportExtension pageCacheExtension = new PageCacheSupportExtension();
    @Inject
    private FileSystemAbstraction fs;
    @Inject
    private TestDirectory directory;
    @Inject
    private DatabaseLayout databaseLayout;
    @Inject
    private Database database;
    @Inject
    private MemoryPools memoryPools;
    private PageCacheWrapper pageCacheWrapper;
    private final AssertableLogProvider logProvider = new AssertableLogProvider();

    DatabaseIT() {
    }

    @ExtensionCallback
    void configure(TestDatabaseManagementServiceBuilder builder) {
        Dependencies dependencies = new Dependencies();
        this.pageCacheWrapper = new PageCacheWrapper(pageCacheExtension.getPageCache(this.fs));
        dependencies.satisfyDependency((Object)this.pageCacheWrapper);
        builder.setInternalLogProvider((LogProvider)this.logProvider).setExternalDependencies((DependencyResolver)dependencies);
    }

    @Test
    void databaseHealthShouldBeHealedOnStart() throws Throwable {
        this.database.stop();
        this.database.init();
        DatabaseHealth databaseHealth = this.database.getDatabaseHealth();
        databaseHealth.panic(new Throwable());
        this.database.start();
        databaseHealth.assertHealthy(Throwable.class);
    }

    @Test
    void dropDataOfNotStartedDatabase() {
        this.database.stop();
        Assertions.assertNotEquals((Object)this.databaseLayout.databaseDirectory(), (Object)this.databaseLayout.getTransactionLogsDirectory());
        Assertions.assertTrue((boolean)this.fs.fileExists(this.databaseLayout.databaseDirectory()));
        Assertions.assertTrue((boolean)this.fs.fileExists(this.databaseLayout.getTransactionLogsDirectory()));
        this.database.drop();
        Assertions.assertFalse((boolean)this.fs.fileExists(this.databaseLayout.databaseDirectory()));
        Assertions.assertFalse((boolean)this.fs.fileExists(this.databaseLayout.getTransactionLogsDirectory()));
    }

    @Test
    void truncateNotStartedDatabase() {
        this.database.stop();
        Path databaseDirectory = this.databaseLayout.databaseDirectory();
        Path transactionLogsDirectory = this.databaseLayout.getTransactionLogsDirectory();
        Assertions.assertTrue((boolean)this.fs.fileExists(databaseDirectory));
        Assertions.assertTrue((boolean)this.fs.fileExists(transactionLogsDirectory));
        Object[] databaseFilesShouldExist = (Path[])DatabaseFileHelper.filesToKeepOnTruncation((DatabaseLayout)this.databaseLayout).stream().filter(x$0 -> Files.exists(x$0, new LinkOption[0])).toArray(Path[]::new);
        this.database.truncate();
        Assertions.assertTrue((boolean)this.fs.fileExists(databaseDirectory));
        Assertions.assertTrue((boolean)this.fs.fileExists(transactionLogsDirectory));
        Object[] currentDatabaseFiles = this.fs.listFiles(databaseDirectory);
        AssertionsForInterfaceTypes.assertThat((Object[])currentDatabaseFiles).contains(databaseFilesShouldExist);
    }

    @Test
    void doNotFlushDataFilesOnDatabaseTruncate() throws IOException {
        this.database.start();
        ArrayList mappingsBefore = new ArrayList(this.pageCacheWrapper.listExistingMappings());
        this.database.truncate();
        ArrayList removedFiles = new ArrayList(this.pageCacheWrapper.listExistingMappings());
        removedFiles.removeAll(mappingsBefore);
        List removedPaths = removedFiles.stream().map(PagedFile::path).collect(Collectors.toList());
        DatabaseLayout databaseLayout = this.database.getDatabaseLayout();
        Set files = databaseLayout.storeFiles();
        files.removeAll(DatabaseFileHelper.filesToKeepOnTruncation((DatabaseLayout)databaseLayout));
        Object[] filesShouldBeDeleted = (Path[])files.stream().filter(x$0 -> Files.exists(x$0, new LinkOption[0])).toArray(Path[]::new);
        AssertionsForInterfaceTypes.assertThat(removedPaths).contains(filesShouldBeDeleted);
    }

    @Test
    void filesRecreatedAfterTruncate() {
        Path databaseDirectory = this.databaseLayout.databaseDirectory();
        Path transactionLogsDirectory = this.databaseLayout.getTransactionLogsDirectory();
        Assertions.assertTrue((boolean)this.fs.fileExists(databaseDirectory));
        Assertions.assertTrue((boolean)this.fs.fileExists(transactionLogsDirectory));
        Object[] databaseFilesBeforeTruncate = this.fs.listFiles(databaseDirectory);
        Object[] logFilesBeforeTruncate = this.fs.listFiles(transactionLogsDirectory);
        this.database.truncate();
        Assertions.assertTrue((boolean)this.fs.fileExists(databaseDirectory));
        Assertions.assertTrue((boolean)this.fs.fileExists(transactionLogsDirectory));
        Object[] databaseFiles = this.fs.listFiles(databaseDirectory);
        Object[] logFiles = this.fs.listFiles(transactionLogsDirectory);
        AssertionsForInterfaceTypes.assertThat((Object[])databaseFilesBeforeTruncate).contains(databaseFiles);
        AssertionsForInterfaceTypes.assertThat((Object[])logFilesBeforeTruncate).contains(logFiles);
    }

    @Test
    void noPageCacheFlushOnDatabaseDrop() {
        this.database.start();
        int flushesBefore = this.pageCacheWrapper.getFlushes();
        this.database.drop();
        Assertions.assertEquals((int)flushesBefore, (int)this.pageCacheWrapper.getFlushes());
    }

    @Test
    void removeDatabaseDataAndLogsOnDrop() {
        Assertions.assertNotEquals((Object)this.databaseLayout.databaseDirectory(), (Object)this.databaseLayout.getTransactionLogsDirectory());
        Assertions.assertTrue((boolean)this.fs.fileExists(this.databaseLayout.databaseDirectory()));
        Assertions.assertTrue((boolean)this.fs.fileExists(this.databaseLayout.getTransactionLogsDirectory()));
        this.database.drop();
        Assertions.assertFalse((boolean)this.fs.fileExists(this.databaseLayout.databaseDirectory()));
        Assertions.assertFalse((boolean)this.fs.fileExists(this.databaseLayout.getTransactionLogsDirectory()));
    }

    @Test
    void flushDatabaseDataOnStop() {
        int flushesBeforeClose = this.pageCacheWrapper.getFileFlushes();
        this.database.stop();
        Assertions.assertNotEquals((int)flushesBeforeClose, (int)this.pageCacheWrapper.getFileFlushes());
    }

    @Test
    void flushOfThePageCacheHappensOnlyOnceDuringShutdown() throws Throwable {
        int databaseFiles = (int)this.database.getDatabaseFileListing().builder().build().stream().count();
        int flushesBefore = this.pageCacheWrapper.getFlushes();
        int filesFlushesBefore = this.pageCacheWrapper.getFileFlushes();
        this.database.stop();
        Assertions.assertEquals((int)flushesBefore, (int)this.pageCacheWrapper.getFlushes());
        AssertionsForInterfaceTypes.assertThat((int)this.pageCacheWrapper.getFileFlushes()).isGreaterThanOrEqualTo(filesFlushesBefore + databaseFiles);
    }

    @Test
    void flushOfThePageCacheOnShutdownDoesNotHappenIfTheDbIsUnhealthy() throws Throwable {
        DatabaseHealth databaseHealth = this.database.getDatabaseHealth();
        databaseHealth.panic(new Throwable("Critical failure"));
        int fileFlushesBefore = this.pageCacheWrapper.getFileFlushes();
        int databaseFiles = (int)this.database.getDatabaseFileListing().builder().excludeLogFiles().build().stream().count();
        this.database.stop();
        AssertionsForInterfaceTypes.assertThat((int)this.pageCacheWrapper.getFileFlushes()).isLessThan(fileFlushesBefore + databaseFiles);
    }

    @Test
    void logModuleSetUpError() {
        RuntimeException exception = new RuntimeException("StartupError");
        this.database.stop();
        this.database.init();
        this.database.getLife().add(LifecycleAdapter.onStart(() -> {
            throw exception;
        }));
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> this.database.start());
        AssertionsForInterfaceTypes.assertThat((Throwable)e).hasRootCause((Throwable)exception);
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).forClass(Database.class).forLevel(AssertableLogProvider.Level.WARN).containsMessageWithException("Exception occurred while starting the database. Trying to stop already started components.", e.getCause());
    }

    @Test
    void shouldAlwaysShutdownLifeEvenWhenSomeComponentFailing() {
        RuntimeException expectedException = new RuntimeException("Failure");
        LifeSupport life = this.database.getLife();
        DatabaseAvailabilityGuard availabilityGuard = this.database.getDatabaseAvailabilityGuard();
        life.add(LifecycleAdapter.onShutdown(() -> {
            throw expectedException;
        }));
        Throwable e = Assertions.assertThrows(Throwable.class, () -> this.database.stop());
        AssertionsForInterfaceTypes.assertThat((Throwable)e).hasCause((Throwable)expectedException);
        Assertions.assertFalse((boolean)availabilityGuard.isAvailable());
    }

    @Test
    void shouldHaveDatabaseLogServiceInDependencyResolver() {
        LogService logService = (LogService)this.database.getDependencyResolver().resolveDependency(LogService.class);
        Assertions.assertEquals((Object)this.database.getLogService(), (Object)logService);
        AssertionsForInterfaceTypes.assertThat((Object)logService).isInstanceOf(DatabaseLogService.class);
    }

    @Test
    void stopShutdownMustOnlyReleaseMemoryOnce() throws Exception {
        MemoryTracker otherMemoryTracker = this.getOtherMemoryTracker();
        long beforeStop = otherMemoryTracker.usedNativeMemory();
        this.database.stop();
        long afterStop = otherMemoryTracker.usedNativeMemory();
        AssertionsForInterfaceTypes.assertThat((long)afterStop).isLessThan(beforeStop);
        this.database.shutdown();
        long afterShutdown = otherMemoryTracker.usedNativeMemory();
        Assertions.assertEquals((long)afterShutdown, (long)afterStop);
    }

    @Test
    void shutdownShutdownMustOnlyReleaseMemoryOnce() throws Exception {
        MemoryTracker otherMemoryTracker = this.getOtherMemoryTracker();
        long beforeShutdown = otherMemoryTracker.usedNativeMemory();
        this.database.shutdown();
        long afterFirstShutdown = otherMemoryTracker.usedNativeMemory();
        AssertionsForInterfaceTypes.assertThat((long)afterFirstShutdown).isLessThan(beforeShutdown);
        this.database.shutdown();
        long afterSecondShutdown = otherMemoryTracker.usedNativeMemory();
        Assertions.assertEquals((long)afterSecondShutdown, (long)afterFirstShutdown);
    }

    @Test
    void shutdownStopMustOnlyReleaseMemoryOnce() throws Exception {
        MemoryTracker otherMemoryTracker = this.getOtherMemoryTracker();
        long beforeShutdown = otherMemoryTracker.usedNativeMemory();
        this.database.shutdown();
        long afterShutdown = otherMemoryTracker.usedNativeMemory();
        AssertionsForInterfaceTypes.assertThat((long)afterShutdown).isLessThan(beforeShutdown);
        this.database.stop();
        long afterStop = otherMemoryTracker.usedNativeMemory();
        Assertions.assertEquals((long)afterStop, (long)afterShutdown);
    }

    private MemoryTracker getOtherMemoryTracker() {
        for (GlobalMemoryGroupTracker pool : this.memoryPools.getPools()) {
            if (!pool.group().equals((Object)MemoryGroup.OTHER)) continue;
            return pool.getPoolMemoryTracker();
        }
        throw new RuntimeException("Could not find memory tracker for group " + MemoryGroup.OTHER);
    }

    private static class PageFileWrapper
    extends DelegatingPagedFile {
        private final AtomicInteger flushCounter;

        PageFileWrapper(PagedFile delegate, AtomicInteger flushCounter) {
            super(delegate);
            this.flushCounter = flushCounter;
        }

        public void flushAndForce() throws IOException {
            this.flushCounter.incrementAndGet();
            super.flushAndForce();
        }

        public void flushAndForce(IOLimiter limiter) throws IOException {
            this.flushCounter.incrementAndGet();
            super.flushAndForce(limiter);
        }
    }

    private static class PageCacheWrapper
    extends DelegatingPageCache {
        private final AtomicInteger flushes = new AtomicInteger();
        private final AtomicInteger fileFlushes = new AtomicInteger();

        PageCacheWrapper(PageCache delegate) {
            super(delegate);
        }

        public PagedFile map(Path path, int pageSize) throws IOException {
            return new PageFileWrapper(super.map(path, pageSize), this.fileFlushes);
        }

        public PagedFile map(Path path, VersionContextSupplier versionContextSupplier, int pageSize, ImmutableSet<OpenOption> openOptions) throws IOException {
            return new PageFileWrapper(super.map(path, versionContextSupplier, pageSize, openOptions), this.fileFlushes);
        }

        public PagedFile map(Path path, int pageSize, ImmutableSet<OpenOption> openOptions) throws IOException {
            return new PageFileWrapper(super.map(path, pageSize, openOptions), this.fileFlushes);
        }

        public PagedFile map(Path path, VersionContextSupplier versionContextSupplier, int pageSize, ImmutableSet<OpenOption> openOptions, String databaseName) throws IOException {
            return new PageFileWrapper(super.map(path, versionContextSupplier, pageSize, openOptions, databaseName), this.fileFlushes);
        }

        public void flushAndForce(IOLimiter limiter) throws IOException {
            this.flushes.incrementAndGet();
            super.flushAndForce(limiter);
        }

        public void flushAndForce() throws IOException {
            this.flushes.incrementAndGet();
            super.flushAndForce();
        }

        public int getFlushes() {
            return this.flushes.get();
        }

        public int getFileFlushes() {
            return this.fileFlushes.get();
        }
    }
}

