/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.store.watch;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.WatchKey;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.ListAssert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.EnabledOnOs;
import org.junit.jupiter.api.condition.OS;
import org.neo4j.common.DependencyResolver;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.graphdb.schema.IndexType;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.FileUtils;
import org.neo4j.io.fs.watcher.FileWatchEventListener;
import org.neo4j.io.fs.watcher.FileWatcher;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.kernel.impl.transaction.log.checkpoint.CheckPointer;
import org.neo4j.kernel.impl.transaction.log.checkpoint.SimpleTriggerInfo;
import org.neo4j.kernel.impl.transaction.log.checkpoint.TriggerInfo;
import org.neo4j.kernel.impl.util.watcher.DefaultFileDeletionEventListener;
import org.neo4j.kernel.impl.util.watcher.FileSystemWatcherService;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogAssertions;
import org.neo4j.logging.LogProvider;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.Neo4jLayoutExtension;
import org.neo4j.test.rule.TestDirectory;

@Neo4jLayoutExtension
@EnabledOnOs(value={OS.LINUX})
class FileWatchIT {
    @Inject
    private TestDirectory testDirectory;
    @Inject
    private DatabaseLayout databaseLayout;
    private AssertableLogProvider logProvider;
    private GraphDatabaseService database;
    private DatabaseManagementService managementService;

    FileWatchIT() {
    }

    @BeforeEach
    void setUp() {
        this.logProvider = new AssertableLogProvider();
        this.managementService = new TestDatabaseManagementServiceBuilder(this.databaseLayout).setInternalLogProvider((LogProvider)this.logProvider).build();
        this.database = this.managementService.database("neo4j");
    }

    @AfterEach
    void tearDown() {
        FileWatchIT.shutdownDatabaseSilently(this.managementService);
    }

    @Test
    void notifyAboutStoreFileDeletion() throws IOException, InterruptedException {
        String fileName = this.databaseLayout.metadataStore().getFileName().toString();
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkpointer = FileWatchIT.getCheckpointer(this.database);
        DeletionLatchEventListener deletionListener = new DeletionLatchEventListener(fileName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)deletionListener);
        do {
            FileWatchIT.createNode(this.database);
            FileWatchIT.forceCheckpoint(checkpointer);
        } while (!deletionListener.awaitModificationNotification());
        FileWatchIT.deleteFile(this.databaseLayout.databaseDirectory(), fileName);
        deletionListener.awaitDeletionNotification();
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).containsMessages(new String[]{"'" + fileName + "' which belongs to the '" + this.databaseLayout.databaseDirectory().getFileName().toString() + "' database was deleted while it was running."});
    }

    @Test
    void notifyWhenFileWatchingFailToStart() {
        AssertableLogProvider logProvider = new AssertableLogProvider(true);
        DatabaseManagementService service = null;
        try {
            service = new TestDatabaseManagementServiceBuilder(this.testDirectory.homePath("failed-start-db")).setInternalLogProvider((LogProvider)logProvider).setFileSystem((FileSystemAbstraction)new NonWatchableFileSystemAbstraction()).build();
            Assertions.assertNotNull((Object)this.managementService.database("neo4j"));
            LogAssertions.assertThat((AssertableLogProvider)logProvider).containsMessages(new String[]{"Can not create file watcher for current file system. File monitoring capabilities for store files will be disabled."});
        }
        catch (Throwable throwable) {
            FileWatchIT.shutdownDatabaseSilently(service);
            throw throwable;
        }
        FileWatchIT.shutdownDatabaseSilently(service);
    }

    @Test
    void doNotNotifyAboutIndexFilesDeletion() throws IOException, InterruptedException {
        DependencyResolver dependencyResolver = ((GraphDatabaseAPI)this.database).getDependencyResolver();
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkPointer = (CheckPointer)dependencyResolver.resolveDependency(CheckPointer.class);
        String propertyStoreName = this.databaseLayout.propertyStore().getFileName().toString();
        AccumulativeDeletionEventListener accumulativeListener = new AccumulativeDeletionEventListener();
        ModificationEventListener modificationListener = new ModificationEventListener(propertyStoreName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)modificationListener);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)accumulativeListener);
        String labelName = "labelName";
        String propertyName = "propertyName";
        Label testLabel = Label.label((String)labelName);
        FileWatchIT.createIndexes(this.database, propertyName, testLabel);
        do {
            FileWatchIT.createNode(this.database, propertyName, testLabel);
            FileWatchIT.forceCheckpoint(checkPointer);
        } while (!modificationListener.awaitModificationNotification());
        fileWatcher.removeFileWatchEventListener((FileWatchEventListener)modificationListener);
        ModificationEventListener afterRemovalListener = new ModificationEventListener(propertyStoreName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)afterRemovalListener);
        FileWatchIT.dropAllIndexes(this.database);
        do {
            FileWatchIT.createNode(this.database, propertyName, testLabel);
            FileWatchIT.forceCheckpoint(checkPointer);
        } while (!afterRemovalListener.awaitModificationNotification());
        accumulativeListener.assertDoesNotHaveAnyDeletions();
    }

    @Test
    void doNotMonitorTransactionLogFiles() throws IOException, InterruptedException {
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkpointer = FileWatchIT.getCheckpointer(this.database);
        String metadataStore = this.databaseLayout.metadataStore().getFileName().toString();
        ModificationEventListener modificationEventListener = new ModificationEventListener(metadataStore);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)modificationEventListener);
        do {
            FileWatchIT.createNode(this.database);
            FileWatchIT.forceCheckpoint(checkpointer);
        } while (!modificationEventListener.awaitModificationNotification());
        String fileName = "neostore.transaction.db.0";
        DeletionLatchEventListener deletionListener = new DeletionLatchEventListener(fileName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)deletionListener);
        FileWatchIT.deleteFile(this.databaseLayout.getTransactionLogsDirectory(), fileName);
        deletionListener.awaitDeletionNotification();
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).forClass(DefaultFileDeletionEventListener.class).forLevel(AssertableLogProvider.Level.INFO).doesNotContainMessage(fileName);
    }

    @Test
    void notifyWhenWholeStoreDirectoryRemoved() throws IOException, InterruptedException {
        String fileName = this.databaseLayout.metadataStore().getFileName().toString();
        FileWatcher fileWatcher = FileWatchIT.getFileWatcher(this.database);
        CheckPointer checkpointer = FileWatchIT.getCheckpointer(this.database);
        ModificationEventListener modificationListener = new ModificationEventListener(fileName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)modificationListener);
        do {
            FileWatchIT.createNode(this.database);
            FileWatchIT.forceCheckpoint(checkpointer);
        } while (!modificationListener.awaitModificationNotification());
        fileWatcher.removeFileWatchEventListener((FileWatchEventListener)modificationListener);
        String storeDirectoryName = this.databaseLayout.databaseDirectory().getFileName().toString();
        DeletionLatchEventListener eventListener = new DeletionLatchEventListener(storeDirectoryName);
        fileWatcher.addFileWatchEventListener((FileWatchEventListener)eventListener);
        FileUtils.deleteDirectory((Path)this.databaseLayout.databaseDirectory());
        eventListener.awaitDeletionNotification();
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).containsMessages(new String[]{"'" + storeDirectoryName + "' which belongs to the '" + this.databaseLayout.databaseDirectory().getFileName().toString() + "' database was deleted while it was running."});
    }

    @Test
    void shouldLogWhenDisabled() {
        AssertableLogProvider logProvider = new AssertableLogProvider(true);
        DatabaseManagementService service = null;
        try {
            service = new TestDatabaseManagementServiceBuilder(this.testDirectory.homePath("failed-start-db")).setInternalLogProvider((LogProvider)logProvider).setFileSystem((FileSystemAbstraction)new NonWatchableFileSystemAbstraction()).setConfig(GraphDatabaseSettings.filewatcher_enabled, (Object)false).build();
            Assertions.assertNotNull((Object)this.managementService.database("neo4j"));
            LogAssertions.assertThat((AssertableLogProvider)logProvider).containsMessages(new String[]{"File watcher disabled by configuration."});
        }
        catch (Throwable throwable) {
            FileWatchIT.shutdownDatabaseSilently(service);
            throw throwable;
        }
        FileWatchIT.shutdownDatabaseSilently(service);
    }

    private static void shutdownDatabaseSilently(DatabaseManagementService managementService) {
        if (managementService != null) {
            try {
                managementService.shutdown();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    private static void dropAllIndexes(GraphDatabaseService database) {
        try (Transaction transaction = database.beginTx();){
            for (IndexDefinition definition : transaction.schema().getIndexes()) {
                if (definition.getIndexType() == IndexType.LOOKUP) continue;
                definition.drop();
            }
            transaction.commit();
        }
    }

    private static void createIndexes(GraphDatabaseService database, String propertyName, Label testLabel) {
        try (Transaction transaction = database.beginTx();){
            transaction.schema().indexFor(testLabel).on(propertyName).create();
            transaction.commit();
        }
        try (Transaction tx = database.beginTx();){
            tx.schema().awaitIndexesOnline(2L, TimeUnit.MINUTES);
        }
    }

    private static void forceCheckpoint(CheckPointer checkPointer) throws IOException {
        checkPointer.forceCheckPoint((TriggerInfo)new SimpleTriggerInfo("testForceCheckPoint"));
    }

    private static void createNode(GraphDatabaseService database, String propertyName, Label testLabel) {
        try (Transaction transaction = database.beginTx();){
            Node node = transaction.createNode(new Label[]{testLabel});
            node.setProperty(propertyName, (Object)"value");
            transaction.commit();
        }
    }

    private static CheckPointer getCheckpointer(GraphDatabaseService database) {
        return (CheckPointer)((GraphDatabaseAPI)database).getDependencyResolver().resolveDependency(CheckPointer.class);
    }

    private static FileWatcher getFileWatcher(GraphDatabaseService database) {
        DependencyResolver dependencyResolver = ((GraphDatabaseAPI)database).getDependencyResolver();
        return ((FileSystemWatcherService)dependencyResolver.resolveDependency(FileSystemWatcherService.class)).getFileWatcher();
    }

    private static void deleteFile(Path storeDir, String fileName) throws IOException {
        Path metadataStore = storeDir.resolve(fileName);
        FileUtils.deleteFile((Path)metadataStore);
    }

    private static void createNode(GraphDatabaseService database) {
        try (Transaction transaction = database.beginTx();){
            transaction.createNode();
            transaction.commit();
        }
    }

    private static class DeletionLatchEventListener
    extends ModificationEventListener {
        private final CountDownLatch deletionLatch = new CountDownLatch(1);

        DeletionLatchEventListener(String expectedFileName) {
            super(expectedFileName);
        }

        public void fileDeleted(WatchKey key, String fileName) {
            if (fileName.endsWith(this.expectedFileName)) {
                this.deletionLatch.countDown();
            }
        }

        void awaitDeletionNotification() throws InterruptedException {
            this.deletionLatch.await();
        }
    }

    private static class ModificationEventListener
    implements FileWatchEventListener {
        final String expectedFileName;
        private final CountDownLatch modificationLatch = new CountDownLatch(1);

        ModificationEventListener(String expectedFileName) {
            this.expectedFileName = expectedFileName;
        }

        public void fileModified(WatchKey key, String fileName) {
            if (this.expectedFileName.equals(fileName)) {
                this.modificationLatch.countDown();
            }
        }

        boolean awaitModificationNotification() throws InterruptedException {
            return this.modificationLatch.await(1L, TimeUnit.SECONDS);
        }
    }

    private static class AccumulativeDeletionEventListener
    implements FileWatchEventListener {
        private final List<String> deletedFiles = new ArrayList<String>();

        private AccumulativeDeletionEventListener() {
        }

        public void fileDeleted(WatchKey key, String fileName) {
            this.deletedFiles.add(fileName);
        }

        void assertDoesNotHaveAnyDeletions() {
            ((ListAssert)LogAssertions.assertThat(this.deletedFiles).as("Should not have any deletions registered", new Object[0])).isEmpty();
        }
    }

    private static class NonWatchableFileSystemAbstraction
    extends DefaultFileSystemAbstraction {
        private NonWatchableFileSystemAbstraction() {
        }

        public FileWatcher fileWatcher() throws IOException {
            throw new IOException("You can't watch me!");
        }
    }
}

