/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.nio.file.Path;
import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.assertj.core.api.Assertions;
import org.eclipse.collections.api.factory.Sets;
import org.junit.jupiter.api.Test;
import org.neo4j.annotations.documented.ReporterFactory;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.helpers.DatabaseReadOnlyChecker;
import org.neo4j.dbms.DatabaseStateService;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.dbms.database.DatabaseStartAbortedException;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.IndexType;
import org.neo4j.index.internal.gbptree.RecoveryCleanupWorkCollector;
import org.neo4j.internal.helpers.ArrayUtil;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.id.DefaultIdGeneratorFactory;
import org.neo4j.internal.id.IdGenerator;
import org.neo4j.internal.id.IdType;
import org.neo4j.internal.kernel.api.IndexQueryConstraints;
import org.neo4j.internal.kernel.api.IndexReadSession;
import org.neo4j.internal.kernel.api.PropertyIndexQuery;
import org.neo4j.internal.kernel.api.RelationshipValueIndexCursor;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.io.pagecache.tracing.DefaultPageCacheTracer;
import org.neo4j.io.pagecache.tracing.PageCacheTracer;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.availability.CompositeDatabaseAvailabilityGuard;
import org.neo4j.kernel.database.DatabaseTracers;
import org.neo4j.kernel.extension.ExtensionFactory;
import org.neo4j.kernel.extension.context.ExtensionContext;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.kernel.impl.store.MetaDataStore;
import org.neo4j.kernel.impl.storemigration.LegacyTransactionLogsLocator;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.checkpoint.DetachedCheckpointAppender;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.kernel.impl.transaction.log.files.checkpoint.CheckpointFile;
import org.neo4j.kernel.impl.transaction.tracing.DatabaseTracer;
import org.neo4j.kernel.impl.transaction.tracing.LogCheckPointEvent;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.kernel.lifecycle.LifecycleAdapter;
import org.neo4j.kernel.recovery.Recovery;
import org.neo4j.kernel.recovery.RecoveryHelpers;
import org.neo4j.kernel.recovery.RecoveryMonitor;
import org.neo4j.kernel.recovery.RecoveryStartInformationProvider;
import org.neo4j.lock.LockTracer;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.LogProvider;
import org.neo4j.logging.NullLogProvider;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.monitoring.Monitors;
import org.neo4j.service.Services;
import org.neo4j.storageengine.api.StorageEngineFactory;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.Neo4jLayoutExtension;
import org.neo4j.test.extension.pagecache.PageCacheExtension;

@PageCacheExtension
@Neo4jLayoutExtension
class RecoveryIT {
    private static final int TEN_KB = (int)ByteUnit.kibiBytes((long)10L);
    @Inject
    private DefaultFileSystemAbstraction fileSystem;
    @Inject
    private PageCache pageCache;
    @Inject
    private Neo4jLayout neo4jLayout;
    @Inject
    private DatabaseLayout databaseLayout;
    private TestDatabaseManagementServiceBuilder builder;
    private DatabaseManagementService managementService;

    RecoveryIT() {
    }

    @Test
    void recoveryRequiredOnDatabaseWithoutCorrectCheckpoints() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(this.databaseLayout));
    }

    @Test
    void recoveryNotRequiredWhenDatabaseNotFound() throws Exception {
        DatabaseLayout absentDatabase = this.neo4jLayout.databaseLayout("absent");
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(absentDatabase));
    }

    @Test
    void recoverEmptyDatabase() throws Throwable {
        Config config = Config.newBuilder().set(GraphDatabaseInternalSettings.skip_default_indexes_on_creation, (Object)true).set(GraphDatabaseSettings.preallocate_logical_logs, (Object)false).build();
        this.managementService = new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(config).build();
        this.managementService.database(this.databaseLayout.getDatabaseName());
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout, config));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithNodes() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfNodes = 10;
        for (int i = 0; i < numberOfNodes; ++i) {
            this.createSingleNode((GraphDatabaseService)database);
        }
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try (Transaction tx = recoveredDatabase.beginTx();){
            org.junit.jupiter.api.Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)tx.getAllNodes()));
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void tracePageCacheAccessOnDatabaseRecovery() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfNodes = 10;
        for (int i = 0; i < numberOfNodes; ++i) {
            this.createSingleNode((GraphDatabaseService)database);
        }
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        DefaultPageCacheTracer pageCacheTracer = new DefaultPageCacheTracer();
        DatabaseTracers tracers = new DatabaseTracers(DatabaseTracer.NULL, LockTracer.NONE, (PageCacheTracer)pageCacheTracer);
        this.recoverDatabase(tracers);
        long pins = pageCacheTracer.pins();
        Assertions.assertThat((long)pins).isGreaterThan(0L);
        Assertions.assertThat((long)pageCacheTracer.unpins()).isEqualTo(pins);
        Assertions.assertThat((long)pageCacheTracer.hits()).isGreaterThan(0L).isLessThanOrEqualTo(pins);
        Assertions.assertThat((long)pageCacheTracer.faults()).isGreaterThan(0L).isLessThanOrEqualTo(pins);
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try (Transaction tx = recoveredDatabase.beginTx();){
            org.junit.jupiter.api.Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)tx.getAllNodes()));
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithNodesAndRelationshipsAndRelationshipTypes() throws Throwable {
        Transaction transaction;
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfRelationships = 10;
        int numberOfNodes = numberOfRelationships * 2;
        for (int i = 0; i < numberOfRelationships; ++i) {
            transaction = database.beginTx();
            try {
                Node start = transaction.createNode();
                Node stop = transaction.createNode();
                start.createRelationshipTo(stop, RelationshipType.withName((String)String.valueOf(i)));
                transaction.commit();
                continue;
            }
            finally {
                if (transaction != null) {
                    transaction.close();
                }
            }
        }
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try {
            transaction = recoveredDatabase.beginTx();
            try {
                org.junit.jupiter.api.Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)transaction.getAllNodes()));
                org.junit.jupiter.api.Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationships()));
                org.junit.jupiter.api.Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationshipTypesInUse()));
            }
            finally {
                if (transaction != null) {
                    transaction.close();
                }
            }
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithProperties() throws Throwable {
        Transaction transaction;
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfRelationships = 10;
        int numberOfNodes = numberOfRelationships * 2;
        for (int i = 0; i < numberOfRelationships; ++i) {
            transaction = database.beginTx();
            try {
                Node start = transaction.createNode();
                Node stop = transaction.createNode();
                start.setProperty("start" + i, (Object)i);
                stop.setProperty("stop" + i, (Object)i);
                start.createRelationshipTo(stop, RelationshipType.withName((String)String.valueOf(i)));
                transaction.commit();
                continue;
            }
            finally {
                if (transaction != null) {
                    transaction.close();
                }
            }
        }
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try {
            transaction = recoveredDatabase.beginTx();
            try {
                org.junit.jupiter.api.Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)transaction.getAllNodes()));
                org.junit.jupiter.api.Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationships()));
                org.junit.jupiter.api.Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationshipTypesInUse()));
                org.junit.jupiter.api.Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)transaction.getAllPropertyKeys()));
            }
            finally {
                if (transaction != null) {
                    transaction.close();
                }
            }
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithIndex() throws Throwable {
        long numberOfPropertyKeys;
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfRelationships = 10;
        int numberOfNodes = numberOfRelationships * 2;
        String startProperty = "start";
        String stopProperty = "stop";
        Label startMarker = Label.label((String)"start");
        Label stopMarker = Label.label((String)"stop");
        try (Transaction transaction = database.beginTx();){
            transaction.schema().indexFor(startMarker).on(startProperty).create();
            transaction.schema().constraintFor(stopMarker).assertPropertyIsUnique(stopProperty).create();
            transaction.commit();
        }
        RecoveryIT.awaitIndexesOnline((GraphDatabaseService)database);
        for (int i = 0; i < numberOfRelationships; ++i) {
            try (Transaction transaction = database.beginTx();){
                Node start = transaction.createNode(new Label[]{startMarker});
                Node stop = transaction.createNode(new Label[]{stopMarker});
                start.setProperty(startProperty, (Object)i);
                stop.setProperty(stopProperty, (Object)i);
                start.createRelationshipTo(stop, RelationshipType.withName((String)String.valueOf(i)));
                transaction.commit();
                continue;
            }
        }
        try (Transaction transaction = database.beginTx();){
            numberOfPropertyKeys = Iterables.count((Iterable)transaction.getAllPropertyKeys());
        }
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        try (Transaction transaction = recoveredDatabase.beginTx();){
            org.junit.jupiter.api.Assertions.assertEquals((long)numberOfNodes, (long)Iterables.count((Iterable)transaction.getAllNodes()));
            org.junit.jupiter.api.Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationships()));
            org.junit.jupiter.api.Assertions.assertEquals((long)numberOfRelationships, (long)Iterables.count((Iterable)transaction.getAllRelationshipTypesInUse()));
            org.junit.jupiter.api.Assertions.assertEquals((long)numberOfPropertyKeys, (long)Iterables.count((Iterable)transaction.getAllPropertyKeys()));
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void recoverDatabaseWithRelationshipIndex() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase();
        int numberOfRelationships = 10;
        RelationshipType type = RelationshipType.withName((String)"TYPE");
        String property = "prop";
        String indexName = "my index";
        try (Transaction transaction = database.beginTx();){
            transaction.schema().indexFor(type).on(property).withIndexType(IndexType.FULLTEXT).withName(indexName).create();
            transaction.commit();
        }
        RecoveryIT.awaitIndexesOnline((GraphDatabaseService)database);
        transaction = database.beginTx();
        try {
            Node start = transaction.createNode();
            Node stop = transaction.createNode();
            for (int i = 0; i < numberOfRelationships; ++i) {
                Relationship relationship = start.createRelationshipTo(stop, type);
                relationship.setProperty(property, (Object)"value");
            }
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.recoverDatabase();
        GraphDatabaseAPI recoveredDatabase = this.createDatabase();
        RecoveryIT.awaitIndexesOnline((GraphDatabaseService)recoveredDatabase);
        try (Transaction transaction = recoveredDatabase.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)transaction).kernelTransaction();
            IndexDescriptor index = ktx.schemaRead().indexGetForName(indexName);
            IndexReadSession indexReadSession = ktx.dataRead().indexReadSession(index);
            int relationshipsInIndex = 0;
            try (RelationshipValueIndexCursor cursor = ktx.cursors().allocateRelationshipValueIndexCursor(ktx.cursorContext(), ktx.memoryTracker());){
                ktx.dataRead().relationshipIndexSeek(indexReadSession, cursor, IndexQueryConstraints.unconstrained(), new PropertyIndexQuery[]{PropertyIndexQuery.fulltextSearch((String)"*")});
                while (cursor.next()) {
                    ++relationshipsInIndex;
                }
            }
            org.junit.jupiter.api.Assertions.assertEquals((int)numberOfRelationships, (int)relationshipsInIndex);
        }
        finally {
            this.managementService.shutdown();
        }
    }

    @Test
    void recoverDatabaseWithFirstTransactionLogFileWithoutShutdownCheckpoint() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        this.managementService.shutdown();
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)this.countCheckPointsInTransactionLogs());
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)this.countCheckPointsInTransactionLogs());
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(this.databaseLayout));
        this.startStopDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout));
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void failToStartDatabaseWithRemovedTransactionLogs() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        this.managementService.shutdown();
        this.removeTransactionLogs();
        GraphDatabaseAPI restartedDb = this.createDatabase();
        try {
            DatabaseStateService dbStateService = (DatabaseStateService)restartedDb.getDependencyResolver().resolveDependency(DatabaseStateService.class);
            Optional failure = dbStateService.causeOfFailure(restartedDb.databaseId());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)failure.isPresent());
            Assertions.assertThat((String)ExceptionUtils.getRootCause((Throwable)((Throwable)failure.get())).getMessage()).contains(new CharSequence[]{"Transaction logs are missing and recovery is not possible."});
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void failToStartDatabaseWithTransactionLogsInLegacyLocation() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        this.managementService.shutdown();
        LogFiles logFiles = this.buildLogFiles();
        Object[] txLogFiles = this.fileSystem.listFiles(logFiles.logFilesDirectory(), path -> path.getFileName().toString().startsWith("neostore.transaction.db"));
        txLogFiles = (Path[])ArrayUtil.concat((Object[])txLogFiles, (Object[])logFiles.getCheckpointFile().getDetachedCheckpointFiles());
        Path databasesDirectory = this.databaseLayout.getNeo4jLayout().databasesDirectory();
        DatabaseLayout legacyLayout = Neo4jLayout.ofFlat((Path)databasesDirectory).databaseLayout(this.databaseLayout.getDatabaseName());
        LegacyTransactionLogsLocator logsLocator = new LegacyTransactionLogsLocator(Config.defaults(), legacyLayout);
        Path transactionLogsDirectory = logsLocator.getTransactionLogsDirectory();
        org.junit.jupiter.api.Assertions.assertNotNull((Object)txLogFiles);
        org.junit.jupiter.api.Assertions.assertTrue((txLogFiles.length > 0 ? 1 : 0) != 0);
        for (Object logFile : txLogFiles) {
            this.fileSystem.moveToDirectory((Path)logFile, transactionLogsDirectory);
        }
        AssertableLogProvider logProvider = new AssertableLogProvider();
        this.builder.setInternalLogProvider((LogProvider)logProvider);
        GraphDatabaseAPI restartedDb = this.createDatabase();
        try {
            DatabaseStateService dbStateService = (DatabaseStateService)restartedDb.getDependencyResolver().resolveDependency(DatabaseStateService.class);
            Optional failure = dbStateService.causeOfFailure(restartedDb.databaseId());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)failure.isPresent());
            Assertions.assertThat((Throwable)((Throwable)failure.get())).hasRootCauseMessage("Transaction logs are missing and recovery is not possible.");
            Assertions.assertThat((String)logProvider.serialize()).contains(new CharSequence[]{txLogFiles[0].getFileName().toString()});
        }
        finally {
            this.managementService.shutdown();
        }
    }

    @Test
    void startDatabaseWithRemovedSingleTransactionLogFile() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase();
        PageCache pageCache = this.getDatabasePageCache(database);
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        org.junit.jupiter.api.Assertions.assertEquals((long)-1L, (long)MetaDataStore.getRecord((PageCache)pageCache, (Path)database.databaseLayout().metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.LAST_MISSING_STORE_FILES_RECOVERY_TIMESTAMP, (String)this.databaseLayout.getDatabaseName(), (CursorContext)CursorContext.NULL));
        this.managementService.shutdown();
        this.removeTransactionLogs();
        this.startStopDatabaseWithForcedRecovery();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout));
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        this.verifyRecoveryTimestampPresent(database);
    }

    @Test
    void startDatabaseWithRemovedMultipleTransactionLogFiles() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase(ByteUnit.mebiBytes((long)1L));
        while (this.countTransactionLogFiles() < 5) {
            RecoveryIT.generateSomeData((GraphDatabaseService)database);
        }
        this.managementService.shutdown();
        this.removeTransactionLogs();
        this.startStopDatabaseWithForcedRecovery();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout));
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
    }

    @Test
    void killAndStartDatabaseAfterTransactionLogsRemoval() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase(ByteUnit.mebiBytes((long)1L));
        while (this.countTransactionLogFiles() < 5) {
            RecoveryIT.generateSomeData((GraphDatabaseService)database);
        }
        this.managementService.shutdown();
        this.removeTransactionLogs();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(this.databaseLayout));
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)this.countTransactionLogFiles());
        DatabaseManagementService forcedRecoveryManagementService = this.forcedRecoveryManagement();
        GraphDatabaseService service = forcedRecoveryManagementService.database("neo4j");
        this.createSingleNode(service);
        forcedRecoveryManagementService.shutdown();
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countTransactionLogFiles());
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.startStopDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout));
        org.junit.jupiter.api.Assertions.assertEquals((int)3, (int)this.countCheckPointsInTransactionLogs());
    }

    @Test
    void killAndStartDatabaseAfterTransactionLogsRemovalWithSeveralFilesWithoutCheckpoint() throws Throwable {
        GraphDatabaseAPI database = this.createDatabase(ByteUnit.mebiBytes((long)1L));
        while (this.countTransactionLogFiles() < 5) {
            RecoveryIT.generateSomeData((GraphDatabaseService)database);
        }
        this.managementService.shutdown();
        this.removeFileWithCheckpoint();
        org.junit.jupiter.api.Assertions.assertEquals((int)4, (int)this.countTransactionLogFiles());
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)this.countCheckPointsInTransactionLogs());
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(this.databaseLayout));
        this.startStopDatabase();
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.startStopDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout));
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
    }

    @Test
    void startDatabaseAfterTransactionLogsRemovalAndKillAfterRecovery() throws Throwable {
        long logThreshold = ByteUnit.mebiBytes((long)1L);
        GraphDatabaseAPI database = this.createDatabase(logThreshold);
        while (this.countTransactionLogFiles() < 5) {
            RecoveryIT.generateSomeData((GraphDatabaseService)database);
        }
        this.managementService.shutdown();
        this.removeFileWithCheckpoint();
        org.junit.jupiter.api.Assertions.assertEquals((int)4, (int)this.countTransactionLogFiles());
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)this.countCheckPointsInTransactionLogs());
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(this.databaseLayout));
        this.startStopDatabase();
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.startStopDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout));
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.countCheckPointsInTransactionLogs());
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.builder = null;
        GraphDatabaseAPI service = this.createDatabase(logThreshold * 2L);
        this.createSingleNode((GraphDatabaseService)service);
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        this.startStopDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout));
        org.junit.jupiter.api.Assertions.assertEquals((int)3, (int)this.countCheckPointsInTransactionLogs());
    }

    @Test
    void recoverDatabaseWithoutOneIdFile() throws Throwable {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        this.fileSystem.deleteFileOrThrow(layout.idRelationshipStore());
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(layout));
        Recovery.performRecovery((FileSystemAbstraction)this.fileSystem, (PageCache)this.pageCache, (DatabaseTracers)DatabaseTracers.EMPTY, (Config)Config.defaults(), (DatabaseLayout)layout, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(layout));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.fileSystem.fileExists(layout.idRelationshipStore()));
    }

    @Test
    void recoverDatabaseWithoutIdFiles() throws Throwable {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        for (Path idFile : layout.idFiles()) {
            this.fileSystem.deleteFileOrThrow(idFile);
        }
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(layout));
        this.recoverDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(layout));
        for (Path idFile : layout.idFiles()) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)this.fileSystem.fileExists(idFile));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void failRecoveryWithMissingStoreFile() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        DatabaseLayout layout = database.databaseLayout();
        this.managementService.shutdown();
        this.fileSystem.deleteFileOrThrow(layout.nodeStore());
        GraphDatabaseAPI restartedDb = this.createDatabase();
        try {
            DatabaseStateService dbStateService = (DatabaseStateService)restartedDb.getDependencyResolver().resolveDependency(DatabaseStateService.class);
            Optional failure = dbStateService.causeOfFailure(restartedDb.databaseId());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)failure.isPresent());
            Assertions.assertThat((Throwable)((Throwable)failure.get()).getCause()).hasMessageContainingAll(new CharSequence[]{"neostore.nodestore.db", "is(are) missing and recovery is not possible"});
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void failRecoveryWithMissingStoreFileAndIdFile() throws Exception {
        GraphDatabaseAPI database = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)database);
        DatabaseLayout layout = database.databaseLayout();
        this.managementService.shutdown();
        this.fileSystem.deleteFileOrThrow(layout.nodeStore());
        this.fileSystem.deleteFileOrThrow(layout.idLabelTokenStore());
        GraphDatabaseAPI restartedDb = this.createDatabase();
        try {
            DatabaseStateService dbStateService = (DatabaseStateService)restartedDb.getDependencyResolver().resolveDependency(DatabaseStateService.class);
            Optional failure = dbStateService.causeOfFailure(restartedDb.databaseId());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)failure.isPresent());
            Assertions.assertThat((Throwable)((Throwable)failure.get()).getCause()).hasMessageContainingAll(new CharSequence[]{"neostore.nodestore.db", "is(are) missing and recovery is not possible"});
        }
        finally {
            this.managementService.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void cancelRecoveryInTheMiddle() throws Throwable {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        RecoveryHelpers.removeLastCheckpointRecordFromLastLogFile((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(layout));
        Monitors monitors = new Monitors();
        final GlobalGuardConsumerTestExtensionFactory guardExtensionFactory = new GlobalGuardConsumerTestExtensionFactory();
        var recoveryMonitor = new RecoveryMonitor(){
            private final AtomicBoolean reverseCompleted = new AtomicBoolean();
            private final AtomicBoolean recoveryCompleted = new AtomicBoolean();

            public void reverseStoreRecoveryCompleted(long lowestRecoveredTxId) {
                try {
                    guardExtensionFactory.getProvidedGuardConsumer().globalGuard.stop();
                }
                catch (Exception exception) {
                    // empty catch block
                }
                this.reverseCompleted.set(true);
            }

            public void recoveryCompleted(int numberOfRecoveredTransactions, long recoveryTimeInMilliseconds) {
                this.recoveryCompleted.set(true);
            }

            public boolean isReverseCompleted() {
                return this.reverseCompleted.get();
            }

            public boolean isRecoveryCompleted() {
                return this.recoveryCompleted.get();
            }
        };
        monitors.addMonitorListener((Object)recoveryMonitor, new String[0]);
        DatabaseManagementService service = new TestDatabaseManagementServiceBuilder(layout.getNeo4jLayout()).addExtension((ExtensionFactory)guardExtensionFactory).setMonitors(monitors).build();
        try {
            GraphDatabaseService database = service.database(layout.getDatabaseName());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)recoveryMonitor.isReverseCompleted());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)recoveryMonitor.isRecoveryCompleted());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)guardExtensionFactory.getProvidedGuardConsumer().globalGuard.isAvailable());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)database.isAvailable(0L));
            Exception e = (Exception)org.junit.jupiter.api.Assertions.assertThrows(Exception.class, () -> ((GraphDatabaseService)database).beginTx());
            Assertions.assertThat((Throwable)ExceptionUtils.getRootCause((Throwable)e)).isInstanceOf(DatabaseStartAbortedException.class);
        }
        finally {
            service.shutdown();
        }
    }

    @Test
    void shouldForceRecoveryEvenThoughNotSeeminglyRequired() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(layout));
        DefaultIdGeneratorFactory idGeneratorFactory = new DefaultIdGeneratorFactory((FileSystemAbstraction)this.fileSystem, RecoveryCleanupWorkCollector.immediate(), "my db");
        try (IdGenerator idGenerator = idGeneratorFactory.open(this.pageCache, layout.idNodeStore(), IdType.NODE, () -> 0L, 10000L, DatabaseReadOnlyChecker.writable(), Config.defaults(), CursorContext.NULL, Sets.immutable.empty());){
            idGenerator.marker(CursorContext.NULL).close();
        }
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(layout));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.idGeneratorIsDirty(layout.idNodeStore(), IdType.NODE));
        final MutableBoolean recoveryRunEvenThoughNoCommitsAfterLastCheckpoint = new MutableBoolean();
        RecoveryStartInformationProvider.Monitor monitor = new RecoveryStartInformationProvider.Monitor(){

            public void noCommitsAfterLastCheckPoint(LogPosition logPosition) {
                recoveryRunEvenThoughNoCommitsAfterLastCheckpoint.setTrue();
            }
        };
        Monitors monitors = new Monitors();
        monitors.addMonitorListener((Object)monitor, new String[0]);
        Recovery.performRecovery((FileSystemAbstraction)this.fileSystem, (PageCache)this.pageCache, (DatabaseTracers)DatabaseTracers.EMPTY, (Config)Config.defaults(), (DatabaseLayout)layout, (StorageEngineFactory)StorageEngineFactory.defaultStorageEngine(), (boolean)true, (LogProvider)NullLogProvider.nullLogProvider(), (Monitors)monitors, (Iterable)Iterables.cast((Iterable)Services.loadAll(ExtensionFactory.class)), Optional.empty(), null, (MemoryTracker)EmptyMemoryTracker.INSTANCE, (Clock)Clock.systemUTC());
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.idGeneratorIsDirty(layout.idNodeStore(), IdType.NODE));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)recoveryRunEvenThoughNoCommitsAfterLastCheckpoint.booleanValue());
    }

    @Test
    void keepCheckpointVersionOnMissingLogFilesWhenValueIsReasonable() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        this.removeTransactionLogs();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(layout));
        MetaDataStore.setRecord((PageCache)this.pageCache, (Path)layout.metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.CHECKPOINT_LOG_VERSION, (long)18L, (String)layout.getDatabaseName(), (CursorContext)CursorContext.NULL);
        this.recoverDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(layout));
        org.junit.jupiter.api.Assertions.assertEquals((long)18L, (long)MetaDataStore.getRecord((PageCache)this.pageCache, (Path)layout.metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.CHECKPOINT_LOG_VERSION, (String)layout.getDatabaseName(), (CursorContext)CursorContext.NULL));
    }

    @Test
    void resetCheckpointVersionOnMissingLogFilesWhenValueIsDefinitelyWrong() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        this.removeTransactionLogs();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(layout));
        MetaDataStore.setRecord((PageCache)this.pageCache, (Path)layout.metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.CHECKPOINT_LOG_VERSION, (long)-42L, (String)layout.getDatabaseName(), (CursorContext)CursorContext.NULL);
        this.recoverDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(layout));
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)MetaDataStore.getRecord((PageCache)this.pageCache, (Path)layout.metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.CHECKPOINT_LOG_VERSION, (String)layout.getDatabaseName(), (CursorContext)CursorContext.NULL));
    }

    @Test
    void recoverySetsCheckpointLogVersionFieldNoCheckpointFiles() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        this.removeFileWithCheckpoint();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(layout));
        MetaDataStore.setRecord((PageCache)this.pageCache, (Path)layout.metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.CHECKPOINT_LOG_VERSION, (long)-5L, (String)layout.getDatabaseName(), (CursorContext)CursorContext.NULL);
        this.recoverDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(layout));
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)MetaDataStore.getRecord((PageCache)this.pageCache, (Path)layout.metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.CHECKPOINT_LOG_VERSION, (String)layout.getDatabaseName(), (CursorContext)CursorContext.NULL));
    }

    @Test
    void recoverySetsCheckpointLogVersionFieldSeveralCheckpointFiles() throws Exception {
        GraphDatabaseAPI db = this.createDatabase();
        RecoveryIT.generateSomeData((GraphDatabaseService)db);
        CheckpointFile checkpointFile = ((LogFiles)db.getDependencyResolver().resolveDependency(LogFiles.class)).getCheckpointFile();
        DetachedCheckpointAppender appender = (DetachedCheckpointAppender)checkpointFile.getCheckpointAppender();
        appender.rotate();
        appender.checkPoint(LogCheckPointEvent.NULL, new LogPosition(0L, 64L), Instant.now(), "test1");
        appender.rotate();
        appender.checkPoint(LogCheckPointEvent.NULL, new LogPosition(0L, 64L), Instant.now(), "test2");
        appender.rotate();
        appender.checkPoint(LogCheckPointEvent.NULL, new LogPosition(0L, 64L), Instant.now(), "test3");
        DatabaseLayout layout = db.databaseLayout();
        this.managementService.shutdown();
        this.removeFileWithCheckpoint();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(layout));
        MetaDataStore.setRecord((PageCache)this.pageCache, (Path)layout.metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.CHECKPOINT_LOG_VERSION, (long)-5L, (String)layout.getDatabaseName(), (CursorContext)CursorContext.NULL);
        this.recoverDatabase();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(layout));
        org.junit.jupiter.api.Assertions.assertEquals((long)2L, (long)MetaDataStore.getRecord((PageCache)this.pageCache, (Path)layout.metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.CHECKPOINT_LOG_VERSION, (String)layout.getDatabaseName(), (CursorContext)CursorContext.NULL));
    }

    private boolean idGeneratorIsDirty(Path path, IdType idType) throws IOException {
        DefaultIdGeneratorFactory idGeneratorFactory = new DefaultIdGeneratorFactory((FileSystemAbstraction)this.fileSystem, RecoveryCleanupWorkCollector.immediate(), "my db");
        try (IdGenerator idGenerator = idGeneratorFactory.open(this.pageCache, path, idType, () -> 0L, 10000L, DatabaseReadOnlyChecker.readOnly(), Config.defaults(), CursorContext.NULL, Sets.immutable.empty());){
            MutableBoolean dirtyOnStartup = new MutableBoolean();
            InvocationHandler invocationHandler = (proxy, method, args) -> {
                if (method.getName().equals("dirtyOnStartup")) {
                    dirtyOnStartup.setTrue();
                }
                return null;
            };
            ReporterFactory reporterFactory = new ReporterFactory(invocationHandler);
            idGenerator.consistencyCheck(reporterFactory, CursorContext.NULL);
            boolean bl = dirtyOnStartup.booleanValue();
            return bl;
        }
    }

    private static void awaitIndexesOnline(GraphDatabaseService database) {
        try (Transaction transaction = database.beginTx();){
            transaction.schema().awaitIndexesOnline(10L, TimeUnit.MINUTES);
            transaction.commit();
        }
    }

    private void createSingleNode(GraphDatabaseService service) {
        try (Transaction transaction = service.beginTx();){
            transaction.createNode();
            transaction.commit();
        }
    }

    private void startStopDatabase() {
        GraphDatabaseAPI db = this.createDatabase();
        db.beginTx().close();
        this.managementService.shutdown();
    }

    private void recoverDatabase() throws Exception {
        this.recoverDatabase(DatabaseTracers.EMPTY);
    }

    void additionalConfiguration(Config config) {
        config.set(GraphDatabaseSettings.fail_on_missing_files, (Object)false);
    }

    TestDatabaseManagementServiceBuilder additionalConfiguration(TestDatabaseManagementServiceBuilder builder) {
        return builder;
    }

    private void recoverDatabase(DatabaseTracers databaseTracers) throws Exception {
        Config config = Config.newBuilder().build();
        this.additionalConfiguration(config);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.isRecoveryRequired(this.databaseLayout, config));
        Recovery.performRecovery((FileSystemAbstraction)this.fileSystem, (PageCache)this.pageCache, (DatabaseTracers)databaseTracers, (Config)config, (DatabaseLayout)this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)this.isRecoveryRequired(this.databaseLayout, config));
    }

    private boolean isRecoveryRequired(DatabaseLayout layout) throws Exception {
        Config config = Config.newBuilder().build();
        this.additionalConfiguration(config);
        return this.isRecoveryRequired(layout, config);
    }

    private boolean isRecoveryRequired(DatabaseLayout layout, Config config) throws Exception {
        return Recovery.isRecoveryRequired((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)layout, (Config)config, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
    }

    private int countCheckPointsInTransactionLogs() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        List checkpoints = logFiles.getCheckpointFile().reachableCheckpoints();
        return checkpoints.size();
    }

    private LogFiles buildLogFiles() throws IOException {
        return LogFilesBuilder.logFilesBasedOnlyBuilder((Path)this.databaseLayout.getTransactionLogsDirectory(), (FileSystemAbstraction)this.fileSystem).withCommandReaderFactory(StorageEngineFactory.defaultStorageEngine().commandReaderFactory()).build();
    }

    private void removeTransactionLogs() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        for (Path logFile : this.fileSystem.listFiles(logFiles.logFilesDirectory())) {
            this.fileSystem.deleteFile(logFile);
        }
    }

    private void removeFileWithCheckpoint() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        this.fileSystem.deleteFileOrThrow(logFiles.getCheckpointFile().getCurrentFile());
    }

    private int countTransactionLogFiles() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        return logFiles.logFiles().length;
    }

    private static void generateSomeData(GraphDatabaseService database) {
        for (int i = 0; i < 10; ++i) {
            try (Transaction transaction = database.beginTx();){
                Node node1 = transaction.createNode();
                Node node2 = transaction.createNode();
                node1.createRelationshipTo(node2, RelationshipType.withName((String)("Type" + i)));
                node2.setProperty("a", (Object)RandomStringUtils.randomAlphanumeric((int)TEN_KB));
                transaction.commit();
                continue;
            }
        }
    }

    private GraphDatabaseAPI createDatabase() {
        return this.createDatabase((Long)GraphDatabaseSettings.logical_log_rotation_threshold.defaultValue());
    }

    private GraphDatabaseAPI createDatabase(long logThreshold) {
        this.createBuilder(logThreshold);
        this.managementService = this.builder.build();
        return (GraphDatabaseAPI)this.managementService.database(this.databaseLayout.getDatabaseName());
    }

    private void createBuilder(long logThreshold) {
        if (this.builder == null) {
            this.builder = new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(GraphDatabaseSettings.preallocate_logical_logs, (Object)false).setConfig(GraphDatabaseSettings.logical_log_rotation_threshold, (Object)logThreshold);
            this.builder = this.additionalConfiguration(this.builder);
        }
    }

    private void startStopDatabaseWithForcedRecovery() {
        DatabaseManagementService forcedRecoveryManagementService = this.forcedRecoveryManagement();
        forcedRecoveryManagementService.shutdown();
    }

    private DatabaseManagementService forcedRecoveryManagement() {
        TestDatabaseManagementServiceBuilder serviceBuilder = new TestDatabaseManagementServiceBuilder(this.neo4jLayout).setConfig(GraphDatabaseSettings.fail_on_missing_files, (Object)false);
        return this.additionalConfiguration(serviceBuilder).build();
    }

    private PageCache getDatabasePageCache(GraphDatabaseAPI databaseAPI) {
        return (PageCache)databaseAPI.getDependencyResolver().resolveDependency(PageCache.class);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyRecoveryTimestampPresent(GraphDatabaseAPI databaseAPI) throws IOException {
        GraphDatabaseAPI restartedDatabase = this.createDatabase();
        try {
            PageCache restartedCache = this.getDatabasePageCache(restartedDatabase);
            long record = MetaDataStore.getRecord((PageCache)restartedCache, (Path)databaseAPI.databaseLayout().metadataStore(), (MetaDataStore.Position)MetaDataStore.Position.LAST_MISSING_STORE_FILES_RECOVERY_TIMESTAMP, (String)this.databaseLayout.getDatabaseName(), (CursorContext)CursorContext.NULL);
            Assertions.assertThat((long)record).isGreaterThan(0L);
        }
        finally {
            this.managementService.shutdown();
        }
    }

    private static class GlobalGuardConsumer
    extends LifecycleAdapter {
        private final CompositeDatabaseAvailabilityGuard globalGuard;

        GlobalGuardConsumer(Dependencies dependencies) {
            this.globalGuard = dependencies.globalGuard();
        }
    }

    private static class GlobalGuardConsumerTestExtensionFactory
    extends ExtensionFactory<Dependencies> {
        private GlobalGuardConsumer providedConsumer;

        GlobalGuardConsumerTestExtensionFactory() {
            super("globalGuardConsumer");
        }

        public Lifecycle newInstance(ExtensionContext context, Dependencies dependencies) {
            this.providedConsumer = new GlobalGuardConsumer(dependencies);
            return this.providedConsumer;
        }

        public GlobalGuardConsumer getProvidedGuardConsumer() {
            return this.providedConsumer;
        }
    }

    static interface Dependencies {
        public CompositeDatabaseAvailabilityGuard globalGuard();
    }
}

