/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphdb;

import java.io.IOException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.LongSupplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.assertj.core.api.Assertions;
import org.eclipse.collections.api.set.ImmutableSet;
import org.glassfish.jersey.internal.guava.Iterators;
import org.junit.jupiter.api.Test;
import org.neo4j.collection.Dependencies;
import org.neo4j.common.DependencyResolver;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.helpers.DatabaseReadOnlyChecker;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.ConstraintViolationException;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.factory.module.id.IdContextFactory;
import org.neo4j.graphdb.factory.module.id.IdContextFactoryBuilder;
import org.neo4j.index.internal.gbptree.RecoveryCleanupWorkCollector;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.id.DefaultIdGeneratorFactory;
import org.neo4j.internal.id.IdGenerator;
import org.neo4j.internal.id.IdType;
import org.neo4j.internal.kernel.api.NodeCursor;
import org.neo4j.internal.kernel.api.PropertyCursor;
import org.neo4j.internal.kernel.api.TokenSet;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.PageSwapperFactory;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.io.pagecache.impl.SingleFilePageSwapperFactory;
import org.neo4j.io.pagecache.impl.muninn.MuninnPageCache;
import org.neo4j.io.pagecache.tracing.PageCacheTracer;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.kernel.impl.scheduler.JobSchedulerFactory;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.kernel.lifecycle.Lifespan;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.scheduler.JobScheduler;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.ImpermanentDbmsExtension;
import org.neo4j.test.extension.Inject;
import org.neo4j.util.concurrent.BinaryLatch;

@ImpermanentDbmsExtension
class LabelsAcceptanceTest {
    @Inject
    private GraphDatabaseAPI db;

    LabelsAcceptanceTest() {
    }

    @Test
    void shouldInsertLabelsWithoutDuplicatingThem() {
        Node node;
        try (Transaction tx = this.db.beginTx();){
            node = tx.createNode();
            node.addLabel((Label)Labels.MY_LABEL);
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.getNodeById(node.getId()).addLabel((Label)Labels.MY_LABEL);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            tx.getNodeById(node.getId()).addLabel(Label.label((String)"BAZQUX"));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            Node labeledNode = tx.getNodeById(node.getId());
            for (Label label : labeledNode.getLabels()) {
                labeledNode.removeLabel(label);
            }
            labeledNode.addLabel(Label.label((String)"BAZQUX"));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        ArrayList<Label> labels = new ArrayList<Label>();
        try (Transaction tx = this.db.beginTx();){
            Node labeledNode = tx.getNodeById(node.getId());
            for (Label label : labeledNode.getLabels()) {
                labels.add(label);
            }
            tx.commit();
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)labels.size(), (String)((Object)labels).toString());
        org.junit.jupiter.api.Assertions.assertEquals((Object)"BAZQUX", (Object)((Label)labels.get(0)).name());
    }

    @Test
    void addingALabelUsingAValidIdentifierShouldSucceed() {
        Node myNode;
        try (Transaction tx = this.db.beginTx();){
            myNode = tx.createNode();
            myNode.addLabel((Label)Labels.MY_LABEL);
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            Node node = transaction.getNodeById(myNode.getId());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)node.hasLabel((Label)Labels.MY_LABEL));
        }
    }

    @Test
    void addingALabelUsingAnInvalidIdentifierShouldFail() {
        try (Transaction tx = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> tx.createNode().addLabel(Label.label((String)"")));
        }
        tx = this.db.beginTx();
        try {
            org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> tx.createNode().addLabel(() -> null));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void addingALabelThatAlreadyExistsBehavesAsNoOp() {
        Node myNode;
        try (Transaction tx = this.db.beginTx();){
            myNode = tx.createNode();
            myNode.addLabel((Label)Labels.MY_LABEL);
            myNode.addLabel((Label)Labels.MY_LABEL);
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            Node node = transaction.getNodeById(myNode.getId());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)node.hasLabel((Label)Labels.MY_LABEL));
        }
    }

    @Test
    void oversteppingMaxNumberOfLabelsShouldFailGracefully() throws IOException {
        JobScheduler scheduler = JobSchedulerFactory.createScheduler();
        try (EphemeralFileSystemAbstraction fileSystem = new EphemeralFileSystemAbstraction();
             Lifespan lifespan = new Lifespan(new Lifecycle[]{scheduler});
             MuninnPageCache pageCache = new MuninnPageCache(LabelsAcceptanceTest.swapper(fileSystem), scheduler, MuninnPageCache.config((int)1000));){
            Dependencies dependencies = new Dependencies();
            dependencies.satisfyDependencies(new Object[]{this.createIdContextFactoryWithMaxedOutLabelTokenIds((FileSystemAbstraction)fileSystem, scheduler)});
            DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder().setFileSystem((FileSystemAbstraction)fileSystem).noOpSystemGraphInitializer().setExternalDependencies((DependencyResolver)dependencies).impermanent().build();
            GraphDatabaseService graphDatabase = managementService.database("neo4j");
            try (Transaction tx = graphDatabase.beginTx();){
                org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> tx.createNode().addLabel((Label)Labels.MY_LABEL));
            }
            managementService.shutdown();
        }
    }

    @Test
    void removingCommittedLabel() {
        Labels label = Labels.MY_LABEL;
        Node myNode = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, label);
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(myNode.getId()).removeLabel((Label)label);
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            Node node = transaction.getNodeById(myNode.getId());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)node.hasLabel((Label)label));
        }
    }

    @Test
    void createNodeWithLabels() {
        Node node;
        try (Transaction tx = this.db.beginTx();){
            node = tx.createNode((Label[])Labels.values());
            tx.commit();
        }
        Set names = Stream.of(Labels.values()).map(Enum::name).collect(Collectors.toSet());
        try (Transaction transaction = this.db.beginTx();){
            Node n = transaction.getNodeById(node.getId());
            for (String labelName : names) {
                org.junit.jupiter.api.Assertions.assertTrue((boolean)n.hasLabel(Label.label((String)labelName)));
            }
        }
    }

    @Test
    void removingNonExistentLabel() {
        Node myNode;
        Labels label = Labels.MY_LABEL;
        try (Transaction tx = this.db.beginTx();){
            myNode = tx.createNode();
            myNode.removeLabel((Label)label);
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            Node node = transaction.getNodeById(myNode.getId());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)node.hasLabel((Label)label));
        }
    }

    @Test
    void removingExistingLabelFromUnlabeledNode() {
        Labels label = Labels.MY_LABEL;
        LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, label);
        Node myNode = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, new Label[0]);
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(myNode.getId()).removeLabel((Label)label);
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            Node node = transaction.getNodeById(myNode.getId());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)node.hasLabel((Label)label));
        }
    }

    @Test
    void removingUncommittedLabel() {
        Labels label = Labels.MY_LABEL;
        try (Transaction tx = this.db.beginTx();){
            Node myNode = tx.createNode();
            myNode.addLabel((Label)label);
            myNode.removeLabel((Label)label);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)myNode.hasLabel((Label)label));
            tx.commit();
        }
    }

    @Test
    void shouldBeAbleToListLabelsForANode() {
        Node node;
        Set expected = org.neo4j.internal.helpers.collection.Iterators.asSet((Object[])new String[]{Labels.MY_LABEL.name(), Labels.MY_OTHER_LABEL.name()});
        try (Transaction tx = this.db.beginTx();){
            node = tx.createNode();
            for (String label : expected) {
                node.addLabel(Label.label((String)label));
            }
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            Node n = transaction.getNodeById(node.getId());
            for (String label : expected) {
                org.junit.jupiter.api.Assertions.assertTrue((boolean)n.hasLabel(Label.label((String)label)));
            }
        }
    }

    @Test
    void shouldReturnEmptyListIfNoLabels() {
        Node node = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, new Label[0]);
        try (Transaction transaction = this.db.beginTx();){
            Node n = transaction.getNodeById(node.getId());
            org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)Iterables.count((Iterable)n.getLabels()));
        }
    }

    @Test
    void getNodesWithLabelCommitted() {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            node.addLabel((Label)Labels.MY_LABEL);
            tx.commit();
        }
        try (Transaction transaction = this.db.beginTx();){
            org.junit.jupiter.api.Assertions.assertTrue((Iterators.size((Iterator)transaction.findNodes((Label)Labels.MY_LABEL)) > 0 ? 1 : 0) != 0);
            org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)Iterators.size((Iterator)transaction.findNodes((Label)Labels.MY_OTHER_LABEL)));
        }
    }

    @Test
    void getNodesWithLabelsWithTxAddsAndRemoves() {
        Set nodesWithMyOtherLabel;
        Set nodesWithMyLabel;
        Node node3;
        Node node1 = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, Labels.MY_LABEL, Labels.MY_OTHER_LABEL);
        Node node2 = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, Labels.MY_LABEL, Labels.MY_OTHER_LABEL);
        try (Transaction tx = this.db.beginTx();){
            node3 = tx.createNode(new Label[]{Labels.MY_LABEL});
            tx.getNodeById(node2.getId()).removeLabel((Label)Labels.MY_LABEL);
            nodesWithMyLabel = org.neo4j.internal.helpers.collection.Iterators.asSet((Iterator)tx.findNodes((Label)Labels.MY_LABEL));
            nodesWithMyOtherLabel = org.neo4j.internal.helpers.collection.Iterators.asSet((Iterator)tx.findNodes((Label)Labels.MY_OTHER_LABEL));
            tx.commit();
        }
        org.junit.jupiter.api.Assertions.assertEquals((Object)org.neo4j.internal.helpers.collection.Iterators.asSet((Object[])new Node[]{node1, node3}), (Object)nodesWithMyLabel);
        org.junit.jupiter.api.Assertions.assertEquals((Object)org.neo4j.internal.helpers.collection.Iterators.asSet((Object[])new Node[]{node1, node2}), (Object)nodesWithMyOtherLabel);
    }

    @Test
    void shouldListAllExistingLabels() {
        List labels;
        LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, Labels.MY_LABEL, Labels.MY_OTHER_LABEL);
        try (Transaction transaction = this.db.beginTx();){
            labels = Iterables.asList((Iterable)transaction.getAllLabels());
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)labels.size());
        Assertions.assertThat((Iterable)Iterables.map(Label::name, (Iterable)labels)).contains((Object[])new String[]{Labels.MY_LABEL.name(), Labels.MY_OTHER_LABEL.name()});
    }

    @Test
    void shouldListAllLabelsInUse() {
        List labels;
        LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, Labels.MY_LABEL);
        Node node = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, Labels.MY_OTHER_LABEL);
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(node.getId()).delete();
            tx.commit();
        }
        try (Transaction tx = this.db.beginTx();){
            labels = Iterables.asList((Iterable)tx.getAllLabelsInUse());
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)labels.size());
        Assertions.assertThat((Iterable)Iterables.map(Label::name, (Iterable)labels)).contains((Object[])new String[]{Labels.MY_LABEL.name()});
    }

    @Test
    void shouldListAllLabelsInUseEvenWhenExclusiveLabelLocksAreTaken() {
        org.junit.jupiter.api.Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(30L), () -> {
            List labels;
            LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, Labels.MY_LABEL);
            Node node = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, Labels.MY_OTHER_LABEL);
            try (Transaction tx = this.db.beginTx();){
                tx.getNodeById(node.getId()).delete();
                tx.commit();
            }
            BinaryLatch indexCreateStarted = new BinaryLatch();
            BinaryLatch indexCreateAllowToFinish = new BinaryLatch();
            Thread indexCreator = new Thread(() -> {
                try (Transaction tx = this.db.beginTx();){
                    tx.schema().indexFor((Label)Labels.MY_LABEL).on("prop").create();
                    indexCreateStarted.release();
                    indexCreateAllowToFinish.await();
                    tx.commit();
                }
            });
            indexCreator.start();
            indexCreateStarted.await();
            try (Transaction tx = this.db.beginTx();){
                labels = Iterables.asList((Iterable)tx.getAllLabelsInUse());
            }
            indexCreateAllowToFinish.release();
            indexCreator.join();
            org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)labels.size());
            Assertions.assertThat((Iterable)Iterables.map(Label::name, (Iterable)labels)).contains((Object[])new String[]{Labels.MY_LABEL.name()});
        });
    }

    @Test
    void shouldListAllRelationshipTypesInUseEvenWhenExclusiveRelationshipTypeLocksAreTaken() {
        org.junit.jupiter.api.Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(30L), () -> {
            List relTypes;
            RelationshipType relType = RelationshipType.withName((String)"REL");
            Node node = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, Labels.MY_LABEL);
            try (Transaction tx = this.db.beginTx();){
                tx.getNodeById(node.getId()).createRelationshipTo(node, relType).setProperty("prop", (Object)"val");
                tx.commit();
            }
            BinaryLatch indexCreateStarted = new BinaryLatch();
            BinaryLatch indexCreateAllowToFinish = new BinaryLatch();
            Thread indexCreator = new Thread(() -> {
                try (Transaction tx = this.db.beginTx();){
                    tx.execute("CREATE FULLTEXT INDEX myIndex FOR ()-[r:REL]-() ON EACH [r.prop]").close();
                    indexCreateStarted.release();
                    indexCreateAllowToFinish.await();
                    tx.commit();
                }
            });
            indexCreator.start();
            indexCreateStarted.await();
            try (Transaction transaction = this.db.beginTx();){
                relTypes = Iterables.asList((Iterable)transaction.getAllRelationshipTypesInUse());
            }
            indexCreateAllowToFinish.release();
            indexCreator.join();
            org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)relTypes.size());
            Assertions.assertThat((Iterable)Iterables.map(RelationshipType::name, (Iterable)relTypes)).contains((Object[])new String[]{relType.name()});
        });
    }

    @Test
    void deleteAllNodesAndTheirLabels() {
        Label label = Label.label((String)"A");
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            node.addLabel(label);
            node.setProperty("name", (Object)"bla");
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            for (Node node : tx.getAllNodes()) {
                node.removeLabel(label);
                node.delete();
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)Iterables.count((Iterable)tx.getAllNodes()));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void removingLabelDoesNotBreakPreviouslyCreatedLabelsIterator() {
        Label label1 = Label.label((String)"A");
        Label label2 = Label.label((String)"B");
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode(new Label[]{label1, label2});
            for (Label next : node.getLabels()) {
                node.removeLabel(next);
            }
            tx.commit();
        }
    }

    @Test
    void removingPropertyDoesNotBreakPreviouslyCreatedNodePropertyKeysIterator() {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            node.setProperty("name", (Object)"Horst");
            node.setProperty("age", (Object)"72");
            for (String key : node.getPropertyKeys()) {
                node.removeProperty(key);
            }
            tx.commit();
        }
    }

    @Test
    void shouldCreateNodeWithLotsOfLabelsAndThenRemoveMostOfThem() {
        Node node;
        int TOTAL_NUMBER_OF_LABELS = 200;
        int NUMBER_OF_PRESERVED_LABELS = 20;
        try (Transaction tx = this.db.beginTx();){
            node = tx.createNode();
            for (int i = 0; i < 200; ++i) {
                node.addLabel(Label.label((String)("label:" + i)));
            }
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            Node labeledNode = tx.getNodeById(node.getId());
            for (int i = 20; i < 200; ++i) {
                labeledNode.removeLabel(Label.label((String)("label:" + i)));
            }
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
        tx = this.db.beginTx();
        try {
            ArrayList<String> labels = new ArrayList<String>();
            Node labeledNode = tx.getNodeById(node.getId());
            for (Label label : labeledNode.getLabels()) {
                labels.add(label.name());
            }
            org.junit.jupiter.api.Assertions.assertEquals((int)20, (int)labels.size(), (String)("labels on node: " + labels));
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldAllowManyLabelsAndPropertyCursor() {
        Node node;
        int propertyCount = 10;
        int labelCount = 15;
        try (Transaction tx = this.db.beginTx();){
            int i;
            node = tx.createNode();
            for (i = 0; i < propertyCount; ++i) {
                node.setProperty("foo" + i, (Object)"bar");
            }
            for (i = 0; i < labelCount; ++i) {
                node.addLabel(Label.label((String)("label" + i)));
            }
            tx.commit();
        }
        HashSet<Integer> seenProperties = new HashSet<Integer>();
        HashSet<Integer> seenLabels = new HashSet<Integer>();
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            try (NodeCursor nodes = ktx.cursors().allocateNodeCursor(CursorContext.NULL);
                 PropertyCursor propertyCursor = ktx.cursors().allocatePropertyCursor(CursorContext.NULL, (MemoryTracker)EmptyMemoryTracker.INSTANCE);){
                ktx.dataRead().singleNode(node.getId(), nodes);
                while (nodes.next()) {
                    nodes.properties(propertyCursor);
                    while (propertyCursor.next()) {
                        seenProperties.add(propertyCursor.propertyKey());
                    }
                    TokenSet labels = nodes.labels();
                    for (int i = 0; i < labels.numberOfTokens(); ++i) {
                        seenLabels.add(labels.token(i));
                    }
                }
            }
            tx.commit();
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)propertyCount, (int)seenProperties.size());
        org.junit.jupiter.api.Assertions.assertEquals((int)labelCount, (int)seenLabels.size());
    }

    @Test
    void nodeWithManyLabels() {
        int labels = 500;
        int halveLabels = labels / 2;
        long nodeId = LabelsAcceptanceTest.createNode((GraphDatabaseService)this.db, new Label[0]).getId();
        this.addLabels(nodeId, 0, halveLabels);
        this.addLabels(nodeId, halveLabels, halveLabels);
        this.verifyLabels(nodeId, 0, labels);
        this.removeLabels(nodeId, halveLabels, halveLabels);
        this.verifyLabels(nodeId, 0, halveLabels);
        this.removeLabels(nodeId, 0, halveLabels - 2);
        this.verifyLabels(nodeId, halveLabels - 2, 2);
    }

    private void addLabels(long nodeId, int startLabelIndex, int count) {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.getNodeById(nodeId);
            int endLabelIndex = startLabelIndex + count;
            for (int i = startLabelIndex; i < endLabelIndex; ++i) {
                node.addLabel(LabelsAcceptanceTest.labelWithIndex(i));
            }
            tx.commit();
        }
    }

    private void verifyLabels(long nodeId, int startLabelIndex, int count) {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.getNodeById(nodeId);
            Set labelNames = Iterables.asList((Iterable)node.getLabels()).stream().map(Label::name).collect(Collectors.toSet());
            org.junit.jupiter.api.Assertions.assertEquals((int)count, (int)labelNames.size());
            int endLabelIndex = startLabelIndex + count;
            for (int i = startLabelIndex; i < endLabelIndex; ++i) {
                org.junit.jupiter.api.Assertions.assertTrue((boolean)labelNames.contains(LabelsAcceptanceTest.labelName(i)));
            }
            tx.commit();
        }
    }

    private void removeLabels(long nodeId, int startLabelIndex, int count) {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.getNodeById(nodeId);
            int endLabelIndex = startLabelIndex + count;
            for (int i = startLabelIndex; i < endLabelIndex; ++i) {
                node.removeLabel(LabelsAcceptanceTest.labelWithIndex(i));
            }
            tx.commit();
        }
    }

    private static Label labelWithIndex(int index) {
        return Label.label((String)LabelsAcceptanceTest.labelName(index));
    }

    private static String labelName(int index) {
        return "Label-" + index;
    }

    private static Node createNode(GraphDatabaseService db, Label ... labels) {
        try (Transaction tx = db.beginTx();){
            Node node = tx.createNode(labels);
            tx.commit();
            Node node2 = node;
            return node2;
        }
    }

    private IdContextFactory createIdContextFactoryWithMaxedOutLabelTokenIds(FileSystemAbstraction fileSystem, JobScheduler jobScheduler) {
        return IdContextFactoryBuilder.of((FileSystemAbstraction)fileSystem, (JobScheduler)jobScheduler, (Config)Config.defaults(), (PageCacheTracer)PageCacheTracer.NULL).withIdGenerationFactoryProvider(any -> new DefaultIdGeneratorFactory(fileSystem, RecoveryCleanupWorkCollector.immediate(), this.db.databaseName()){

            public IdGenerator open(PageCache pageCache, Path fileName, IdType idType, LongSupplier highId, long maxId, DatabaseReadOnlyChecker readOnlyChecker, Config config, CursorContext cursorContext, ImmutableSet<OpenOption> openOptions) throws IOException {
                return super.open(pageCache, fileName, idType, highId, this.maxId(idType, maxId, highId), readOnlyChecker, config, cursorContext, openOptions);
            }

            public IdGenerator create(PageCache pageCache, Path fileName, IdType idType, long highId, boolean throwIfFileExists, long maxId, DatabaseReadOnlyChecker readOnlyChecker, Config config, CursorContext cursorContext, ImmutableSet<OpenOption> openOptions) throws IOException {
                return super.create(pageCache, fileName, idType, highId, throwIfFileExists, this.maxId(idType, maxId, () -> highId), readOnlyChecker, config, cursorContext, openOptions);
            }

            private long maxId(IdType idType, long maxId, LongSupplier highId) {
                return idType == IdType.LABEL_TOKEN ? highId.getAsLong() - 1L : maxId;
            }
        }).build();
    }

    private static PageSwapperFactory swapper(EphemeralFileSystemAbstraction fileSystem) {
        return new SingleFilePageSwapperFactory((FileSystemAbstraction)fileSystem);
    }

    private static enum Labels implements Label
    {
        MY_LABEL,
        MY_OTHER_LABEL;

    }
}

