/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.graphdb;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.DatabaseShutdownException;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.TransactionTerminatedException;
import org.neo4j.kernel.DeadlockDetectedException;
import org.neo4j.kernel.availability.DatabaseAvailability;
import org.neo4j.kernel.impl.MyRelTypes;
import org.neo4j.test.Barrier;
import org.neo4j.test.OtherThreadExecutor;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.ExtensionCallback;
import org.neo4j.test.extension.ImpermanentDbmsExtension;
import org.neo4j.test.extension.Inject;

@ImpermanentDbmsExtension(configurationCallback="configure")
public class GraphDatabaseServiceTest {
    private final OtherThreadExecutor t2 = new OtherThreadExecutor("T2-" + this.getClass().getName());
    private final OtherThreadExecutor t3 = new OtherThreadExecutor("T3-" + this.getClass().getName());
    @Inject
    private DatabaseManagementService managementService;
    @Inject
    private GraphDatabaseService database;

    @ExtensionCallback
    void configure(TestDatabaseManagementServiceBuilder builder) {
        builder.setConfig(GraphDatabaseSettings.shutdown_transaction_end_timeout, (Object)Duration.ofSeconds(10L));
    }

    @AfterEach
    public void tearDown() {
        this.t2.close();
        this.t3.close();
    }

    @Test
    void givenShutdownDatabaseWhenBeginTxThenExceptionIsThrown() {
        this.managementService.shutdown();
        org.junit.jupiter.api.Assertions.assertThrows(DatabaseShutdownException.class, () -> this.database.beginTx());
    }

    @Test
    void givenDatabaseAndStartedTxWhenShutdownThenWaitForTxToFinish() throws Exception {
        Barrier.Control barrier = new Barrier.Control();
        Future txFuture = this.t2.executeDontWait(() -> {
            try (Transaction tx = this.database.beginTx();){
                barrier.reached();
                tx.createNode();
                tx.commit();
            }
            return null;
        });
        barrier.await();
        Future shutdownFuture = this.t3.executeDontWait(() -> {
            this.managementService.shutdown();
            return null;
        });
        this.t3.waitUntilWaiting(location -> location.isAt(DatabaseAvailability.class, "stop"));
        barrier.release();
        org.junit.jupiter.api.Assertions.assertDoesNotThrow(txFuture::get);
        shutdownFuture.get();
    }

    @Test
    void terminateTransactionThrowsExceptionOnNextOperation() {
        try (Transaction tx = this.database.beginTx();){
            tx.terminate();
            org.junit.jupiter.api.Assertions.assertThrows(TransactionTerminatedException.class, () -> ((Transaction)tx).createNode());
        }
    }

    @Test
    void givenDatabaseAndStartedTxWhenShutdownAndStartNewTxThenBeginTxTimesOut() throws Exception {
        Barrier.Control barrier = new Barrier.Control();
        this.t2.executeDontWait(() -> {
            try (Transaction tx = this.database.beginTx();){
                barrier.reached();
            }
            return null;
        });
        barrier.await();
        Future shutdownFuture = this.t3.executeDontWait(() -> {
            this.managementService.shutdown();
            return null;
        });
        this.t3.waitUntilWaiting(location -> location.isAt(DatabaseAvailability.class, "stop"));
        barrier.release();
        shutdownFuture.get();
        org.junit.jupiter.api.Assertions.assertThrows(DatabaseShutdownException.class, () -> ((GraphDatabaseService)this.database).beginTx());
    }

    @Test
    void shouldLetDetectedDeadlocksDuringCommitBeThrownInTheirOriginalForm() throws Exception {
        Node n1 = GraphDatabaseServiceTest.createNode(this.database);
        Node n2 = GraphDatabaseServiceTest.createNode(this.database);
        Relationship r3 = GraphDatabaseServiceTest.createRelationship(this.database, n1);
        Relationship r2 = GraphDatabaseServiceTest.createRelationship(this.database, n1);
        Relationship r1 = GraphDatabaseServiceTest.createRelationship(this.database, n1);
        List<Node> emptyNodes = List.of(GraphDatabaseServiceTest.createNode(this.database), GraphDatabaseServiceTest.createNode(this.database), GraphDatabaseServiceTest.createNode(this.database), GraphDatabaseServiceTest.createNode(this.database), GraphDatabaseServiceTest.createNode(this.database), GraphDatabaseServiceTest.createNode(this.database), GraphDatabaseServiceTest.createNode(this.database), GraphDatabaseServiceTest.createNode(this.database));
        Transaction t1Tx = this.database.beginTx();
        Transaction t2Tx = (Transaction)this.t2.executeDontWait(GraphDatabaseServiceTest.beginTx(this.database)).get();
        t1Tx.getNodeById(n2.getId()).setProperty("locked", (Object)"indeed");
        this.t2.executeDontWait(GraphDatabaseServiceTest.setProperty((Entity)t2Tx.getRelationshipById(r1.getId()), "locked", "absolutely")).get();
        for (Node emptyNode : emptyNodes) {
            this.t2.executeDontWait(GraphDatabaseServiceTest.setProperty((Entity)t2Tx.getNodeById(emptyNode.getId()), "locked", "absolutely")).get();
        }
        Future t2n2Wait = this.t2.executeDontWait(GraphDatabaseServiceTest.setProperty((Entity)t2Tx.getNodeById(n2.getId()), "locked", "In my dreams"));
        this.t2.waitUntilWaiting();
        t1Tx.getRelationshipById(r2.getId()).delete();
        org.junit.jupiter.api.Assertions.assertThrows(DeadlockDetectedException.class, () -> ((Transaction)t1Tx).commit());
        t2n2Wait.get();
        this.t2.executeDontWait(GraphDatabaseServiceTest.close(t2Tx)).get();
    }

    @Test
    void terminationOfClosedTransactionDoesNotInfluenceNextTransaction() {
        Transaction transaction;
        try (Transaction tx = this.database.beginTx();){
            tx.createNode();
            tx.commit();
        }
        try (Transaction tx = transaction = this.database.beginTx();){
            tx.createNode();
            tx.commit();
        }
        transaction.terminate();
        tx = this.database.beginTx();
        try {
            Assertions.assertThat((Iterable)tx.getAllNodes()).hasSize(2);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private static Callable<Transaction> beginTx(GraphDatabaseService db) {
        return () -> ((GraphDatabaseService)db).beginTx();
    }

    private static Callable<Void> setProperty(Entity entity, String key, String value) {
        return () -> {
            entity.setProperty(key, (Object)value);
            return null;
        };
    }

    private static Callable<Void> close(Transaction tx) {
        return () -> {
            tx.close();
            return null;
        };
    }

    private static Relationship createRelationship(GraphDatabaseService db, Node node) {
        try (Transaction tx = db.beginTx();){
            Relationship rel = tx.getNodeById(node.getId()).createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
            tx.commit();
            Relationship relationship = rel;
            return relationship;
        }
    }

    private static Node createNode(GraphDatabaseService db) {
        try (Transaction tx = db.beginTx();){
            Node node = tx.createNode();
            tx.commit();
            Node node2 = node;
            return node2;
        }
    }
}

