/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.IOException;
import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.TimeUnit;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.EnumSource;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.internal.kernel.api.security.LoginContext;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.kernel.api.Kernel;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.impl.transaction.log.checkpoint.CheckPointer;
import org.neo4j.kernel.impl.transaction.log.checkpoint.CheckPointerImpl;
import org.neo4j.kernel.impl.transaction.log.checkpoint.SimpleTriggerInfo;
import org.neo4j.kernel.impl.transaction.log.checkpoint.TriggerInfo;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.EphemeralFileSystemExtension;
import org.neo4j.test.extension.Inject;
import org.neo4j.token.TokenHolders;
import org.neo4j.token.api.TokenHolder;

@ExtendWith(value={EphemeralFileSystemExtension.class})
class TestRecoveryScenarios {
    @Inject
    private EphemeralFileSystemAbstraction fs;
    private final Label label = Label.label((String)"label");
    private GraphDatabaseAPI db;
    private DatabaseManagementService managementService;

    TestRecoveryScenarios() {
    }

    @BeforeEach
    void before() {
        this.managementService = TestRecoveryScenarios.databaseFactory((FileSystemAbstraction)this.fs).impermanent().build();
        this.db = (GraphDatabaseAPI)this.managementService.database("neo4j");
    }

    @AfterEach
    void after() {
        this.managementService.shutdown();
    }

    @ParameterizedTest
    @EnumSource(value=FlushStrategy.class)
    void shouldRecoverTransactionWhereNodeIsDeletedInTheFuture(FlushStrategy strategy) throws Exception {
        Node node = this.createNodeWithProperty("key", "value", this.label);
        this.checkPoint();
        this.setProperty(node, "other-key", 1);
        this.deleteNode(node);
        strategy.flush(this.db);
        this.crashAndRestart();
        try (Transaction tx = this.db.beginTx();){
            node = tx.getNodeById(node.getId());
            tx.commit();
            Assertions.fail((String)"Should not exist");
        }
        catch (NotFoundException e) {
            Assertions.assertEquals((Object)("Node " + node.getId() + " not found"), (Object)e.getMessage());
        }
    }

    @ParameterizedTest
    @EnumSource(value=FlushStrategy.class)
    void shouldRecoverTransactionWherePropertyIsRemovedInTheFuture(FlushStrategy strategy) throws Exception {
        this.createIndex(this.label, "key");
        Node node = this.createNodeWithProperty("key", "value", new Label[0]);
        this.checkPoint();
        this.addLabel(node, this.label);
        this.removeProperty(node, "key");
        strategy.flush(this.db);
        this.crashAndRestart();
        try (Transaction tx = this.db.beginTx();){
            Assertions.assertEquals(Collections.emptyList(), (Object)Iterators.asList((Iterator)tx.findNodes(this.label, "key", (Object)"value")), (String)"Updates not propagated correctly during recovery");
            tx.commit();
        }
    }

    @ParameterizedTest
    @EnumSource(value=FlushStrategy.class)
    void shouldRecoverTransactionWhereManyLabelsAreRemovedInTheFuture(FlushStrategy strategy) throws Exception {
        Node node;
        this.createIndex(this.label, "key");
        Label[] labels = new Label[16];
        for (int i = 0; i < labels.length; ++i) {
            labels[i] = Label.label((String)("Label" + Integer.toHexString(i)));
        }
        try (Transaction tx = this.db.beginTx();){
            node = tx.createNode(labels);
            node.addLabel(this.label);
            tx.commit();
        }
        this.checkPoint();
        this.setProperty(node, "key", "value");
        this.removeLabels(node, labels);
        strategy.flush(this.db);
        this.crashAndRestart();
        tx = this.db.beginTx();
        try {
            Assertions.assertEquals((Object)node, (Object)tx.findNode(this.label, "key", (Object)"value"));
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    @Test
    void shouldRecoverCounts() throws Exception {
        Node node = this.createNode(this.label);
        this.checkPoint();
        this.deleteNode(node);
        this.crashAndRestart();
        Kernel kernel = (Kernel)this.db.getDependencyResolver().resolveDependency(Kernel.class);
        try (KernelTransaction tx = kernel.beginTransaction(KernelTransaction.Type.EXPLICIT, LoginContext.AUTH_DISABLED);){
            Assertions.assertEquals((long)0L, (long)tx.dataRead().countsForNode(-1));
            TokenHolder holder = ((TokenHolders)this.db.getDependencyResolver().resolveDependency(TokenHolders.class)).labelTokens();
            int labelId = holder.getIdByName(this.label.name());
            Assertions.assertEquals((long)0L, (long)tx.dataRead().countsForNode(labelId));
            tx.commit();
        }
    }

    private void removeLabels(Node node, Label ... labels) {
        try (Transaction tx = this.db.beginTx();){
            Node targetNode = tx.getNodeById(node.getId());
            for (Label label : labels) {
                targetNode.removeLabel(label);
            }
            tx.commit();
        }
    }

    private void removeProperty(Node node, String key) {
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(node.getId()).removeProperty(key);
            tx.commit();
        }
    }

    private void addLabel(Node node, Label label) {
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(node.getId()).addLabel(label);
            tx.commit();
        }
    }

    private Node createNode(Label ... labels) {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode(labels);
            tx.commit();
            Node node2 = node;
            return node2;
        }
    }

    private Node createNodeWithProperty(String key, String value, Label ... labels) {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode(labels);
            node.setProperty(key, (Object)value);
            tx.commit();
            Node node2 = node;
            return node2;
        }
    }

    private void createIndex(Label label, String key) {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().indexFor(label).on(key).create();
            tx.commit();
        }
        tx = this.db.beginTx();
        try {
            tx.schema().awaitIndexesOnline(10L, TimeUnit.SECONDS);
            tx.commit();
        }
        finally {
            if (tx != null) {
                tx.close();
            }
        }
    }

    private void checkPoint() throws IOException {
        ((CheckPointer)this.db.getDependencyResolver().resolveDependency(CheckPointer.class)).forceCheckPoint((TriggerInfo)new SimpleTriggerInfo("test"));
    }

    private void deleteNode(Node node) {
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(node.getId()).delete();
            tx.commit();
        }
    }

    private void setProperty(Node node, String key, Object value) {
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(node.getId()).setProperty(key, value);
            tx.commit();
        }
    }

    private static TestDatabaseManagementServiceBuilder databaseFactory(FileSystemAbstraction fs) {
        return new TestDatabaseManagementServiceBuilder().setFileSystem(fs);
    }

    private void crashAndRestart() throws Exception {
        EphemeralFileSystemAbstraction uncleanFs = this.fs.snapshot();
        try {
            this.managementService.shutdown();
        }
        finally {
            this.fs.close();
            this.fs = uncleanFs;
        }
        this.managementService = TestRecoveryScenarios.databaseFactory((FileSystemAbstraction)uncleanFs).impermanent().build();
        this.db = (GraphDatabaseAPI)this.managementService.database("neo4j");
    }

    public static enum FlushStrategy {
        FORCE_EVERYTHING{

            @Override
            void flush(GraphDatabaseAPI db) throws IOException {
                ((CheckPointerImpl.ForceOperation)db.getDependencyResolver().resolveDependency(CheckPointerImpl.ForceOperation.class)).flushAndForce(CursorContext.NULL);
            }
        }
        ,
        FLUSH_PAGE_CACHE{

            @Override
            void flush(GraphDatabaseAPI db) throws IOException {
                ((PageCache)db.getDependencyResolver().resolveDependency(PageCache.class)).flushAndForce();
            }
        };


        abstract void flush(GraphDatabaseAPI var1) throws IOException;
    }
}

