/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.core;

import java.util.Arrays;
import java.util.HashSet;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.graphdb.ConstraintViolationException;
import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.ResourceIterable;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.helpers.collection.PrefetchingIterator;
import org.neo4j.kernel.impl.AbstractNeo4jTestCase;
import org.neo4j.kernel.impl.MyRelTypes;

class TestLoopRelationships
extends AbstractNeo4jTestCase {
    TestLoopRelationships() {
    }

    @Test
    void canCreateRelationshipBetweenTwoNodesWithLoopsThenDeleteOneOfTheNodesAndItsRelationships() {
        Node source = TestLoopRelationships.createNode();
        Node target = TestLoopRelationships.createNode();
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            source = transaction.getNodeById(source.getId());
            target = transaction.getNodeById(target.getId());
            source.createRelationshipTo(source, (RelationshipType)MyRelTypes.TEST);
            target.createRelationshipTo(target, (RelationshipType)MyRelTypes.TEST);
            source.createRelationshipTo(target, (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        transaction = TestLoopRelationships.getGraphDb().beginTx();
        try {
            target = transaction.getNodeById(target.getId());
            Iterables.forEach((Iterable)target.getRelationships(), Entity::delete);
            target.delete();
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void canDeleteNodeAfterDeletingItsRelationshipsIfThoseRelationshipsIncludeLoops() {
        Node node = TestLoopRelationships.createNode();
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            node = transaction.getNodeById(node.getId());
            TestLoopRelationships.txCreateLoop(node);
            TestLoopRelationships.txCreateRel(transaction, node);
            TestLoopRelationships.txCreateLoop(node);
            Iterables.forEach((Iterable)node.getRelationships(), Entity::delete);
            node.delete();
            transaction.commit();
        }
    }

    private static void txCreateRel(Transaction transaction, Node node) {
        node.createRelationshipTo(transaction.createNode(), (RelationshipType)MyRelTypes.TEST);
    }

    private static void txCreateLoop(Node node) {
        node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
    }

    @Test
    void canAddLoopRelationship() {
        Node node = TestLoopRelationships.createNode();
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            node = transaction.getNodeById(node.getId());
            node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        transaction = TestLoopRelationships.getGraphDb().beginTx();
        try {
            node = transaction.getNodeById(node.getId());
            for (Direction dir : Direction.values()) {
                int count = 0;
                try (ResourceIterable relationships = node.getRelationships(dir);){
                    for (Relationship rel : relationships) {
                        org.junit.jupiter.api.Assertions.assertEquals((Object)node, (Object)rel.getStartNode(), (String)"start node");
                        org.junit.jupiter.api.Assertions.assertEquals((Object)node, (Object)rel.getEndNode(), (String)"end node");
                        org.junit.jupiter.api.Assertions.assertEquals((Object)node, (Object)rel.getOtherNode(node), (String)"other node");
                        ++count;
                    }
                }
                org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)count, (String)(dir.name() + " relationship count"));
            }
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void canAddManyLoopRelationships() {
        TestLoopRelationships.testAddManyLoopRelationships(2);
        TestLoopRelationships.testAddManyLoopRelationships(3);
        TestLoopRelationships.testAddManyLoopRelationships(5);
    }

    private static void testAddManyLoopRelationships(int count) {
        for (boolean[] loop : TestLoopRelationships.permutations(count)) {
            Relationship[] relationships = new Relationship[count];
            Node root = TestLoopRelationships.createNode();
            try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
                root = transaction.getNodeById(root.getId());
                for (int i = 0; i < count; ++i) {
                    relationships[i] = loop[i] ? root.createRelationshipTo(root, (RelationshipType)MyRelTypes.TEST) : root.createRelationshipTo(transaction.createNode(), (RelationshipType)MyRelTypes.TEST);
                }
                transaction.commit();
            }
            TestLoopRelationships.verifyRelationships(Arrays.toString(loop), root, loop, relationships);
        }
    }

    @Test
    void canAddLoopRelationshipAndOtherRelationships() {
        TestLoopRelationships.testAddLoopRelationshipAndOtherRelationships(2);
        TestLoopRelationships.testAddLoopRelationshipAndOtherRelationships(3);
        TestLoopRelationships.testAddLoopRelationshipAndOtherRelationships(5);
    }

    private static void testAddLoopRelationshipAndOtherRelationships(int size) {
        for (int i = 0; i < size; ++i) {
            Node root = TestLoopRelationships.createNode();
            Relationship[] relationships = TestLoopRelationships.createRelationships(size, i, root);
            TestLoopRelationships.verifyRelationships(String.format("loop on %s of %s", i, size), root, i, relationships);
        }
    }

    @Test
    void canAddAndRemoveLoopRelationshipAndOtherRelationships() {
        TestLoopRelationships.testAddAndRemoveLoopRelationshipAndOtherRelationships(2);
        TestLoopRelationships.testAddAndRemoveLoopRelationshipAndOtherRelationships(3);
        TestLoopRelationships.testAddAndRemoveLoopRelationshipAndOtherRelationships(5);
    }

    @Test
    void getSingleRelationshipOnNodeWithOneLoopOnly() {
        Relationship singleRelationship;
        Node node = TestLoopRelationships.createNode();
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            node = transaction.getNodeById(node.getId());
            singleRelationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.OUTGOING));
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.BOTH));
            transaction.commit();
        }
        transaction = TestLoopRelationships.getGraphDb().beginTx();
        try {
            node = transaction.getNodeById(node.getId());
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.OUTGOING));
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
            org.junit.jupiter.api.Assertions.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.BOTH));
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    @Test
    void cannotDeleteNodeWithLoopStillAttached() {
        Node node;
        GraphDatabaseService db = TestLoopRelationships.getGraphDb();
        try (Transaction tx = db.beginTx();){
            node = tx.createNode();
            node.createRelationshipTo(node, RelationshipType.withName((String)"MAYOR_OF"));
            tx.commit();
        }
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            transaction.getNodeById(node.getId()).delete();
            ConstraintViolationException e = (ConstraintViolationException)org.junit.jupiter.api.Assertions.assertThrows(ConstraintViolationException.class, () -> ((Transaction)transaction).commit());
            Assertions.assertThat((String)e.getMessage()).isEqualTo("Cannot delete node<" + node.getId() + ">, because it still has relationships. To delete this node, you must first delete its relationships.");
        }
    }

    @Test
    void getOtherNodeFunctionsCorrectly() {
        Relationship relationship;
        Node node = TestLoopRelationships.createNode();
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            node = transaction.getNodeById(node.getId());
            relationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        for (int i = 0; i < 2; ++i) {
            try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
                node = transaction.getNodeById(node.getId());
                Relationship rel = transaction.getRelationshipById(relationship.getId());
                org.junit.jupiter.api.Assertions.assertEquals((Object)node, (Object)rel.getOtherNode(node));
                org.junit.jupiter.api.Assertions.assertEquals(Arrays.asList(node, node), Arrays.asList(rel.getNodes()));
                org.junit.jupiter.api.Assertions.assertThrows(NotFoundException.class, () -> rel.getOtherNode(transaction.createNode()));
                transaction.commit();
                continue;
            }
        }
    }

    @Test
    void getNewlyCreatedLoopRelationshipFromCache() {
        Relationship relationship;
        Node node = TestLoopRelationships.createNode();
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            node = transaction.getNodeById(node.getId());
            node.createRelationshipTo(transaction.createNode(), (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            node = transaction.getNodeById(node.getId());
            relationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
            transaction.commit();
        }
        transaction = TestLoopRelationships.getGraphDb().beginTx();
        try {
            node = transaction.getNodeById(node.getId());
            org.junit.jupiter.api.Assertions.assertEquals((Object)relationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
            transaction.commit();
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    private static void testAddAndRemoveLoopRelationshipAndOtherRelationships(int size) {
        for (boolean[] delete : TestLoopRelationships.permutations(size)) {
            for (int i = 0; i < size; ++i) {
                Node root = TestLoopRelationships.createNode();
                Relationship[] relationships = TestLoopRelationships.createRelationships(size, i, root);
                try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
                    for (int j = 0; j < size; ++j) {
                        if (!delete[j]) continue;
                        transaction.getRelationshipById(relationships[j].getId()).delete();
                        relationships[j] = null;
                    }
                    transaction.commit();
                }
                TestLoopRelationships.verifyRelationships(String.format("loop on %s of %s, delete %s", i, size, Arrays.toString(delete)), root, i, relationships);
            }
        }
    }

    private static Iterable<boolean[]> permutations(final int size) {
        final int max = 1 << size;
        return () -> new PrefetchingIterator<boolean[]>(){
            int pos;

            protected boolean[] fetchNextOrNull() {
                if (this.pos < max) {
                    int cur = this.pos++;
                    boolean[] result = new boolean[size];
                    for (int i = 0; i < size; ++i) {
                        result[i] = (cur & 1) == 1;
                        cur >>= 1;
                    }
                    return result;
                }
                return null;
            }
        };
    }

    private static Relationship[] createRelationships(int count, int loop, Node root) {
        Node[] nodes = new Node[count];
        for (int i = 0; i < count; ++i) {
            nodes[i] = loop == i ? root : TestLoopRelationships.createNode();
        }
        Relationship[] relationships = new Relationship[count];
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            Node node = transaction.getNodeById(root.getId());
            for (int i = 0; i < count; ++i) {
                relationships[i] = node.createRelationshipTo(nodes[i], (RelationshipType)MyRelTypes.TEST);
            }
            transaction.commit();
        }
        return relationships;
    }

    private static void verifyRelationships(String message, Node root, int loop, Relationship ... relationships) {
        boolean[] loops = new boolean[relationships.length];
        for (int i = 0; i < relationships.length; ++i) {
            loops[i] = i == loop;
        }
        TestLoopRelationships.verifyRelationships(message, root, loops, relationships);
    }

    private static void verifyRelationships(String message, Node node, boolean[] loop, Relationship ... relationships) {
        try (Transaction transaction = TestLoopRelationships.getGraphDb().beginTx();){
            Node root = transaction.getNodeById(node.getId());
            for (Direction dir : Direction.values()) {
                HashSet<Relationship> expected = new HashSet<Relationship>();
                for (int i = 0; i < relationships.length; ++i) {
                    if (relationships[i] == null || dir == Direction.INCOMING && !loop[i]) continue;
                    expected.add(relationships[i]);
                }
                try (ResourceIterable rootRels = root.getRelationships(dir);){
                    for (Relationship rel : rootRels) {
                        org.junit.jupiter.api.Assertions.assertTrue((boolean)expected.remove(rel), (String)(message + ": unexpected relationship: " + rel));
                    }
                }
                org.junit.jupiter.api.Assertions.assertTrue((boolean)expected.isEmpty(), (String)(message + ": expected relationships not seen " + expected));
            }
            transaction.commit();
        }
    }
}

