/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.database;

import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.BooleanSupplier;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.dbms.DbmsRuntimeVersionProvider;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.dbms.database.DbmsRuntimeVersion;
import org.neo4j.graphdb.DatabaseShutdownException;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.ResourceIterable;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.event.TransactionData;
import org.neo4j.graphdb.event.TransactionEventListener;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.kernel.KernelVersion;
import org.neo4j.kernel.KernelVersionProvider;
import org.neo4j.kernel.impl.locking.forseti.ForsetiClient;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.internal.event.InternalTransactionEventListener;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.InternalLogProvider;
import org.neo4j.logging.LogAssertions;
import org.neo4j.storageengine.api.TransactionIdStore;
import org.neo4j.test.LatestVersions;
import org.neo4j.test.OtherThreadExecutor;
import org.neo4j.test.Race;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.UpgradeTestUtil;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;
import org.neo4j.util.concurrent.BinaryLatch;

@TestDirectoryExtension
class UpgradeToFutureVersionIT {
    @Inject
    private TestDirectory testDirectory;
    private final AssertableLogProvider logProvider = new AssertableLogProvider();
    private DatabaseManagementService dbms;
    private GraphDatabaseAPI db;
    private GraphDatabaseAPI systemDb;

    UpgradeToFutureVersionIT() {
    }

    @BeforeEach
    void setUp() {
        this.startDbms();
    }

    @AfterEach
    void tearDown() {
        this.shutdownDbms();
    }

    private TestDatabaseManagementServiceBuilder configureGloriousFutureAsLatest(TestDatabaseManagementServiceBuilder builder) {
        return builder.setConfig(GraphDatabaseInternalSettings.latest_runtime_version, (Object)DbmsRuntimeVersion.GLORIOUS_FUTURE.getVersion()).setConfig(GraphDatabaseInternalSettings.latest_kernel_version, (Object)KernelVersion.GLORIOUS_FUTURE.version());
    }

    @Test
    void shouldUpgradeDatabaseAutomaticallyToLatestIfSet() throws Exception {
        long startTransaction = this.lastCommittedTransactionId();
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)LatestVersions.LATEST_KERNEL_VERSION);
        this.shutdownDbms();
        this.startDbms(this::configureGloriousFutureAsLatest, true);
        Assertions.assertThat((Comparable)this.dbmsRuntimeVersion()).isEqualTo((Object)DbmsRuntimeVersion.GLORIOUS_FUTURE);
        this.createReadTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)LatestVersions.LATEST_KERNEL_VERSION);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)KernelVersion.GLORIOUS_FUTURE);
        this.assertUpgradeTransactionInOrder(startTransaction);
    }

    @Test
    void shouldUpgradeDatabaseToLatestVersionOnFirstWriteTransaction() throws Exception {
        this.shutdownDbms();
        this.startDbms(this::configureGloriousFutureAsLatest, false);
        long startTransaction = this.lastCommittedTransactionId();
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)LatestVersions.LATEST_KERNEL_VERSION);
        this.systemDb.executeTransactionally("CALL dbms.upgrade()");
        Assertions.assertThat((Comparable)this.dbmsRuntimeVersion()).isEqualTo((Object)DbmsRuntimeVersion.GLORIOUS_FUTURE);
        this.createReadTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)LatestVersions.LATEST_KERNEL_VERSION);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)KernelVersion.GLORIOUS_FUTURE);
        this.assertUpgradeTransactionInOrder(startTransaction);
    }

    @Test
    void shouldNotBeAbleToReadFutureVersions() {
        this.shutdownDbms();
        this.startDbms(this::configureGloriousFutureAsLatest, true);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)KernelVersion.GLORIOUS_FUTURE);
        this.shutdownDbms();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(this::startDbms).isInstanceOf(RuntimeException.class)).hasRootCauseMessage("Unsupported component state for 'dbms-runtime': The sub-graph is unsupported because it is a newer version, this component cannot function");
        this.startDbms(builder -> builder.setConfig(GraphDatabaseInternalSettings.latest_runtime_version, (Object)DbmsRuntimeVersion.GLORIOUS_FUTURE.getVersion()), false);
        DatabaseShutdownException e = (DatabaseShutdownException)org.junit.jupiter.api.Assertions.assertThrows(DatabaseShutdownException.class, () -> ((GraphDatabaseAPI)this.db).beginTx());
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)e).cause().isInstanceOf(RuntimeException.class)).hasMessageContaining("Checkpoint log file with version 0 has some data available after last readable log entry");
    }

    @Test
    void shouldUpgradeDatabaseToLatestVersionOnFirstWriteTransactionStressTest() throws Throwable {
        this.shutdownDbms();
        this.startDbms(this::configureGloriousFutureAsLatest, false);
        long startTransaction = this.lastCommittedTransactionId();
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)LatestVersions.LATEST_KERNEL_VERSION);
        Assertions.assertThat((Comparable)this.dbmsRuntimeVersion()).isEqualTo((Object)LatestVersions.LATEST_RUNTIME_VERSION);
        Race race = new Race().withRandomStartDelays().withEndCondition(new BooleanSupplier[]{() -> KernelVersion.GLORIOUS_FUTURE.equals((Object)this.kernelVersion())});
        race.addContestant(() -> this.systemDb.executeTransactionally("CALL dbms.upgrade()"), 1);
        race.addContestants(Integer.max(Runtime.getRuntime().availableProcessors() - 1, 2), Race.throwing(() -> {
            this.createWriteTransaction();
            Thread.sleep(ThreadLocalRandom.current().nextInt(0, 2));
        }));
        race.go(1L, TimeUnit.MINUTES);
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)KernelVersion.GLORIOUS_FUTURE);
        Assertions.assertThat((Comparable)this.dbmsRuntimeVersion()).isEqualTo((Object)DbmsRuntimeVersion.GLORIOUS_FUTURE);
        this.assertUpgradeTransactionInOrder(startTransaction);
    }

    @Test
    void shouldNotUpgradePastDbmsRuntime() {
        this.shutdownDbms();
        this.startDbms(this::configureGloriousFutureAsLatest, false);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)LatestVersions.LATEST_KERNEL_VERSION);
    }

    @Test
    void shouldHandleDeadlocksOnUpgradeTransaction() throws Exception {
        this.shutdownDbms();
        this.startDbms(this::configureGloriousFutureAsLatest, false);
        final String lockNode1 = this.createWriteTransaction();
        final String lockNode2 = this.createWriteTransaction();
        final BinaryLatch l1 = new BinaryLatch();
        final BinaryLatch l2 = new BinaryLatch();
        long numNodesBefore = this.getNodeCount();
        this.dbms.registerTransactionEventListener(this.db.databaseName(), (TransactionEventListener)new InternalTransactionEventListener.Adapter<Object>(){

            public Object beforeCommit(TransactionData data, Transaction transaction, GraphDatabaseService databaseService) {
                UpgradeToFutureVersionIT.this.dbms.unregisterTransactionEventListener(UpgradeToFutureVersionIT.this.db.databaseName(), (TransactionEventListener)this);
                l2.release();
                l1.await();
                transaction.acquireWriteLock((Entity)transaction.getNodeByElementId(lockNode2));
                transaction.acquireWriteLock((Entity)transaction.getNodeByElementId(lockNode1));
                return null;
            }
        });
        try (OtherThreadExecutor executor = new OtherThreadExecutor("Executor");){
            Future f1 = executor.executeDontWait(this::createWriteTransaction);
            l2.await();
            this.systemDb.executeTransactionally("CALL dbms.upgrade()");
            try (Transaction tx = this.db.beginTx();){
                tx.acquireWriteLock((Entity)tx.getNodeByElementId(lockNode1));
                tx.createNode();
                l1.release();
                executor.waitUntilWaiting(details -> details.isAt(ForsetiClient.class, "acquireExclusive"));
                tx.commit();
            }
            executor.awaitFuture(f1);
        }
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).containsMessageWithArguments("Upgrade transaction from %s to %s not possible right now due to conflicting transaction, will retry on next write", new Object[]{LatestVersions.LATEST_KERNEL_VERSION, KernelVersion.GLORIOUS_FUTURE}).doesNotContainMessageWithArguments("Upgrade transaction from %s to %s started", new Object[]{LatestVersions.LATEST_KERNEL_VERSION, KernelVersion.GLORIOUS_FUTURE});
        ((AbstractLongAssert)Assertions.assertThat((long)this.getNodeCount()).as("Both transactions succeeded", new Object[0])).isEqualTo(numNodesBefore + 2L);
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)LatestVersions.LATEST_KERNEL_VERSION);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.kernelVersion()).isEqualTo((Object)KernelVersion.GLORIOUS_FUTURE);
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).containsMessageWithArguments("Upgrade transaction from %s to %s started", new Object[]{LatestVersions.LATEST_KERNEL_VERSION, KernelVersion.GLORIOUS_FUTURE}).containsMessageWithArguments("Upgrade transaction from %s to %s completed", new Object[]{LatestVersions.LATEST_KERNEL_VERSION, KernelVersion.GLORIOUS_FUTURE});
    }

    private long getNodeCount() {
        try (Transaction tx = this.db.beginTx();){
            long l = Iterables.count((Iterable)tx.getAllNodes());
            return l;
        }
    }

    private void createReadTransaction() {
        try (Transaction tx = this.db.beginTx();
             ResourceIterable allNodes = tx.getAllNodes();){
            allNodes.forEach(Entity::getAllProperties);
            tx.commit();
        }
    }

    private String createWriteTransaction() {
        try (Transaction tx = this.db.beginTx();){
            String nodeId = tx.createNode().getElementId();
            tx.commit();
            String string = nodeId;
            return string;
        }
    }

    private void startDbms() {
        this.startDbms(builder -> builder, false);
    }

    private void startDbms(Configuration configuration, boolean allowAutomaticUpgrade) {
        this.dbms = configuration.configure(new TestDatabaseManagementServiceBuilder().setDatabaseRootDirectory(this.testDirectory.homePath()).setInternalLogProvider((InternalLogProvider)this.logProvider).setConfig(GraphDatabaseInternalSettings.automatic_upgrade_enabled, (Object)allowAutomaticUpgrade)).build();
        this.db = (GraphDatabaseAPI)this.dbms.database("neo4j");
        this.systemDb = (GraphDatabaseAPI)this.dbms.database("system");
    }

    private void shutdownDbms() {
        if (this.dbms != null) {
            this.dbms.shutdown();
            this.dbms = null;
        }
    }

    private long lastCommittedTransactionId() {
        return this.get(this.db, TransactionIdStore.class).getLastCommittedTransactionId();
    }

    private KernelVersion kernelVersion() {
        return this.get(this.db, KernelVersionProvider.class).kernelVersion();
    }

    private DbmsRuntimeVersion dbmsRuntimeVersion() {
        return this.get(this.systemDb, DbmsRuntimeVersionProvider.class).getVersion();
    }

    private <T> T get(GraphDatabaseAPI db, Class<T> cls) {
        return (T)db.getDependencyResolver().resolveDependency(cls);
    }

    private void assertUpgradeTransactionInOrder(long fromTxId) throws Exception {
        UpgradeTestUtil.assertUpgradeTransactionInOrder((KernelVersion)LatestVersions.LATEST_KERNEL_VERSION, (KernelVersion)KernelVersion.GLORIOUS_FUTURE, (long)fromTxId, (GraphDatabaseAPI)this.db);
    }

    @FunctionalInterface
    static interface Configuration {
        public TestDatabaseManagementServiceBuilder configure(TestDatabaseManagementServiceBuilder var1);
    }
}

