/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.IOException;
import java.nio.file.Path;
import java.util.function.Predicate;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.neo4j.common.DependencyResolver;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.config.Configuration;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.CommonDatabaseStores;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.kernel.database.DatabaseTracers;
import org.neo4j.kernel.impl.transaction.log.CheckpointInfo;
import org.neo4j.kernel.impl.transaction.log.checkpoint.CheckPointer;
import org.neo4j.kernel.impl.transaction.log.checkpoint.SimpleTriggerInfo;
import org.neo4j.kernel.impl.transaction.log.checkpoint.TriggerInfo;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.recovery.RecoveryRequiredChecker;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.storageengine.api.StorageEngineFactory;
import org.neo4j.test.RandomSupport;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.RandomExtension;
import org.neo4j.test.extension.pagecache.PageCacheSupportExtension;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;

@TestDirectoryExtension
@ExtendWith(value={RandomExtension.class})
class RecoveryRequiredCheckerTest {
    @RegisterExtension
    static PageCacheSupportExtension pageCacheExtension = new PageCacheSupportExtension();
    @Inject
    private TestDirectory testDirectory;
    @Inject
    private FileSystemAbstraction fileSystem;
    @Inject
    private RandomSupport random;
    private DatabaseLayout databaseLayout;
    private Path storeDir;
    private StorageEngineFactory storageEngineFactory;

    RecoveryRequiredCheckerTest() {
    }

    @BeforeEach
    void setup() {
        this.storeDir = this.testDirectory.homePath();
    }

    @Test
    void shouldNotWantToRecoverIntactStore() throws Exception {
        this.startStopAndCreateDefaultData();
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker recoverer = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            Assertions.assertThat((boolean)recoverer.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE)).isEqualTo(false);
        }
    }

    @Test
    void shouldWantToRecoverBrokenStore() throws Exception {
        try (EphemeralFileSystemAbstraction ephemeralFs = this.createAndCrashWithDefaultConfig();
             PageCache pageCache = pageCacheExtension.getPageCache((FileSystemAbstraction)ephemeralFs);){
            RecoveryRequiredChecker recoverer = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig((FileSystemAbstraction)ephemeralFs, pageCache, this.storageEngineFactory);
            Assertions.assertThat((boolean)recoverer.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE)).isEqualTo(true);
        }
    }

    @Test
    void shouldBeAbleToRecoverBrokenStore() throws Exception {
        try (EphemeralFileSystemAbstraction ephemeralFs = this.createAndCrashWithDefaultConfig();
             PageCache pageCache = pageCacheExtension.getPageCache((FileSystemAbstraction)ephemeralFs);){
            RecoveryRequiredChecker recoverer = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig((FileSystemAbstraction)ephemeralFs, pageCache, this.storageEngineFactory);
            Assertions.assertThat((boolean)recoverer.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE)).isEqualTo(true);
            RecoveryRequiredCheckerTest.startStopDatabase((FileSystemAbstraction)ephemeralFs, this.storeDir);
            Assertions.assertThat((boolean)recoverer.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE)).isEqualTo(false);
        }
    }

    @Test
    void shouldBeAbleToRecoverBrokenStoreWithLogsInSeparateAbsoluteLocation() throws Exception {
        Path customTransactionLogsLocation = this.testDirectory.directory("transactions");
        Config config = Config.newBuilder().set(GraphDatabaseSettings.neo4j_home, (Object)this.testDirectory.homePath()).set(GraphDatabaseSettings.transaction_logs_root_path, (Object)customTransactionLogsLocation.toAbsolutePath()).build();
        this.recoverBrokenStoreWithConfig(config);
    }

    @Test
    void shouldNotWantToRecoverEmptyStore() throws Exception {
        DatabaseLayout databaseLayout = DatabaseLayout.ofFlat((Path)this.testDirectory.directory("dir-without-store"));
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, StorageEngineFactory.selectStorageEngine((FileSystemAbstraction)this.fileSystem, (DatabaseLayout)databaseLayout, (Configuration)Config.defaults()));
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    @Test
    void shouldWantToRecoverStoreWithoutOneIdFile() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertAllIdFilesExist();
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
            this.fileSystem.deleteFileOrThrow((Path)Iterables.first((Iterable)this.databaseLayout.idFiles()));
            org.junit.jupiter.api.Assertions.assertTrue((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    @Test
    void shouldWantToRecoverStoreWithoutAllIdFiles() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertAllIdFilesExist();
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
            for (Path idFile : this.databaseLayout.idFiles()) {
                this.fileSystem.deleteFileOrThrow(idFile);
            }
            org.junit.jupiter.api.Assertions.assertTrue((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void doNotRequireCheckpointWhenOldestNotCompletedPositionIsEqualToCheckpointedPosition() throws IOException {
        DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(this.testDirectory.directory("test")).build();
        try {
            GraphDatabaseAPI db = (GraphDatabaseAPI)managementService.database("neo4j");
            this.databaseLayout = db.databaseLayout();
            DependencyResolver dependencyResolver = db.getDependencyResolver();
            this.storageEngineFactory = (StorageEngineFactory)dependencyResolver.resolveDependency(StorageEngineFactory.class);
            LogFiles logFiles = (LogFiles)dependencyResolver.resolveDependency(LogFiles.class);
            CheckPointer checkPointer = (CheckPointer)dependencyResolver.resolveDependency(CheckPointer.class);
            try (Transaction tx = db.beginTx();){
                tx.createNode();
                tx.commit();
            }
            checkPointer.forceCheckPoint((TriggerInfo)new SimpleTriggerInfo("Test"));
            CheckpointInfo latestCheckpoint = (CheckpointInfo)logFiles.getCheckpointFile().findLatestCheckpoint().orElseThrow();
            org.junit.jupiter.api.Assertions.assertEquals((Object)latestCheckpoint.transactionLogPosition(), (Object)latestCheckpoint.oldestNotVisibleTransactionLogPosition());
        }
        finally {
            managementService.shutdown();
        }
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    @Test
    void shouldNotWantToRecoveryWhenStoreExistenceFileIsMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
            this.fileSystem.deleteFileOrThrow(this.databaseLayout.pathForExistsMarker());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    @Test
    void recoveryRequiredWhenAnyMandatoryStoreFileIsMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
            Path path = (Path)this.random.among(this.databaseLayout.mandatoryStoreFiles().stream().filter(Predicate.not(this.databaseLayout.pathForExistsMarker()::equals)).toList());
            this.fileSystem.deleteFileOrThrow(path);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    @Test
    void recoveryRequiredWhenSeveralStoreFileAreMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
            this.fileSystem.deleteFileOrThrow(this.databaseLayout.pathForStore(CommonDatabaseStores.COUNTS));
            this.fileSystem.deleteFileOrThrow(this.databaseLayout.pathForStore(CommonDatabaseStores.SCHEMAS));
            this.fileSystem.deleteFileOrThrow(this.databaseLayout.pathForStore(CommonDatabaseStores.RELATIONSHIP_TYPE_TOKENS));
            org.junit.jupiter.api.Assertions.assertTrue((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    @Test
    void recoveryNotRequiredWhenCountStoreIsMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
            this.fileSystem.deleteFileOrThrow(this.databaseLayout.pathForStore(CommonDatabaseStores.COUNTS));
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    @Test
    void recoveryNotRequiredWhenIndexStatisticStoreIsMissing() throws Exception {
        this.startStopAndCreateDefaultData();
        this.assertStoreFilesExist();
        try (PageCache pageCache = pageCacheExtension.getPageCache(this.fileSystem);){
            RecoveryRequiredChecker checker = RecoveryRequiredCheckerTest.getRecoveryCheckerWithDefaultConfig(this.fileSystem, pageCache, this.storageEngineFactory);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
            this.fileSystem.deleteFileOrThrow(this.databaseLayout.pathForStore(CommonDatabaseStores.INDEX_STATISTICS));
            org.junit.jupiter.api.Assertions.assertFalse((boolean)checker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        }
    }

    private void recoverBrokenStoreWithConfig(Config config) throws IOException {
        try (EphemeralFileSystemAbstraction ephemeralFs = this.createSomeDataAndCrash(this.storeDir, config);
             PageCache pageCache = pageCacheExtension.getPageCache((FileSystemAbstraction)ephemeralFs);){
            RecoveryRequiredChecker recoveryChecker = RecoveryRequiredCheckerTest.getRecoveryChecker((FileSystemAbstraction)ephemeralFs, pageCache, this.storageEngineFactory, config);
            Assertions.assertThat((boolean)recoveryChecker.isRecoveryRequiredAt(DatabaseLayout.of((Config)config), (MemoryTracker)EmptyMemoryTracker.INSTANCE)).isEqualTo(true);
            DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(this.storeDir).setFileSystem((FileSystemAbstraction)ephemeralFs).setConfig(config).build();
            managementService.shutdown();
            Assertions.assertThat((boolean)recoveryChecker.isRecoveryRequiredAt(this.databaseLayout, (MemoryTracker)EmptyMemoryTracker.INSTANCE)).isEqualTo(false);
        }
    }

    private EphemeralFileSystemAbstraction createAndCrashWithDefaultConfig() throws IOException {
        return this.createSomeDataAndCrash(this.storeDir, Config.defaults());
    }

    private void assertAllIdFilesExist() {
        for (Path idFile : this.databaseLayout.idFiles()) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)this.fileSystem.fileExists(idFile), (String)("ID file " + idFile + " does not exist"));
        }
    }

    private void assertStoreFilesExist() {
        for (Path file : this.databaseLayout.storeFiles()) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)this.fileSystem.fileExists(file), (String)("Store file " + file + " does not exist"));
        }
    }

    private static RecoveryRequiredChecker getRecoveryCheckerWithDefaultConfig(FileSystemAbstraction fileSystem, PageCache pageCache, StorageEngineFactory storageEngineFactory) {
        return RecoveryRequiredCheckerTest.getRecoveryChecker(fileSystem, pageCache, storageEngineFactory, Config.defaults());
    }

    private static RecoveryRequiredChecker getRecoveryChecker(FileSystemAbstraction fileSystem, PageCache pageCache, StorageEngineFactory storageEngineFactory, Config config) {
        return new RecoveryRequiredChecker(fileSystem, pageCache, config, storageEngineFactory, DatabaseTracers.EMPTY);
    }

    private EphemeralFileSystemAbstraction createSomeDataAndCrash(Path store, Config config) throws IOException {
        try (EphemeralFileSystemAbstraction ephemeralFs = new EphemeralFileSystemAbstraction();){
            DatabaseManagementService managementService = new TestDatabaseManagementServiceBuilder(store).setFileSystem((FileSystemAbstraction)ephemeralFs).setConfig(config).build();
            GraphDatabaseService db = managementService.database("neo4j");
            try (Transaction tx = db.beginTx();){
                tx.createNode();
                tx.commit();
            }
            this.databaseLayout = ((GraphDatabaseAPI)db).databaseLayout();
            this.storageEngineFactory = (StorageEngineFactory)((GraphDatabaseAPI)db).getDependencyResolver().resolveDependency(StorageEngineFactory.class);
            EphemeralFileSystemAbstraction snapshot = ephemeralFs.snapshot();
            managementService.shutdown();
            EphemeralFileSystemAbstraction ephemeralFileSystemAbstraction = snapshot;
            return ephemeralFileSystemAbstraction;
        }
    }

    private static DatabaseManagementService startDatabase(FileSystemAbstraction fileSystem, Path storeDir) {
        return new TestDatabaseManagementServiceBuilder(storeDir).setFileSystem(fileSystem).build();
    }

    private static void startStopDatabase(FileSystemAbstraction fileSystem, Path storeDir) {
        DatabaseManagementService managementService = RecoveryRequiredCheckerTest.startDatabase(fileSystem, storeDir);
        managementService.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void startStopAndCreateDefaultData() {
        DatabaseManagementService managementService = RecoveryRequiredCheckerTest.startDatabase(this.fileSystem, this.storeDir);
        try {
            GraphDatabaseService database = managementService.database("neo4j");
            try (Transaction transaction = database.beginTx();){
                transaction.createNode();
                transaction.commit();
            }
            this.databaseLayout = ((GraphDatabaseAPI)database).databaseLayout();
            this.storageEngineFactory = (StorageEngineFactory)((GraphDatabaseAPI)database).getDependencyResolver().resolveDependency(StorageEngineFactory.class);
        }
        finally {
            managementService.shutdown();
        }
    }
}

