/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.api.index;

import java.lang.invoke.CallSite;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.exceptions.KernelException;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.ResourceIterable;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.kernel.api.IndexMonitor;
import org.neo4j.internal.kernel.api.IndexQueryConstraints;
import org.neo4j.internal.kernel.api.IndexReadSession;
import org.neo4j.internal.kernel.api.NodeValueIndexCursor;
import org.neo4j.internal.kernel.api.PropertyIndexQuery;
import org.neo4j.internal.kernel.api.exceptions.schema.IndexNotFoundKernelException;
import org.neo4j.internal.schema.IndexDescriptor;
import org.neo4j.internal.schema.IndexPrototype;
import org.neo4j.internal.schema.LabelSchemaDescriptor;
import org.neo4j.internal.schema.SchemaDescriptor;
import org.neo4j.internal.schema.SchemaDescriptors;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.api.index.IndexSample;
import org.neo4j.kernel.impl.api.index.UpdatesTracker;
import org.neo4j.kernel.impl.api.index.stats.IndexStatisticsStore;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.monitoring.Monitors;
import org.neo4j.test.Barrier;
import org.neo4j.test.RandomSupport;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.DbmsController;
import org.neo4j.test.extension.DbmsExtension;
import org.neo4j.test.extension.ExtensionCallback;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.RandomExtension;
import org.neo4j.util.concurrent.Futures;
import org.neo4j.values.storable.Values;

@ExtendWith(value={RandomExtension.class})
@DbmsExtension(configurationCallback="configure")
class IndexStatisticsTest {
    private static final String[] NAMES = new String[]{"Andres", "Davide", "Jakub", "Chris", "Tobias", "Stefan", "Petra", "Rickard", "Mattias", "Emil", "Chris", "Chris"};
    private static final double UNIQUE_NAMES = Arrays.stream(NAMES).distinct().count();
    private static final int CREATION_MULTIPLIER = Integer.getInteger(IndexStatisticsTest.class.getName() + ".creationMultiplier", 1000);
    private static final String PERSON_LABEL = "Person";
    private static final String NAME_PROPERTY = "name";
    @Inject
    private DbmsController dbmsController;
    @Inject
    private GraphDatabaseAPI db;
    @Inject
    private RandomSupport random;
    private final IndexOnlineMonitor indexOnlineMonitor = new IndexOnlineMonitor();

    IndexStatisticsTest() {
    }

    @ExtensionCallback
    void configure(TestDatabaseManagementServiceBuilder builder) {
        builder.setConfig(GraphDatabaseSettings.index_background_sampling_enabled, (Object)false);
        builder.setConfig(GraphDatabaseInternalSettings.index_population_print_debug, (Object)true);
        int batchSize = this.random.nextInt(1, 6);
        builder.setConfig(GraphDatabaseInternalSettings.index_population_queue_threshold, (Object)batchSize);
        Monitors monitors = new Monitors();
        monitors.addMonitorListener((Object)this.indexOnlineMonitor, new String[0]);
        builder.setMonitors(monitors);
    }

    @Test
    void shouldProvideIndexStatisticsForDataCreatedWhenPopulationBeforeTheIndexIsOnline() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        org.junit.jupiter.api.Assertions.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)0.0);
        org.junit.jupiter.api.Assertions.assertEquals((long)4L, (long)this.indexSize(index));
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    void shouldUpdateIndexStatisticsForDataCreatedAfterCleanRestart() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        long indexUpdatesBeforeRestart = this.indexUpdates(index);
        this.dbmsController.restartDatabase(this.db.databaseName());
        this.createSomePersons();
        Assertions.assertThat((long)this.indexUpdates(index)).isGreaterThan(indexUpdatesBeforeRestart);
    }

    @Test
    void shouldNotSeeDataCreatedAfterPopulation() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.createSomePersons();
        org.junit.jupiter.api.Assertions.assertEquals((double)1.0, (double)this.indexSelectivity(index), (double)0.0);
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)this.indexSize(index));
        org.junit.jupiter.api.Assertions.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    void shouldProvideIndexStatisticsForDataSeenDuringPopulationAndIgnoreDataCreatedAfterPopulation() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.createSomePersons();
        org.junit.jupiter.api.Assertions.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)0.0);
        org.junit.jupiter.api.Assertions.assertEquals((long)4L, (long)this.indexSize(index));
        org.junit.jupiter.api.Assertions.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    void shouldRemoveIndexStatisticsAfterIndexIsDeleted() throws KernelException {
        this.indexOnlineMonitor.initialize(0);
        this.createSomePersons();
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.dropIndex(index);
        try {
            this.indexSelectivity(index);
            org.junit.jupiter.api.Assertions.fail((String)"Expected IndexNotFoundKernelException to be thrown");
        }
        catch (IndexNotFoundKernelException e) {
            IndexSample sample = this.getIndexingStatisticsStore().indexSample(index.getId());
            org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)sample.uniqueValues());
            org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)sample.sampleSize());
        }
        IndexSample indexSample = this.getIndexingStatisticsStore().indexSample(index.getId());
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)indexSample.indexSize());
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)indexSample.updates());
    }

    @Test
    void shouldProvideIndexSelectivityWhenThereAreManyDuplicates() throws Exception {
        this.indexOnlineMonitor.initialize(0);
        int created = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER).length;
        IndexDescriptor index = this.createPersonNameIndex();
        this.awaitIndexesOnline();
        this.assertCorrectIndexSelectivity(index, created);
        this.assertCorrectIndexSize(created, this.indexSize(index));
        org.junit.jupiter.api.Assertions.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        int initialNodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER).length;
        IndexDescriptor index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreations(CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedUpdates = updatesTracker.createdAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedUpdates, this.indexUpdates(index));
    }

    @Test
    void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndDeletions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexDescriptor index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsAndDeletions(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChanges() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexDescriptor index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsAndUpdates(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.createdAfterPopulation() + updatesTracker.updatedAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChangesAndDeletions() throws Exception {
        this.indexOnlineMonitor.initialize(1);
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexDescriptor index = this.createPersonNameIndex();
        UpdatesTracker updatesTracker = this.executeCreationsDeletionsAndUpdates(nodes, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        this.assertIndexedNodesMatchesStoreNodes(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation() + updatesTracker.updatedAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    void shouldWorkWhileHavingHeavyConcurrentUpdates() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        int threads = 5;
        this.indexOnlineMonitor.initialize(threads);
        ExecutorService executorService = Executors.newFixedThreadPool(threads);
        IndexDescriptor index = this.createPersonNameIndex();
        ArrayList<Callable<UpdatesTracker>> jobs = new ArrayList<Callable<UpdatesTracker>>(threads);
        for (int i = 0; i < threads; ++i) {
            jobs.add(() -> this.executeCreationsDeletionsAndUpdates(nodes, CREATION_MULTIPLIER));
        }
        List futures = executorService.invokeAll(jobs);
        UpdatesTracker result = new UpdatesTracker();
        result.notifyPopulationCompleted();
        for (UpdatesTracker jobTracker : Futures.getAllResults(futures)) {
            result.add(jobTracker);
        }
        this.awaitIndexesOnline();
        executorService.shutdown();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)executorService.awaitTermination(1L, TimeUnit.MINUTES));
        this.assertIndexedNodesMatchesStoreNodes(index);
        int seenWhilePopulating = initialNodes + result.createdDuringPopulation() - result.deletedDuringPopulation();
        this.assertCorrectIndexSelectivity(index, seenWhilePopulating);
        this.assertCorrectIndexSize("Tracker had " + String.valueOf(result), seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = result.deletedAfterPopulation() + result.createdAfterPopulation() + result.updatedAfterPopulation() + Math.toIntExact(this.indexOnlineMonitor.indexSampleOnCompletion.updates());
        IndexStatisticsTest.assertCorrectIndexUpdates("Tracker had " + String.valueOf(result), expectedIndexUpdates, this.indexUpdates(index));
    }

    private void assertIndexedNodesMatchesStoreNodes(IndexDescriptor index) throws Exception {
        int nodesInStore = 0;
        Label label = Label.label((String)PERSON_LABEL);
        try (Transaction transaction = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)transaction).kernelTransaction();
            ArrayList<CallSite> mismatches = new ArrayList<CallSite>();
            int propertyKeyId = ktx.tokenRead().propertyKey(NAME_PROPERTY);
            IndexReadSession indexSession = ktx.dataRead().indexReadSession(index);
            try (NodeValueIndexCursor cursor = ktx.cursors().allocateNodeValueIndexCursor(ktx.cursorContext(), ktx.memoryTracker());
                 ResourceIterable allNodes = transaction.getAllNodes();){
                for (Node node : Iterables.filter(n -> n.hasLabel(label) && n.hasProperty(NAME_PROPERTY), (Iterable)allNodes)) {
                    ++nodesInStore;
                    String name = (String)node.getProperty(NAME_PROPERTY);
                    ktx.dataRead().nodeIndexSeek(ktx.queryContext(), indexSession, cursor, IndexQueryConstraints.unconstrained(), new PropertyIndexQuery[]{PropertyIndexQuery.exact((int)propertyKeyId, (Object)name)});
                    boolean found = false;
                    while (cursor.next()) {
                        long indexedNode = cursor.nodeReference();
                        if (indexedNode != node.getId()) continue;
                        if (found) {
                            mismatches.add((CallSite)((Object)("Index has multiple entries for " + name + " and " + indexedNode)));
                        }
                        found = true;
                    }
                    if (found) continue;
                    mismatches.add((CallSite)((Object)("Index is missing entry for " + name + " " + String.valueOf(node))));
                }
                if (!mismatches.isEmpty()) {
                    org.junit.jupiter.api.Assertions.fail((String)String.join((CharSequence)String.format("%n", new Object[0]), mismatches));
                }
                ktx.dataRead().nodeIndexSeek(ktx.queryContext(), indexSession, cursor, IndexQueryConstraints.unconstrained(), new PropertyIndexQuery[]{PropertyIndexQuery.exists((int)propertyKeyId)});
                int nodesInIndex = 0;
                while (cursor.next()) {
                    ++nodesInIndex;
                }
                org.junit.jupiter.api.Assertions.assertEquals((int)nodesInStore, (int)nodesInIndex);
            }
        }
    }

    private void deleteNode(long nodeId) {
        try (Transaction tx = this.db.beginTx();){
            tx.getNodeById(nodeId).delete();
            tx.commit();
        }
    }

    private boolean changeName(long nodeId, Object newValue) {
        boolean changeIndexedNode = false;
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.getNodeById(nodeId);
            Object oldValue = node.getProperty(NAME_PROPERTY);
            if (!oldValue.equals(newValue)) {
                changeIndexedNode = true;
            }
            node.setProperty(NAME_PROPERTY, newValue);
            tx.commit();
        }
        return changeIndexedNode;
    }

    private int createNamedPeople(long[] nodes, int offset) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            for (String name : NAMES) {
                long nodeId = IndexStatisticsTest.createPersonNode(ktx, name);
                if (nodes == null) continue;
                nodes[offset++] = nodeId;
            }
            tx.commit();
        }
        return NAMES.length;
    }

    private long[] repeatCreateNamedPeopleFor(int totalNumberOfPeople) throws Exception {
        long[] nodes = new long[totalNumberOfPeople];
        int threads = 100;
        int peoplePerThread = totalNumberOfPeople / 100;
        ExecutorService service = Executors.newFixedThreadPool(100);
        AtomicReference exception = new AtomicReference();
        ArrayList<Callable<Void>> jobs = new ArrayList<Callable<Void>>(100);
        int i = 0;
        while (i < 100) {
            int finalI = i++;
            jobs.add(() -> {
                for (int offset = finalI * peoplePerThread; offset < (finalI + 1) * peoplePerThread; offset += this.createNamedPeople(nodes, offset)) {
                    try {
                        continue;
                    }
                    catch (KernelException e) {
                        exception.compareAndSet(null, e);
                        throw new RuntimeException(e);
                    }
                }
                return null;
            });
        }
        Futures.getAllResults(service.invokeAll(jobs));
        service.awaitTermination(1L, TimeUnit.SECONDS);
        service.shutdown();
        Exception ex = (Exception)exception.get();
        if (ex != null) {
            throw ex;
        }
        return nodes;
    }

    private void dropIndex(IndexDescriptor index) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            ktx.schemaWrite().indexDrop(index);
            tx.commit();
        }
    }

    private long indexSize(IndexDescriptor reference) {
        return this.resolveDependency(IndexStatisticsStore.class).indexSample(reference.getId()).indexSize();
    }

    private long indexUpdates(IndexDescriptor reference) {
        return this.resolveDependency(IndexStatisticsStore.class).indexSample(reference.getId()).updates();
    }

    private double indexSelectivity(IndexDescriptor reference) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            double selectivity = IndexStatisticsTest.getSelectivity(tx, reference);
            tx.commit();
            double d = selectivity;
            return d;
        }
    }

    private static double getSelectivity(Transaction tx, IndexDescriptor reference) throws IndexNotFoundKernelException {
        return ((InternalTransaction)tx).kernelTransaction().schemaRead().indexUniqueValuesSelectivity(reference);
    }

    private IndexStatisticsStore getIndexingStatisticsStore() {
        return this.resolveDependency(IndexStatisticsStore.class);
    }

    private void createSomePersons() throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            IndexStatisticsTest.createPersonNode(ktx, "Davide");
            IndexStatisticsTest.createPersonNode(ktx, "Stefan");
            IndexStatisticsTest.createPersonNode(ktx, "John");
            IndexStatisticsTest.createPersonNode(ktx, "John");
            tx.commit();
        }
    }

    private static long createPersonNode(KernelTransaction ktx, Object value) throws KernelException {
        int labelId = ktx.tokenWrite().labelGetOrCreateForName(PERSON_LABEL);
        int propertyKeyId = ktx.tokenWrite().propertyKeyGetOrCreateForName(NAME_PROPERTY);
        long nodeId = ktx.dataWrite().nodeCreate();
        ktx.dataWrite().nodeAddLabel(nodeId, labelId);
        ktx.dataWrite().nodeSetProperty(nodeId, propertyKeyId, Values.of((Object)value));
        return nodeId;
    }

    private IndexDescriptor createPersonNameIndex() throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = ((InternalTransaction)tx).kernelTransaction();
            int labelId = ktx.tokenWrite().labelGetOrCreateForName(PERSON_LABEL);
            int propertyKeyId = ktx.tokenWrite().propertyKeyGetOrCreateForName(NAME_PROPERTY);
            LabelSchemaDescriptor schema = SchemaDescriptors.forLabel((int)labelId, (int[])new int[]{propertyKeyId});
            IndexDescriptor index = ktx.schemaWrite().indexCreate(IndexPrototype.forSchema((SchemaDescriptor)schema).withName("my index"));
            tx.commit();
            IndexDescriptor indexDescriptor = index;
            return indexDescriptor;
        }
    }

    private <T> T resolveDependency(Class<T> clazz) {
        return (T)this.db.getDependencyResolver().resolveDependency(clazz);
    }

    private void awaitIndexesOnline() {
        try (Transaction tx = this.db.beginTx();){
            tx.schema().awaitIndexesOnline(3L, TimeUnit.MINUTES);
        }
    }

    private UpdatesTracker executeCreations(int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(null, numberOfCreations, false, false);
    }

    private UpdatesTracker executeCreationsAndDeletions(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, true, false);
    }

    private UpdatesTracker executeCreationsAndUpdates(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, false, true);
    }

    private UpdatesTracker executeCreationsDeletionsAndUpdates(long[] nodes, int numberOfCreations) throws KernelException, InterruptedException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, numberOfCreations, true, true);
    }

    private UpdatesTracker internalExecuteCreationsDeletionsAndUpdates(long[] nodes, int numberOfCreations, boolean allowDeletions, boolean allowUpdates) throws KernelException, InterruptedException {
        if (this.random.nextBoolean()) {
            this.indexOnlineMonitor.startSignal.await();
        }
        ThreadLocalRandom random = ThreadLocalRandom.current();
        UpdatesTracker updatesTracker = new UpdatesTracker();
        int offset = 0;
        while (updatesTracker.created() < numberOfCreations) {
            int created = this.createNamedPeople(nodes, offset);
            offset += created;
            updatesTracker.increaseCreated(created);
            this.notifyIfPopulationCompleted(updatesTracker);
            if (allowDeletions && updatesTracker.created() % 24 == 0) {
                long nodeId = nodes[((Random)random).nextInt(nodes.length)];
                try {
                    this.deleteNode(nodeId);
                    updatesTracker.increaseDeleted(1);
                }
                catch (NotFoundException notFoundException) {
                    // empty catch block
                }
                this.notifyIfPopulationCompleted(updatesTracker);
            }
            if (!allowUpdates || updatesTracker.created() % 24 != 0) continue;
            int randomIndex = ((Random)random).nextInt(nodes.length);
            try {
                if (this.changeName(nodes[randomIndex], NAMES[((Random)random).nextInt(NAMES.length)])) {
                    updatesTracker.increaseUpdated(1);
                }
            }
            catch (NotFoundException notFoundException) {
                // empty catch block
            }
            this.notifyIfPopulationCompleted(updatesTracker);
        }
        this.notifyPopulationCompleted(updatesTracker);
        return updatesTracker;
    }

    private void notifyPopulationCompleted(UpdatesTracker updatesTracker) {
        this.indexOnlineMonitor.updatesDone();
        updatesTracker.notifyPopulationCompleted();
    }

    private void notifyIfPopulationCompleted(UpdatesTracker updatesTracker) {
        if (this.isCompletedPopulation(updatesTracker)) {
            this.notifyPopulationCompleted(updatesTracker);
        }
    }

    private boolean isCompletedPopulation(UpdatesTracker updatesTracker) {
        return !updatesTracker.isPopulationCompleted() && this.indexOnlineMonitor.isIndexOnline();
    }

    private void assertCorrectIndexSize(long expected, long actual) {
        this.assertCorrectIndexSize("", expected, actual);
    }

    private void assertCorrectIndexSize(String info, long expected, long actual) {
        long updatesAfterCompletion = this.indexOnlineMonitor.indexSampleOnCompletion.updates();
        String message = String.format("Expected number of entries to not differ (expected: %d actual: %d) %s", expected, actual, info);
        ((AbstractLongAssert)Assertions.assertThat((long)Math.abs(expected - actual)).withFailMessage(message, new Object[0])).isLessThanOrEqualTo(updatesAfterCompletion);
    }

    private static void assertCorrectIndexUpdates(long expected, long actual) {
        IndexStatisticsTest.assertCorrectIndexUpdates("", expected, actual);
    }

    private static void assertCorrectIndexUpdates(String info, long expected, long actual) {
        String message = String.format("Expected number of index updates to not differ (expected: %d actual: %d). %s", expected, actual, info);
        ((AbstractLongAssert)Assertions.assertThat((long)Math.abs(expected - actual)).as(message, new Object[0])).isLessThanOrEqualTo(actual / 10L);
    }

    private void assertCorrectIndexSelectivity(IndexDescriptor index, long numberOfEntries) throws KernelException {
        double expected = UNIQUE_NAMES / (double)numberOfEntries;
        double actual = this.indexSelectivity(index);
        double maxDelta = Double.max(1.0E-4, (double)this.indexOnlineMonitor.indexSampleOnCompletion.updates() / (double)numberOfEntries);
        String message = String.format("Expected number of entries to not differ (expected: %f actual: %f)", expected, actual);
        org.junit.jupiter.api.Assertions.assertEquals((double)expected, (double)actual, (double)maxDelta, (String)message);
    }

    private class IndexOnlineMonitor
    extends IndexMonitor.MonitorAdapter {
        private CountDownLatch updateTrackerCompletionLatch;
        private final CountDownLatch startSignal = new CountDownLatch(1);
        private volatile boolean isOnline;
        private Barrier.Control barrier;
        private IndexSample indexSampleOnCompletion;

        private IndexOnlineMonitor() {
        }

        void initialize(int numberOfUpdateTrackers) {
            this.updateTrackerCompletionLatch = new CountDownLatch(numberOfUpdateTrackers);
            if (numberOfUpdateTrackers > 0) {
                this.barrier = new Barrier.Control();
            }
        }

        void updatesDone() {
            this.updateTrackerCompletionLatch.countDown();
            try {
                this.updateTrackerCompletionLatch.await();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            if (this.barrier != null) {
                this.barrier.reached();
            }
        }

        public void indexPopulationScanStarting(IndexDescriptor[] indexDescriptors) {
            this.startSignal.countDown();
        }

        public void indexPopulationScanComplete(IndexDescriptor[] indexDescriptors) {
            this.isOnline = true;
            if (this.barrier != null) {
                try {
                    this.barrier.await();
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        public void populationCompleteOn(IndexDescriptor descriptor) {
            this.indexSampleOnCompletion = IndexStatisticsTest.this.getIndexingStatisticsStore().indexSample(descriptor.getId());
            if (this.barrier != null) {
                this.barrier.release();
            }
        }

        boolean isIndexOnline() {
            return this.isOnline;
        }
    }
}

