/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.database;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BooleanSupplier;
import java.util.stream.Stream;
import org.apache.commons.lang3.mutable.MutableLong;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.Assumptions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.dbms.database.DbmsRuntimeRepository;
import org.neo4j.dbms.database.DbmsRuntimeVersion;
import org.neo4j.dbms.database.SystemGraphComponent;
import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.ResourceIterable;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.event.TransactionData;
import org.neo4j.graphdb.event.TransactionEventListener;
import org.neo4j.internal.helpers.collection.Iterables;
import org.neo4j.internal.recordstorage.Command;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.kernel.DeadlockDetectedException;
import org.neo4j.kernel.KernelVersion;
import org.neo4j.kernel.impl.locking.forseti.ForsetiClient;
import org.neo4j.kernel.impl.store.MetaDataStore;
import org.neo4j.kernel.impl.transaction.CommittedTransactionRepresentation;
import org.neo4j.kernel.impl.transaction.log.CompleteTransaction;
import org.neo4j.kernel.impl.transaction.log.LogicalTransactionStore;
import org.neo4j.kernel.impl.transaction.log.TransactionCursor;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.internal.event.InternalTransactionEventListener;
import org.neo4j.logging.AssertableLogProvider;
import org.neo4j.logging.InternalLogProvider;
import org.neo4j.logging.LogAssertions;
import org.neo4j.storageengine.api.KernelVersionRepository;
import org.neo4j.storageengine.api.TransactionIdStore;
import org.neo4j.test.OtherThreadExecutor;
import org.neo4j.test.Race;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;
import org.neo4j.util.concurrent.BinaryLatch;

@TestDirectoryExtension
class DatabaseUpgradeTransactionIT {
    @Inject
    private TestDirectory testDirectory;
    private final AssertableLogProvider logProvider = new AssertableLogProvider();
    private DatabaseManagementService dbms;
    private GraphDatabaseAPI db;

    DatabaseUpgradeTransactionIT() {
    }

    @BeforeEach
    void setUp() throws IOException {
        Assumptions.assumeThat((Comparable)KernelVersion.V5_0).isLessThan((Comparable)KernelVersion.LATEST);
        this.restartDbms();
    }

    @AfterEach
    void tearDown() {
        if (this.dbms != null) {
            this.dbms.shutdown();
        }
    }

    @Test
    void shouldUpgradeDatabaseToLatestVersionOnFirstWriteTransaction() throws Exception {
        this.setKernelVersion(KernelVersion.V5_0);
        this.setDbmsRuntime(DbmsRuntimeVersion.V5_0);
        this.restartDbms();
        long startTransaction = ((TransactionIdStore)this.db.getDependencyResolver().resolveDependency(TransactionIdStore.class)).getLastCommittedTransactionId();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
        this.createWriteTransaction();
        this.setDbmsRuntime(DbmsRuntimeVersion.LATEST_DBMS_RUNTIME_COMPONENT_VERSION);
        this.createReadTransaction();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.LATEST);
        this.assertUpgradeTransactionInOrder(KernelVersion.V5_0, KernelVersion.LATEST, startTransaction);
    }

    @Test
    void shouldUpgradeDatabaseToMaxKernelVersionForDbmsRuntimeVersionOnFirstWriteTransaction() throws Exception {
        this.setKernelVersion(KernelVersion.V5_0);
        this.setDbmsRuntime(DbmsRuntimeVersion.V5_0);
        this.restartDbms();
        long startTransaction = ((TransactionIdStore)this.db.getDependencyResolver().resolveDependency(TransactionIdStore.class)).getLastCommittedTransactionId();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
        this.createWriteTransaction();
        this.setDbmsRuntime(DbmsRuntimeVersion.V5_0);
        this.createReadTransaction();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
        this.assertUpgradeTransactionInOrder(KernelVersion.V5_0, KernelVersion.V5_0, startTransaction);
    }

    @Test
    void shouldUpgradeDatabaseToLatestVersionOnFirstWriteTransactionStressTest() throws Throwable {
        this.setKernelVersion(KernelVersion.V5_0);
        this.setDbmsRuntime(DbmsRuntimeVersion.V5_0);
        this.restartDbms();
        long startTransaction = ((TransactionIdStore)this.db.getDependencyResolver().resolveDependency(TransactionIdStore.class)).getLastCommittedTransactionId();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
        Assertions.assertThat((Comparable)this.getDbmsRuntime()).isEqualTo((Object)DbmsRuntimeVersion.V5_0);
        this.createWriteTransaction();
        Race race = new Race().withRandomStartDelays().withEndCondition(new BooleanSupplier[]{() -> KernelVersion.LATEST.equals((Object)this.getKernelVersion())});
        race.addContestant(() -> this.dbms.database("system").executeTransactionally("CALL dbms.upgrade()"), 1);
        race.addContestants(Integer.max(Runtime.getRuntime().availableProcessors() - 1, 2), Race.throwing(() -> {
            this.createWriteTransaction();
            Thread.sleep(ThreadLocalRandom.current().nextInt(0, 2));
        }));
        race.go(1L, TimeUnit.MINUTES);
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.LATEST);
        Assertions.assertThat((Comparable)this.getDbmsRuntime()).isEqualTo((Object)DbmsRuntimeVersion.LATEST_DBMS_RUNTIME_COMPONENT_VERSION);
        this.assertUpgradeTransactionInOrder(KernelVersion.V5_0, KernelVersion.LATEST, startTransaction);
    }

    @Test
    void shouldUpgradeDatabaseToLatestVersionOnDenseNodeTransactionStressTest() throws Throwable {
        this.setKernelVersion(KernelVersion.V5_0);
        this.setDbmsRuntime(DbmsRuntimeVersion.V5_0);
        this.restartDbms();
        long startTransaction = ((TransactionIdStore)this.db.getDependencyResolver().resolveDependency(TransactionIdStore.class)).getLastCommittedTransactionId();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
        Assertions.assertThat((Comparable)this.getDbmsRuntime()).isEqualTo((Object)DbmsRuntimeVersion.V5_0);
        long nodeId = this.createDenseNode();
        Race race = new Race().withRandomStartDelays().withEndCondition(new BooleanSupplier[]{new BooleanSupplier(){
            private final AtomicLong timeOfUpgrade = new AtomicLong();

            @Override
            public boolean getAsBoolean() {
                if (KernelVersion.LATEST.equals((Object)DatabaseUpgradeTransactionIT.this.getKernelVersion())) {
                    this.timeOfUpgrade.compareAndSet(0L, System.currentTimeMillis());
                }
                return this.timeOfUpgrade.get() != 0L && System.currentTimeMillis() - this.timeOfUpgrade.get() > 1000L;
            }
        }});
        race.addContestant(Race.throwing(() -> {
            while (true) {
                try {
                    Thread.sleep(ThreadLocalRandom.current().nextInt(0, 1000));
                    this.dbms.database("system").executeTransactionally("CALL dbms.upgrade()");
                    return;
                }
                catch (DeadlockDetectedException deadlockDetectedException) {
                    continue;
                }
                break;
            }
        }), 1);
        race.addContestants(Integer.max(Runtime.getRuntime().availableProcessors() - 1, 2), Race.throwing(() -> {
            while (true) {
                try (Transaction tx = this.db.beginTx();){
                    tx.getNodeById(nodeId).createRelationshipTo(tx.createNode(), RelationshipType.withName((String)("TYPE_" + ThreadLocalRandom.current().nextInt(3))));
                    tx.commit();
                    Thread.sleep(ThreadLocalRandom.current().nextInt(0, 2));
                    return;
                }
                catch (DeadlockDetectedException deadlockDetectedException) {
                    continue;
                }
                break;
            }
        }));
        race.go(10L, TimeUnit.MINUTES);
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.LATEST);
        Assertions.assertThat((Comparable)this.getDbmsRuntime()).isEqualTo((Object)DbmsRuntimeVersion.LATEST_DBMS_RUNTIME_COMPONENT_VERSION);
        this.assertUpgradeTransactionInOrder(KernelVersion.V5_0, KernelVersion.LATEST, startTransaction);
        this.assertDegrees(nodeId);
    }

    @Test
    void shouldNotUpgradePastDbmsRuntime() throws IOException {
        this.setKernelVersion(KernelVersion.V5_0);
        this.restartDbms();
        this.setDbmsRuntime(DbmsRuntimeVersion.V5_0);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
    }

    @Test
    void shouldHandleDeadlocksOnUpgradeTransaction() throws Exception {
        this.setKernelVersion(KernelVersion.V5_0);
        this.setDbmsRuntime(DbmsRuntimeVersion.V5_0);
        this.restartDbms();
        final long lockNode1 = this.createWriteTransaction();
        final long lockNode2 = this.createWriteTransaction();
        final BinaryLatch l1 = new BinaryLatch();
        final BinaryLatch l2 = new BinaryLatch();
        long numNodesBefore = this.getNodeCount();
        this.dbms.registerTransactionEventListener(this.db.databaseName(), (TransactionEventListener)new InternalTransactionEventListener.Adapter<Object>(){

            public Object beforeCommit(TransactionData data, Transaction transaction, GraphDatabaseService databaseService) {
                DatabaseUpgradeTransactionIT.this.dbms.unregisterTransactionEventListener(DatabaseUpgradeTransactionIT.this.db.databaseName(), (TransactionEventListener)this);
                l2.release();
                l1.await();
                transaction.acquireWriteLock((Entity)transaction.getNodeById(lockNode2));
                transaction.acquireWriteLock((Entity)transaction.getNodeById(lockNode1));
                return null;
            }
        });
        try (OtherThreadExecutor executor = new OtherThreadExecutor("Executor");){
            Future f1 = executor.executeDontWait(this::createWriteTransaction);
            l2.await();
            this.setDbmsRuntime(DbmsRuntimeVersion.LATEST_DBMS_RUNTIME_COMPONENT_VERSION);
            try (Transaction tx = this.db.beginTx();){
                tx.acquireWriteLock((Entity)tx.getNodeById(lockNode1));
                tx.createNode();
                l1.release();
                executor.waitUntilWaiting(details -> details.isAt(ForsetiClient.class, "acquireExclusive"));
                tx.commit();
            }
            executor.awaitFuture(f1);
        }
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).containsMessageWithArguments("Upgrade transaction from %s to %s not possible right now due to conflicting transaction, will retry on next write", new Object[]{KernelVersion.V5_0, KernelVersion.LATEST}).doesNotContainMessageWithArguments("Upgrade transaction from %s to %s started", new Object[]{KernelVersion.V5_0, KernelVersion.LATEST});
        ((AbstractLongAssert)Assertions.assertThat((long)this.getNodeCount()).as("Both transactions succeeded", new Object[0])).isEqualTo(numNodesBefore + 2L);
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.V5_0);
        this.createWriteTransaction();
        Assertions.assertThat((Comparable)this.getKernelVersion()).isEqualTo((Object)KernelVersion.LATEST);
        LogAssertions.assertThat((AssertableLogProvider)this.logProvider).containsMessageWithArguments("Upgrade transaction from %s to %s started", new Object[]{KernelVersion.V5_0, KernelVersion.LATEST}).containsMessageWithArguments("Upgrade transaction from %s to %s completed", new Object[]{KernelVersion.V5_0, KernelVersion.LATEST});
    }

    private long getNodeCount() {
        try (Transaction tx = this.db.beginTx();){
            long l = Iterables.count((Iterable)tx.getAllNodes());
            return l;
        }
    }

    private void createReadTransaction() {
        try (Transaction tx = this.db.beginTx();
             ResourceIterable allNodes = tx.getAllNodes();){
            allNodes.forEach(Entity::getAllProperties);
            tx.commit();
        }
    }

    private long createWriteTransaction() {
        try (Transaction tx = this.db.beginTx();){
            long nodeId = tx.createNode().getId();
            tx.commit();
            long l = nodeId;
            return l;
        }
    }

    private void restartDbms() throws IOException {
        boolean restart = this.db != null;
        DatabaseLayout databaseLayout = null;
        if (restart) {
            databaseLayout = this.db.databaseLayout();
        }
        if (this.dbms != null) {
            this.dbms.shutdown();
        }
        if (restart) {
            this.testDirectory.getFileSystem().delete(databaseLayout.getTransactionLogsDirectory());
        }
        this.dbms = new TestDatabaseManagementServiceBuilder(this.testDirectory.homePath()).setConfig(GraphDatabaseInternalSettings.allow_single_automatic_upgrade, (Object)false).setConfig(GraphDatabaseSettings.fail_on_missing_files, (Object)false).setInternalLogProvider((InternalLogProvider)this.logProvider).build();
        this.db = (GraphDatabaseAPI)this.dbms.database("neo4j");
    }

    private void setKernelVersion(KernelVersion version) {
        MetaDataStore metaDataStore = (MetaDataStore)this.db.getDependencyResolver().resolveDependency(MetaDataStore.class);
        metaDataStore.setKernelVersion(version);
    }

    private KernelVersion getKernelVersion() {
        return ((KernelVersionRepository)this.db.getDependencyResolver().resolveDependency(KernelVersionRepository.class)).kernelVersion();
    }

    private void setDbmsRuntime(DbmsRuntimeVersion runtimeVersion) {
        GraphDatabaseAPI system = (GraphDatabaseAPI)this.dbms.database("system");
        try (Transaction tx = system.beginTx();
             Stream nodes = tx.findNodes(SystemGraphComponent.VERSION_LABEL).stream();){
            nodes.forEach(dbmsRuntimeNode -> dbmsRuntimeNode.setProperty("dbms-runtime", (Object)runtimeVersion.getVersion()));
            tx.commit();
        }
    }

    private DbmsRuntimeVersion getDbmsRuntime() {
        GraphDatabaseAPI system = (GraphDatabaseAPI)this.dbms.database("system");
        return ((DbmsRuntimeRepository)system.getDependencyResolver().resolveDependency(DbmsRuntimeRepository.class)).getVersion();
    }

    private void assertUpgradeTransactionInOrder(KernelVersion from, KernelVersion to, long fromTxId) throws Exception {
        LogicalTransactionStore lts = (LogicalTransactionStore)this.db.getDependencyResolver().resolveDependency(LogicalTransactionStore.class);
        ArrayList<KernelVersion> transactionVersions = new ArrayList<KernelVersion>();
        ArrayList<CommittedTransactionRepresentation> transactions = new ArrayList<CommittedTransactionRepresentation>();
        try (TransactionCursor transactionCursor = lts.getTransactions(fromTxId + 1L);){
            while (transactionCursor.next()) {
                CommittedTransactionRepresentation representation = (CommittedTransactionRepresentation)transactionCursor.get();
                transactions.add(representation);
                transactionVersions.add(representation.startEntry().getVersion());
            }
        }
        Assertions.assertThat(transactionVersions).hasSizeGreaterThanOrEqualTo(2);
        Assertions.assertThat(transactionVersions).isSortedAccordingTo(Comparator.comparingInt(KernelVersion::version));
        Assertions.assertThat((Comparable)((KernelVersion)transactionVersions.get(0))).isEqualTo((Object)from);
        Assertions.assertThat((Comparable)((KernelVersion)transactionVersions.get(transactionVersions.size() - 1))).isEqualTo((Object)to);
        CommittedTransactionRepresentation upgradeTransaction = (CommittedTransactionRepresentation)transactions.get(transactionVersions.indexOf(to));
        CompleteTransaction commandBatch = (CompleteTransaction)upgradeTransaction.commandBatch();
        commandBatch.accept(element -> {
            Assertions.assertThat((Object)element).isInstanceOf(Command.MetaDataCommand.class);
            return true;
        });
    }

    private long createDenseNode() {
        MutableLong nodeId = new MutableLong();
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            for (int i = 0; i < 100; ++i) {
                node.createRelationshipTo(tx.createNode(), RelationshipType.withName((String)("TYPE_" + i % 3)));
            }
            nodeId.setValue(node.getId());
            tx.commit();
        }
        return nodeId.longValue();
    }

    private void assertDegrees(long nodeId) {
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.getNodeById(nodeId);
            HashMap<String, Map> actualDegrees = new HashMap<String, Map>();
            Iterables.forEach((Iterable)node.getRelationships(), r -> actualDegrees.computeIfAbsent(r.getType().name(), t -> new HashMap()).computeIfAbsent(DatabaseUpgradeTransactionIT.directionOf(node, r), d -> new MutableLong()).increment());
            MutableLong actualTotalDegree = new MutableLong();
            actualDegrees.forEach((typeName, directions) -> {
                long actualTotalDirectionDegree = 0L;
                for (Map.Entry actualDirectionDegree : directions.entrySet()) {
                    Assertions.assertThat((int)node.getDegree(RelationshipType.withName((String)typeName), (Direction)actualDirectionDegree.getKey())).isEqualTo(((MutableLong)actualDirectionDegree.getValue()).longValue());
                    actualTotalDirectionDegree += ((MutableLong)actualDirectionDegree.getValue()).longValue();
                }
                Assertions.assertThat((int)node.getDegree(RelationshipType.withName((String)typeName))).isEqualTo(actualTotalDirectionDegree);
                actualTotalDegree.add(actualTotalDirectionDegree);
            });
            Assertions.assertThat((int)node.getDegree()).isEqualTo(actualTotalDegree.longValue());
        }
    }

    private static Direction directionOf(Node node, Relationship relationship) {
        return relationship.getStartNode().equals(node) ? (relationship.getEndNode().equals(node) ? Direction.BOTH : Direction.OUTGOING) : Direction.INCOMING;
    }
}

