/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.log.pruning;

import java.io.IOException;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Transaction;
import org.neo4j.io.fs.EphemeralFileSystemAbstraction;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.UncloseableDelegatingFileSystemAbstraction;
import org.neo4j.kernel.impl.transaction.log.CommittedCommandBatchCursor;
import org.neo4j.kernel.impl.transaction.log.LogVersionBridge;
import org.neo4j.kernel.impl.transaction.log.LogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.ReadAheadLogChannel;
import org.neo4j.kernel.impl.transaction.log.ReadableClosablePositionAwareChecksumChannel;
import org.neo4j.kernel.impl.transaction.log.checkpoint.CheckPointer;
import org.neo4j.kernel.impl.transaction.log.checkpoint.SimpleTriggerInfo;
import org.neo4j.kernel.impl.transaction.log.checkpoint.TriggerInfo;
import org.neo4j.kernel.impl.transaction.log.entry.LogEntryReader;
import org.neo4j.kernel.impl.transaction.log.entry.VersionAwareLogEntryReader;
import org.neo4j.kernel.impl.transaction.log.files.LogFile;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.tracing.LogAppendEvent;
import org.neo4j.kernel.impl.transaction.tracing.LogRotateEvents;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.storageengine.api.StorageEngineFactory;
import org.neo4j.test.TestDatabaseManagementServiceBuilder;

class TestLogPruning {
    private DatabaseManagementService managementService;
    private GraphDatabaseAPI db;
    private FileSystemAbstraction fs;
    private LogFiles logFiles;
    private int rotateEveryNTransactions;
    private int performedTransactions;

    TestLogPruning() {
    }

    @AfterEach
    void after() throws Exception {
        if (this.db != null) {
            this.managementService.shutdown();
        }
        this.fs.close();
    }

    @Test
    void noPruning() throws Exception {
        this.newDb("true", 2);
        for (int i = 0; i < 100; ++i) {
            this.doTransaction();
        }
        LogFile logFile = this.logFiles.getLogFile();
        long currentVersion = logFile.getHighestLogVersion();
        for (long version = 0L; version < currentVersion; ++version) {
            org.junit.jupiter.api.Assertions.assertTrue((boolean)this.fs.fileExists(logFile.getLogFileForVersion(version)), (String)("Version " + version + " has been unexpectedly pruned"));
        }
    }

    @Test
    void pruneByFileSize() throws Exception {
        int transactionByteSize = this.figureOutSampleTransactionSizeBytes();
        int transactionsPerFile = 3;
        int logThreshold = transactionByteSize * transactionsPerFile;
        this.newDb(logThreshold + " size", 1);
        for (int i = 0; i < 100; ++i) {
            this.doTransaction();
        }
        int totalLogFileSize = this.logFileSize();
        double totalTransactions = (double)totalLogFileSize / (double)transactionByteSize;
        org.junit.jupiter.api.Assertions.assertTrue((totalTransactions >= 3.0 && totalTransactions < 4.0 ? 1 : 0) != 0);
    }

    @Test
    void pruneByFileCount() throws Exception {
        int logsToKeep = 5;
        this.newDb(logsToKeep + " files", 3);
        for (int i = 0; i < 100; ++i) {
            this.doTransaction();
        }
        org.junit.jupiter.api.Assertions.assertEquals((int)logsToKeep, (int)this.logCount());
    }

    @Test
    void pruneByTransactionCount() throws Exception {
        int transactionsToKeep = 100;
        int transactionsPerLog = 3;
        this.newDb(transactionsToKeep + " txs", 3);
        for (int i = 0; i < 100; ++i) {
            this.doTransaction();
        }
        int transactionCount = this.transactionCount();
        org.junit.jupiter.api.Assertions.assertTrue((transactionCount >= transactionsToKeep && transactionCount <= transactionsToKeep + transactionsPerLog ? 1 : 0) != 0, (String)("Transaction count expected to be within " + transactionsToKeep + " <= txs <= " + (transactionsToKeep + transactionsPerLog) + ", but was " + transactionCount));
    }

    @Test
    void shouldKeepAtLeastOneTransactionAfterRotate() throws Exception {
        this.newDb("1 size", 1);
        for (int i = 0; i < 2; ++i) {
            this.doTransaction();
        }
        this.rotate();
        Assertions.assertThat((int)this.transactionCount()).isGreaterThanOrEqualTo(1);
    }

    private GraphDatabaseAPI newDb(String logPruning, int rotateEveryNTransactions) {
        this.rotateEveryNTransactions = rotateEveryNTransactions;
        this.fs = new EphemeralFileSystemAbstraction();
        this.managementService = new TestDatabaseManagementServiceBuilder().setFileSystem((FileSystemAbstraction)new UncloseableDelegatingFileSystemAbstraction(this.fs)).setConfig(GraphDatabaseSettings.keep_logical_logs, (Object)logPruning).build();
        this.db = (GraphDatabaseAPI)this.managementService.database("neo4j");
        this.logFiles = (LogFiles)this.db.getDependencyResolver().resolveDependency(LogFiles.class);
        return this.db;
    }

    private void doTransaction() throws IOException {
        if (++this.performedTransactions >= this.rotateEveryNTransactions) {
            this.rotate();
            this.performedTransactions = 0;
        }
        try (Transaction tx = this.db.beginTx();){
            Node node = tx.createNode();
            node.setProperty("name", (Object)"a somewhat lengthy string of some sort, right?");
            tx.commit();
        }
        this.checkPoint();
    }

    private void rotate() throws IOException {
        this.logFiles.getLogFile().getLogRotation().rotateLogFile((LogRotateEvents)LogAppendEvent.NULL);
    }

    private void checkPoint() throws IOException {
        SimpleTriggerInfo triggerInfo = new SimpleTriggerInfo("test");
        ((CheckPointer)this.db.getDependencyResolver().resolveDependency(CheckPointer.class)).forceCheckPoint((TriggerInfo)triggerInfo);
    }

    private int figureOutSampleTransactionSizeBytes() throws IOException {
        this.db = this.newDb("true", 5);
        this.doTransaction();
        this.managementService.shutdown();
        return (int)this.fs.getFileSize(this.logFiles.getLogFile().getLogFileForVersion(0L));
    }

    private int aggregateLogData(Extractor extractor) throws IOException {
        int total = 0;
        LogFile logFile = this.logFiles.getLogFile();
        for (long i = logFile.getHighestLogVersion(); i >= 0L && logFile.versionExists(i); --i) {
            total += extractor.extract(i);
        }
        return total;
    }

    private int logCount() throws IOException {
        return this.aggregateLogData(from -> 1);
    }

    private int logFileSize() throws IOException {
        return this.aggregateLogData(from -> (int)this.fs.getFileSize(this.logFiles.getLogFile().getLogFileForVersion(from)));
    }

    private int transactionCount() throws IOException {
        return this.aggregateLogData(version -> {
            int counter = 0;
            LogVersionBridge bridge = LogVersionBridge.NO_MORE_CHANNELS;
            PhysicalLogVersionedStoreChannel versionedStoreChannel = this.logFiles.getLogFile().openForVersion(version);
            StorageEngineFactory storageEngineFactory = (StorageEngineFactory)this.db.getDependencyResolver().resolveDependency(StorageEngineFactory.class);
            try (ReadAheadLogChannel channel = new ReadAheadLogChannel((LogVersionedStoreChannel)versionedStoreChannel, bridge, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
                 CommittedCommandBatchCursor physicalTransactionCursor = new CommittedCommandBatchCursor((ReadableClosablePositionAwareChecksumChannel)channel, (LogEntryReader)new VersionAwareLogEntryReader(storageEngineFactory.commandReaderFactory()));){
                while (physicalTransactionCursor.next()) {
                    ++counter;
                }
            }
            return counter;
        });
    }

    @FunctionalInterface
    private static interface Extractor {
        public int extract(long var1) throws IOException;
    }
}

