/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.security;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.neo4j.dbms.api.DatabaseManagementService;
import org.neo4j.function.Suppliers;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Transaction;
import org.neo4j.internal.kernel.api.security.AbstractSecurityLog;
import org.neo4j.kernel.impl.security.Credential;
import org.neo4j.kernel.impl.security.User;
import org.neo4j.server.security.SecureHasher;
import org.neo4j.server.security.SystemGraphCredential;
import org.neo4j.server.security.systemgraph.SecurityGraphHelper;
import org.neo4j.server.security.systemgraph.versions.KnownCommunitySecurityComponentVersion;
import org.neo4j.string.UTF8;
import org.neo4j.test.extension.ImpermanentDbmsExtension;
import org.neo4j.test.extension.Inject;

@ImpermanentDbmsExtension
public class SecurityGraphHelperTest {
    @Inject
    private DatabaseManagementService dbms;
    private SecurityGraphHelper securityGraphHelper;
    private AbstractSecurityLog securityLog;
    GraphDatabaseService system;

    @BeforeEach
    void setUp() {
        this.system = this.dbms.database("system");
        this.securityLog = (AbstractSecurityLog)Mockito.mock(AbstractSecurityLog.class);
        this.securityGraphHelper = new SecurityGraphHelper(Suppliers.lazySingleton(() -> this.system), new SecureHasher(), this.securityLog);
    }

    @AfterEach
    void tearDown() {
        try (Transaction tx = this.system.beginTx();){
            List users = tx.execute("SHOW USERS YIELD user").map(m -> (String)m.get("user")).stream().toList();
            for (String user : users) {
                tx.execute("DROP USER " + user);
            }
            tx.commit();
        }
    }

    @Test
    void getAuthShouldFindAuthObjectForUserNameAndProvider() {
        this.createUser(new User("alice", "aliceId", null, false, false, Set.of(new User.Auth("provider1", "sub123"))));
        String result = this.securityGraphHelper.getAuthId("provider1", "alice");
        Assertions.assertThat((String)result).isEqualTo("sub123");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up 'provider1' auth for user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=aliceId, credential=null, passwordChangeRequired=false, suspended=false, auth=[Auth[provider=provider1, id=sub123]]]");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getAuthShouldFindNativeAuth() {
        this.createUser(new User("alice", "uuid", null, false, false, Set.of(new User.Auth("native", "aliceId"))));
        String result = this.securityGraphHelper.getAuthId("native", "alice");
        Assertions.assertThat((String)result).isEqualTo("aliceId");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up 'native' auth for user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=uuid, credential=null, passwordChangeRequired=false, suspended=false, auth=[Auth[provider=native, id=aliceId]]]");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getAuthShouldReturnNullAndLogWhenUserDoesNotExist() {
        String result = this.securityGraphHelper.getAuthId("provider1", "alice");
        Assertions.assertThat((String)result).isNull();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up 'provider1' auth for user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("User 'alice' not found");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getAuthShouldReturnNullAndLogWhenAuthDoesNotExists() {
        this.createUser(new User("alice", "userId", null, false, false));
        String result = this.securityGraphHelper.getAuthId("provider1", "alice");
        Assertions.assertThat((String)result).isNull();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up 'provider1' auth for user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=userId, credential=null, passwordChangeRequired=false, suspended=false, auth=[]]");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("'provider1' auth not found for user 'alice'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void hasExternalAuthShouldReturnTrueWhenUserHasExternalAuth() {
        this.createUser(new User("alice", "aliceId", null, false, false, Set.of(new User.Auth("provider1", "sub123"))));
        boolean result = this.securityGraphHelper.hasExternalAuth("alice");
        Assertions.assertThat((boolean)result).isTrue();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Checking whether user 'alice' has an external auth");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=aliceId, credential=null, passwordChangeRequired=false, suspended=false, auth=[Auth[provider=provider1, id=sub123]]]");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("External auth found for user 'alice'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void hasExternalAuthObjectShouldReturnFalseWhenUserHasNativeAuthObject() {
        this.createUser(new User("alice", "aliceId", null, false, false, Set.of(new User.Auth("native", "aliceId"))));
        boolean result = this.securityGraphHelper.hasExternalAuth("alice");
        Assertions.assertThat((boolean)result).isFalse();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Checking whether user 'alice' has an external auth");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=aliceId, credential=null, passwordChangeRequired=false, suspended=false, auth=[Auth[provider=native, id=aliceId]]]");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("No external auth found for user 'alice'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void hasAuthObjectShouldReturnFalseWhenUserDoesNotExist() {
        boolean result = this.securityGraphHelper.hasExternalAuth("alice");
        Assertions.assertThat((boolean)result).isFalse();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Checking whether user 'alice' has an external auth");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("User 'alice' not found");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void hasAuthObjectShouldReturnFalseWhenNoAuthObjectExists() {
        this.createUser(new User("alice", "userId", null, false, false));
        boolean result = this.securityGraphHelper.hasExternalAuth("alice");
        Assertions.assertThat((boolean)result).isFalse();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Checking whether user 'alice' has an external auth");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user 'alice'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=userId, credential=null, passwordChangeRequired=false, suspended=false, auth=[]]");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("No external auth found for user 'alice'");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getUserByAuthShouldReturnUser() {
        this.createUser(new User("alice", "userId", null, false, false, Set.of(new User.Auth("provider1", "sub123"))));
        User result = this.securityGraphHelper.getUserByAuth("provider1", "sub123");
        Assertions.assertThat((String)result.name()).isEqualTo("alice");
        Assertions.assertThat((String)result.id()).isEqualTo("userId");
        Assertions.assertThat((boolean)result.passwordChangeRequired()).isFalse();
        Assertions.assertThat((boolean)result.suspended()).isFalse();
        Assertions.assertThat((Collection)result.auth()).isEqualTo(Set.of(new User.Auth("provider1", "sub123")));
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user with auth provider: provider1, auth id: sub123");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=userId, credential=null, passwordChangeRequired=false, suspended=false, auth=[Auth[provider=provider1, id=sub123]]]");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getUserByAuthShouldReturnNullWhenAuthDoesNotExist() {
        this.createUser(new User("alice", "userId", null, false, false, Set.of(new User.Auth("provider1", "sub456"))));
        User result = this.securityGraphHelper.getUserByAuth("provider1", "sub123");
        Assertions.assertThat((Object)result).isNull();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user with auth provider: provider1, auth id: sub123");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("No auth for auth provider: provider1, auth id: sub123");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getUserByIdShouldReturnNullUserIdIsNull() {
        User result = this.securityGraphHelper.getUserById(null);
        Assertions.assertThat((Object)result).isNull();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user with id 'null'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Cannot look up user with id = null");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getUserByIdShouldReturnUserWithNativeAuthForLegacyUserWhenCredentialsSet() {
        SystemGraphCredential credential = SystemGraphCredential.createCredentialForPassword((byte[])UTF8.encode((String)"password"), (SecureHasher)new SecureHasher());
        this.createUser(new User("alice", "userId", (Credential)credential, false, false));
        User result = this.securityGraphHelper.getUserById("userId");
        Assertions.assertThat((String)result.id()).isEqualTo("userId");
        Assertions.assertThat((String)result.name()).isEqualTo("alice");
        Assertions.assertThat((Collection)result.auth()).isEqualTo(Set.of(new User.Auth("native", "userId")));
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user with id 'userId'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=userId, credential=*****, passwordChangeRequired=false, suspended=false, auth=[Auth[provider=native, id=userId]]]");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getUserByIdShouldReturnUserWithoutNativeAuthForLegacyUserWhenCredentialsNotSet() {
        this.createUser(new User("alice", "userId", null, false, false));
        User result = this.securityGraphHelper.getUserById("userId");
        Assertions.assertThat((String)result.id()).isEqualTo("userId");
        Assertions.assertThat((String)result.name()).isEqualTo("alice");
        Assertions.assertThat((Collection)result.auth()).isEqualTo(Set.of());
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user with id 'userId'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Found user: User[name=alice, id=userId, credential=null, passwordChangeRequired=false, suspended=false, auth=[]]");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    @Test
    void getUserByIdShouldReturnNullWhenUserDoesNotExist() {
        User result = this.securityGraphHelper.getUserById("userId");
        Assertions.assertThat((Object)result).isNull();
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("Looking up user with id 'userId'");
        ((AbstractSecurityLog)Mockito.verify((Object)this.securityLog)).debug("User with id 'userId' not found");
        Mockito.verifyNoMoreInteractions((Object[])new Object[]{this.securityLog});
    }

    void createUser(User user) {
        try (Transaction tx = this.system.beginTx();){
            Node userNode = tx.createNode(new Label[]{KnownCommunitySecurityComponentVersion.USER_LABEL});
            userNode.setProperty("name", (Object)user.name());
            userNode.setProperty("id", (Object)user.id());
            if (user.credential() != null && user.credential().value() != null) {
                userNode.setProperty("credentials", (Object)user.credential().value().serialize());
                userNode.setProperty("passwordChangeRequired", (Object)user.passwordChangeRequired());
            }
            userNode.setProperty("suspended", (Object)user.suspended());
            for (User.Auth auth : user.auth()) {
                Node authNode = tx.createNode(new Label[]{KnownCommunitySecurityComponentVersion.AUTH_LABEL});
                authNode.setProperty("provider", (Object)auth.provider());
                authNode.setProperty("id", (Object)auth.id());
                userNode.createRelationshipTo(authNode, KnownCommunitySecurityComponentVersion.HAS_AUTH);
            }
            tx.commit();
        }
    }
}

