/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.cache;

import java.util.Collection;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import org.neo4j.kernel.impl.cache.Cache;
import org.neo4j.kernel.impl.cache.EntityWithSize;
import org.neo4j.kernel.impl.util.StringLogger;
import org.neo4j.kernel.info.DiagnosticsPhase;
import org.neo4j.kernel.info.DiagnosticsProvider;

public class GCResistantCache<E extends EntityWithSize>
implements Cache<E>,
DiagnosticsProvider {
    public static final long MIN_SIZE = 1L;
    private final AtomicReferenceArray<E> cache;
    private final long maxSize;
    private long closeToMaxSize;
    private long purgeStopSize;
    private long purgeHandoffSize;
    private final AtomicLong currentSize = new AtomicLong(0L);
    private final long minLogInterval;
    private final String name;
    private final AtomicLong highestIdSet = new AtomicLong();
    private long hitCount = 0L;
    private long missCount = 0L;
    private long totalPuts = 0L;
    private long collisions = 0L;
    private long purgeCount = 0L;
    private final StringLogger logger;
    private final AtomicBoolean purging = new AtomicBoolean();
    private final AtomicInteger avertedPurgeWaits = new AtomicInteger();
    private final AtomicInteger forcedPurgeWaits = new AtomicInteger();
    private long purgeTime;
    private long putTimeStamp = 0L;
    private long lastPurgeLogTimestamp = 0L;

    GCResistantCache(AtomicReferenceArray<E> cache) {
        this.cache = cache;
        this.minLogInterval = Long.MAX_VALUE;
        this.maxSize = 0x40000000L;
        this.name = "test cache";
        this.logger = null;
        this.calculateSizes();
    }

    public GCResistantCache(long maxSizeInBytes, float arrayHeapFraction, long minLogInterval, String name, StringLogger logger) {
        if (logger == null) {
            throw new IllegalArgumentException("Null logger");
        }
        this.minLogInterval = minLogInterval;
        if (arrayHeapFraction < 1.0f || arrayHeapFraction > 10.0f) {
            throw new IllegalArgumentException("The heap fraction used by a GC resistant cache must be between 1% and 10%, not " + arrayHeapFraction + "%");
        }
        long memToUse = (long)((double)arrayHeapFraction * (double)Runtime.getRuntime().maxMemory() / 100.0);
        long maxElementCount = (int)(memToUse / 8L);
        if (memToUse > Integer.MAX_VALUE) {
            maxElementCount = Integer.MAX_VALUE;
        }
        if (maxSizeInBytes < 1L) {
            throw new IllegalArgumentException("Max size can not be " + maxSizeInBytes);
        }
        this.cache = new AtomicReferenceArray((int)maxElementCount);
        this.maxSize = maxSizeInBytes;
        this.name = name == null ? super.toString() : name;
        this.logger = logger;
        this.calculateSizes();
    }

    private void calculateSizes() {
        this.closeToMaxSize = (long)((double)this.maxSize * 0.95);
        this.purgeStopSize = (long)((double)this.maxSize * 0.9);
        this.purgeHandoffSize = (long)((double)this.maxSize * 1.05);
    }

    private int getPosition(EntityWithSize obj) {
        return (int)(obj.getId() % (long)this.cache.length());
    }

    private int getPosition(long id) {
        return (int)(id % (long)this.cache.length());
    }

    public void put(E obj) {
        int pos;
        EntityWithSize oldObj;
        long time = System.currentTimeMillis();
        if (time - this.putTimeStamp > this.minLogInterval) {
            this.putTimeStamp = time;
            this.printStatistics();
        }
        if ((oldObj = (EntityWithSize)this.cache.get(pos = this.getPosition((EntityWithSize)obj))) != obj) {
            int objectSize = obj.size();
            if (this.cache.compareAndSet(pos, oldObj, (EntityWithSize)obj)) {
                this.setHighest(pos);
                int oldObjSize = 0;
                if (oldObj != null) {
                    oldObjSize = oldObj.getRegisteredSize();
                }
                long size = this.currentSize.addAndGet(objectSize - oldObjSize);
                obj.setRegisteredSize(objectSize);
                if (oldObj != null) {
                    ++this.collisions;
                }
                ++this.totalPuts;
                if (size > this.closeToMaxSize) {
                    this.purgeFrom(pos);
                }
            }
        }
    }

    private void setHighest(long id) {
        long highest;
        while (id > (highest = this.highestIdSet.get()) && !this.highestIdSet.compareAndSet(highest, id)) {
        }
    }

    public E remove(long id) {
        int pos = this.getPosition(id);
        EntityWithSize obj = (EntityWithSize)this.cache.get(pos);
        if (obj != null && this.cache.compareAndSet(pos, obj, null)) {
            this.currentSize.addAndGet(obj.getRegisteredSize() * -1);
        }
        return (E)obj;
    }

    public E get(long id) {
        int pos = this.getPosition(id);
        EntityWithSize obj = (EntityWithSize)this.cache.get(pos);
        if (obj != null && obj.getId() == id) {
            ++this.hitCount;
            return (E)obj;
        }
        ++this.missCount;
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void purgeFrom(int pos) {
        long myCurrentSize = this.currentSize.get();
        if (myCurrentSize <= this.closeToMaxSize) {
            return;
        }
        if (this.purging.compareAndSet(false, true)) {
            try {
                this.doPurge(pos);
            }
            finally {
                this.purging.set(false);
            }
        } else {
            if (myCurrentSize < this.purgeHandoffSize) {
                this.avertedPurgeWaits.incrementAndGet();
                return;
            }
            this.forcedPurgeWaits.incrementAndGet();
            this.waitForCurrentPurgeToComplete();
        }
    }

    private synchronized void waitForCurrentPurgeToComplete() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void doPurge(int pos) {
        long myCurrentSize = this.currentSize.get();
        if (myCurrentSize <= this.closeToMaxSize) {
            return;
        }
        long startTime = System.currentTimeMillis();
        ++this.purgeCount;
        long sizeBefore = this.currentSize.get();
        try {
            int index = 1;
            do {
                if (pos - index >= 0) {
                    int minusPos = pos - index;
                    this.remove(minusPos);
                    if (this.currentSize.get() <= this.purgeStopSize) {
                        return;
                    }
                }
                if (pos + index >= this.cache.length()) continue;
                int plusPos = pos + index;
                this.remove(plusPos);
                if (this.currentSize.get() > this.purgeStopSize) continue;
                return;
            } while (pos - ++index >= 0 || pos + index < this.cache.length());
            this.remove(pos);
        }
        finally {
            long timestamp = System.currentTimeMillis();
            this.purgeTime += timestamp - startTime;
            if (timestamp - this.lastPurgeLogTimestamp > this.minLogInterval) {
                this.lastPurgeLogTimestamp = timestamp;
                long sizeAfter = this.currentSize.get();
                String sizeBeforeStr = this.getSize(sizeBefore);
                String sizeAfterStr = this.getSize(sizeAfter);
                String diffStr = this.getSize(sizeBefore - sizeAfter);
                String missPercentage = (float)this.missCount / (float)(this.hitCount + this.missCount) * 100.0f + "%";
                String colPercentage = (float)this.collisions / (float)this.totalPuts * 100.0f + "%";
                this.logger.logMessage(this.name + " purge (nr " + this.purgeCount + ") " + sizeBeforeStr + " -> " + sizeAfterStr + " (" + diffStr + ") " + missPercentage + " misses, " + colPercentage + " collisions (" + this.collisions + ").", true);
                this.printAccurateStatistics();
            }
        }
    }

    private void printAccurateStatistics() {
        int elementCount = 0;
        long actualSize = 0L;
        long registeredSize = 0L;
        for (int i = 0; i < this.cache.length(); ++i) {
            EntityWithSize obj = (EntityWithSize)this.cache.get(i);
            if (obj == null) continue;
            ++elementCount;
            actualSize += (long)obj.size();
            registeredSize += (long)obj.getRegisteredSize();
        }
        this.logger.logMessage(this.name + " purge (nr " + this.purgeCount + "): elementCount:" + elementCount + " and sizes actual:" + this.getSize(actualSize) + ", perceived:" + this.getSize(this.currentSize.get()) + " (diff:" + this.getSize(this.currentSize.get() - actualSize) + "), registered:" + this.getSize(registeredSize), true);
    }

    public void printStatistics() {
        this.logStatistics(this.logger);
    }

    public String getDiagnosticsIdentifier() {
        return this.getName();
    }

    public void acceptDiagnosticsVisitor(Object visitor) {
    }

    public void dump(DiagnosticsPhase phase, StringLogger log) {
        if (phase.isExplicitlyRequested()) {
            this.logStatistics(log);
        }
    }

    private void logStatistics(StringLogger log) {
        log.debug(this.toString());
    }

    public String toString() {
        String currentSizeStr = this.getSize(this.currentSize.get());
        String missPercentage = (float)this.missCount / (float)(this.hitCount + this.missCount) * 100.0f + "%";
        String colPercentage = (float)this.collisions / (float)this.totalPuts * 100.0f + "%";
        return this.name + " array:" + this.cache.length() + " purge:" + this.purgeCount + " size:" + currentSizeStr + " misses:" + missPercentage + " collisions:" + colPercentage + " (" + this.collisions + ") av.purge waits:" + this.avertedPurgeWaits.get() + " purge waits:" + this.forcedPurgeWaits.get() + " avg. purge time:" + (this.purgeCount > 0L ? this.purgeTime / this.purgeCount + "ms" : "N/A");
    }

    private String getSize(long size) {
        if (size > 0x40000000L) {
            float value = (float)size / 1024.0f / 1024.0f / 1024.0f;
            return value + "Gb";
        }
        if (size > 0x100000L) {
            float value = (float)size / 1024.0f / 1024.0f;
            return value + "Mb";
        }
        if (size > 1024L) {
            float value = (float)size / 1024.0f / 1024.0f;
            return value + "kb";
        }
        return size + "b";
    }

    public void clear() {
        int i = 0;
        while ((long)i <= this.highestIdSet.get()) {
            this.cache.set(i, null);
            ++i;
        }
        this.currentSize.set(0L);
        this.highestIdSet.set(0L);
    }

    public void putAll(Collection<E> objects) {
        for (EntityWithSize obj : objects) {
            this.put(obj);
        }
    }

    public String getName() {
        return this.name;
    }

    public long size() {
        return this.currentSize.get();
    }

    public long hitCount() {
        return this.hitCount;
    }

    public long missCount() {
        return this.missCount;
    }

    public void updateSize(E obj, int newSize) {
        int pos = this.getPosition((EntityWithSize)obj);
        EntityWithSize existingObj = (EntityWithSize)this.cache.get(pos);
        if (existingObj != obj) {
            return;
        }
        long size = this.currentSize.addAndGet(newSize - existingObj.getRegisteredSize());
        obj.setRegisteredSize(newSize);
        if (size > this.closeToMaxSize) {
            this.purgeFrom(pos);
        }
    }
}

