package org.nervos.ckb.sign.signer;

import org.nervos.ckb.crypto.Blake2b;
import org.nervos.ckb.crypto.secp256k1.ECKeyPair;
import org.nervos.ckb.crypto.secp256k1.Sign;
import org.nervos.ckb.sign.Context;
import org.nervos.ckb.sign.ScriptGroup;
import org.nervos.ckb.sign.ScriptSigner;
import org.nervos.ckb.type.Script;
import org.nervos.ckb.type.Transaction;
import org.nervos.ckb.type.WitnessArgs;
import org.nervos.ckb.utils.MoleculeConverter;

import java.util.Arrays;
import java.util.List;

public class Secp256k1Blake160SighashAllSigner implements ScriptSigner {
  private static final int WITNESS_OFFSET_IN_BYTE = 20;
  private static final int SIGNATURE_LENGTH_IN_BYTE = 65;

  private static Secp256k1Blake160SighashAllSigner INSTANCE;

  private Secp256k1Blake160SighashAllSigner() {
  }

  public static Secp256k1Blake160SighashAllSigner getInstance() {
    if (INSTANCE == null) {
      INSTANCE = new Secp256k1Blake160SighashAllSigner();
    }
    return INSTANCE;
  }

  @Override
  public boolean signTransaction(
      Transaction transaction, ScriptGroup scriptGroup, Context context) {
    Script script = scriptGroup.getScript();
    ECKeyPair keyPair = context.getKeyPair();
    if (isMatched(keyPair, script.args)) {
      return signScriptGroup(transaction, scriptGroup, keyPair);
    } else {
      return false;
    }
  }

  public boolean signScriptGroup(
      Transaction transaction, ScriptGroup scriptGroup, ECKeyPair keyPair) {
    byte[] txHash = transaction.computeHash();
    List<byte[]> witnesses = transaction.witnesses;
    Blake2b blake2b = new Blake2b();
    blake2b.update(txHash);

    for (int i : scriptGroup.getInputIndices()) {
      byte[] witness = witnesses.get(i);
      blake2b.update(MoleculeConverter.packUint64(witness.length).toByteArray());
      blake2b.update(witness);
    }
    for (int i = transaction.inputs.size(); i < transaction.witnesses.size(); i++) {
      byte[] witness = witnesses.get(i);
      blake2b.update(MoleculeConverter.packUint64(witness.length).toByteArray());
      blake2b.update(witness);
    }

    byte[] message = blake2b.doFinal();
    byte[] signature = Sign.signMessage(message, keyPair).getSignature();

    int index = scriptGroup.getInputIndices().get(0);
    WitnessArgs witnessArgs = WitnessArgs.unpack(witnesses.get(index));
    witnessArgs.setLock(signature);
    witnesses.set(index, witnessArgs.pack().toByteArray());
    return true;
  }

  // Check if the script with `scriptArgs` is generated by and can be unlocked by `privateKey`
  public boolean isMatched(ECKeyPair keyPair, byte[] scriptArgs) {
    if (scriptArgs == null || keyPair == null) {
      return false;
    }
    byte[] hash = Blake2b.digest(keyPair.getEncodedPublicKey(true));
    hash = Arrays.copyOfRange(hash, 0, 20);
    return Arrays.equals(scriptArgs, hash);
  }
}
