/*
 * Decompiled with CFR 0.152.
 */
package org.nervos.ckb.utils.address;

import java.io.ByteArrayOutputStream;
import java.util.Arrays;
import java.util.Objects;
import org.nervos.ckb.Network;
import org.nervos.ckb.type.Script;
import org.nervos.ckb.utils.address.AddressFormatException;
import org.nervos.ckb.utils.address.Bech32;

public class Address {
    Script script;
    Network network;

    public Address() {
    }

    public Address(Script script, Network network) {
        this.script = script;
        this.network = network;
    }

    public Script getScript() {
        return this.script;
    }

    public Address setScript(Script script) {
        Objects.requireNonNull(script);
        this.script = script;
        return this;
    }

    public Network getNetwork() {
        return this.network;
    }

    public Address setNetwork(Network network) {
        Objects.requireNonNull(network);
        this.network = network;
        return this;
    }

    public static Address decode(String address) {
        Objects.requireNonNull(address);
        Bech32.Bech32Data bech32Data = Bech32.decode(address);
        Bech32.Encoding encoding = bech32Data.encoding;
        Network network = Address.network(bech32Data.hrp);
        byte[] payload = bech32Data.data;
        payload = Address.convertBits(payload, 0, payload.length, 5, 8, false);
        switch (payload[0]) {
            case 0: {
                if (encoding != Bech32.Encoding.BECH32M) {
                    throw new AddressFormatException("Payload header 0x00 must have encoding bech32");
                }
                return Address.decodeLongBech32m(payload, network);
            }
            case 1: {
                if (encoding != Bech32.Encoding.BECH32) {
                    throw new AddressFormatException("Payload header 0x01 must have encoding bech32");
                }
                return Address.decodeShort(payload, network);
            }
            case 2: 
            case 4: {
                if (encoding != Bech32.Encoding.BECH32) {
                    throw new AddressFormatException("Payload header 0x02 or 0x04 must have encoding bech32m");
                }
                return Address.decodeLongBech32(payload, network);
            }
        }
        throw new AddressFormatException("Unknown format type");
    }

    private static Address decodeShort(byte[] payload, Network network) {
        byte[] codeHash;
        byte codeHashIndex = payload[1];
        byte[] args = Arrays.copyOfRange(payload, 2, payload.length);
        if (codeHashIndex == 0) {
            if (args.length != 20) {
                throw new AddressFormatException("Invalid args length " + args.length);
            }
            codeHash = Script.SECP256K1_BLAKE160_SIGNHASH_ALL_CODE_HASH;
        } else if (codeHashIndex == 1) {
            if (args.length != 20) {
                throw new AddressFormatException("Invalid args length " + args.length);
            }
            codeHash = Script.SECP256K1_BLAKE160_MULTISIG_ALL_CODE_HASH;
        } else if (codeHashIndex == 2) {
            if (args.length < 20 || args.length > 22) {
                throw new AddressFormatException("Invalid args length " + args.length);
            }
            codeHash = network == Network.MAINNET ? Script.ANY_CAN_PAY_CODE_HASH_MAINNET : Script.ANY_CAN_PAY_CODE_HASH_TESTNET;
        } else {
            throw new AddressFormatException("Unknown code hash index");
        }
        Script script = new Script(codeHash, args, Script.HashType.TYPE);
        return new Address(script, network);
    }

    private static Address decodeLongBech32(byte[] payload, Network network) {
        Script.HashType hashType;
        if (payload[0] == 4) {
            hashType = Script.HashType.TYPE;
        } else if (payload[0] == 2) {
            hashType = Script.HashType.DATA;
        } else {
            throw new AddressFormatException("Unknown script hash type");
        }
        byte[] codeHash = Arrays.copyOfRange(payload, 1, 33);
        byte[] args = Arrays.copyOfRange(payload, 33, payload.length);
        Script script = new Script(codeHash, args, hashType);
        return new Address(script, network);
    }

    private static Address decodeLongBech32m(byte[] payload, Network network) {
        if (payload[0] != 0) {
            throw new AddressFormatException("Invalid payload header 0x" + payload[0]);
        }
        byte[] codeHash = Arrays.copyOfRange(payload, 1, 33);
        Script.HashType hashType = Script.HashType.unpack(payload[33]);
        byte[] args = Arrays.copyOfRange(payload, 34, payload.length);
        Script script = new Script(codeHash, args, hashType);
        return new Address(script, network);
    }

    public String toString() {
        return this.encode();
    }

    public String encode() {
        return this.encodeFullBech32m();
    }

    @Deprecated
    public String encodeShort() {
        int codeHashIndex;
        byte[] payload = new byte[2 + this.script.args.length];
        byte[] codeHash = this.script.codeHash;
        if (Arrays.equals(codeHash, Script.SECP256K1_BLAKE160_SIGNHASH_ALL_CODE_HASH)) {
            codeHashIndex = 0;
        } else if (Arrays.equals(codeHash, Script.SECP256K1_BLAKE160_MULTISIG_ALL_CODE_HASH)) {
            codeHashIndex = 1;
        } else if (this.network == Network.MAINNET && Arrays.equals(codeHash, Script.ANY_CAN_PAY_CODE_HASH_MAINNET) || this.network == Network.TESTNET && Arrays.equals(codeHash, Script.ANY_CAN_PAY_CODE_HASH_TESTNET)) {
            codeHashIndex = 2;
        } else {
            throw new AddressFormatException("Encoding to short address for given script is unsupported");
        }
        payload[0] = 1;
        payload[1] = codeHashIndex;
        System.arraycopy(this.script.args, 0, payload, 2, this.script.args.length);
        payload = Address.convertBits(payload, 0, payload.length, 8, 5, true);
        return Bech32.encode(Bech32.Encoding.BECH32, Address.hrp(this.network), payload);
    }

    @Deprecated
    public String encodeFullBech32() {
        byte[] payload = new byte[1 + this.script.codeHash.length + this.script.args.length];
        if (this.script.hashType == Script.HashType.TYPE) {
            payload[0] = 4;
        } else if (this.script.hashType == Script.HashType.DATA) {
            payload[0] = 2;
        } else {
            throw new AddressFormatException("Unknown script hash type");
        }
        int pos = 1;
        System.arraycopy(this.script.codeHash, 0, payload, pos, this.script.codeHash.length);
        System.arraycopy(this.script.args, 0, payload, pos += this.script.codeHash.length, this.script.args.length);
        payload = Address.convertBits(payload, 0, payload.length, 8, 5, true);
        return Bech32.encode(Bech32.Encoding.BECH32, Address.hrp(this.network), payload);
    }

    public String encodeFullBech32m() {
        byte[] payload = new byte[1 + this.script.codeHash.length + 1 + this.script.args.length];
        payload[0] = 0;
        int pos = 1;
        System.arraycopy(this.script.codeHash, 0, payload, pos, this.script.codeHash.length);
        payload[pos += this.script.codeHash.length] = this.script.hashType.pack();
        System.arraycopy(this.script.args, 0, payload, ++pos, this.script.args.length);
        payload = Address.convertBits(payload, 0, payload.length, 8, 5, true);
        return Bech32.encode(Bech32.Encoding.BECH32M, Address.hrp(this.network), payload);
    }

    private static Network network(String hrp) {
        switch (hrp) {
            case "ckb": {
                return Network.MAINNET;
            }
            case "ckt": {
                return Network.TESTNET;
            }
        }
        throw new AddressFormatException("Invalid hrp");
    }

    private static String hrp(Network network) {
        Objects.requireNonNull(network);
        switch (network) {
            case MAINNET: {
                return "ckb";
            }
            case TESTNET: {
                return "ckt";
            }
        }
        throw new AddressFormatException("Unknown network");
    }

    private static byte[] convertBits(byte[] in, int inStart, int inLen, int fromBits, int toBits, boolean pad) throws AddressFormatException {
        int acc = 0;
        int bits = 0;
        ByteArrayOutputStream out = new ByteArrayOutputStream(64);
        int maxv = (1 << toBits) - 1;
        int max_acc = (1 << fromBits + toBits - 1) - 1;
        for (int i = 0; i < inLen; ++i) {
            int value = in[i + inStart] & 0xFF;
            if (value >>> fromBits != 0) {
                throw new AddressFormatException(String.format("Input value '%X' exceeds '%d' bit size", value, fromBits));
            }
            acc = (acc << fromBits | value) & max_acc;
            bits += fromBits;
            while (bits >= toBits) {
                out.write(acc >>> (bits -= toBits) & maxv);
            }
        }
        if (pad) {
            if (bits > 0) {
                out.write(acc << toBits - bits & maxv);
            }
        } else if (bits >= fromBits || (acc << toBits - bits & maxv) != 0) {
            throw new AddressFormatException("Could not convert bits, invalid padding");
        }
        return out.toByteArray();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Address address = (Address)o;
        if (!this.script.equals(address.script)) {
            return false;
        }
        return this.network == address.network;
    }

    public int hashCode() {
        int result = this.script.hashCode();
        result = 31 * result + this.network.hashCode();
        return result;
    }
}

