/* 
Copyright (c) 2010, NHIN Direct Project
All rights reserved.

Authors:
   Greg Meyer      gm2552@cerner.com
 
Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer 
in the documentation and/or other materials provided with the distribution.  Neither the name of the The NHIN Direct Project (nhindirect.org). 
nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.nhindirect.policy.x509;

import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;

import javax.security.auth.x500.X500Principal;

import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.TBSCertificate;
import org.nhindirect.policy.PolicyProcessException;
import org.nhindirect.policy.PolicyValueFactory;

/**
 * Subject field of TBS section of certificate
 * <p>
 * The policy value of this field is returned as a string containing the value of a specific attribute in the subject relative distinguished name (RDN).
 * <br>
 * If the requested attribute does not exist in the issue RDN, the policy value returned by this class
 * evaluates to an empty string.
 * @author Greg Meyer
 * @since 1.0
 */
public class SubjectAttributeField extends IssuerAttributeField
{
	static final long serialVersionUID = -6488771961800809924L;
	
	/**
	 * Constructor
	 * @param required Indicates if the field is required to be present in the certificate to be compliant with the policy.
	 * @param rdnAttributeId Id of the attribute to extract from the subject RDN
	 */	
	public SubjectAttributeField(boolean required, RDNAttributeIdentifier rdnAttributeId)
	{
		super(required, rdnAttributeId);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void injectReferenceValue(X509Certificate value) throws PolicyProcessException 
	{
		this.certificate = value;
		
		if (rdnAttributeId.equals(RDNAttributeIdentifier.DISTINGUISHED_NAME))
		{
			final Collection<String> str = Arrays.asList(certificate.getSubjectX500Principal().getName(X500Principal.RFC2253));
			this.policyValue = PolicyValueFactory.getInstance(str);
			return;
		}
		
		super.injectReferenceValue(value);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public TBSFieldName getFieldName() 
	{
		return TBSFieldName.SUBJECT;
	}
	
	/**
	 * Gets the subject field as an X509Name from the certificate TBS structure.
	 * @param tbsStruct The TBS structure of the certificate
	 * @return the subject field as an X509Name from the certificate TBS structure.
	 */
	protected X500Name getX509Name(TBSCertificate tbsStruct)
	{
		return tbsStruct.getSubject();
	}
}
