
package org.nkjmlab.sorm4j.extension;

import static org.nkjmlab.sorm4j.core.util.StringUtils.*;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.nkjmlab.sorm4j.core.util.SqlTypeUtils;
import org.nkjmlab.sorm4j.core.util.StringUtils;

/**
 * Default implementation of {@link ColumnFieldMapper}
 *
 * @author nkjm
 *
 */

public class DefaultColumnFieldMapper implements ColumnFieldMapper {

  @Override
  public List<Column> getAutoGeneratedColumns(DatabaseMetaData metaData, String tableName)
      throws SQLException {
    try (ResultSet resultSet =
        metaData.getColumns(null, getSchemaPattern(metaData), tableName, "%")) {
      final List<Column> columnsList = new ArrayList<>();
      while (resultSet.next()) {
        String columnName = resultSet.getString(4);
        String isAutoIncrement = resultSet.getString(23);
        if (isAutoIncrement.equals("YES")) {
          columnsList.add(new Column(columnName));
        }
      }
      return columnsList;
    }
  }


  @Override
  public List<Column> getColumns(DatabaseMetaData metaData, String tableName) throws SQLException {
    try (ResultSet resultSet =
        metaData.getColumns(null, getSchemaPattern(metaData), tableName, "%")) {
      final List<Column> columnsList = new ArrayList<>();
      while (resultSet.next()) {
        String columnName = resultSet.getString(4);
        int dataType = resultSet.getInt(5);
        columnsList.add(new ColumnOnTable(columnName, dataType));
      }
      return columnsList;
    }
  }


  @Override
  public List<Column> getPrimaryKeys(DatabaseMetaData metaData, String tableName)
      throws SQLException {
    final List<Column> primaryKeysList = new ArrayList<>();
    try (ResultSet resultSet =
        metaData.getPrimaryKeys(null, getSchemaPattern(metaData), tableName)) {
      while (resultSet.next()) {
        final String columnName = resultSet.getString(4);
        primaryKeysList.add(new Column(columnName));
      }
      return primaryKeysList;
    }
  }


  protected String getSchemaPattern(DatabaseMetaData metaData) throws SQLException {
    // oracle expects a pattern such as "%" to work
    return "Oracle".equalsIgnoreCase(metaData.getDatabaseProductName()) ? "%" : null;
  }


  @Override
  public List<Column> getColumnNameCandidates(List<FieldName> fieldNames) {
    return fieldNames.stream().flatMap(fieldName -> guessColumnNameCandidates(fieldName).stream())
        .collect(Collectors.toList());
  }

  protected List<Column> guessColumnNameCandidates(FieldName fieldName) {
    return List.of(new Column(toUpperSnakeCase(fieldName.getName())),
        new Column(toUpperCase(fieldName.getName())));
  }

  @Override
  public Optional<FieldName> getFieldNameByColumnName(Column column, List<FieldName> fieldNames) {
    for (FieldName fieldName : fieldNames) {
      if (StringUtils.containsIgnoreCase(guessColumnNameCandidates(fieldName).stream()
          .map(s -> s.toString()).collect(Collectors.toList()), column.getName())) {
        return Optional.of(fieldName);
      }
    }
    return Optional.empty();
  }

  /**
   * Column name and data type for message.
   *
   * @author nkjm
   *
   */
  private static final class ColumnOnTable extends Column {

    private int dataType;

    public ColumnOnTable(String name, int dataType) {
      super(name);
      this.dataType = dataType;
    }

    @Override
    public String toString() {
      return getName() + "(" + SqlTypeUtils.sqlTypeToString(dataType) + ")";
    }
  }

}
