package org.nkjmlab.sorm4j.internal.mapping;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Sqls generated by {@link TableMapping}.
 *
 * @author nkjm
 *
 */
public final class TableMappingSql {

  private static final Map<String, String> multiRowSqlMap = new ConcurrentHashMap<>();

  private final String tableName;

  private final List<String> columns;
  private final List<String> primaryKeys;
  private final String[] autoGeneratedColumnsArray;
  private final List<String> notAutoGeneratedColumns;
  private final List<String> columnsForUpdate;

  private final boolean hasPrimaryKey;
  private final boolean hasAutoGeneratedColumns;

  private final String insertOrMergePlaceholders;

  private final String selectByPrimaryKeySql;
  private final String selectAllSql;
  private final String insertSql;
  private final String updateSql;
  private final String deleteSql;
  private final String mergeSql;
  private final String insertSqlPrefix;
  private final String mergeSqlPrefix;


  public TableMappingSql(String tableName, List<String> columns, List<String> primaryKeys,
      List<String> autoGeneratedColumns) {
    this.tableName = tableName;
    this.columns = columns;
    this.primaryKeys = primaryKeys;

    this.notAutoGeneratedColumns = columns.stream()
        .filter(col -> !autoGeneratedColumns.contains(col)).collect(Collectors.toList());


    this.hasPrimaryKey = primaryKeys.size() != 0;
    this.hasAutoGeneratedColumns = autoGeneratedColumns.size() != 0;
    this.autoGeneratedColumnsArray = autoGeneratedColumns.toArray(String[]::new);

    this.insertSqlPrefix =
        "insert into " + tableName + " (" + toColumList(notAutoGeneratedColumns) + ") values";
    this.mergeSqlPrefix =
        "merge into " + tableName + " (" + toColumList(notAutoGeneratedColumns) + ") values";

    this.insertOrMergePlaceholders =
        " (" + generatePlaceholders(notAutoGeneratedColumns.size()) + ") ";

    this.insertSql = insertSqlPrefix + insertOrMergePlaceholders;
    this.mergeSql = mergeSqlPrefix + insertOrMergePlaceholders;


    this.selectAllSql = "select " + toColumList(columns) + " from " + tableName;
    this.selectByPrimaryKeySql =
        selectAllSql + " " + createWhereClauseIdentifyByPrimaryKeys(primaryKeys);

    List<String> notPrimaryKeys =
        columns.stream().filter(col -> !primaryKeys.contains(col)).collect(Collectors.toList());

    this.columnsForUpdate =
        Stream.concat(notPrimaryKeys.stream(), primaryKeys.stream()).collect(Collectors.toList());
    this.updateSql = "update " + tableName + createUpdateSetClause(notPrimaryKeys)
        + createWhereClauseIdentifyByPrimaryKeys(primaryKeys);
    this.deleteSql =
        "delete from " + tableName + createWhereClauseIdentifyByPrimaryKeys(primaryKeys);
  }

  private String createUpdateSetClause(List<String> notPrimaryKeys) {
    return " set " + String.join(",",
        notPrimaryKeys.stream().map(npk -> npk + "=?").collect(Collectors.toList()));
  }

  private String createWhereClauseIdentifyByPrimaryKeys(List<String> primaryKeys) {
    return " where " + String.join(" and ",
        primaryKeys.stream().map(pk -> pk + "=?").collect(Collectors.toList()));
  }

  private String generatePlaceholders(int num) {
    return String.join(",", Stream.generate(() -> "?").limit(num).collect(Collectors.toList()));
  }


  public String[] getAutoGeneratedColumnsArray() {
    return autoGeneratedColumnsArray;
  }

  public List<String> getColumns() {
    return columns;
  }

  public List<String> getColumnsForUpdate() {
    return columnsForUpdate;
  }

  public String getDeleteSql() {
    return deleteSql;
  }

  public String getInsertSql() {
    return insertSql;
  }

  public String getMergeSql() {
    return mergeSql;
  }


  public String getMultirowInsertSql(int num) {
    return getSqlWithMultirowPlaceholders(insertSqlPrefix, num);
  }

  public String getMultirowMergeSql(int num) {
    return getSqlWithMultirowPlaceholders(mergeSqlPrefix, num);
  }

  public List<String> getNotAutoGeneratedColumns() {
    return notAutoGeneratedColumns;
  }

  private String getSqlWithMultirowPlaceholders(String sqlPrefix, int num) {
    return multiRowSqlMap.computeIfAbsent(sqlPrefix + num, n -> sqlPrefix + String.join(",",
        Stream.generate(() -> insertOrMergePlaceholders).limit(num).collect(Collectors.toList())));
  }

  public List<String> getPrimaryKeys() {
    return primaryKeys;
  }

  public String getSelectAllSql() {
    return selectAllSql;
  }

  public String getSelectByPrimaryKeySql() {
    return selectByPrimaryKeySql;
  }

  public String getTableName() {
    return tableName;
  }

  public String getUpdateSql() {
    return updateSql;
  }

  public boolean hasAutoGeneratedColumns() {
    return hasAutoGeneratedColumns;
  }

  public boolean hasPrimaryKey() {
    return hasPrimaryKey;
  }

  private String toColumList(List<String> columns) {
    return String.join(",", columns);
  }

}

