package org.nkjmlab.sorm4j.internal.mapping;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.nkjmlab.sorm4j.extension.ColumnNameWithMetaData;
import org.nkjmlab.sorm4j.sql.TableMetaData;

public class TableMetaDataImpl implements TableMetaData {

  private final String tableName;
  private final List<String> columns;
  private final List<String> primaryKeys;
  private final List<String> autoGeneratedColumns;
  private final boolean hasPrimaryKey;
  private final boolean hasAutoGeneratedColumns;
  private final List<String> notAutoGeneratedColumns;

  private final List<String> columnsForUpdate;
  private final String[] autoGeneratedColumnsArray;
  private final List<String> notPrimaryKeys;
  private final String columnsWithAliasPrefix;
  private final List<ColumnNameWithMetaData> columnsWithMetaData;


  public TableMetaDataImpl(String tableName, String tableNameAliasPrefix,
      List<ColumnNameWithMetaData> columnsWithMetaData, List<String> primaryKeys,
      List<String> autoGeneratedColumns) {
    this.tableName = tableName;

    this.columnsWithMetaData = columnsWithMetaData;
    this.columns = Collections.unmodifiableList(
        columnsWithMetaData.stream().map(c -> c.getName()).collect(Collectors.toList()));
    this.primaryKeys = Collections.unmodifiableList(primaryKeys);
    this.autoGeneratedColumns = Collections.unmodifiableList(autoGeneratedColumns);

    this.notAutoGeneratedColumns = Collections.unmodifiableList(columns.stream()
        .filter(col -> !autoGeneratedColumns.contains(col)).collect(Collectors.toList()));
    this.hasPrimaryKey = primaryKeys.size() != 0;
    this.hasAutoGeneratedColumns = autoGeneratedColumns.size() != 0;

    this.autoGeneratedColumnsArray = autoGeneratedColumns.toArray(String[]::new);

    this.notPrimaryKeys =
        columns.stream().filter(col -> !primaryKeys.contains(col)).collect(Collectors.toList());

    this.columnsForUpdate =
        Stream.concat(notPrimaryKeys.stream(), primaryKeys.stream()).collect(Collectors.toList());

    this.columnsWithAliasPrefix = String.join(", ",
        columns.stream().map(col -> tableName + "." + col + " as " + tableNameAliasPrefix + col)
            .toArray(String[]::new));

  }

  @Override
  public String getTableName() {
    return tableName;
  }

  @Override
  public List<String> getColumns() {
    return columns;
  }

  @Override
  public List<String> getPrimaryKeys() {
    return primaryKeys;
  }

  @Override
  public List<String> getAutoGeneratedColumns() {
    return autoGeneratedColumns;
  }

  @Override
  public boolean hasPrimaryKey() {
    return hasPrimaryKey;
  }

  @Override
  public boolean hasAutoGeneratedColumns() {
    return hasAutoGeneratedColumns;
  }

  @Override
  public List<String> getNotAutoGeneratedColumns() {
    return notAutoGeneratedColumns;
  }

  @Override
  public List<String> getColumnsForUpdate() {
    return columnsForUpdate;
  }

  @Override
  public String[] getAutoGeneratedColumnsArray() {
    return autoGeneratedColumnsArray;
  }

  @Override
  public List<String> getNotPrimaryKeys() {
    return notPrimaryKeys;
  }

  @Override
  public String getColumnAliases() {
    return columnsWithAliasPrefix;
  }

  @Override
  public List<ColumnNameWithMetaData> getColumnsWithMetaData() {
    return columnsWithMetaData;
  }

}
