package org.nkjmlab.sorm4j.context;

import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.nkjmlab.sorm4j.internal.util.ConcurrentCache;

/**
 * SQL statements generated by {@link TableSqlFactory}.
 *
 * @author yuu_nkjm
 *
 */
public final class TableSql {

  private final Map<String, String> multiRowSqlMap = new ConcurrentCache<>(256);

  private final String insertPlaceholders;
  private final String mergePlaceholders;
  private final String selectByPrimaryKeySql;
  private final String selectAllSql;
  private final String insertSql;
  private final String updateSql;
  private final String deleteSql;
  private final String mergeSql;
  private final String existsSql;
  private final String insertSqlPrefix;
  private final String mergeSqlPrefix;

  /**
   * <pre>
   * selectByPrimaryKeySql=select * from GUESTS  where ID=?
   * selectAllSql=select * from GUESTS
   * insertSql=insert into GUESTS (NAME, ADDRESS) values (?,?)
   * updateSql=update GUESTS set NAME=?, ADDRESS=? where ID=?
   * deleteSql=delete from GUESTS where ID=?
   * mergeSql=merge into GUESTS (ID, NAME, ADDRESS) key (ID) values (?,?,?)
   * existsSql=select 1 from GUESTS where ID=?
   * insertSqlPrefix=insert into GUESTS (NAME, ADDRESS) values
   * insertPlaceholders= (?,?)
   * mergeSqlPrefix=merge into GUESTS (ID, NAME, ADDRESS) key (ID) values
   * mergePlaceholders= (?,?,?)
   * </pre>
   *
   * @param inserPlaceholders
   * @param mergePlaceholders
   * @param selectByPrimaryKeySql
   * @param selectAllSql
   * @param insertSql
   * @param updateSql
   * @param deleteSql
   * @param mergeSql
   * @param existsSql
   * @param insertSqlPrefix
   * @param mergeSqlPrefix
   */
  public TableSql(String inserPlaceholders, String mergePlaceholders, String selectByPrimaryKeySql,
      String selectAllSql, String insertSql, String updateSql, String deleteSql, String mergeSql,
      String existsSql, String insertSqlPrefix, String mergeSqlPrefix) {
    this.insertPlaceholders = inserPlaceholders;
    this.mergePlaceholders = mergePlaceholders;
    this.selectByPrimaryKeySql = selectByPrimaryKeySql;
    this.selectAllSql = selectAllSql;
    this.insertSql = insertSql;
    this.updateSql = updateSql;
    this.deleteSql = deleteSql;
    this.mergeSql = mergeSql;
    this.existsSql = existsSql;
    this.insertSqlPrefix = insertSqlPrefix;
    this.mergeSqlPrefix = mergeSqlPrefix;
  }

  public String getDeleteSql() {
    return deleteSql;
  }

  public String getInsertSql() {
    return insertSql;
  }

  public String getMergeSql() {
    return mergeSql;
  }


  public String getMultirowInsertSql(int num) {
    return getSqlWithMultirowPlaceholders(insertSqlPrefix, insertPlaceholders, num);
  }

  public String getMultirowMergeSql(int num) {
    return getSqlWithMultirowPlaceholders(mergeSqlPrefix, mergePlaceholders, num);
  }

  /**
   *
   * @param sqlPrefix
   * @param placeHolders e.g. (?,?,?)
   * @param repeat
   * @return
   */
  private String getSqlWithMultirowPlaceholders(String sqlPrefix, String placeHolders, int repeat) {
    return multiRowSqlMap.computeIfAbsent(sqlPrefix + repeat, n -> sqlPrefix + String.join(",",
        Stream.generate(() -> placeHolders).limit(repeat).collect(Collectors.toList())));
  }

  public String getSelectAllSql() {
    return selectAllSql;
  }

  public String getSelectByPrimaryKeySql() {
    return selectByPrimaryKeySql;
  }

  public String getUpdateSql() {
    return updateSql;
  }

  public String getExistsSql() {
    return existsSql;
  }

  @Override
  public String toString() {
    return "TableSql [insertPlaceholders=" + insertPlaceholders + ", mergePlaceholders="
        + mergePlaceholders + ", selectByPrimaryKeySql=" + selectByPrimaryKeySql + ", selectAllSql="
        + selectAllSql + ", insertSql=" + insertSql + ", updateSql=" + updateSql + ", deleteSql="
        + deleteSql + ", mergeSql=" + mergeSql + ", existsSql=" + existsSql + ", insertSqlPrefix="
        + insertSqlPrefix + ", mergeSqlPrefix=" + mergeSqlPrefix + "]";
  }

}

