package org.nkjmlab.sorm4j.internal.context.impl;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.nkjmlab.sorm4j.common.container.RowMap;
import org.nkjmlab.sorm4j.internal.context.TableSqlFactory;
import org.nkjmlab.sorm4j.internal.context.impl.DefaultTableSqlFactory.UpdateSqlFactory;
import org.nkjmlab.sorm4j.internal.util.ConcurrentCache;
import org.nkjmlab.sorm4j.sql.TableSql;

/**
 * SQL statements generated by {@link TableSqlFactory}.
 *
 * @author yuu_nkjm
 */
public final class DefaultTableSql implements TableSql {

  private final Map<String, String> multiRowSqlMap = new ConcurrentCache<>(256);

  private final String insertPlaceholders;
  private final String mergePlaceholders;
  private final String selectByPrimaryKeySql;
  private final String selectAllSql;
  private final String insertSql;
  private final String updateSql;
  private final String deleteSql;
  private final String mergeSql;
  private final String existsSql;
  private final String insertSqlPrefix;
  private final String mergeSqlPrefix;
  private final UpdateSqlFactory updateSqlFactory;

  /**
   *
   *
   * <pre>
   * selectByPrimaryKeySql=select * from GUESTS  where ID=?
   * selectAllSql=select * from GUESTS
   * insertSql=insert into GUESTS (NAME, ADDRESS) values (?,?)
   * updateSql=update GUESTS set NAME=?, ADDRESS=? where ID=?
   * deleteSql=delete from GUESTS where ID=?
   * mergeSql=merge into GUESTS (ID, NAME, ADDRESS) key (ID) values (?,?,?)
   * existsSql=select 1 from GUESTS where ID=?
   * insertSqlPrefix=insert into GUESTS (NAME, ADDRESS) values
   * insertPlaceholders= (?,?)
   * mergeSqlPrefix=merge into GUESTS (ID, NAME, ADDRESS) key (ID) values
   * mergePlaceholders= (?,?,?)
   * </pre>
   *
   * @param inserPlaceholders
   * @param mergePlaceholders
   * @param selectByPrimaryKeySql
   * @param selectAllSql
   * @param insertSql
   * @param updateSql
   * @param deleteSql
   * @param mergeSql
   * @param existsSql
   * @param insertSqlPrefix
   * @param mergeSqlPrefix
   * @param updateSqlFactory
   * @param primaryKeys
   */
  public DefaultTableSql(
      String inserPlaceholders,
      String mergePlaceholders,
      String selectByPrimaryKeySql,
      String selectAllSql,
      String insertSql,
      String updateSql,
      String deleteSql,
      String mergeSql,
      String existsSql,
      String insertSqlPrefix,
      String mergeSqlPrefix,
      UpdateSqlFactory updateSqlFactory,
      List<String> primaryKeys) {
    this.insertPlaceholders = inserPlaceholders;
    this.mergePlaceholders = mergePlaceholders;
    this.selectByPrimaryKeySql = selectByPrimaryKeySql;
    this.selectAllSql = selectAllSql;
    this.insertSql = insertSql;
    this.updateSql = updateSql;
    this.deleteSql = deleteSql;
    this.mergeSql = mergeSql;
    this.existsSql = existsSql;
    this.insertSqlPrefix = insertSqlPrefix;
    this.mergeSqlPrefix = mergeSqlPrefix;
    this.updateSqlFactory = updateSqlFactory;
  }

  @Override
public String getDeleteSql() {
    return deleteSql;
  }

  @Override
public String getInsertSql() {
    return insertSql;
  }

  @Override
public String getMergeSql() {
    return mergeSql;
  }

  @Override
public String getMultirowInsertSql(int num) {
    return getSqlWithMultirowPlaceholders(insertSqlPrefix, insertPlaceholders, num);
  }

  @Override
public String getMultirowMergeSql(int num) {
    return getSqlWithMultirowPlaceholders(mergeSqlPrefix, mergePlaceholders, num);
  }

  /**
   * @param sqlPrefix
   * @param placeHolders e.g. (?,?,?)
   * @param repeat
   * @return
   */
  private String getSqlWithMultirowPlaceholders(String sqlPrefix, String placeHolders, int repeat) {
    return multiRowSqlMap.computeIfAbsent(
        sqlPrefix + repeat,
        n ->
            sqlPrefix
                + String.join(
                    ",",
                    Stream.generate(() -> placeHolders)
                        .limit(repeat)
                        .collect(Collectors.toList())));
  }

  @Override
public String getSelectAllSql() {
    return selectAllSql;
  }

  @Override
public String getSelectByPrimaryKeySql() {
    return selectByPrimaryKeySql;
  }

  @Override
public String getUpdateSql() {
    return updateSql;
  }

  @Override
public String getExistsSql() {
    return existsSql;
  }

  @Override
  public String toString() {
    return "TableSql [insertPlaceholders="
        + insertPlaceholders
        + ", mergePlaceholders="
        + mergePlaceholders
        + ", selectByPrimaryKeySql="
        + selectByPrimaryKeySql
        + ", selectAllSql="
        + selectAllSql
        + ", insertSql="
        + insertSql
        + ", updateSql="
        + updateSql
        + ", deleteSql="
        + deleteSql
        + ", mergeSql="
        + mergeSql
        + ", existsSql="
        + existsSql
        + ", insertSqlPrefix="
        + insertSqlPrefix
        + ", mergeSqlPrefix="
        + mergeSqlPrefix
        + "]";
  }

  @Override
public String getUpdateSql(RowMap object) {
    return updateSqlFactory.createUpdateSql(object.keySet());
  }
}
